#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Import used to generate dependency on Launcher
# pylint: disable=import-error
# pylint: disable=unused-import
import LauncherPlugin.Launcher

def Plugin( context ):
   # -------------------------------------------------------------------------
   # Set up the Smbus agent's AgentConfig
   #
   # Smbus runs in two modes:
   #    1. On a given cell when Hardware::SmbusEngine exists under
   #       hardware/smbus/cell/<cellId>
   #    2. on the active supervisor for a slice when Hardware::SmbusEngine
   #       exists at 'hardware/smbus/slice/<sliceId>
   #
   # We tell Launcher this by setting the AgentConfig's "qualPath" to
   # reference the full path description.
   # -------------------------------------------------------------------------

   # We want to add two different AgentConfig for this agent, one for cell and
   # one for slice, on the same cell. Reason is, in case of modular system
   # a cell should run one instance of Smbus for cell/supervisor and one instance
   # of Smbus per slice/linecard.
   # Now goal is to have common name for all the Smbus instances running on a given
   # cell as "Smbus" (same as name in passive mount profile).
   # So a modular system with one linecard will have
   #   "Smbus"           -> instance for cell/sup running on cell 1 (active sup)
   #   "Smbus-Linecard1" -> instance for linecard 1 running on cell 1 (active sup)

   # Since key 'name' is used as collection key in Launcher tac model, we can't have
   # two different configs with same 'name' value. While creating processName,
   # Launcher uses 'launchName' if it's present. So let's use same 'launchName'
   # and different 'name' to achieve our goal.

   # What's in the 'name'?
   # For cell config,
   # 'name' should be "Smbus", which should be same as following two instances:
   # 1. StageMgr registers Stage::AgentConfig with this name under cell path
   #    stageInput/boot. Check registerStage call for this agent in
   #    src/Asu/LauncherPlugin/EosStageEvents.py
   #    I suspect, for some boot sequences, Smbus is started when AgentConfig entry
   #    for Smbus is present with name same as stage config entry name under above
   #    mentioned boot config path. If AgentConfig has different name than the
   #    boot config path, Smbus won't start at all.
   # 2. Smbus mounts cell path stageInput/boot/Smbus and sets the complete
   #    flag which marks the event complete. If AgentConfig has a different name
   #    Smbus agent will be marking 'Smbus' entry and not the entry with new name.
   #    This will result in subsequent Asu stages getting blocked forever.
   # So if you really need to change the 'name', make sure you take care of
   # above two instances as well.
   libConfig = context.lookup( 'hardware/pluto/library/config' )

   cellCfg = { 'name' : 'Smbus',
               'launchName' : 'Smbus',
               'exe' : '/usr/bin/Smbus',
               'argv' : [],
               'heartbeatPeriod' : 30,
               'startupGracePeriod' : 90,
               'runnability': [ { 'qualPath' :
                                  "hardware/smbus/cell/%cellId.accel/%*" },
                                { 'entity' : libConfig,
                                  'qualAttr' : 'isWhitebox' } ] }

   # When a 'sliceId' parameter is configured here for the agent, it
   # calls SmbusAgent.Smbus.__init__ with a 'sliceId' parameter. If
   # the option is not provided (as above for the cell case), the
   # parameter defaults to the empty string "".
   sliceCfg = { 'name' : 'SmbusSlice',
                'launchName' : 'Smbus',
                'exe' : '/usr/bin/Smbus',
                'argv' : [ "--sliceId=%(sliceId)s" ],
                'heartbeatPeriod' : 30,
                'startupGracePeriod' : 90,
                'runnability' :
                   [ { 'qualPath' :
                       "hardware/smbus/slice/%sliceId.accel/%*" } ] }

   activeSupe = context.activeSupervisorRoleName
   allSupes = context.allSupervisorsRoleName

   context.agentConfigIs( allSupes, cellCfg )
   context.agentConfigIs( activeSupe, sliceCfg )
