#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Import used to generate dependency on Launcher
# pylint: disable=import-error
# pylint: disable=unused-import
import LauncherPlugin.Launcher

def Plugin( context ):

   # ScdAgent runs on a given cell when the Hardware::Scd for that cell exists in
   # "hardware/cell/<cellId>".

   # We want to add two different AgentConfig for this agent, one for cell and
   # one for slice, on the same cell. Reason is, in case of modular system
   # a cell should run one instance of Scd for cell/supervisor and one instance
   # of Scd per slice/linecard.
   # Now goal is to have common name for all the Scd instances running on a given
   # cell as "Scd" (same as name in passive mount profile).
   # So a modular system with one linecard will have
   #   "Scd"           -> instance for cell/sup running on cell 1
   #   "Scd-Linecard1" -> instance for linecard 1 running on cell 1

   # Since key 'name' is used as collection key in Launcher tac model, we can't have
   # two different configs with same 'name' value. While creating processName,
   # Launcher uses 'launchName' if it's present. So let's use same 'launchName'
   # and different 'name' to achieve our goal.

   # What's in the 'name'?
   # For cell config,
   # 'name' should be "Scd", which should be same as following two instances:
   # 1. Asu registers Stage::AgentConfig with this name under cell path
   #    stageInput/boot.
   #    I suspect, for some boot sequences, Scd is started when AgentConfig entry
   #    for Scd is present with name same as stage config entry name under above
   #    mentioned boot config path. If AgentConfig has different name than the
   #    boot config path, Scd won't start at all.
   # 2. Scd mounts cell path stageInput/boot/Scd and sets the complete
   #    flag which marks the event complete. If AgentConfig has a different name
   #    Scd agent will be marking 'Scd' entry and not the entry with the new name.
   #    This will result in subsequent Asu stages getting blocked forever.
   # So if you really need to change the 'name', make sure you take care of
   # above two instances as well.
   agentCfg = { "name": "Scd",
                "launchName": "Scd",
                "exe": "/usr/bin/Scd",
                "argv": [],
                "heartbeatPeriod": 30,
                "runnability": [ { 
                   "qualPath": "hardware/cell/%cellId/scd/config.scdConfig/%*"
                 } ]
               }

   scdRoleName = context.allCellsRoleName
   context.agentConfigIs( scdRoleName, agentCfg )

   # Launch the slice-specific version of the agent when the 'scd'
   # directory has been created in that slice by the FruPlugin.
   sliceAgentCfg = { "name": "ScdSlice",
                     "launchName": "Scd",
                     "exe": "/usr/bin/Scd",
                     "argv": [ "--sliceId=%(sliceId)s" ],
                     "heartbeatPeriod": 30,
                     "runnability":
                        [ { 'qualPath' : 
                           "hardware/scd/config/slice/%sliceId.scdConfig/%*" } ]
                     }

   # ScdSlice runs on allCells to speed up SSO switchover. It only
   # _needs_ to run on Active as it does nothing while running on
   # standby.
   scdSliceRoleName = context.allCellsRoleName
   context.agentConfigIs( scdSliceRoleName, sliceAgentCfg )
