# Copyright (c) 2009, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Import used to generate dependency on Launcher
# pkgeps : import LauncherPlugin.Launcher

# Because of its 'runnability' setting, this agent is added to
# conditionalStartAgentsToIgnoreLauncherMsg in ArosTestPlugin to prevent spurious
# ptest failures due to unrecognized logs triggered by config change/tampering
from LauncherPlugin.Maintenance import getMmodeRunnability

def registerAgent( context, name, runPath, argv, oneTimeLaunch=False,
                   exeName=None, hbPeriod=300 ):
   if exeName is None:
      exeName = "/usr/bin/%s" % name
   agentCfg = { "name" : name,
                "exe" : exeName,
                "argv" : argv,
                "heartbeatPeriod" : hbPeriod,
                "oomScoreAdj": -300, # see AID3426
                "coredumpTimeout" : 600,
                "oneTimeLaunch" : oneTimeLaunch,
                "runnability" : runPath }

   roleName = context.allSupervisorsRoleName
   context.agentConfigIs( roleName, agentCfg )


def Plugin( context ):
   # Rib depends on qualPath maintenancemode/ready whenever MaintenanceMode
   # agent is runnable and Rib hasn't already come up. If the MaintenanceMode
   # agent is runnable, this qualPath is set after it comes up and is done with 
   # its work. Considering the MaintenanceMode agent is runnable, the various
   # stages when the qualPath is set in different cases is:
   # 1. In case of normalBoot, normalBootXp and supervisorColdBoot,
   #    maintenancemode/ready is set once all the plugins are loaded,
   #    maintenance/mapping is setup  and onBoot processing is done.
   # 2. In case of ASU reboot( Hitless/Fastboot ), MaintenanceMode agent
   #    participates in early stages ( CriticalAgent stage in hitless bootMode and
   #    PrimaryAgent stage in fastboot mode ) and sets this qualPath at the end of
   #    doMountsComplete.
   # 3. In case of RPR standby, MaintenanceMode agent sets this qualPath and
   #    prevents all the SMs from starting. It reacts to redundancyStatus()->mode()
   #    and start all SMs if it is running on active supe.
   # 4. In case of SSO standby, MaintenanceMode agent does not run and
   #    maintenancemode/ready qualPath is synced between active and standby.
   #    So when switchover happens, path is already set and Rib will start.
   defaultVrfRunnability = ( { 
      "qualPath" : "cell/%cellId/routing/defaultVrfProtocolLaunch/Rib" }, )

   # MaintenanceMode agent is now conditionally runnable on configuration of 
   # "maintenance" command.
   # So, Rib should not wait on MaintenaceMode agent to set "ready" qualPath
   # if the agent is not runnable.

   # getMmodeRunnability returns the runnability criteria for MaintenanceMode
   # agent
   mmodeRunnable = getMmodeRunnability()
   
   # If MaintenanceMode is runnable, Rib should wait for "ready" qualPath
   mmodeReady = { "qualPath" : "maintenance/agent/maintenancemode/ready" } 

   # getMmodeRunnability with negate as True returns the conditions which
   # evaluate to True when MaintenanceMode agent is not runnable.
   mmodeNotRunnable = getMmodeRunnability( negate=True )

   # There could be a case where Rib is already running and MaintenanceMode
   # comes up later. In that case, we need to ensure that Rib continues to run
   # and does not go down until MaintenanceMode agent sets ready qualPath.
   # To ensure this, we add another qualPath, which specifies whether Rib is 
   # already running
   ribRanOnce = ( { "qualPath" : "cell/%cellId/agent/config/Rib" } )
   
   # The final runnability is: 
   # ((mmodeRunnable AND mmodeReady) OR mmodeNotRunnable)) AND 
   #                                            defaultVrfRunnability)
   #                      OR
   # ((ribRanOnce AND defaultVrfRunnability))
   runnability = ( [ ( [ ( mmodeRunnable, mmodeReady ), mmodeNotRunnable ], 
                       defaultVrfRunnability ),
                     ( ribRanOnce, defaultVrfRunnability ) ] )

   # Register Main Rib ( BR )
   registerAgent( context, "Rib", runnability, [] )

   # Vrf Rib ( VR ) is always conditional on the existence of vrf status
   nonDefaultVrfRunnability = ( {
      "qualPath" : "cell/%cellId/routing/VrfLaunchIndicator/%sliceId" }, )
   
   # Vrf Rib ran once needs to be defined
   ribVrfRanOnce = { "qualPath" : "cell/%cellId/agent/config/Rib-vrf-%sliceId" }

   # The final runnability is: 
   # ((mmodeRunnable AND mmodeReady) OR mmodeNotRunnable) AND 
   #                                            nonDefaultVrfRunnability)
   #                      OR
   # (ribRanOnce AND nonDefaultVrfRunnability)
   vrfRunnability = ( [ ( [ ( mmodeRunnable, mmodeReady ), mmodeNotRunnable ], 
                          nonDefaultVrfRunnability ),
                        ( ribVrfRanOnce, nonDefaultVrfRunnability ) ] )

   argv = "--z", "'client name %(sliceId)s ns-name ns-%(sliceId)s " \
          "vrfname %(sliceId)s servername vre_server'", "--sliceId=%(sliceId)s"
   registerAgent( context, "Rib-vrf", vrfRunnability, argv,
                  exeName="/usr/bin/Rib-vrf" )
