#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import LauncherUtil
import SysdbLauncherUtil
import Tac
import os
import Tracing
from LauncherLib import isSsoStandby

launcherTraceHandle = Tracing.Handle( "Launcher" )
t0 = launcherTraceHandle.trace0
t9 = launcherTraceHandle.trace9

class LauncherContext( object ):
   def __init__( self, em ):
      self.em = em
      self.mg = em.mountGroup()
      self.activeSupervisorRoleName = LauncherUtil.activeSupervisorRoleName
      self.allSupervisorsRoleName = LauncherUtil.allSupervisorsRoleName
      self.allCellsRoleName = LauncherUtil.allCellsRoleName

   def agentConfigIs( self, roleStrOrDict, agentCfg ):
      if os.environ.get( 'SYSDB_WRITE_AGENTCONFIGS' ) != '1':
         agentConfigIs( self.em, roleStrOrDict, agentCfg )

   # Most plugins should only use the default mountFlags. For certain
   # rare scenarios where Launcher is mounting a high scale mount point
   # we might need to use scheduled attrlogging mountflags to prevent
   # socket buffer overflow. Those specific users will pass in customized
   # mount flags for the same
   def lookup( self, lookupStr, mountFlags='r' ):
      # Mount the path. Some paths are of the form
      # foo/barDir.bar/baz, in which case we want to mount
      # foo/barDir.
      splitpath = lookupStr.split( '.' )
      mountpath = splitpath[ 0 ]
      self.mg.mount( mountpath, '', mountFlags )
      return lookupStr

   def closeMounts( self, callback, overrideBootComplete ):
      def _pluginFinish():
         t0( "Finished mounting plugin paths" )
         launcherState = self.em.launcherSysdbState
         agentCfgs = launcherState.agentConfigDir.agent
         agentSts = launcherState.agentStatusDir.agent
         launcherState.bootComplete = overrideBootComplete or \
                                      launcherState.bootComplete

         for entry in launcherState.cfgToRoles:
            agentCfg = entry[ 0 ]
            roleStrOrDict = entry[ 1 ]
            LauncherUtil.launcherAgentConfigIs(
               self.em, agentCfgs, agentSts, roleStrOrDict, agentCfg,
               launcherState.launchCandidateDir,
               launcherState.agentValgrindConfigDir, userDaemon=False,
               updateRoleCfg=launcherState.bootComplete )
         callback()
      self.mg.close( _pluginFinish )

def agentConfigIs( entMan, roleStrOrDict, agentCfg ):
   SysdbLauncherUtil.createLauncherSysdbState( entMan )

   agentName = agentCfg[ "name" ]
   t9( "LauncherPlugin::agentConfigIs()", agentName )

   redStatus = entMan.redundancyStatus()
   launcherState = entMan.launcherSysdbState
   agentSts = launcherState.agentStatusDir.agent

   # We'll set the agentConfigs after mounts have completed
   launcherState.cfgToRoles.append( ( agentCfg, roleStrOrDict ) )

   if isSsoStandby( redStatus ):
      # By this time, we should have agentStatus and agentConfigs populated
      agentStatusEntity = agentSts.get( agentName )
      assert agentStatusEntity, \
            "agent %s not found in agent status dir" % agentName
      agentConfigEntity = agentStatusEntity.agentConfig
      assert agentConfigEntity, \
            "agent %s not found in agent config dir" % agentName
