# Copyright (c) 2007-2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import os
import re
import uuid

import CliApi
from CliApi import CapiStatus
from CliCommon import EditConfigOptions
from CliCommon import JsonRpcErrorCodes
from CliCommon import ResponseFormats
from CliCommon import JSONRPC_VERSION_LATEST
from JsonRpcBase import JsonRpcBase
from JsonRpcBase import InvalidParamsError
from JsonRpcBase import JsonRpcError
import Tracing
import QuickTrace

traceHandle = Tracing.Handle( 'JsonCli' )
warn = traceHandle.trace1
info = traceHandle.trace2
trace = traceHandle.trace3
debug = traceHandle.trace4
nitty = traceHandle.trace5

qt0 = QuickTrace.trace0
qv = QuickTrace.Var
qtc = QuickTrace.traceLongString

EXCLUDED_AUTOCOMPLETES = frozenset( [ '<cr>', '|', '>', '>>' ] )
COMMAND_ATTRIBUTES = ( ( 'cmd', basestring, 'a string' ),
                       ( 'revision', int, 'a number' ),
                       ( 'input', basestring, 'a string' ) )
VALID_RESPONSE_FORMATS = ( ResponseFormats.JSON, ResponseFormats.TEXT )
VALID_VERSIONS = [ 1 ]

class JsonCli( JsonRpcBase ):
   def __init__( self, capiExecutor ):
      """Give the base class a name for the plugin instance."""
      JsonRpcBase.__init__( self )
      self.capiExecutor_ = capiExecutor

   def _validateStringOption( self, option, optionName ):
      if not isinstance( option, basestring ):
         msg = 'Invalid parameter "%s": expected a string' % optionName
         qt0( "Error:", 'invalid param "%s"' % optionName )
         trace( '_validateStringOption exit', msg )
         raise InvalidParamsError( msg )

   def _validateOption( self, option, optionName, possibleValues ):
      # Validate the argument.
      self._validateStringOption( option, optionName )

      if option not in possibleValues:
         msg = ( 'Invalid parameter "%s": unrecognized option "%s", '
                 'expected one of: %s' % ( optionName, option,
                                           ', '.join( possibleValues ) ) )
         qt0( "Error:", 'invalid param "%s"' % optionName )
         trace( '_validateOption exit', msg )
         raise InvalidParamsError( msg )

   def _validateCmds( self, cmds ):
      # The argument cmds must be a list.
      if not isinstance( cmds, list ):
         msg = 'Invalid parameter "cmds": expected an array'
         qt0( "Error:", 'invalid param "cmds"' )
         trace( '_validateCmds exit', msg )
         raise InvalidParamsError( msg )

   def _validateVersion( self, version ):
      # internally, version 0 means "pick latest revision"
      if type( version ) == str and version == "latest":
         version = JSONRPC_VERSION_LATEST
      elif version not in VALID_VERSIONS:
         msg = 'Invalid value for parameter "version": expected %s' % str(
                  VALID_VERSIONS ) + ' or "latest"'
         qt0( "Error:", 'invalid param "version"' )
         trace( '_validateVersion exit', msg )
         raise InvalidParamsError( msg )
      return version

   def _generateCmdList( self, cmds ):
      # validate commands sent first
      self._validateCmds( cmds )

      # The Cli API expects a list of strings; each a CLI command.
      cliCommandList = []

      for i, cmdObj in enumerate( cmds ):
         if isinstance( cmdObj, basestring ):
            cmd = CliApi.CliCommand( cmdObj )
         elif isinstance( cmdObj, dict ):
            cmd = self._validateComplexCommand( cmdObj, i )
         else:
            msg = ( 'Invalid parameter "cmds": unrecognized structure '
                    'for command %d: %r' % ( i, cmdObj ) )
            qt0( "Error:", 'invalid param cmd', qv( i ) )
            trace( 'runCmds exit', msg )
            raise InvalidParamsError( msg )
         cliCommandList.append( cmd )

      return cliCommandList

   def _processCliResults( self, cliResults, cliCommandList, stopOnError,
                           includeErrorDetail ):
      # Process the result and raise an application specific error if
      # any command failed
      cmdResults = []
      failedCmds = []
      for i, cmdResponse in enumerate( cliResults ):
         # Make sure we get the result before checking the status,
         # because the status may change in the case of a bad
         # GeneratorList/Dict:
         cmdResults.append( cmdResponse.result )
         if cmdResponse.status != CapiStatus.SUCCESS:
            trace( 'cli exit json' )
            qt0( "Error:", "cmd", i, "status:", qv( cmdResponse.status ) )
            failedCmds.append( ( cmdResponse, i ) )
            if stopOnError:
               break

      if failedCmds:
         raise self._createCommandException( failedCmds, cmdResults, cliCommandList,
                                             includeErrorDetail )
      return cmdResults

   def _executeCommands( self, cliCommandList, textFlag, timestamps, stopOnError,
                         version, preCommandsFn=None, postCommandsFn=None,
                         autoComplete=False, expandAliases=False,
                         requestTimeout=None, includeErrorDetail=False ):
      nitty( 'invoking CliApi executeCommands',
             'cmds', cliCommandList,
             'textFlag', textFlag,
             'timestamps', timestamps,
             'version', version )
      if self.outputFd() is not None and textFlag:
         raise JsonRpcError( code=JsonRpcErrorCodes.COMMAND_INCOMPATIBLE,
                             msg="Streaming is not supported with format='text'" )
      cliResults = self.capiExecutor_.executeCommands(
         cliCommandList,
         textOutput=textFlag,
         timestamps=timestamps,
         stopOnError=stopOnError,
         preCommandsFn=preCommandsFn,
         postCommandsFn=postCommandsFn,
         globalVersion=version,
         autoComplete=autoComplete,
         expandAliases=expandAliases,
         requestTimeout=requestTimeout,
         streamFd=self.outputFd() )

      nitty( 'executeCommands returned', cliResults )
      assert len( cliResults ) <= len( cliCommandList ), \
              "Received %s cliResults, expected a max of %s" % \
              ( len( cliResults ), len( cliCommandList ) )

      cmdResults = self._processCliResults( cliResults, cliCommandList, stopOnError,
                                            includeErrorDetail )

      # Update the information in the request context
      self.setRequestCount( 1 )
      self.setCommandCount( len( cliCommandList ) )

      qt0( "Response: success, response length:", qv( len( str( cmdResults ) ) ) )
      trace( 'cli exit json', cmdResults )
      return cmdResults

   @JsonRpcBase.rpc
   def getCommandCompletions( self, command, preamble=None ):
      """ Given a string representing a CLI command, return a list
      fully expanded commands that this string could refer to. This
      is done for all commands in enable mode. For
      example, 'sh' could lead to a reply of [ 'show', 'shutdown'].
      Similarly, if fed 'sh int', this function would return
      [ 'interfaces' ].  If no matches for the string were found,
      this function returns an empty list"""
      trace( 'Getting completions for command', command )
      assert self.capiExecutor_
      qt0( "getCommandCompletions: Request:", "command:", qv( str( command ) ) )
      self._validateStringOption( command, "command" )

      errs, compl = self.capiExecutor_.getCompletions( command,
                                                       preamble=preamble )
      complete = bool( [ c for c in compl if c.name == '<cr>' ] )
      filtered = [ c for c in compl if c.name not in EXCLUDED_AUTOCOMPLETES ]
      return { 'errors': errs,
               'complete': complete,
               'completions': dict( ( c.name, c.help ) for c in filtered ) }

   @JsonRpcBase.rpc
   def getPrompt( self ):
      return { 'prompt': self.capiExecutor_.getPrompt() }

   @JsonRpcBase.rpc
   def getCommandHelp( self, command ):
      """Getting help information from the CliAPI"""
      trace( 'Getting help for command', command )
      return self.capiExecutor_.getHelpInfo( command=command )

   # Ignore pylint's complaints that we redefine format
   # pylint: disable-msg=W0622
   @JsonRpcBase.rpc
   def editConfig( self, version, cmds, format=ResponseFormats.JSON,
                   timestamps=False, enablePassword="",
                   operation=EditConfigOptions.DEFAULT_OPERATION,
                   testOption=EditConfigOptions.DEFAULT_TEST_OPTION,
                   stopOnError=True, sessionName=None, writeMemoryOnCommit=False ):
      assert self.capiExecutor_
      qt0( "editConfig: Request:", "version:", qv( version ),
           "response format:", qv( str( format ) ),
           "operation:", qv( str( operation ) ),
           "testOption:", qv( str( testOption ) ),
           "stopOnError:", qv( str( stopOnError ) ),
           "enablePwSet:", qv( enablePassword != "" ) )
      qtc( qt0, cmds )
      self._validateVersion( version )
      self._validateOption( operation, "operation",
                            EditConfigOptions.VALID_OPERATIONS )
      self._validateOption( testOption, "testOption",
                            EditConfigOptions.VALID_TEST_OPTIONS )
      self._validateStringOption( enablePassword, "enablePassword" )
      self._validateOption( format, "format", VALID_RESPONSE_FORMATS )
      cliCommandList = self._generateCmdList( cmds )
      if sessionName is None:
         cliSessionName = "capi-%s-%s" % ( os.getpid(), uuid.uuid1().hex )
      else:
         cliSessionName = sessionName

      def preCommandsFn():
         enableInput = enablePassword if enablePassword else None
         preCommands = [ CliApi.CliCommand( "enable", input=enableInput ) ]
         preCommands.append( CliApi.CliCommand( "configure session %s" %
                                                cliSessionName ) )
         if operation == EditConfigOptions.OPERATION_REPLACE: # else we are merging
            preCommands.append( CliApi.CliCommand( "rollback clean-config" ) )

         return preCommands

      def postCommandsFn( errorSeen ):
         postCommands = []
         postCommands.append( "configure session %s" % cliSessionName )
         if testOption == EditConfigOptions.TEST_OPTION_TEST_ONLY:
            postCommands.append( "abort" )
         elif testOption == EditConfigOptions.TEST_OPTION_TEST_THEN_SET:
            if errorSeen:
               postCommands.append( "abort" )
            else:
               postCommands.append( "commit" )
               if writeMemoryOnCommit:
                  postCommands.append( "write memory" )
         elif testOption == EditConfigOptions.TEST_OPTION_SET:
            postCommands.append( "commit" )
            if writeMemoryOnCommit:
               postCommands.append( "write memory" )

         postCommands = [ CliApi.CliCommand( cmd ) for cmd in postCommands ]
         assert len( postCommands ) > 1, \
                "No testOption %s found, would leak session" % testOption
         return postCommands

      return self._executeCommands( cliCommandList, format == ResponseFormats.TEXT,
                                    timestamps, stopOnError, version,
                                    preCommandsFn=preCommandsFn,
                                    postCommandsFn=postCommandsFn,
                                    includeErrorDetail=False )

   # Ignore pylint's complaints that we redefine format
   # pylint: disable-msg=W0622
   @JsonRpcBase.rpc
   def verifyConfig( self, configList, format=ResponseFormats.JSON,
                   timestamps=False, stopOnError=True, version=1 ):
      """The verifyConfig api takes a list of configlets and returns
      a list of aggregated normalized configlets. For eg. if
      configList=[ c1, c2 ] then result will be [ normalized(c1),
      normalized(c1+c2) ]. This function much faster as it applies the
      configlets in a config session with AAA disabled. The optional
      `format` parameter is either 'json' or 'text', and controls the
      response format. `format` defaults to 'json' if not specified"""
      assert self.capiExecutor_
      qt0( "verifyConfig: Request:", "version:", qv( version ),
           "response format:", qv( str( format ) ),
           "stopOnError:", qv( str( stopOnError ) ) )
      qtc( qt0, configList )
      self._validateVersion( version )
      self._validateOption( format, "format", VALID_RESPONSE_FORMATS )
      if not isinstance( configList, list ):
         raise InvalidParamsError( 'configList value %s is not a list' % configList )
      cliSessionName = "capiVerify-%s-%s" % ( os.getpid(), uuid.uuid1().hex )

      def preCommandsFn():
         preCommands = [ CliApi.CliCommand( "enable" ) ]
         preCommands.append( CliApi.CliCommand( "configure session %s no-commit" %
                                                cliSessionName ) )
         preCommands.append( CliApi.CliCommand( "rollback clean-config" ) )
         return preCommands

      def postCommandsFn( errorSeen ):
         postCommands = []
         postCommands.append( CliApi.CliCommand( "abort" ) )
         return postCommands

      configCmds = []
      showSessionCmd = 'show session-config'
      for config in configList:
         cmd = { "cmd": "copy terminal: session-config",
                 "input": "%s" % config }
         configCmds += [ cmd, showSessionCmd ]
      cliCommandList = self._generateCmdList( configCmds )
      aaa = ( self.capiExecutor_.session_.authenticationEnabled() or
              self.capiExecutor_.session_.authorizationEnabled() )
      try:
         self.capiExecutor_.session_.disableAaaIs( True )
         return self._executeCommands( cliCommandList,
                                       format == ResponseFormats.TEXT,
                                       timestamps, stopOnError, version,
                                       preCommandsFn=preCommandsFn,
                                       postCommandsFn=postCommandsFn,
                                       includeErrorDetail=False )
      finally:
         self.capiExecutor_.session_.disableAaaIs( not aaa )

   # Ignore pylint's complaints that we redefine format
   # pylint: disable-msg=W0622
   @JsonRpcBase.rpc
   def runCmds( self, version, cmds,
                format=ResponseFormats.JSON,
                timestamps=False,
                stopOnError=True,
                autoComplete=False,
                requestTimeout=None,
                expandAliases=False,
                includeErrorDetail=False,
                streaming=False ):
      """The runCmds procedure executes an list of CLI commands. `cmds`
      is an array of strings; each a valid CLI command that we'll
      execute in order. The optional `format` parameter is either
      either 'json' or 'text', and controls the response
      format. `format` defaults to 'json' if not specified"""
      assert self.capiExecutor_
      qt0( "runCmds: Request:", "version:", qv( str( version ) ),
           "response format:", qv( str( format ) ) )
      qtc( qt0, cmds )
      version = self._validateVersion( version )
      self._validateOption( format, "format", VALID_RESPONSE_FORMATS )
      cliCommandList = self._generateCmdList( cmds )

      return self._executeCommands( cliCommandList, format == ResponseFormats.TEXT,
                                    timestamps, stopOnError, version,
                                    autoComplete=autoComplete,
                                    requestTimeout=requestTimeout,
                                    expandAliases=expandAliases,
                                    includeErrorDetail=includeErrorDetail )

   def handlePostRawText( self, request ):
      """ Override handlePostRawText to accept HTTP POST raw text
      methods.  The caller may supply the command list as text, one
      command per line.  Synthesize the arguments to the runCmds()
      procedure and invoke. Currently assumes the commands use version
      number 1, though eventually this assumption will change to
      always use the latest version number. """
      trace( 'handlePostRawText entry' )

      # Get the command list from the request body.
      cmdList = []
      contentSplit = re.split( '[\n\r;]+', request )
      for contentElem in contentSplit:
         cmd = contentElem.strip()
         if len( cmd ):
            cmdList.append( cmd )

      # Call the cli procedure.
      response = self.runCmds( version=1, cmds=cmdList )
      trace( 'handlePostRawText exit' )
      return response

   def _validateComplexCommand( self, cmdObj, cmdIndex ):
      """ Given a dictionary representing a complex command, return a
      CliApi.CliCommand if the command is valid. Otherwise, this
      raises an InvalidParamsError with an appropriate message. """

      invalidCmdAttrMsg = ( 'Invalid parameter "cmds": unrecognized structure '
                            'for "%s" attribute in command %d, expected %s' )

      if 'cmd' not in cmdObj:
         msg = ( 'Invalid parameter "cmds": no "cmd" attribute '
                 'specified in command %d' % cmdIndex )
         trace( 'runCmds exit', msg )
         raise InvalidParamsError( msg )

      # Build up a list of parameters to pass to the CliCommand from a
      # predefined allowed list:
      params = {}
      for attr, attrType, strForm in COMMAND_ATTRIBUTES:
         if attr not in cmdObj:
            continue
         if not isinstance( cmdObj[ attr ], attrType ):
            msg = invalidCmdAttrMsg % ( attr, cmdIndex, strForm )
            trace( 'runCmds exit', msg )
            raise InvalidParamsError( msg )
         params[ attr ] = cmdObj[ attr ]

      return CliApi.CliCommand( **params )

   def _createCommandException( self, failedCmds, cmdResults, cliCommandList,
                                includeErrorDetail ):
      """ Generates a JsonRpcError with the appropriate error code,
      msg, and data """
      messages = []
      errorDetailList = []
      for failedInfo in failedCmds:
         ( failedCmdResponse, failedCmdIndex ) = failedInfo
         # Build error message.
         errorCode, msgDetails = self._capiStatusToError( failedCmdResponse.status )
         msg = "CLI command %d of %d %s failed: %s" % \
               ( failedCmdIndex + 1,
                 len( cliCommandList ),
                 cliCommandList[ failedCmdIndex ],
                 msgDetails )
         messages.append( msg )
         if includeErrorDetail:
            errorDetailList.append(
                  { 'index': failedCmdIndex,
                    'code': errorCode,
                    'message': msgDetails,
                    'cmd': str( cliCommandList[ failedCmdIndex ].cmdStr ) } )

      # we always return the first error code, not the last
      firstErrorCode, _ = self._capiStatusToError( failedCmds[ 0 ][ 0 ].status )
      if includeErrorDetail:
         data = { 'result': cmdResults,
                  'errorDetail': errorDetailList }
      else:
         data = cmdResults
      return JsonRpcError( code=firstErrorCode, msg='\n'.join( messages ),
                           data=data )

   def _capiStatusToError( self, errorStatus ):
      assert errorStatus != CapiStatus.SUCCESS
      if errorStatus == CapiStatus.ERROR:
         return ( JsonRpcErrorCodes.COMMAND_FAILED, 'could not run command' )
      if errorStatus == CapiStatus.UNAUTHORIZED:
         return ( JsonRpcErrorCodes.COMMAND_UNAUTHORIZED,
                  'permission to run command denied' )
      if errorStatus == CapiStatus.FORBIDDEN:
         return ( JsonRpcErrorCodes.COMMAND_INCOMPATIBLE, 'incompatible command' )
      if errorStatus == CapiStatus.INTERNAL_ERROR:
         return ( JsonRpcErrorCodes.COMMAND_EXCEPTION, 'internal error' )
      if errorStatus == CapiStatus.NOT_FOUND:
         return ( JsonRpcErrorCodes.COMMAND_INVALID, 'invalid command' )
      if errorStatus == CapiStatus.NOT_CAPI_READY:
         return ( JsonRpcErrorCodes.COMMAND_UNCONVERTED, 'unconverted command' )
      if errorStatus == CapiStatus.CONFIG_LOCKED:
         return ( JsonRpcErrorCodes.COMMAND_CONFIG_LOCKED, 'configuration locked' )
      return ( JsonRpcErrorCodes.COMMAND_EXCEPTION,
               'unexpected error %s' % str( errorStatus ) )
