#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Arnet

def _ndPrefixOrPrefixProfileParameters( entry, validLifetime, preferredLifetime,
                                        flagsList ):
   if validLifetime is not None:
      entry.validLifetime = validLifetime

   if preferredLifetime is not None:
      entry.preferredLifetime = preferredLifetime

   for flag in flagsList or []:
      if flag == 'no-onlink':
         entry.noOnLink = True
      elif flag == 'no-autoconfig':
         entry.noAutoConfig = True
      else:
         assert False, "Unexpected Flag: %s" % flag

class Ip6NdConfig():
   def __init__( self, routerAdvtConfig ):
      self.routerAdvtConfig = routerAdvtConfig

   def ndConfigProfile( self, profileName ):
      if profileName in self.routerAdvtConfig.prefixProfiles:
         del self.routerAdvtConfig.prefixProfiles[ profileName ]
      return self.routerAdvtConfig.prefixProfiles.newMember( profileName )

   def ndPrefixProfileNoAdvertise( self, profileName ):
      prefixProfile = self.ndConfigProfile( profileName )
      prefixProfile.noAdvertise = True

   def ndPrefixProfileParameters( self, profileName, validLifetime=None,
                                  preferredLifetime=None, flagsList=None ):
      prefixProfile = self.ndConfigProfile( profileName )
      _ndPrefixOrPrefixProfileParameters( prefixProfile,
                                          validLifetime=validLifetime,
                                          preferredLifetime=preferredLifetime,
                                          flagsList=flagsList )

   def noNdPrefixProfile( self, profileName ):
      if profileName not in self.routerAdvtConfig.prefixProfiles:
         return True

      # Enforce the profile to be removed from all prefixes before removing
      # the profile definition.
      for intfConfig in self.routerAdvtConfig.intf.itervalues():
         for prefixConfigEntry in intfConfig.prefixes.itervalues():
            if prefixConfigEntry.profile == profileName:
               return False

      del self.routerAdvtConfig.prefixProfiles[ profileName ]

      return True

   def defaultNdPrefixProfile( self, profileName ):
      if profileName not in self.routerAdvtConfig.prefixProfiles:
         return

      # When defaulting, the profile would still exist, but have default values
      del self.routerAdvtConfig.prefixProfiles[ profileName ]
      self.routerAdvtConfig.prefixProfiles.newMember( profileName )

class Ip6NdIntfConfig():
   def __init__( self, routerAdvtConfig, intfName ):
      self.routerAdvtConfig = routerAdvtConfig
      self.intfName = intfName

   def ndRaConfigIntf( self ):
      intfName = self.intfName
      if intfName in self.routerAdvtConfig.intf:
         return self.routerAdvtConfig.intf[ intfName ]
      ndRaConfigIntf_ = self.routerAdvtConfig.intf.newMember( intfName )
      if intfName.startswith( 'Management' ):
         ndRaConfigIntf_.raSuppress = True
         ndRaConfigIntf_.raSuppressRouterSolicitationResponse = True
      return ndRaConfigIntf_

   def ndRaSuppress( self, allRa ):
      intfConfig = self.ndRaConfigIntf()

      intfConfig.raSuppress = True
      
      if allRa:
         intfConfig.raSuppressRouterSolicitationResponse = True
      else:
         intfConfig.raSuppressRouterSolicitationResponse = False

   def noNdRaSuppress( self ):
      intfConfig = self.ndRaConfigIntf( )
      if not intfConfig:
         return
      intfConfig.raSuppress = False
      intfConfig.raSuppressRouterSolicitationResponse = False

   def ndRaInterval( self, intervalMaxMsecValue, intervalMinMsecValue=None ):
      if intervalMinMsecValue:
         if intervalMinMsecValue >= intervalMaxMsecValue:
            return

      intfConfig = self.ndRaConfigIntf()
         
      intfConfig.raIntervalMaxMsec = intervalMaxMsecValue
      if intervalMinMsecValue:
         intfConfig.raIntervalMinMsec = intervalMinMsecValue
      else:
         intfConfig.raIntervalMinMsec = 0

   def noNdRaInterval( self ):
      intfConfig = self.ndRaConfigIntf()
      if not intfConfig:
         return
      intfConfig.raIntervalMaxMsec = intfConfig.raIntervalMaxDefault
      intfConfig.raIntervalMinMsec = intfConfig.raIntervalMinDefault

   def ndRaLifetime( self, raLifetimeValue ):
      intfConfig = self.ndRaConfigIntf()
      intfConfig.raLifetime = raLifetimeValue

   def noNdRaLifetime( self ):
      intfConfig = self.ndRaConfigIntf()
      if not intfConfig:
         return
      intfConfig.raLifetime = intfConfig.raLifetimeDefault

   def ndRaHopLimitIs( self, hopLimitValue ):
      intfConfig = self.ndRaConfigIntf()
      intfConfig.hopLimit = hopLimitValue

   def noNdRaHopLimitIs( self ):
      intfConfig = self.ndRaConfigIntf()
      if not intfConfig:
         return
      intfConfig.hopLimit = intfConfig.hopLimitDefault

   def ndRaMtuSuppress( self ):
      intfConfig = self.ndRaConfigIntf()
      intfConfig.raMtuSuppress = True

   def noNdRaMtuSuppress( self ):
      intfConfig = self.ndRaConfigIntf()
      if not intfConfig:
         return
      intfConfig.raMtuSuppress = intfConfig.raMtuSuppressDefault

   def ndManagedConfigFlag( self ):
      intfConfig = self.ndRaConfigIntf()
      intfConfig.managedConfigFlag = True

   def noNdManagedConfigFlag ( self ):
      intfConfig = self.ndRaConfigIntf()
      if not intfConfig:
         return
      intfConfig.managedConfigFlag = intfConfig.managedConfigFlagDefault

   def ndOtherConfigFlag( self ):
      intfConfig = self.ndRaConfigIntf()
      intfConfig.otherConfigFlag = True

   def noNdOtherConfigFlag ( self ):
      intfConfig = self.ndRaConfigIntf()
      if not intfConfig:
         return
      intfConfig.otherConfigFlag = intfConfig.otherConfigFlagDefault

   def ndReachableTime( self, ndReachableTimeValue ):
      intfConfig = self.ndRaConfigIntf()
      intfConfig.reachableTime = ndReachableTimeValue

   def noNdReachableTime ( self ):
      intfConfig = self.ndRaConfigIntf()
      if not intfConfig:
         return
      intfConfig.reachableTime = intfConfig.reachableTimeDefault

   def ndRouterPreference ( self, routerPreferenceValue ):
      intfConfig = self.ndRaConfigIntf()
      assert routerPreferenceValue in [ 'high', 'medium', 'low' ]
      intfConfig.routerPreference = routerPreferenceValue

   def noNdRouterPreference ( self ):
      intfConfig = self.ndRaConfigIntf()
      if not intfConfig:
         return
      intfConfig.routerPreference = 'medium'

   # Creates a new prefix object. If one already exists, it deletes it and
   # recreates the object.
   def ndIntfConfigPrefix( self, ip6Prefix ):
      if type( ip6Prefix ) == Tac.Type( 'Arnet::Ip6Prefix' ):
         ip6Prefix = Arnet.Ip6AddrWithMask( ip6Prefix.stringValue )
      intfConfig = self.ndRaConfigIntf()
      if ip6Prefix in intfConfig.prefixes:
         del intfConfig.prefixes[ ip6Prefix ]
      return intfConfig.prefixes.newMember( ip6Prefix )

   def ndPrefixNoAdvertise( self, ip6Prefix ):
      prefix = self.ndIntfConfigPrefix( ip6Prefix )
      prefix.noAdvertise = True

   def ndPrefixParameters( self, ip6Prefix, validLifetime=None,
                           preferredLifetime=None, flagsList=None ):
      prefix = self.ndIntfConfigPrefix( ip6Prefix )
      _ndPrefixOrPrefixProfileParameters( prefix,
                                          validLifetime=validLifetime,
                                          preferredLifetime=preferredLifetime,
                                          flagsList=flagsList )

   def ndPrefixProfile( self, ip6Prefix, profileName ):
      prefix = self.ndIntfConfigPrefix( ip6Prefix )
      if profileName in self.routerAdvtConfig.prefixProfiles:
         prefix.profile = profileName
      else:
         prefix.profile = ''

   def noNdPrefix( self, ip6Prefix ):
      if type( ip6Prefix ) == Tac.Type( 'Arnet::Ip6Prefix' ):
         ip6Prefix = Arnet.Ip6AddrWithMask( ip6Prefix.stringValue )
      intfConfig = self.ndRaConfigIntf()

      if not intfConfig:
         return
      if ip6Prefix in intfConfig.prefixes:
         del intfConfig.prefixes[ ip6Prefix ]

   def ndNetworkBoot( self, url ):
      intfConfig = self.ndRaConfigIntf()
      if not intfConfig:
         return
      intfConfig.networkBootFileUrl = url

   def noNdNetworkBoot( self ):
      intfConfig = self.ndRaConfigIntf()
      if not intfConfig:
         return
      intfConfig.networkBootFileUrl = ''
