# Copyright (c) 2005-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ConfigMount, itertools
import Toggles.IntfToggleLib
import Intf.IntfRange as IntfRange
import Tac
import Tracing

t0 = Tracing.trace0

ethPhyIntfConfigSliceDir = None

#-------------------------------------------------------------------------------
# Register the EthPhyIntf class as able to participate in interface ranges.
#-------------------------------------------------------------------------------

class EthPhyRangeIntfTypeBase( IntfRange.AutoIntfType ):
   def __init__( self, name, shortName, desc, subSupported, subIntfGuard,
                 helpdesc=None, **kwargs ):
      IntfRange.AutoIntfType.__init__(
         self, self.ethPhyIntfCollection, name, shortName, desc, helpdesc=helpdesc,
         collectionVersionFunc=self.ethPhyIntfCollectionVersion,
         subSupported=subSupported, subIntfGuard=subIntfGuard, **kwargs )
      # List of class, function pairs.
      self.clazz = None
      self.mountForced = False

   def getCliIntf( self, mode, name ):
      if self.subSupported_ and '.' in name and self.cliSubIntfClazz_:
         return self.cliSubIntfClazz_( name, mode )
      else:
         return self.clazz( name, mode )
   
   def registerEthPhyIntfClass( self, clazz ):
      """
      clazz is the Cli Intf class that should be instantiated if
      func( name, mode ) is true
      """
      self.clazz = clazz 

   def ethPhyIntfCollection( self ):
      # Build an array of iter objects (an iter object for each EthPhyIntfConfigDir)
      # and return an itertool chain (of the array objects)
      allIntfConfigs = [ iter( ethPhyDir.intfConfig )
                         for ethPhyDir in ethPhyIntfConfigSliceDir.itervalues() ]
      return itertools.chain.from_iterable( allIntfConfigs )

   def ethPhyIntfCollectionVersion( self ):
      # This function implements versioning. Using this method the IntfRange
      # infrastructure code can figure out if the collection returned by
      # ethPhyIntfCollection() has changed
      # This method concatenates the version of the intfConfig collection of each
      # EthPhyIntfConfigDir. The version attribute is incremented every time the 
      # collection is updated

      if self.mountForced:
         sliceDir = ethPhyIntfConfigSliceDir
      else:
         sliceDir = ConfigMount.force( ethPhyIntfConfigSliceDir )
         self.mountForced = True

      # In entity copy we make sure the destination dir has a new version
      # from the source dir, so if we switch between Sysdb and session we'll
      # rebuild the interface range tree.
      versionList = [ sliceDir.entityPtr.version() ]

      for ethPhyDir in sliceDir.itervalues():
         versionList.append( ethPhyDir.intfConfig.version() )

      t0( "EthIntf version", versionList )
      return versionList

# Ethernet interfaces
class EthPhyRangeIntfType( EthPhyRangeIntfTypeBase ):
   def __init__( self, helpdesc=None, **kwargs ):
      EthPhyRangeIntfTypeBase.__init__( self, name="Ethernet", 
         shortName="Et", desc="hardware Ethernet interface",
         subSupported=True, subIntfGuard=IntfRange.subintfSupportedGuard,
         **kwargs )

if Toggles.IntfToggleLib.toggleDynamicIfIndexEnabled():
   SubIntfRangeDetails = Tac.Type( "Interface::SubIntfIdConstants" )
   SubExtendedMin = SubIntfRangeDetails.subIntfExtendedIdMin
   SubExtendedMax = SubIntfRangeDetails.subIntfExtendedIdMax
   EthPhyAutoIntfType = EthPhyRangeIntfType( intfSubLowerBound=SubExtendedMin,
         intfSubUpperBound=SubExtendedMax )
else:
   EthPhyAutoIntfType = EthPhyRangeIntfType( )
intfOptions = IntfRange.IntfOptions( ipSupport=True, routingProtoSupport=True )
IntfRange.registerIntfType( EthPhyAutoIntfType, intfOptions )

class ManagementRangeIntfType( IntfRange.AutoIntfType ):
   def __init__( self, helpdesc=None ):
      IntfRange.AutoIntfType.__init__(
         self, self.ethPhyIntfCollection,
         "Management", "Ma", "Management interface", helpdesc=helpdesc,
         collectionVersionFunc=self.ethPhyIntfCollectionVersion )
      # List of class, function pairs.
      self.cliIntfClasses_ = []
      self.extraIntfs_ = set()

   def getCliIntf( self, mode, name ):
      for clazz, func in self.cliIntfClasses_:
         if func( name, mode ):
            return clazz( name, mode )
      return self.cliIntfClazz_( name, mode )

   def registerManagementIntfClass( self, clazz, func, extraIntfs ):
      """
      clazz is the Cli Intf class that should be instantiated if
      func( name, mode ) is true
      """
      self.cliIntfClasses_.append( ( clazz, func ) )
      self.extraIntfs_.update( extraIntfs )

   def ethPhyIntfCollection( self ):
      # This is a generator of two collections
      allIntfConfigs = [ iter( ethPhyDir.intfConfig ) \
                         for ethPhyDir in ethPhyIntfConfigSliceDir.values() ]
      allIntfConfigs.append( iter( self.extraIntfs_ ) )
      return itertools.chain.from_iterable( allIntfConfigs )

   def ethPhyIntfCollectionVersion( self ):
      # XXX_ATANDON
      # NEED TO ADD A TEST FOR THIS
      # This is the original change. Need to confirm that my approach of returning
      # a list works (especially from a performance standpoint)
      #return ( ethPhyIntfConfigDir.intfConfig.version() |
      #         len( self.extraIntfs_ ) << 24 )

      # This function is expected to return a unique number that changes whenever
      # the set of interface changes.
      # Using this method the IntfRange infrastructure code can figure out if the
      # collection returned by ethPhyIntfCollection() has changed
      # This method concatenates the version of the intfConfig collection of each
      # EthPhyIntfConfigDir. The version attribute is incremented every time the 
      # collection is updated

      sliceDir = ConfigMount.force( ethPhyIntfConfigSliceDir )

      versionList = list()
      versionList.append( id( sliceDir) )
      versionList.append( sliceDir.entityPtr.version() )

      for ethPhyDir in sliceDir.values():
         versionList.append( ethPhyDir.intfConfig.version() )

      versionList.append( len( self.extraIntfs_ ) )

      return versionList

MgmtAutoIntfType = ManagementRangeIntfType( )
intfOptions = IntfRange.IntfOptions( ipSupport=True, routingProtoSupport=False )
IntfRange.registerIntfType( MgmtAutoIntfType, intfOptions )

# This is the physical mgmt collection (nobody should call 
# registerManagementIntfClass on this)
MgmtPhyAutoIntfType = ManagementRangeIntfType( helpdesc=[ 'Port number' ] )

# Unconnected Ethernet Interfaces
class UnconnectedEthPhyRangeIntfType( EthPhyRangeIntfTypeBase ):
   def __init__( self, helpdesc=None ):
      EthPhyRangeIntfTypeBase.__init__( self, name="UnconnectedEthernet",
         shortName="Ue", desc="Unconnected internal Ethernet interface",
         subSupported=False, subIntfGuard=None )

UnconnectedEthPhyAutoIntfType = UnconnectedEthPhyRangeIntfType()
intfOptions = IntfRange.IntfOptions( ipSupport=True, routingProtoSupport=False )
IntfRange.registerIntfType( UnconnectedEthPhyAutoIntfType, intfOptions )

_em = None

def Plugin( em ):
   global ethPhyIntfConfigSliceDir, _em

   if _em == em:
      return

   _em = em
   ethPhyIntfConfigSliceDir = ConfigMount.mount(
                                         em,
                                         "interface/config/eth/phy/slice", 
                                         "Tac::Dir",
                                         "wi" )
