# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

def getNonProfileCmds( currentConfig, profileCmds ):
   """ Process the currentConfig so profile commands don't show up in the 
   resulting config, and config that overrides profile configs are explicit.

   currentConfig - the raw unprocessed interface config text
   profileCmds - list of profile commands
   """
   lines = currentConfig.splitlines()
   assert lines[ 0 ].startswith( 'profile' ), \
          "'profile' command must be first command in mode"
   for pCmd in profileCmds:
      if pCmd in lines:
         lines.remove( pCmd )
      else:
         if pCmd.startswith( 'default ' ):
            continue
         if pCmd.startswith( 'no ' ):
            pCmd = pCmd[ 3: ]
         # interface negated the profile config.
         lines.insert( 1, 'default ' + pCmd )
   return '\n'.join( lines ) + '\n'

def filterIntfConfig( config, intfNames ):
   """ Returns a dictionary mapping { intfName : config under that interface } """
   intfConfig = {}
   for line in re.split( '^interface ', config, flags=re.MULTILINE ):
      line = line.splitlines()
      intf = line[ 0 ]
      if intf in intfNames:
         intfCmds = ''
         for cmd in line[ 1: ]:
            if cmd.startswith( '   ' ):
               # strip the 3 leading spaces from intf commands
               intfCmds += cmd[ 3: ] + "\n"
            else:
               # The rest of these aren't cmds under the intf mode, save and move on
               intfConfig[ intf ] = intfCmds
               break
   return intfConfig

class AppliedProfileConfig( object ):
   """ Functions to generate the config text that will be used to apply a profile 
   to an interface.

   newProfileConfig() - generates the rollback config and the profile commands
                        config, call this one first
   nonProfileConfig() - generate the non-profile config, this goes after the
                        profile config
   """

   def __init__( self, intfName, config, oldProfCmds, newProfCmds ):
      """
      intfName - name of interface to apply profile
      config - the raw unprocessed interface config text
      oldProfileCmds - list of previously applied profile commands
      newProfileCmds - list of profile commands to apply
      """
      self.intfName = intfName
      self.config = config
      self.oldProfileCmds = oldProfCmds
      self.newProfileCmds = newProfCmds

   def _nothingToApply( self ):
      return not self.oldProfileCmds and not self.newProfileCmds

   def newProfileConfig( self ):
      if self._nothingToApply():
         return ''
      config = 'default interface %s\n' % self.intfName
      config += 'interface %s\n' % self.intfName
      for newProfCmd in self.newProfileCmds:
         config += '%s\n' % newProfCmd
      return config

   def nonProfileConfig( self ):
      if self._nothingToApply():
         return ''
      config = 'interface %s\n' % self.intfName
      if self.oldProfileCmds:
         oldNonProfileCmds = getNonProfileCmds( self.config, self.oldProfileCmds )
         # Remove the "profile" line
         oldNonProfileCmds = "".join( oldNonProfileCmds.splitlines( True )[ 1: ] )
         return config + oldNonProfileCmds
      else:
         if self.config.startswith( 'profile' ):
            # Remove the "profile" line
            return config + "".join( self.config.splitlines( True )[ 1: ] )
         return config + self.config
