# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
This module contains templates for the interface interactions CLI command
(AID4425, AID7499). These templates are imported by the CLI to product the output
of 'show interfaces interactions'. They are also used in tests that use/mock
interactions CLI output in some way.
'''

from __future__ import absolute_import, division, print_function

from collections import namedtuple

import CliPlugin.AlePhyIntfInteractionsModel as IxnModel
import EthIntfLib
from TypeFuture import TacLazyType

################################################################################
# Tac Type declarations

EthLinkModeSet = TacLazyType( "Interface::EthLinkModeSet" )

################################################################################
# Global declarations

# Speed tokens
s400g8 = "400G-8"
s200g4 = "200G-4"
s100g4 = "100G-4"
s100g2 = "100G-2"
s50g2 = "50G-2"
s50g1 = "50G-1"
s40g = "40G"
s25g = "25G"
s10g = "10G"
s1g = "1G"
s100m = "100M"

# Speed-group rate tokens
sg50g = "50g"
sg25g = "25g"
sg10g = "10g"

################################################################################
# CAPI Model Helper
################################################################################

class IntfInteractionsModelHelper( object ):
   """
   Provides interface to populate the interactions CAPI model. Primary goal is
   to reduce repetitious model instantiation code.

   Parameters
   ----------
   modes: EthLinkModeSet
     In some cases, models will need to filter out modes listed in the template
     based on the capabilities of the system (see `compatibleRateList`). `modes`
     should describe the union of capabilities of the template's member interfaces.

   Note
   ----
   More of a note-to-self. Perhaps these belong as helper functions of the
   CAPI model itself. CAPI models providing mutators for attributes is a
   thing. Perhaps evaluate as a side-project...

   Also, to get certain renders to combine intfIds correctly, need to do
   something like the following during a test run to avoid importing a whole
   bunch of IntfRangePlugin and EthIntf stuff.

   # make render work right
   def dummyCombine( intfs, optarg=False ):
      return Utils.combineInterfaces( intfs )[ 0 ].replace( "Et", "Ethernet" )
   CliPlugin.AlePhyIntfInteractionsModel._combineIntfNames = dummyCombine

   Reason being that the combining method used relies on certain Sysdb structures
   to be in place to validate intfIds. This method substitution will allow
   tests to run mostly the same without building out all that Sysdb state.
   """
   def __init__( self, modes=None ):
      self.model = IxnModel.InterfacesInteractions()
      self.filterModes = modes or EthLinkModeSet.allCapsSet

   def compatibleRateList( self, rates ):
      """
      There are cases where when we populate a model with a list of rates, we
      should limit the rates to those supported by the reference's supported
      link modes. Consider the following contrived example:

      Ethernet1/1:
        For speed 100G:
           Ethernet1/2 becomes inactive
        For speed 50G:
           No interactions with other interfaces
        For speed 40G:
           No interactions with other interfaces
      Ethernet1/2:
        For speed 50G:
           Ethernet1/1 must be configured to 50G/40G
        For speed 40G:
           Ethernet1/1 must be configured to 50G/40G

      In this example, Et1/1 and Et1/2 make up a serdes-domain. The capabilities
      of the domain are 100G, 50G, and 40G. If we imagine that there is another
      flavor of this domain that can only do 100G and 40G, we would want the output
      to change to:

      Ethernet1/1:
        For speed 100G:
           Ethernet1/2 becomes inactive
        For speed 40G:
           No interactions with other interfaces
      Ethernet1/2:
        For speed 40G:
           Ethernet1/1 must be configured to 40G  <-- no 50G listed

      In other words, the rates of Et1/1 that are compatible with Et1/2 must be
      filtered from the template's superset based on the capabilities of the domain.
      This helper function provides that filtering.

      Parameters
      ----------
      rates: list of strings, values of EthIntfLib.speedLanestoSpeedLanesStr
         This function has the fun job of filtering a list of strings based on
         the contents of an EthLinkModeSet (lots of conversions incoming)
      """
      groupRates = EthIntfLib.linkModeSetToSpeedLaneList( self.filterModes )
      # The membership check will handle cases where a speed-lane combination
      # is not populated in speedLanestoSpeedLanesStr. I initially thought this
      # was overly cautious, but at the time of writing, it did not contain a
      # mapping for ( speed2p5Gbps, laneCount1 ).
      groupRateStrs = [ EthIntfLib.speedLanestoSpeedLanesStr[ rate ]
                        for rate in groupRates
                        if rate in EthIntfLib.speedLanestoSpeedLanesStr ]
      filteredRates = [ rate for rate in rates if rate in groupRateStrs ]
      return filteredRates

   def intfData( self, intf ):
      """
      Accessor for interactions data associated with intfId. If data does
      not exist, new object will be created.

      Parameters
      ----------
      intf: string representation of Arnet::IntfId
      """
      intfData = self.model.interfaces.setdefault( intf,
                  IxnModel.InterfacesInteractionsDataBySpeed() )
      return intfData

   def addIntfMode( self, intf, speed ):
      """
      Creates speed entry for a given interface.

      Parameters
      ----------
      intf: string representation of Arnet::IntfId
      speed: string representation of speed; expected values are from
         EthIntfLib.speedLanestoSpeedLanesStr

      Notes
      -----
      Not quite sure why I decided 'intfData' should be a create-if-not-exist and
      this method should be strictly create-on-request. I think it works well, but
      I'm not sure how to justify it.
      """
      intfData = self.intfData( intf )
      intfData.speeds[ speed ] = IxnModel.InterfacesInteractionsData()

   def inactiveIxn( self, parentIntf, speed, inactives ):
      intfData = self.intfData( parentIntf )
      inactivesModel = intfData.speeds[ speed ].inactiveInterfaces
      if not inactivesModel:
         inactivesModel = IxnModel.InactiveInterfacesInteraction()
         intfData.speeds[ speed ].inactiveInterfaces = inactivesModel
      inactivesModel.interfaces.extend( inactives )

   def speedLimitation( self, parentIntf, speed, subLimitations ):
      """
      Notes
      -----
      This interaction type doesn't lend itself to being called for the same
      parentIntf-speed combination more than once. I haven't had to yet, but
      it is something to look out for in the future.
      """
      intfData = self.intfData( parentIntf )
      limitationModel = intfData.speeds[ speed ].speedLimitedInterfaces
      if not limitationModel:
         limitationModel = IxnModel.SpeedLimitedInterfacesInteraction()
         intfData.speeds[ speed ].speedLimitedInterfaces = limitationModel
      for subIntf, speeds in subLimitations.items():
         compatibleSpeeds = self.compatibleRateList( speeds )
         speedStr = "/".join( compatibleSpeeds )
         limitationModel.interfaces[ subIntf ] = speedStr

   def speedRequirement( self, subIntf, speed, parentRequirements ):
      """
      Note
      ----
      This may need some finagling to work with both
      'compatibleParentInterfaceConfigurations' and its outdated alternative
      'primaryInterfaceConfigurationRequirement'
      """
      lineShortener = self.intfData( subIntf ).speeds[ speed ]  # too snarky?
      parentRequirement = lineShortener.compatibleParentInterfaceConfigurations
      if not parentRequirement:
         parentRequirement = IxnModel.CompatibleParentInterfaceConfigurations()
         lineShortener.compatibleParentInterfaceConfigurations = parentRequirement
      for parentIntf, configs in parentRequirements.items():
         configListModel = IxnModel.SpeedConfigurationList()
         compatibleConfigs = self.compatibleRateList( configs )
         for config in compatibleConfigs:
            configModel = IxnModel.SpeedConfiguration()
            configModel.populateFromStr( config )
            configListModel.configurations.append( configModel )
         parentRequirement.interfaces[ parentIntf ] = configListModel

   def primaryIntfSpeedRequirement( self, intf, primary, speed ):
      """
      Helps populate the 'primaryInterfaceConfigurationRequirement' member
      of the interactions model. This is more restrictive that the
      'compatibleParentInterfaceConfigurations' attribute since it assumes
      a single parent intf, and it assumes that the parent intf and the
      subordinate intf are restricted to the same speed.

      Note
      ----
      For newer types of interfaces, you likely want to use the
      'speedRequirement' function above instead. Long-story short, the
      corresponding attribute in the model could not accommodate the feedback
      given by reviewers of the output for newer systems. However, the attribute
      has to stick around of CAPI compatibility reasons, so we still use it for
      the port types that released with it.
      """
      lineShortener = self.intfData( intf ).speeds[ speed ]
      parentRequirement = lineShortener.primaryInterfaceConfigurationRequirement
      if not parentRequirement:
         parentRequirement = IxnModel.PrimaryInterfaceConfigurationRequirement()
         lineShortener.primaryInterfaceConfigurationRequirement = parentRequirement
      parentRequirement.interface = primary
      parentRequirement.requiredConfiguration = speed

   def speedGroupRequirement( self, intf, speed, speedGroupName, setting ):
      intfData = self.intfData( intf )
      speedGroupModel = intfData.speeds[ speed ].speedGroupRequirement
      if not speedGroupModel:
         speedGroupModel = IxnModel.SpeedGroupRequirement()
         intfData.speeds[ speed ].speedGroupRequirement = speedGroupModel
      speedGroupModel.speedGroup = str( speedGroupName )
      speedGroupModel.serdes = setting

   def logicalPortDescription( self, intf, poolId, memberIntfs, maxPorts ):
      intfData = self.intfData( intf )
      logicalPortModel = intfData.interfaceHardwareResourcePool
      if not logicalPortModel:
         logicalPortModel = IxnModel.InterfaceHardwareResourcePool()
         intfData.interfaceHardwareResourcePool = logicalPortModel
      logicalPortModel.poolId = poolId
      logicalPortModel.memberInterfaces = memberIntfs
      logicalPortModel.capacity = maxPorts

################################################################################
# Reference/Template objects
################################################################################

# Unlike for speed-groups, logical port interactions need more than just a name
# to populate the IntfInteractionReference object.
LogicalPortInteractionDesc = namedtuple( "LogicalPortInteractionDesc",
                                         [ "poolId", "memberIntfs", "maxPorts" ] )

class SerdesDomain( object ):
   """
   SerdesDomain objects store a list of interfaces, which are used to populate an
   IntfInteractionReference
   """
   def __init__( self, members ):
      self.memberIntfs = members

class IntfInteractionReference( object ):
   """
   Base-class for modularly generated intf interactions CLI model references
   for use in tests. Interactions modeled are for a particular resource group
   type. Each group type should be implemented in a base class.

   Common Members
   --------------
   groupLinkModes: Interface::EthLinkModeSet
      Speeds that interfaces in this group are capable of. Note that these are
      the capabilities of the group, not individual interfaces. The interfaces
      used to service a given speed are implied based on group type. For
      example, if a Babbage group lists 50G-2 as a capability, it is assumed
      that the configuration is serviced via master/1+3 and slave/1+3. By
      default, all capabilities are assumed servicable.

   speedGroupName: str or None
      If set, reference will use this speed-group name to populate fields for
      the relevant link modes. 

   logicalPortPoolDesc : LogicalPortInteractionDesc or None
      If set, reference will use information to populate logical port fields.
      If not set, the product is assumed to not have logical port limitations.

   Note
   ----
   For speed-group references: each template will hard-code the speed-group
      settings for a link-mode (e.g. 400G-8 will be assumed to require the
      50g serdes rate from the appropriate speed-group). If future products
      require templates with customizable speed-group parameters, we can
      extend the architecture to support it relatively painlessly.
   """
   def __init__( self ):
      self.groupLinkModes = EthLinkModeSet.allCapsSet
      self.speedGroupName = None
      self.logicalPortPoolDesc = None

   def model( self ):
      """
      Should be overwritten by child classes
      """
      return IxnModel.InterfacesInteractions()

   def json( self ):
      return self.model().toDict()

   def render( self ):
      # BUG479185: should capture STDOUT here, probably, to return string directly.
      return self.model().render()

################################################################################
# Interaction Template Definitions
################################################################################

class BabbageQsfp28Pair( IntfInteractionReference ):
   """
   Provides interactions CLI model reference for master+slave interfaces of a
   Babbage QSFP28 pair.

   Options (enabled by default unless specified):
     * 10G/25G retimer-mode
     * 40G/50G/100G gearbox-mode
     * Speed-group membership
     * Logical port pool membership (disabled)
   """
   def __init__( self, serdesDomain ):
      """
      Instantiate interaction models provided master-slave pair.

      Parameters
      ----------
      master: string representation of master port (i.e. EthernetX, without lane)
      slave: string representation of slave port (again, without lane)
      """
      IntfInteractionReference.__init__( self )
      self.master1 = serdesDomain.memberIntfs[ "master1" ]
      self.master2 = serdesDomain.memberIntfs[ "master2" ]
      self.master3 = serdesDomain.memberIntfs[ "master3" ]
      self.master4 = serdesDomain.memberIntfs[ "master4" ]
      self.slave1 = serdesDomain.memberIntfs[ "slave1" ]
      self.slave3 = serdesDomain.memberIntfs[ "slave3" ]

   def model( self ):
      """
      Create models for all member interfaces based on provided options.

      Note
      ----
      To optimize, cache models as class member as well as a dirty flag. Dirty
      flag should be set by default and in base-class's option mutators.
      """
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      # Rename to save me some typing...
      m1, m2, m3, m4 = [ self.master1, self.master2, self.master3, self.master4 ]
      s1, s3 = [ self.slave1, self.slave3 ]
      if self.groupLinkModes.mode200GbpsFull4Lane:
         # 200G-4 interactions:
         # * Applied on M/1; M/2-4,S/1,S/3 become inactive
         helper.addIntfMode( m1, s200g4 )
         if self.speedGroupName:
            helper.speedGroupRequirement( m1, s200g4, self.speedGroupName, sg50g )
         helper.inactiveIxn( m1, s200g4, ( m2, m3, m4, s1, s3 ) )
      if self.groupLinkModes.mode100GbpsFull2Lane:
         # 100G-2 interactions:
         # * Applied on M/1; M/2,M/4,S/1,S/3 become inactive
         # * Applied on M/3; M/4 becomes inactive
         # * M/1 limits M/3 to 100G-2
         # * M/3 requires M/1 to be 100g-2
         for intf in ( m1, m3 ):
            helper.addIntfMode( intf, s100g2 )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s100g2, self.speedGroupName,
                                             sg50g )
         helper.inactiveIxn( m1, s100g2, ( m2, m4, s1, s3 ) )
         helper.inactiveIxn( m3, s100g2, ( m4, ) )
         helper.speedLimitation( m1, s100g2, { m3: ( s100g2, ) } )
         helper.speedRequirement( m3, s100g2, { m1: ( s100g2, ) } )
      if self.groupLinkModes.mode100GbpsFull:
         # 100G-4 interactions:
         # * Applied on M/1 and S/1; M/2-4 and S/3 become inactive
         # * S/1 requires M/1 to be either 100G-4/50G-2/40G
         for intf in ( m1, s1 ):
            helper.addIntfMode( intf, s100g4 )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s100g4, self.speedGroupName,
                                             sg50g )
         helper.inactiveIxn( m1, s100g4, ( m2, m3, m4 ) )
         helper.inactiveIxn( s1, s100g4, ( s3, ) )
         helper.speedRequirement( s1, s100g4, { m1: ( s100g4, s50g2, s40g ) } )
      if self.groupLinkModes.mode50GbpsFull1Lane:
         # 50G-1 interactions:
         # * Applied on M/1-4; S/1,3 become inactive
         # * M/1 limits M/3 to 50G-1/25G/10G
         # * M/2-4 require M/1 to be either 50G-1/25G/10G
         # * M/4 also requires M/3 to be either 50G-1/25G/10G
         for intf in ( m1, m2, m3, m4 ):
            helper.addIntfMode( intf, s50g1 )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s50g1, self.speedGroupName,
                                             sg50g )
         helper.inactiveIxn( m1, s50g1, ( s1, s3 ) )
         rates = ( s50g1, s25g, s10g, )
         helper.speedLimitation( m1, s50g1, { m3: rates } )
         for intf in ( m2, m3, m4 ):
            helper.speedRequirement( intf, s50g1, { m1: rates } )
         helper.speedRequirement( m4, s50g1, { m3: rates } )
      if self.groupLinkModes.mode50GbpsFull:
         # 50G-2 interactions:
         # * Applied on M/1,3 and S/1,3; M/2,4 become inactive.
         # * M/1 limits M/3 to 50G
         # * M/3 requires M/1 to be at 50G
         # * S/1,3 require M/1 to be either 100G-4/50G-2/40G
         # * S/3 additionally requires S/1 to be 50G-2
         for intf in ( m1, m3, s1, s3 ):
            helper.addIntfMode( intf, s50g2 )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s50g2, self.speedGroupName,
                                             sg50g )
         helper.inactiveIxn( m1, s50g2, ( m2, m4 ) )
         helper.inactiveIxn( m3, s50g2, ( m4, ) )
         helper.speedLimitation( m1, s50g2, { m3: ( s50g2, ) } )
         helper.speedRequirement( m3, s50g2, { m1: ( s50g2, ) } )
         for intf in ( s1, s3 ):
            helper.speedRequirement( intf, s50g2, { m1: ( s100g4, s50g2, s40g ) } )
         helper.speedRequirement( s3, s50g2, { s1: ( s50g2, ) } )
      if self.groupLinkModes.mode40GbpsFull:
         # 40G interactions (same as 100G-4):
         # * Applied on M/1 and S/1; M/2-4 and S/3 become inactive
         # * S/1 requires M/1 to be either 100G-4/50G-2/40G
         for intf in ( m1, s1 ):
            helper.addIntfMode( intf, s40g )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s40g, self.speedGroupName, sg10g )
         helper.inactiveIxn( m1, s40g, ( m2, m3, m4 ) )
         helper.inactiveIxn( s1, s40g, ( s3, ) )
         helper.speedRequirement( s1, s40g, { m1: ( s100g4, s50g2, s40g ) } )
      if self.groupLinkModes.mode25GbpsFull:
         # 25G interactions:
         # * Applies on M/1-4; S/1,3 become inactive
         # * M/1 limits M/3 to either 50G-1/25G/10G
         # * M/2-4 require M/1 to be either 50G-1/25G/10G
         # * M/4 also requires M/3 to be either 50G-1/25G/10G
         for intf in ( m1, m2, m3, m4 ):
            helper.addIntfMode( intf, s25g )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s25g, self.speedGroupName, sg25g )
         helper.inactiveIxn( m1, s25g, ( s1, s3 ) )
         rates = ( s50g1, s25g, s10g )
         helper.speedLimitation( m1, s25g, { m3: rates } )
         for intf in ( m2, m3, m4 ):
            helper.speedRequirement( intf, s25g, { m1: rates } )
         helper.speedRequirement( m4, s25g, { m3: rates } )
      if self.groupLinkModes.mode10GbpsFull:
         # 10G interactions (same as 25G):
         # * Applies on M/1-4; S/1,3 become inactive
         # * M/1 limits M/3 to either 50G-1/25G/10G
         # * M/2-4 require M/1 to be either 50G-1/25G/10G
         # * M/4 also requires M/3 to be either 50G-1/25G/10G
         for intf in ( m1, m2, m3, m4 ):
            helper.addIntfMode( intf, s10g )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s10g, self.speedGroupName, sg10g )
         helper.inactiveIxn( m1, s10g, ( s1, s3 ) )
         rates = ( s50g1, s25g, s10g )
         helper.speedLimitation( m1, s10g, { m3: rates } )
         for intf in ( m2, m3, m4 ):
            helper.speedRequirement( intf, s10g, { m1: rates } )
         helper.speedRequirement( m4, s10g, { m3: rates } )

      # Populate logical port description, if applicable
      if self.logicalPortPoolDesc:
         for intf in ( m1, m2, m3, m4, s1, s3 ):
            # If set, self.logicalPortPoolDesc will be an iterable
            # pylint: disable=not-an-iterable
            helper.logicalPortDescription( intf, *self.logicalPortPoolDesc )

      return helper.model

class EightLaneIntfs( IntfInteractionReference ):
   """
   Provides reference for single eight-lane port (e.g. OSFP, QSFP-DD).

   Options:
     * Speeds: 10G/25G/40G/50G-1/50G-2/100G-2/100G-4/200G-4/200G-8/400G-8
     * Speed-group membership
     * Logical port pool membership (disabled by default)
   """
   def __init__( self, serdesDomain ):
      """
      Parameters
      ----------
      port: string representation of Ethernet port (i.e. EthernetX, without lane)
      """
      IntfInteractionReference.__init__( self )
      self.lane1 = serdesDomain.memberIntfs[ "lane1" ]
      self.lane2 = serdesDomain.memberIntfs[ "lane2" ]
      self.lane3 = serdesDomain.memberIntfs[ "lane3" ]
      self.lane4 = serdesDomain.memberIntfs[ "lane4" ]
      self.lane5 = serdesDomain.memberIntfs[ "lane5" ]
      self.lane6 = serdesDomain.memberIntfs[ "lane6" ]
      self.lane7 = serdesDomain.memberIntfs[ "lane7" ]
      self.lane8 = serdesDomain.memberIntfs[ "lane8" ]

   def model( self ):
      """
      Note
      ----
      Since these interactions are primarily lane/serdes-based, many speeds will
      share the exact same interactions. For example, from an interface-activeness
      perspective, 100G-4 and 200G-4 are the same. Logic here needs to be
      revisited if that ever becomes false.
      """
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      # This seems silly, but it gets me the variable names I want. eh.
      allIntfs = [ self.lane1, self.lane2, self.lane3, self.lane4,
                   self.lane5, self.lane6, self.lane7, self.lane8 ]
      lane1, lane2, lane3, lane4, lane5, lane6, lane7, lane8 = allIntfs
      # Given that this is a very lane-based interaction group, subordinate intfs
      # can only access the serdes it needs if one of its parent intfs doesn't
      # take it. This applies to the subordinate intfs at each of its
      # configurations. I'll list them here to get them out of the way.
      compatibleParentConfigs = {
         # These entries are in the format
         #    subIntf: { parent1: ( compatible speeds ),
         #               parent2: ( compatible speeds ),
         #               ... }
         lane2: { lane1: ( s50g1, s25g, s10g ) },
         lane3: { lane1: ( s100g2, s50g1, s50g2, s25g, s10g ) },
         lane4: { lane1: ( s100g2, s50g1, s50g2, s25g, s10g ),
                  lane3: ( s50g1, s25g, s10g ) },
         lane5: { lane1: ( s200g4, s100g2, s100g4, s50g1, s50g2, s40g, s25g,
                           s10g ) },
         lane6: { lane1: ( s200g4, s100g2, s100g4, s50g1, s50g2, s40g, s25g,
                           s10g ),
                  lane5: ( s50g1, s25g, s10g ) },
         lane7: { lane1: ( s200g4, s100g2, s100g4, s50g1, s50g2, s40g, s25g,
                           s10g ),
                  lane5: ( s100g2, s50g1, s50g2, s25g, s10g ) },
         lane8: { lane1: ( s200g4, s100g2, s100g4, s50g1, s50g2, s40g, s25g,
                           s10g ),
                  lane5: ( s100g2, s50g1, s50g2, s25g, s10g ),
                  lane7: ( s50g1, s25g, s10g ) }
      }
      if self.groupLinkModes.mode400GbpsFull8Lane:
         # 400G-8 interactions:
         # * Applied on lane1; all others go inactive
         helper.addIntfMode( lane1, s400g8 )
         if self.speedGroupName:
            helper.speedGroupRequirement( lane1, s400g8, self.speedGroupName, sg50g )
         helper.inactiveIxn( lane1, s400g8, ( lane2, lane3, lane4, lane5, lane6,
                                              lane7, lane8 ) )
      supportedFourLaneSpeeds = []
      if self.groupLinkModes.mode200GbpsFull4Lane:
         supportedFourLaneSpeeds.append( s200g4 )
      if self.groupLinkModes.mode100GbpsFull:
         supportedFourLaneSpeeds.append( s100g4 )
      if self.groupLinkModes.mode40GbpsFull:
         supportedFourLaneSpeeds.append( s40g )
      for speed in supportedFourLaneSpeeds:
         # 4-lane speed interactions:
         # * Applied on lanes 1 and 5; lanes 2-4 go inactive if configured on
         #   lane 1, and lanes 6-8 do the same if configured on lane 5.
         # * Lane 5 needs Lane 1 to not use the bottom four serdes
         for intf in ( lane1, lane5 ):
            helper.addIntfMode( intf, speed )
            if self.speedGroupName:
               sgRate = { s200g4: sg50g,
                          s100g4: sg25g,
                          s40g: sg10g }[ speed ]
               helper.speedGroupRequirement( intf, speed, self.speedGroupName,
                                             sgRate )
         helper.inactiveIxn( lane1, speed, ( lane2, lane3, lane4 ) )
         helper.inactiveIxn( lane5, speed, ( lane6, lane7, lane8 ) )
         helper.speedRequirement( lane5, speed, compatibleParentConfigs[ lane5 ] )
      supportedTwoLaneSpeeds = []
      if self.groupLinkModes.mode100GbpsFull2Lane:
         supportedTwoLaneSpeeds.append( s100g2 )
      if self.groupLinkModes.mode50GbpsFull:
         supportedTwoLaneSpeeds.append( s50g2 )
      for speed in supportedTwoLaneSpeeds:
         # 2-lane speed interactions:
         # * Applied on odd-numbered lanes; even-numbered lanes go inactive if
         #   preceding odd lane is configured for one of these speeds.
         # * All odd-numbered lanes must not be made inactive by one of the
         #   larger odd-numbered lanes.
         for intf in ( lane1, lane3, lane5, lane7 ):
            helper.addIntfMode( intf, speed )
            if self.speedGroupName:
               sgRate = { s100g2: sg50g,
                          s50g2: sg25g }[ speed ]
               helper.speedGroupRequirement( intf, speed, self.speedGroupName,
                                             sgRate )
         helper.inactiveIxn( lane1, speed, ( lane2, ) )
         helper.inactiveIxn( lane3, speed, ( lane4, ) )
         helper.inactiveIxn( lane5, speed, ( lane6, ) )
         helper.inactiveIxn( lane7, speed, ( lane8, ) )
         for intf in ( lane3, lane5, lane7 ):
            helper.speedRequirement( intf, speed, compatibleParentConfigs[ intf ] )
      supportedOneLaneSpeeds = []
      if self.groupLinkModes.mode50GbpsFull1Lane:
         supportedOneLaneSpeeds.append( s50g1 )
      if self.groupLinkModes.mode25GbpsFull:
         supportedOneLaneSpeeds.append( s25g )
      if self.groupLinkModes.mode10GbpsFull:
         supportedOneLaneSpeeds.append( s10g )
      for speed in supportedOneLaneSpeeds:
         # 1-lane speed interactions:
         # * All lanes greater than 1 must not be made inactive by one of the
         #   other lanes.
         for intf in allIntfs:
            helper.addIntfMode( intf, speed )
            if self.speedGroupName:
               sgRate = { s50g1: sg50g,
                          s25g: sg25g,
                          s10g: sg10g }[ speed ]
               helper.speedGroupRequirement( intf, speed, self.speedGroupName,
                                             sgRate )
         for intf in allIntfs[ 1 : ]:  # exclude lane1 in this section
            helper.speedRequirement( intf, speed, compatibleParentConfigs[ intf ] )

      # Populate logical port description, if applicable
      if self.logicalPortPoolDesc:
         for intf in allIntfs:
            # If set, self.logicalPortPoolDesc will be an iterable
            # pylint: disable=not-an-iterable
            helper.logicalPortDescription( intf, *self.logicalPortPoolDesc )

      return helper.model

class Sfp( IntfInteractionReference ):
   """
   Provides reference model for single SFP port

   Options:
    * 100M/1G/10G speeds
    * That's it
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.port = serdesDomain.memberIntfs[ "lane0" ]

   def model( self ):
      # TODO: this assumes a very simple SFP+ port, since that is the only type
      # of SFP port exposed to the interactions CLI at the moment.
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      helper.addIntfMode( self.port, s10g )
      helper.addIntfMode( self.port, s1g )
      helper.addIntfMode( self.port, s100m )
      helper.intfData( self.port ).includesSlowerSpeeds10gIs( True )
      return helper.model

class BasicQsfp( IntfInteractionReference ):
   """
   Provides reference model for single QSFP port, not connected to any gearbox.

   Options:
    * (10M/100M/1G/)10G/25G/40G/50G-2/100G-4

   Regarding <10G speeds: these will only affect the output for lane1. There's
   no QSFP module that we support that would allow <10G speeds to run on
   lanes2-4.

   Note
   ----
   Options can be enabled/disabled to represent QSFP+ or QSFP28
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.lane1 = serdesDomain.memberIntfs[ "lane1" ]
      self.lane2 = serdesDomain.memberIntfs[ "lane2" ]
      self.lane3 = serdesDomain.memberIntfs[ "lane3" ]
      self.lane4 = serdesDomain.memberIntfs[ "lane4" ]

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      allIntfs = [ self.lane1, self.lane2, self.lane3, self.lane4 ]
      lane1, lane2, lane3, lane4 = allIntfs
      supportedFourLaneSpeeds = []
      if self.groupLinkModes.mode100GbpsFull:
         supportedFourLaneSpeeds.append( s100g4 )
      if self.groupLinkModes.mode40GbpsFull:
         supportedFourLaneSpeeds.append( s40g )
      for speed in supportedFourLaneSpeeds:
         helper.addIntfMode( lane1, speed )
         helper.inactiveIxn( lane1, speed, ( lane2, lane3, lane4 ) )
      if self.groupLinkModes.mode50GbpsFull:
         helper.addIntfMode( lane1, s50g2 )
         helper.addIntfMode( lane3, s50g2 )
         helper.inactiveIxn( lane1, s50g2, ( lane2, lane4 ) )
         helper.inactiveIxn( lane3, s50g2, ( lane4, ) )
         helper.speedLimitation( lane1, s50g2, { lane3: ( s50g2, ) } )
         helper.primaryIntfSpeedRequirement( lane3, lane1, s50g2 )
      supportedOneLaneSpeeds = []
      if self.groupLinkModes.mode25GbpsFull:
         supportedOneLaneSpeeds.append( s25g )
      if self.groupLinkModes.mode10GbpsFull:
         supportedOneLaneSpeeds.append( s10g )
      for speed in supportedOneLaneSpeeds:
         for intf in allIntfs:
            helper.addIntfMode( intf, speed )
         for intf in allIntfs[ 1 : ]:
            helper.primaryIntfSpeedRequirement( intf, lane1, speed )
         if len( supportedOneLaneSpeeds ) > 1:
            # Weird conditional. For QSFP28s, since all the ones we have are
            # connected to Falcon-cores, /1 at a 1-lane speed limits the other three
            # to the same speed, so the template describes as such. For QSFP+, when
            # /1 is at 10G, the others are naturally also at 10G. However, because
            # /2-4 in this case can only ever be 10G, they are not "limited" to it.
            # I made the call in the past to omit the speed-limitation line here,
            # and this conditional is accounting for that.
            helper.speedLimitation( lane1, speed,
                                    { intf: ( speed, ) for intf in allIntfs[ 1: ] } )
      # As mentioned earlier, only lane1 needs an indicator that says its <10G
      # interactions are identical to its 10G interactions.
      helper.intfData( lane1 ).includesSlowerSpeeds10gIs( True )
      return helper.model

class QsfpMixedRate( IntfInteractionReference ):
   """
   Provides reference model for single QSFP port that supports mixed rates (e.g.
   mix of 25G and 10G across the four interfaces)

   Options:
    * 10G/25G/40G/50G-2/100G-4

   Note
   ----
   Options can be enabled/disabled to represent QSFP+ or QSFP28
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.lane1 = serdesDomain.memberIntfs[ "lane1" ]
      self.lane2 = serdesDomain.memberIntfs[ "lane2" ]
      self.lane3 = serdesDomain.memberIntfs[ "lane3" ]
      self.lane4 = serdesDomain.memberIntfs[ "lane4" ]

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      allIntfs = [ self.lane1, self.lane2, self.lane3, self.lane4 ]
      lane1, lane2, lane3, lane4 = allIntfs
      supportedFourLaneSpeeds = []
      if self.groupLinkModes.mode100GbpsFull:
         supportedFourLaneSpeeds.append( s100g4 )
      if self.groupLinkModes.mode40GbpsFull:
         supportedFourLaneSpeeds.append( s40g )
      for speed in supportedFourLaneSpeeds:
         helper.addIntfMode( lane1, speed )
         helper.inactiveIxn( lane1, speed, ( lane2, lane3, lane4 ) )
      # Main distinction between this template and BasicQsfp is that this template
      # allows mixing non-four-lane speeds across the interfaces.
      if self.groupLinkModes.mode50GbpsFull:
         helper.addIntfMode( lane1, s50g2 )
         helper.addIntfMode( lane3, s50g2 )
         helper.inactiveIxn( lane1, s50g2, ( lane2, ) )
         helper.inactiveIxn( lane3, s50g2, ( lane4, ) )
         helper.speedRequirement( lane3, s50g2, { lane1: ( s50g2, s25g, s10g ) } )
      supportedOneLaneSpeeds = []
      if self.groupLinkModes.mode25GbpsFull:
         supportedOneLaneSpeeds.append( s25g )
      if self.groupLinkModes.mode10GbpsFull:
         supportedOneLaneSpeeds.append( s10g )
      for speed in supportedOneLaneSpeeds:
         for intf in allIntfs:
            helper.addIntfMode( intf, speed )
         helper.speedRequirement( lane2, speed, { lane1: ( s25g, s10g ) } )
         helper.speedRequirement( lane3, speed, { lane1: ( s50g2, s25g, s10g ) } )
         helper.speedRequirement( lane4, speed, { lane1: ( s50g2, s25g, s10g ),
                                                  lane3: ( s25g, s10g ) } )
      helper.intfData( lane1 ).includesSlowerSpeeds10gIs( True )
      return helper.model

class Qsfp40gOnly( IntfInteractionReference ):
   """
   Provides reference model for a single 40G-only QSFP port. This QSFP port has
   no associated subordinate interfaces (i.e. /2-4 don't exist).
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      # There was a point where 40G-only ports did not include the /1 in the intfId.
      # For more recent products, we include the /1.
      lane = "lane1"
      if lane not in serdesDomain.memberIntfs:
         lane = "lane0"
         assert lane in serdesDomain.memberIntfs
      self.port = serdesDomain.memberIntfs[ lane ]

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      helper.addIntfMode( self.port, s40g )
      return helper.model

class Sesto2Pair( IntfInteractionReference ):
   """
   Provides reference model for QSFP port pair connected to a Sesto2
   gearbox
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.master1 = serdesDomain.memberIntfs[ "master1" ]
      self.master2 = serdesDomain.memberIntfs[ "master2" ]
      self.master3 = serdesDomain.memberIntfs[ "master3" ]
      self.master4 = serdesDomain.memberIntfs[ "master4" ]
      self.slave1 = serdesDomain.memberIntfs[ "slave1" ]

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      m1, m2, m3, m4 = [ self.master1, self.master2, self.master3, self.master4 ] 
      s1 = self.slave1
      if self.groupLinkModes.mode100GbpsFull:
         # 100G-4 interactions:
         # * Applied on M/1; M/2-4,S/1 become inactive
         helper.addIntfMode( m1, s100g4 )
         helper.inactiveIxn( m1, s100g4, ( m2, m3, m4, s1 ) )
      if self.groupLinkModes.mode50GbpsFull:
         # 50G-2 interactions:
         # * Applied on M/1,M/3; M/2,M/4,S/1 become inactive
         # * M/3 is limited to 50G
         for intf in ( m1, m3 ):
            helper.addIntfMode( intf, s50g2 )
         helper.inactiveIxn( m1, s50g2, ( m2, m4, s1 ) )
         helper.inactiveIxn( m3, s50g2, ( m4, ) )
         helper.speedLimitation( m1, s50g2, { m3: ( s50g2, ) } )
         helper.primaryIntfSpeedRequirement( m3, m1, s50g2 )
      if self.groupLinkModes.mode40GbpsFull:
         # 40G interactions:
         # * Applied on M/1,S/1; M/2-4 become inactive
         for intf in ( m1, s1 ):
            helper.addIntfMode( intf, s40g )
         helper.inactiveIxn( m1, s40g, ( m2, m3, m4 ) )
         helper.primaryIntfSpeedRequirement( s1, m1, s40g )
      # 10G/25G interactions are identical, so group them together
      supportedOneLaneSpeeds = []
      if self.groupLinkModes.mode25GbpsFull:
         supportedOneLaneSpeeds.append( s25g )
      if self.groupLinkModes.mode10GbpsFull:
         supportedOneLaneSpeeds.append( s10g )
      for speed in supportedOneLaneSpeeds:
         # 10G/25G interactions:
         # * Applies on M/1-4; S/1 becomes inactive
         # * M/2-4 require M/1 to be 10G/25G (tightly coupled)
         for intf in ( m1, m2, m3, m4 ):
            helper.addIntfMode( intf, speed )
         helper.inactiveIxn( m1, speed, ( s1, ) )
         helper.speedLimitation( m1, speed, { sub: ( speed, )
                                              for sub in ( m2, m3, m4 ) } )
         for sub in ( m2, m3, m4 ):
            helper.primaryIntfSpeedRequirement( sub, m1, speed )

      return helper.model

