# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#----------------------------------------------------------------------
# This module implements:
#   - IntfRangeMatcher (interface range parsing)
#   - IntfType base class (must be derived from for an intf type to
#            support interface ranges)
#   - AutoIntfType class (helper for implementing IntfType)
#----------------------------------------------------------------------

import re
import sys
from collections import namedtuple

import Arnet
import CliMatcher
import MultiRangeRule
from CliCommon import GuardError
from CliParser import subLowerBound, subUpperBound
from Tracing import Handle, t0

__defaultTraceHandle__ = Handle( 'IntfRange' )

allIntfTypes_ = {}
allIntfTypeGuards_ = {}
intfTypesWithIpSupport = {}
intfTypesWithRoutingProtoSupport = {}
allGroupTypes_ = {}
subintfSupportedGuard = []

IntfOptions = namedtuple( "IntfOptions",
                          [ 'ipSupport', 'routingProtoSupport' ] )

class IntfType( object ):
   '''Each interface type (such as Ethernet, Management, Port-Channel,
   Vlan, etc) must implement this class in order to participate in
   interface ranges.  The key operation is to tell the intfRange
   infrastructure (e.g. IntfRangeRule) what interfaces are available,
   and to fetch an IntfCli.Intf-compatible object given a name.  See
   AutoIntfType below for the most convenient implementation, and
   for example usage, see EthIntfCli.py (search for AutoIntfType).

   tagLong is something like "Ethernet" and tagShort is something
   like "Et".  tagLong is used in parsing, and tagShort is used to
   generate the collapsed string representation "Et2-5,10-14".

   helpText is a string for the type's help description.

   helpDesc is a list of strings for the number part of the interface,
   each being a help description for one number in the name. For example,
   for modular interfaces in the format of 'slot/port', you can pass
   helpDesc=[ 'Slot number', 'Port number' ].

   After constructing one of these, you'll generally want to call
   "IntfRange.registerIntfType".'''

   def __init__( self, tagLong, tagShort, helpText, helpDesc=None,
                 subSupported=False, subIntfGuard=None, dynamicRange=False ):
      self.tagLong = tagLong
      self.tagShort = tagShort
      self.helpText = helpText
      if helpDesc is None:
         self.helpDesc = [ '%s number' % helpText ]
      else:
         self.helpDesc = helpDesc
      self.subHelpDesc = "Subinterface number"
      self.subSupported_ = subSupported
      self.subIntfGuard_ = subIntfGuard
      # cliIntfClazz is a class object (or factory function) that
      # can be called with a name and the mode, and returns an instance of
      # IntfCli.Intf. It should be initialized before fetCliIntf() is called.
      self.cliIntfClazz_ = None
      # .cliSubIntfClazz is a class object (or factory function) that
      # this IntfType will use to instantiate the sub-interface's
      # corresponding instance. We separate this from the cliIntfClazz
      # because we (for most cases) want support different commands
      # for the sub-interface, than the real interface. If the
      # IntfType chooses to not specify the cliSubIntfClazz, then we
      # will use cliIntfClazz instead.
      self.cliSubIntfClazz_ = None
      self.dynamicRange_ = dynamicRange

   def close( self ):
      del allIntfTypes_[ self.tagLong ]
      del allIntfTypeGuards_[ self.tagLong ]

   def invalidate( self ):
      '''Called once before calling endpoints a lot of times.
      Refreshes or invalidates any intermediate data structures
      that "endpoints" might have built up.  Basically, call this
      once to make sure that any changes to the set of available
      interfaces are seen.'''
      pass

   def endpoints( self, partialPath, hasSub=False, mode=None ):
      '''This function takes a partial path (as a sequence of
      integers).  A "partial path" for "Ethernet1/4" would be (1,4).
      This function returns the following pair of values:

          first: what is the first legal value
          last: what is the last legal value

      "first" is generally either 0 or 1.  Note that "last" should be
      limited to physical interfaces that exist, but not for virtual
      interfaces.  For virtual interfaces, "last" should return the
      last possible interface that may be created.  An extra "create"
      keyword is required to trigger any actual interface creation.
      Note that these are inclusive, unlike the python range()
      primitive.

      Instead of returning a pair of values, this function may also
      return one of these special values:

          IntfRange.TERMINAL: The "partial path" that was passed in
                              is actually a full path; no further
                              slash is legal.

          IntfRange.INVALID_PREFIX: The "partial path" that was passed
                                    in is invalid.  It is not in itself
                                    a valid interface, nor is it a prefix
                                    of any valid interface.  It is either
                                    out of range, too deep, or there is
                                    simply nothing plugged in there.

      For example, consider a modular system with eight slots where
      linecard 3 is inserted with 12 ports, and 3/7 is connected to
      a port expander with 48 ports.  Thus:

             endpoints( () )          returns (1, 8)
             endpoints( (2,) )        returns IntfRange.INVALID_PREFIX
                                      (because nothing is in slot 2)
             endpoints( (3,) )        returns (1, 12)
             endpoints( (3,6) )       returns IntfRange.TERMINAL
             endpoints( (3,7) )       returns (1, 48)
             endpoints( (3,7,49) )    returns IntfRange.INVALID_PREFIX
             endpoints( (3,7,48) )    returns IntfRange.TERMINAL
             endpoints( (3,7,48,1) )  returns IntfRange.INVALID_PREFIX

      The "mode" parameter is a Cli mode, which is needed by some
      implementations of IntfType to get to the sysdb root, which is
      needed to figure out which interfaces are actually available
      from here.
      '''

      raise NotImplementedError

   def cliIntfClazzIs( self, cliIntfClazz ):
      self.cliIntfClazz_ = cliIntfClazz

   def cliSubIntfClazzIs( self, cliSubIntfClazz ):
      self.cliSubIntfClazz_ = cliSubIntfClazz

   def getCliIntf( self, mode, name ):
      '''Take a string like "Ethernet3/7" and return an object like
      EthPhyIntf("Ethernet3/7"), i.e., something that derives from
      IntfCli.Intf. If this supports sub-interface and the name
      contains a dot, and it has a separate cliSubIntfClazz_, it will
      use that one instead.'''

      if self.subSupported_ and '.' in name and self.cliSubIntfClazz_:
         return self.cliSubIntfClazz_( name, mode )
      else:
         return self.cliIntfClazz_( name, mode )

def registerIntfType( intfType, intfOptions ): # pylint: disable-msg=W0621
   tag = intfType.tagLong
   assert tag not in allIntfTypes_
   if intfOptions.ipSupport:
      intfTypesWithIpSupport[ tag ] = intfType
   if intfOptions.routingProtoSupport:
      intfTypesWithRoutingProtoSupport[ tag ] = intfType
   allIntfTypes_[ tag ] = intfType
   allIntfTypeGuards_[ tag ] = None

def registerIntfTypeGuard( intfType, guard ):
   allIntfTypeGuards_[ intfType.tagLong ] = guard

def registerGroupType( groupType ):
   tag = groupType.tagLong
   assert tag not in allGroupTypes_
   allGroupTypes_[ tag ] = groupType

class GenericRangeIntfType( IntfType ):
   '''This class provides a type with one single component in the path
   (most virtual interfaces). It takes a range function similar to the
   GenericRangeType.'''

   def __init__( self, rangeFunc, tagLong, tagShort, helpText, helpdesc=None,
                 subSupported=False, subIntfGuard=None ):
      IntfType.__init__( self, tagLong, tagShort, helpText, helpdesc, subSupported,
                         subIntfGuard )
      self.rangeFunc_ = rangeFunc

   def endpoints( self, partialPath, hasSub=False, mode=None ):
      eps = MultiRangeRule.genericEndpoints( partialPath, self.rangeFunc_ )
      if self.subSupported_:
         if eps.terminal:
            return MultiRangeRule.Endpoint( None, None, subLowerBound,
                                            subUpperBound, terminal=True )
         elif hasSub and len( partialPath ) == 2:
            v0 = int( partialPath[ 0 ] )
            v1 = int( partialPath[ 1 ] )
            if ( MultiRangeRule.genericEndpoints( [ v0 ], self.rangeFunc_ ).terminal
                 and subLowerBound <= v1 <= subUpperBound ):
               return MultiRangeRule.TERMINAL
            else:
               return MultiRangeRule.INVALID_PREFIX

      return eps

class SubRangeInfo( object ):
   '''This class serves to associate a range function with an allowed upper and lower
   bound for allowed subinterface range.'''

   def __init__( self, rangeFunc, subLower=subLowerBound, subUpper=subUpperBound ):
      self.rangeFunc_ = rangeFunc
      self.subLower_ = subLower
      self.subUpper_ = subUpper

   def rangeFunc( self ):
      return self.rangeFunc_

   def subLower( self ):
      return self.subLower_

   def subUpper( self ):
      return self.subUpper_

class MultiSubRangeIntfType( IntfType ):
   '''This class provides a type with one single component in the path
   (most virtual interfaces), but features multiple allowed sub-interface
   ranges, depending on the prefix. It takes a range function for the overall
   allowed base range, and a list of SubRangeInfo objects to determine the
   corresponding allowed subinterface ranges.'''

   def __init__( self, rangeFunc, tagLong, tagShort, helpText, rangeInfo,
                 helpdesc=None,
                 subSupported=False, subIntfGuard=None, dynamicRange=False ):
      IntfType.__init__( self, tagLong, tagShort, helpText, helpdesc, subSupported,
                         subIntfGuard, dynamicRange=dynamicRange )
      self.rangeFunc_ = rangeFunc
      self.rangeInfo_ = rangeInfo

   def endpoints( self, partialPath, hasSub=False, mode=None ):
      if self.dynamicRange_:
         rangeArgs = ( mode, )
      else:
         rangeArgs = ()
      baseEps = MultiRangeRule.genericEndpoints( partialPath, self.rangeFunc_,
                                                 rangeArgs=rangeArgs )
      epsList = []
      for info in self.rangeInfo_:
         eps = MultiRangeRule.genericEndpoints( partialPath, info.rangeFunc() )
         epsList.append( ( eps, info ) )

      if self.subSupported_:
         for eps, info in epsList:
            if eps.terminal:
               return MultiRangeRule.Endpoint( None, None, info.subLower(),
                                               info.subUpper(), terminal=True )
         if hasSub and len( partialPath ) == 2:
            v0 = int( partialPath[ 0 ] )
            v1 = int( partialPath[ 1 ] )
            for eps, info in epsList:
               if ( MultiRangeRule.genericEndpoints( [ v0 ],
                     info.rangeFunc() ).terminal
                     and info.subLower() <= v1 <= info.subUpper() ):
                  return MultiRangeRule.TERMINAL
            return MultiRangeRule.INVALID_PREFIX

      return baseEps

class AutoIntfType( IntfType ):
   '''This class provides IntfRange.IntfType simply by allowing
   anything that actually exists in some collection (typically a sysdb
   interface status collection) to be used in a range.'''
   numberRe_ = re.compile( r'\d+(\/\d+)*$' )

   def __init__( self, collectionFunc,
                 tagLong, tagShort, helpText, helpdesc=None,
                 collectionVersionFunc=None, subSupported=False,
                 subIntfGuard=None, intfSubLowerBound=subLowerBound,
                 intfSubUpperBound=subUpperBound ):
      '''Give me a collection (e.g. an EthIntfStatus collection) and I will
      make those interfaces available as a range.  I need the tag (e.g. "Ethernet")
      because I cannot infer it because the collection might be empty.

      Actually, the "collectionFunc" parameter is not an actual collection, but
      rather a function that returns an unbound collection conntaining interfaces
      that could exist (not necessarily already exist). The collection is used
      to build a grammar tree for parsing interface ranges. If the collection
      changes, it's expected the tree needs to be rebuilt. Since building a tree
      is potentially expensive (and the collection is potentially huge), the
      parser tries to only rebuild the tree until the collection actually changes.
      There are a few cases:
      1. The collection is always the same. This is always true for virtual
      interfaces (port-channel, vlan interfaces, etc).
      2. The collection dynamically changes. Typically the collectionFunc returns
      a reference to a collection in Sysdb. The collectionVersionFunc can be used
      to return a unique version id if the collection has changes since last time 
      the function is called. The parser remembers the previous version, and if
      the new version changes, it rebuilds the tree.
      3. The invalidate() function can also be called manually to force rebuilding
      the tree.
      '''

      IntfType.__init__( self, tagLong, tagShort, helpText, helpdesc, subSupported,
                         subIntfGuard )
      self.collectionFunc_ = collectionFunc
      self.collectionVersionFunc_ = collectionVersionFunc
      self.tree_ = None
      self.treeVersion_ = None
      self.subLowerBound_ = intfSubLowerBound
      self.subUpperBound_ = intfSubUpperBound

   def invalidate( self ):
      # This will force revalidate() to invalidate the tree.
      # Note this function may be called from activity loop, so we
      # do not set self.tree_ to None as it may be in use.
      self.treeVersion_ = None

   def revalidate( self ):
      # Building a tree is expensive, so we try to be smarter in detecting
      # when we need to rebuild it, and only do it when collectionVersionFunc()
      # returns a different version.
      if self.tree_ is not None and self.collectionVersionFunc_ is not None:
         ver = self.collectionVersionFunc_()
         if self.treeVersion_ != ver:
            t0( "revalidate: version changed ", self.treeVersion_, "==>", ver )
            self.tree_ = None
            self.treeVersion_ = ver

   def endpoints( self, partialPath, hasSub=False, mode=None ):
      self.revalidate( )
      if self.tree_ is None:
         self.tree_ = {}
         for k in self.collectionFunc_():
            if not k.startswith( self.tagLong ):
               # This name does not start with our tag (e.g., "Ethernet").
               continue
            k = k[ len( self.tagLong ) : ]
            if not self.numberRe_.match( k ):
               continue
            pp = tuple( int( l ) for l in k.split( "/" ) )
            if self.subSupported_:
               subTerminal = MultiRangeRule.Endpoint( None, None,
                                                      self.subLowerBound_,
                                                      self.subUpperBound_,
                                                      terminal=True )
               epp = self.tree_.get( pp, subTerminal )
               epp.subMin, epp.subMax = self.subLowerBound_, self.subUpperBound_
            else:
               epp = self.tree_.get( pp, MultiRangeRule.TERMINAL )
            # epp could already exist; we need to set terminal to True.
            epp.terminal = True
            self.tree_[ pp ] = epp

            while pp:
               n = pp[ -1 ]
               pp = pp[ : -1 ]
               r = self.tree_.get( pp, MultiRangeRule.Endpoint( n, n ) )
               if r.vmin is None:
                  assert r.terminal
                  self.tree_[ pp ] = MultiRangeRule.Endpoint( n, n, terminal=True )
               else:
                  self.tree_[ pp ] = MultiRangeRule.Endpoint(
                     min( n, r.vmin ), max( n, r.vmax ),
                     terminal=r.terminal )
               self.tree_[ pp ].subMin, self.tree_[ pp ].subMax = r.subMin, r.subMax
      if hasSub and partialPath:
         if not self.subSupported_:
            return MultiRangeRule.INVALID_PREFIX
         pp = tuple( partialPath[ : -1 ] )
         result = self.tree_.get( pp, MultiRangeRule.INVALID_PREFIX )
         if not result.invalid and \
               result.subMin <= partialPath[ -1 ] <= result.subMax:
            return MultiRangeRule.TERMINAL
         return result
      else:
         pp = tuple( partialPath ) # because tuples can index dicts sensibly.
         return self.tree_.get( pp, MultiRangeRule.INVALID_PREFIX )

class IntfRangeMatcher( CliMatcher.Matcher ):
   '''A matcher that matches a range of interfaces.'''
   rangeOpRe = re.compile( r'[-,\$]' )

   def __init__( self, noSingletons=False, explicitIntfTypes=None,
                 earlySingletonReject=False, dollarCompHelpText=None, **kwargs ):
      '''By default, I will accept a single interface and return it as
      a range of size 1.  By passing noSingletons=True, I will not
      regard a singleton as legal.  This is useful when you want to
      use IntfRangeMatcher in a disjunction with IntfCli.IntfMatcher.  By
      default, this will result in grammatical ambiguity as either me
      or IntfCli.IntfMatcher will match "Et1".  By using me in my
      noSingletons mode, I will reject "Et1" and let IntfCli.IntfMatcher
      have it.

      Note, we parse interfaces that may not have existed. In the Cli handler,
      we can use IntfCli.Intf.getAll() to filter nonexistent interfaces.
      '''
      CliMatcher.Matcher.__init__( self, **kwargs )
      self.noSingletons_ = noSingletons
      self.explicitIntfTypes_ = ( explicitIntfTypes
                                  if explicitIntfTypes is not None
                                  else allIntfTypes_ )
      self.earlySingletonReject_ = earlySingletonReject
      self.dollarCompHelpText_ = dollarCompHelpText

   def __str__( self ):
      return "<if-range>"

   def _initState( self, context ):
      return MultiRangeRule.IrrState(
         noSingletons=self.noSingletons_,
         explicitIntfTypes=self.explicitIntfTypes_,
         dollarCompHelpText=self.dollarCompHelpText_ )

   def match( self, mode, context, token ):
      if ( not mode.session_.interactive_ and
           not context.state and
           self.earlySingletonReject_ and
           context.finalToken and
           not self.rangeOpRe.search( token ) ):
         return CliMatcher.noMatch

      firstToken = ( context.state is None )
      s = context.state or self._initState( context )
      assert s.__class__ is MultiRangeRule.IrrState

      r = s.match( mode, token )
      if r is CliMatcher.noMatch:
         return r
      if firstToken and s.type_:
         guardFn = allIntfTypeGuards_.get( s.type_.tagLong )
         if guardFn and mode.session_.guardsEnabled():
            guardCode = guardFn( mode, token )
            if guardCode:
               raise GuardError( guardCode )
      context.state = s
      return r

   def completions( self, mode, context, token ):
      firstToken = ( context.state is None )
      s = context.state or self._initState( context )
      assert s.__class__ is MultiRangeRule.IrrState
      compl = s.completions( mode, token )
      if firstToken and mode.session_.guardsEnabled():
         for c in compl:
            guardFn = allIntfTypeGuards_.get( c.name )
            if guardFn:
               c.guardCode = guardFn( mode, token )
      return compl

IntfList = MultiRangeRule.IntfList

intfTagRe_ = re.compile( r'([^\d]+)(\d.*)' )

def intfTypeFromName( intfName ):
   # from interface name get the registered type
   # both long and short names are OK
   # returns a tuple of ( intfType, intfNumber )
   m = intfTagRe_.match( intfName )
   if m:
      tag = m.group( 1 )
      if tag in allIntfTypes_:
         return allIntfTypes_[ tag ], m.group( 2 )
      elif tag in allGroupTypes_:
         return allGroupTypes_[ tag ], m.group( 2 )
      else:
         tag = tag.lower()
         # see if it's a valid short tag
         for t in allIntfTypes_.itervalues():
            if t.tagShort.lower() == tag:
               return t, m.group( 2 )
   # cannot match type, just return the same string
   return None, intfName

def _rangeStrLimit( tag, intfRange, strLimit ):
   # Note this is a best-effort: if the limit is too low
   # that we cannot fit everything, it will just go over.
   if len( intfRange ) <= strLimit:
      return [ intfRange ]
   number = intfRange[ len( tag ) : ].split( ',' )
   strLimit -= len( tag )
   assert strLimit > 0
   rangeSet = []
   rangeStr = ""
   for n in number:
      if rangeStr:
         if len( rangeStr ) + len( n ) < strLimit: # need space for comma
            rangeStr += ',' + n
         else:
            rangeSet.append( tag + rangeStr )
            rangeStr = n
      else:
         rangeStr = n
   rangeSet.append( tag + rangeStr )
   return rangeSet

def _splitIntfsIntoConsecutiveGroups( intfType, intfNumberList,
                                      verifyIntfHoles ):
   # Helper function to split the given interface list into groups of consecutive 
   # interfaces (no holes inside each group).

   if not verifyIntfHoles:
      # No need to validate presence of holes in the intfs list. Only one group.
      return [ intfNumberList ]

   assert hasattr( intfType, 'collectionFunc_' ), \
          "verifyIntfHoles requires AutoIntfType based interfaces."

   coll = Arnet.sortIntf( intfType.collectionFunc_() )
   sortedIntfs = Arnet.sortIntf( intfNumberList )

   # 'groups' is a list of groups of interfaces which are consecutive based on
   # their backing collection and their leading component.
   # 'consec' is the group of consecutive interfaces which we are currently adding
   # to. consec will be appended to groups once the group is complete.
   groups = []
   consec = []
   intfListCursor = 0
   collCursor = 0

   def leadComp( intfNumber_ ):
      return int( intfNumber_.split( '/' )[ 0 ] )

   # This finds intfs in coll, collecting consecutive ones into lists.
   while intfListCursor < len( sortedIntfs ):
      intf = sortedIntfs[ intfListCursor ]
      _, collIntf = intfTypeFromName( coll[ collCursor ] )
      collCursor += 1
      if collIntf != intf:
         if consec:
            # Reset group, we lost consecutiveness
            groups.append( consec )
            consec = []
         continue

      # Found in the collection, can move on to next intf in next iter.
      intfListCursor += 1

      # No pred to check, start the group.
      if not consec:
         consec.append( intf )
         continue

      maybePred = consec[ -1 ]
      # They have the same leading component, or leading component + 1, and
      # they're consecutive in the collection. assume they're truly consecutive.
      if ( leadComp( maybePred ) == leadComp( intf ) or
           leadComp( maybePred ) + 1 == leadComp( intf ) ):
         consec.append( intf )
      # Otherwise, they're not consecutive ( e.g. linecard3 to linecard5 ) put in
      # new group.
      else:
         groups.append( consec )
         consec = [ intf ]

   # Add the final consecutive intf group.
   groups.append( consec )
   return groups

def _convertIntfList( intfs, canonical, shortName=False,
                      strLimit=sys.maxint, noHoleRange=False,
                      verifyIntfHoles=False, mode=None ):

   assert ( ( not noHoleRange or canonical ) and
            ( not verifyIntfHoles or noHoleRange ) ), \
          "unexpected combination of arguments"

   rangeList = []
   intfSet = {} # type -> sequence

   for i in intfs:
      intfType, intfNumber = intfTypeFromName( str( i ) )
      if intfType is None:
         # unknown interface, just use the original string
         rangeList.append( str( i ) )
      else:
         if intfType in intfSet:
            intfSet[ intfType ].append( intfNumber )
         else:
            intfSet[ intfType ] = [ intfNumber ]

   # Convert each list of interfaces of each type to a range string
   for intfType, intfNumberList in intfSet.iteritems():
      numericRangeStrings = []
      tagLen = len( intfType.tagShort )
      # Split them into groups of consecutive intfs if necessary
      for group in _splitIntfsIntoConsecutiveGroups( intfType, intfNumberList,
                                                     verifyIntfHoles ):
         buf = intfType.tagShort + ','.join( group )
         doParse = MultiRangeRule.DoParse( mode, buf + ' ', False, [ intfType ],
                                           noHoleRange=noHoleRange )
         if doParse.errorMsg:
            rangeList.append( buf )
         else:
            numericRangeStrings.append( str( doParse.intfList )[ tagLen: ] )
            if not canonical:
               # return the individual list names (e.g., 'Et1','Et2','Et3','Et6')
               rangeList += doParse.intfList.intfNames( shortName=shortName )

      if canonical and numericRangeStrings:
         tag = intfType.tagShort if shortName else intfType.tagLong
         # Prepend the requested tag (long or short) and join each range by a comma
         rangeString = tag + ",".join( numericRangeStrings )
         # Resize to within strLimit and add it to the list of ranges.
         rangeList += _rangeStrLimit( tag, rangeString, strLimit )

   return sorted( rangeList )


def intfListToCanonical( intfs, strLimit=sys.maxint, noHoleRange=False,
                         verifyIntfHoles=False, shortName=True ):
   # convert a list of valid interfaces to a list of interface ranges.
   # return: a list of strings, each being a type of interface ranges.
   #
   # For example:
   # intfListToRange( [ 'Et2', 'Et1', 'Port-Channel3', 'Port-Channel5' ] )
   # will return: [ 'Et1-2', 'Po3,5' ]
   # both short and long names can be used as interface name
   #
   # If strLimit is specified, a range string over the limit is broken up
   # into sub ranges with each length lower than the limit. For example,
   # 'Et1,3,5,7,9,11' may be broken up into 'Et1,3,4' and 'Et7,9,11' if
   # strLimit is 8.
   #
   # if noHoleRange is set to True, a range string is calculated assuming
   # interfaces are consecutive.
   #
   # if verifyIntfHoles and noHoleRange are set to True, the interfaces passed in
   # must be based on AutoIntfTypes, and everything consecutive in the
   # Arnet.sortIntf'd collection is assumed to be consecutive. e.g. on Sperry
   # linecards, Et1/1/1 and Et1/1/3 are consecutive in their collection and Et1/1/2
   # does not exist, but IntfList considers there to be a hole. The flag will remove
   # the hole and return the range Et1/1/1-1/1/3.
   return _convertIntfList( intfs, True,
                            shortName=shortName,
                            strLimit=strLimit,
                            noHoleRange=noHoleRange,
                            verifyIntfHoles=verifyIntfHoles )


def intfListFromCanonical( intfs, shortName=False, mode=None ):
   # this is the reverse of intfListToCanonical:
   # passing [ 'Et1-2', 'po3-5' ] would return
   # [ 'Ethernet1', 'Ethernet2', 'Port-Channel3', 'Port-Channel4',
   #   'Port-Channel5' ]
   # Note, by default long names are returned (unless shortName=True is
   # specified), and there is no strLimit applicable.
   return _convertIntfList( intfs, False,
                            shortName=shortName,
                            strLimit=sys.maxint,
                            mode=mode )
