#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
@ Summary:
Variables used for building EOS images
Previously defined in EosImage/Makefile.am

@ Vars List:
- BASE PACKAGES
  Base packages to install into EOS. These packages aim to aggregate dependencies of
  RPMs that should be installed in EOS for platform-independent feature support.

- PRODUCT PACKAGES
  Product packages ( platform-dependent )
  These RPMs come from various SKU Family Packages and are aggregators of
  RPMs needed to support various SKUs belonging to that SKU Family.

- ASIC PACKAGES
  Packages containing support for ASICs and related features

- BAD PACKAGES
  Bad packages is a list of packages whose immediate dependencies must not
  appear in the image. Note that it is added to in the SHIPPING case.

- BLOCKED PACKAGES
  A list of package name regex that must never appear in the image

- ALLOWED PACKAGES
  A list of package name regex that we allow in the SWI, used to
  override our badpackage and blocked checks.

- BADPLUGINS
  A list of plugin types, where there must be no
  %Plugin directory in /usr/lib*/python*/site-packages (or it must be empty)

- YUM REPOS
  A default list of repos to be enabled or disabled while using the Yum command
  to build images.
'''

from __future__ import absolute_import, division, print_function

# BASE PACKAGES
basePkgs = [ 'Eos' ]

# PRODUCT PACKAGES without future
productPkgsNoFuture = [
   'Baremetal-EosImage',
   'Bodega-EosImage',
   'Hollywood-EosImage',
   'Mendocino-EosImage',
   'NorCal-EosImage',
   'VEosSku-EosImage',
]

# If this image is being shipped to customers, then we exclude all support for
# products that have not been released yet ( e.g. we exclude Yosemite-future )
# For internal builds, we include support for all products into the swi.

# BAD PACKAGES
badPkgs = [ 'Eos-devel', ]

# map of packages with future to their future
pkgToFutureMap = {
   'basePkgs' : {
      'Eos' : 'Eos-future'
   },
   'productPkgs' : {
      'Alameda-EosImage' : 'Alameda-future',
      'BfnFixed-EosImage' : 'BfnFixed-future',
      'DenaliModular-EosImage' : 'DenaliModular-future',
      'Glacier-EosImage' : 'Glacier-future',
      'JerichoPlusFixed-EosImage' : 'JerichoPlusFixed-future',
      'Jericho2Fixed-EosImage' : 'Jericho2Fixed-future',
      'MakoBfnFixed-EosImage' : 'MakoBfnFixed-future',
      'MakoFixed-EosImage' : 'MakoFixed-future',
      'MakoJericho2Fixed-EosImage' : 'MakoJericho2Fixed-future',
      'PtReyes-EosImage' : 'PtReyes-future',
      'SaltLakeFixed-EosImage' : 'SaltLakeFixed-future',
      'Sequoia-EosImage' : 'Sequoia-future',
      'Tomahawk2Fixed-EosImage' : 'Tomahawk2Fixed-future',
      'Tomahawk3Fixed-EosImage' : 'Tomahawk3Fixed-future',
      'Tundra-EosImage' : 'Tundra-future',
      'Yosemite-EosImage' : 'Yosemite-future',
   },
   'asicPkgs' : {
      'BfnEosImage-full' : 'BfnEosImage-future',
      'SandEosImage-full' : 'SandEosImage-future',
      'StrataEosImage-full' : 'StrataEosImage-future',
      'XpEosImage-full' : 'XpEosImage-future',
   },
   'miscPkgs' : {
      'StrataSdkShim-Full' : 'StrataSdkShim-future'
   }
}

def getBasePkgs( shipping ):
   if not shipping:
      return basePkgs + pkgToFutureMap[ 'basePkgs' ].values()
   else:
      return basePkgs

def getProductPkgs( shipping ):
   if not shipping:
      return productPkgsNoFuture + pkgToFutureMap[ 'productPkgs' ].keys() + \
             pkgToFutureMap[ 'productPkgs' ].values()
   else:
      return productPkgsNoFuture + pkgToFutureMap[ 'productPkgs' ].keys()

def getAsicPkgs( shipping ):
   if not shipping:
      return pkgToFutureMap[ 'asicPkgs' ].keys() + \
             pkgToFutureMap[ 'asicPkgs' ].values() 
   else:
      return pkgToFutureMap[ 'asicPkgs' ].keys()

def getRootfsPkgs( shipping ):
   rootfsPkgs = []
   rootfsPkgs += getBasePkgs( shipping )
   rootfsPkgs += getProductPkgs( shipping )
   rootfsPkgs += getAsicPkgs( shipping )
   return rootfsPkgs

def getPkgsWithFuture():
   pkgs = []
   for _, v in pkgToFutureMap.items():
      pkgs += v.keys()
   return pkgs

def getAllFuturePkgs():
   pkgs = []
   for _, v in pkgToFutureMap.items():
      pkgs += v.values()
   return pkgs

def getFuturePkgs( pkgWithFuture=None ):
   if pkgWithFuture:
      for _, v in pkgToFutureMap.items():
         if pkgWithFuture in v:
            return v[ pkgWithFuture ]
      return []
   else:
      return getAllFuturePkgs()

def getBadPkgs( shipping ):
   if shipping:
      return badPkgs + getAllFuturePkgs()
   else:
      return badPkgs

# BLOCKED PACKAGES
# pylint: disable-msg=W1401
blockedPkgs = [
   ".*-devel",
   ".*-no-swi",
   ".*sim",
   ".*[Tt]est.*",
   ".*-etba",
   ".*Dut",
   "Aboot.*",
   "Art",
   "Artools",
   "GoExternal",
   "Hello",
   "World",
   "ProjectReport.*",
   "Winston.*"
   "gotacc",
   "goworkspace",
   "libstdc\+\+8",
   "tacc",
]

# We still want to include Dos.*-pysrc in non-shipping images
blockedPkgsShipping = [
   ".*-pysrc",
   "Dos.*-src",
]

blockedPkgsNonShipping = [
   "libtac-pysrc",
   "libtacutils-pysrc",
   "TaccPyUtils-pysrc",
]

def getBlockedPkgs( shipping ):
   if shipping:
      return blockedPkgs + blockedPkgsShipping
   else:
      return blockedPkgs + blockedPkgsNonShipping

# ALLOWED PACKAGES
allowedPkgs = [
   # Only allow specific aboot RPMs to go into the SWI
   'Aboot-utils',
   'AbootEos-cli',
   'AbootEos-swi',
   'AbootEos-toggles',
   'AbootGpl-swi',
   # Allow this as it's needed by yum in CentOS
   'yum-plugin-fastestmirror',
   'mtest',
]

def getAllowedPkgs():
   return allowedPkgs

# BADPLUGINS
badPlugins = [
   'ArosTest',
   'ArosTestUi',
   'CliSaveTest',
   'EbraTestBridge',
   'EosTortureTest',
   'Riota',
   'Scapy',
   'SilvercreekTest',
   'Simulation',
   'Skegness',
   'TestRiota',
]

def getBadPlugins():
   return badPlugins

def getDefaultYumRepos( mock=False, fmtArgs=False ):
   enableRepos = []
   disableRepos = []

   if not mock:
      enableRepos += [
         'local',
      ]
      disableRepos += [
         'ToolsV2',
      ]
   else: # MOCK
      enableRepos += [
         'Abuild*',
      ]
      disableRepos += [
         'local',
         'prebuilt',
         'ToolsV2',
      ]

   if fmtArgs:
      enableRepos = [ '--enablerepo=' + repo for repo in enableRepos ]
      disableRepos = [ '--disablerepo=' + repo for repo in disableRepos ]

   return enableRepos, disableRepos
