#!/usr/bin/env python
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import MlagMountHelper
t0 = Tracing.trace0
t2 = Tracing.trace2

from GenericReactor import GenericReactor

mlagStatusReactor = None
pluginCtx = None
peerIntfStatusDir = None
mlagConfig = None
mlagStatus = None
mlagProtoStatus = None
mlagProtoStatusReactor = None
mlagIntfStatusReactor = None
mlagIntfStatusSnapshot = {}
igmpSnoopingInputDir = None
mlagTranslateStatusSm = None
peerDeletedTimer = None 
tunnel = False
timerRunning = False

currentMode = None
supportedModes = ( 'legacy', 'mlagSync', 'directSync' )

# MLAG VERSION
mlagSyncSupportVersion = 7
# Code Disabled until the entire Protocol state of IgmpSnooping is exported
directSyncSupportVersion = 9

def Plugin( ctx ):
   global pluginCtx
   global currentMode
   pluginCtx = ctx
   mg = ctx.entityManager().mountGroup()
   global mlagStatus
   # Mount mlag/status, Mlag::Status and its dependent paths
   mlagStatus = MlagMountHelper.mountMlagStatus( mg )
   global peerIntfStatusDir
   peerIntfStatusDir = mg.mount( 'interface/status/eth/peer',
                                 'Interface::PeerIntfStatusDir' )
   global mlagProtoStatus
   mlagProtoStatus = mg.mount( 'mlag/proto', 'Mlag::ProtoStatus', "rO" )

   global igmpSnoopingInputDir 
   igmpSnoopingInputDir = mg.mount( 'bridging/igmpsnooping/input/mlag', 
                                    'Bridging::IgmpSnooping::SyncStatus', 'r' )
   global mlagConfig
   # Mount mlag/config, Mlag::Config and its dependent paths
   mlagConfig = MlagMountHelper.mountMlagConfig( mg )

   currentMode = DisabledMode()

   def finishMounts():
      global mlagStatusReactor
      mlagStatusReactor = GenericReactor(
         mlagStatus, ['mlagState'], handleMlagState, callBackNow=True )
      global mlagIntfStatusReactor
      mlagIntfStatusReactor = GenericReactor(
         mlagStatus, ['intfStatus'], handleMlagIntfStatus, callBackNow=True )
      global mlagProtoStatusReactor
      mlagProtoStatusReactor = GenericReactor(
            mlagProtoStatus, ['mlagVersion'], handleMlagVersion, callBackNow=True )
   mg.close( finishMounts )
   
def createPeerDeletedTimer():
   global peerDeletedTimer
   global timerRunning

   t2( "Create peer deleted timer" )

   peerDeletedTimer = Tac.ClockNotifiee( handlePeerDeleted )
   peerDeletedTimer.timeMin = Tac.endOfTime
   timerRunning = False

def startPeerDeletedTimer():
   global timerRunning

   if not peerDeletedTimer:
      createPeerDeletedTimer()

   if not timerRunning:
      t2( "Start peer deleted timer" )
      peerDeletedTimer.timeMin = Tac.now() + 180
      timerRunning = True
   else:
      t2( "Timer is already running - not starting" )

def cancelPeerDeletedTimer():
   global timerRunning
   t2( "Cancel peer deleted timer" )

   if peerDeletedTimer:
      peerDeletedTimer.timeMin = Tac.endOfTime
      timerRunning = False

def handlePeerDeleted():
   setMode( DisabledMode )
   cancelPeerDeletedTimer()   

def getOperationalMode( mProtoStatus ):
   if mProtoStatus :
      if mProtoStatus.mlagVersion >= directSyncSupportVersion:
         return 'directSync'
      elif mProtoStatus.mlagVersion >= mlagSyncSupportVersion:
         return 'mlagSync'
      else:
         return 'legacy'

   return 'mlagSync'

def handleMlagVersion( n ):
   processOperationalMode( mlagStatus, n.notifier() )

def handleMlagState( n ):
   processOperationalMode( n.notifier(), mlagProtoStatus )

def processOperationalMode( mStatus, mProtoStatus ):
   state = mStatus.mlagState
   operationalMode = getOperationalMode( mProtoStatus )
   t0( 'Current operational mode is ' + operationalMode )

   state = mlagStatus.mlagState
   if state in ( 'primary', 'secondary', 'inactive' ):
      assert operationalMode in supportedModes
      if operationalMode == 'legacy':
         setMode( LegacyMode )
      elif operationalMode == 'mlagSync':
         setMode( MlagSync )
      elif operationalMode == 'directSync':
         setMode( DirectAgentSync )
      else:
         assert False, "Missing mode specification"
   if state in ('primary', 'secondary' ):
      t0( "Processing Active state" )
      currentMode.handleActive( state )

   elif state in ( 'inactive', 'disabling' ):
      t0( "Processing an Inactive state" )
      currentMode.handleInactive()
   else:
      if not timerRunning:
         handlePeerDeleted()

def handleMlagIntfStatus( n, intf=None ):
   status = n.notifier()
   if not intf:
      mlagIntfStatusSnapshot.clear()
      for ( k, v ) in status.intfStatus.iteritems():
         mlagIntfStatusSnapshot[k] = v.status
      return

   intfStatus = status.intfStatus.get( intf )
   if not intfStatus:
      mlagIntfStatusSnapshot.pop( intf, None )
      return

   # Update the snapshot
   statusWas = mlagIntfStatusSnapshot.get( intf )
   mlagIntfStatusSnapshot[ intf ] = intfStatus.status

   if statusWas == None:
      return

   # If an mlag interface's state became inactive from active because
   # all the local links are down, tell IgmpSnooping to pretend that
   # the port-channel is down.
   if intfStatus.status == 'linkInactive' and \
          statusWas == 'linkActive' \
          and intfStatus.peerLinkStatus == 'linkUp':
      pluginCtx.handleDisabledIntf( intf )


# Mlag Igmp Snooping operational mode

def setMode( newMode ):
   global currentMode
   if getMode() is newMode :
      return
   t2( 'Leaving mode %s ' % currentMode )
   currentMode.leave()
   currentMode = newMode()
   t2( 'Entering mode %s ' % currentMode )
   currentMode.enter()

def getMode():
   return type( currentMode )


class BaseMode( object ):
   def enter( self ):
      pass
   
   def leave( self ):
      pass

   def handleActive( self, state ):
      pass

   def handleInactive( self ):
      pass


# MlagState is either 'disabled'
class DisabledMode( BaseMode ):
   pass

# MlagState is 'primary' or 'secondary' or 'inactive'
class ConfiguredBaseMode( BaseMode ):
   def __init__( self ):
      self.stateControl = None
   def enableStateControl( self ):
      if not self.stateControl:
         self.stateControl = Tac.newInstance( 
                                       "IgmpSnooping::MlagIntfStateControl",
                                       "mlag", mlagStatus )
         pluginCtx.intfStateControlIs( self.stateControl )
   def disableStateControl( self ):
      if self.stateControl:
         pluginCtx.intfStateControlDel( "mlag" )
         self.stateControl = None
   def handleActive( self, state ):
      super( ConfiguredBaseMode, self ).handleActive( state )
      self.enableStateControl()

   def handleInactive( self ):
      self.disableStateControl()
      super( ConfiguredBaseMode, self ).handleInactive()

   def leave( self ):
      self.disableStateControl()
      super( ConfiguredBaseMode, self ).leave()


# Legacy Mlag Operational mode
class LegacyMode( ConfiguredBaseMode ):
   def handleActive( self, state ):
      super( LegacyMode, self ).handleActive( state )
      pluginCtx.floodControlIs( Tac.newInstance( "IgmpSnooping::MlagFloodControl",
                                          "mlag", mlagStatus,
                                          peerIntfStatusDir, mlagProtoStatus ) )
   def leave( self ):
      pluginCtx.floodControlDel( "mlag" )
      super( LegacyMode, self ).leave()

# Common for all State StateSync based methods
class SyncBaseMode( ConfiguredBaseMode ):
   def __init__( self, legacy=True):
      super( SyncBaseMode, self ).__init__()
      self.mlagInfo = None
      self.floodControl = None
      self.legacy = legacy

   def enableFloodControl( self ):
      if not self.floodControl :
         self.floodControl = Tac.newInstance( 
                                          "IgmpSnooping::MlagSyncFloodControl",
                                          "mlag", mlagStatus, peerIntfStatusDir,
                                          mlagProtoStatus )
         pluginCtx.floodControlIs( self.floodControl )

   def disableFloodControl( self ):
      if self.floodControl:
         pluginCtx.floodControlDel( "mlag" )
         self.floodControl = None

   def enableMlagInfo( self ):
      if not self.mlagInfo:
         self.mlagInfo = Tac.newInstance( "IgmpSnooping::MlagInfoForSnooping",
                                          "mlag", mlagStatus, mlagProtoStatus, 
                                          mlagConfig )
         pluginCtx.mlagInfoIs( self.mlagInfo )

   def disableMlagInfo( self ):
      if self.mlagInfo:
         pluginCtx.mlagInfoDel()
         self.mlagInfo = None

   def enter( self ):
      super( SyncBaseMode, self ).enter()
      self.enableMlagInfo()

   def handleActive( self, state ):
      super( SyncBaseMode, self ).handleActive( state )
      cancelPeerDeletedTimer()   
      self.enableFloodControl()
      pluginCtx.mlagActiveIs( True, self.mlagInfo )

   def handleInactive ( self ):
      super( SyncBaseMode, self ).handleInactive()
      self.disableFloodControl()
      startPeerDeletedTimer()
      pluginCtx.mlagActiveIs( False, None )
 
   def leave( self ):
      pluginCtx.mlagActiveIs( False, None )
      self.disableMlagInfo()
      self.disableFloodControl()
      super( SyncBaseMode, self ).leave()

# Sync using Sysdb and Mlag
class MlagSync( SyncBaseMode ):
   def enter( self ):
      super( MlagSync, self ).enter()
      pluginCtx.snoopingSyncEnable()

   def setupPeerStatus( self ):
      if self.mlagInfo.translateStatusSm:
         t2( "Reusing existing TranslateSm" )
         return
      t2( "Creating new peer status" )
      peerStatus = Tac.newInstance( "Bridging::IgmpSnooping::SyncStatus", "mlag" )
      pluginCtx.peerStatusIs( peerStatus )

      t2( "Creating new mlag translate status sm" )
      self.mlagInfo.translateStatusSm = \
                  Tac.newInstance( "IgmpSnooping::MlagTranslateStatusSm",
                        mlagStatus, igmpSnoopingInputDir, peerStatus )

   def cleanupPeerStatus( self ):
      if self.mlagInfo:
         self.mlagInfo.translateStatusSm = None

      pluginCtx.peerStatusDel()

   def handleActive( self, state ):
      super( MlagSync, self ).handleActive( state )
      self.setupPeerStatus()

   def handleInactive( self ):
      super( MlagSync, self ).handleInactive()
      startPeerDeletedTimer()

   def leave( self ):
      pluginCtx.snoopingSyncDisable()
      self.cleanupPeerStatus()
      super( MlagSync, self ).leave()


# Direct Sync between IgmpSnooping agents running on two Mlags
class DirectAgentSync( SyncBaseMode ):
   def __init__( self ):
      super( DirectAgentSync, self ).__init__( legacy=False )

   def handleActive( self, state ):
      super( DirectAgentSync, self ).handleActive( state )
      pluginCtx.agentSyncEnabledIs( True )

   def leave( self ):
      pluginCtx.agentSyncEnabledIs( False )
      super( DirectAgentSync, self ).leave()
