# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from CliModel import Model, List, Dict, Enum, Bool, Submodel, Int, Float, \
      DeferredModel
from ArnetModel import IpGenericAddress, Ip4Address, MacAddress
from IntfModels import Interface
from CliPlugin import IntfCli
from CliPlugin import BridgingCliModel
import Tac, Arnet
import StringIO

vlanWidth = 8
groupAddressWidth = 17
intfWidth = 25
sourceListWidth = 20

statusEnumToStr = {
      "enabled" : "Enabled",
      "disabled" : "Disabled",
      "default" : "Default",
}

stateEnumToStr = {
      "enabled" : "Enabled",
      "disabled" : "Disabled",
}

operationalStateEnumToStr = {
      "querier" : "Querier",
      "non-querier" : "Non-Querier",
}

filterMode = {
      "include" : "IN",
      "exclude" : "EX",
}

'''
   Legend :
      L - State from IGMP/MLD reports from hosts
      S - Static configured IGMP/MLD groups
      E - State from EVPN RouteType 6, 7, 8
      M - State synced from MLAG peer
      P - State from PIMSM
'''
clientStateEnumToStr = {
      'gmpsnooping' : 'L',
      'gmpSnoopingStatic' : 'S',
      'evpn' : 'E',
      'mlag' : 'M',
      'pimsm' : 'P',
 }

class FormattedPrinting( object ):
   '''helpful for printing something like
      sss aaa ssss,ssss,sss
              ssaa,aas,asasss,
              asas,asa,asasa,
              ss
   '''
   def __init__( self, margin, startOffset, termwidth=79 ):
      self.margin_ = margin
      self.currOffset_ = startOffset
      self.termwidth_ = termwidth
      self.sep_ = ""
      self.buffer_ = StringIO.StringIO()
   def add( self, string ):
      spaceNeeded = len( string ) + len( self.sep_ )
      if self.currOffset_ + spaceNeeded > self.termwidth_:
         self.buffer_.write( ',\n' )
         self.buffer_.write( ' ' * self.margin_ )
         self.currOffset_ = self.margin_
         self.sep_ = ""
         spaceNeeded = len( string )
      self.buffer_.write( self.sep_ )
      self.buffer_.write( string )
      self.currOffset_ = self.currOffset_ + spaceNeeded
      self.sep_ = ', '
   def display( self ):
      print self.buffer_.getvalue()

def timeDiff( newtime, oldtime, displayDays=True):
   '''Take floating point time values, and convert 
   the diff into day/hour/min/sec format.'''
   difftime = int( newtime ) - int( oldtime )

   # Check if diff is negative, note this down
   if difftime < 0:
      negativeTime = True
      difftime = -1 * difftime
   else:
      negativeTime = False

   # Choose format based on flag
   if displayDays == False:
      string = '%02dh%02dm%02d' % \
      ( ( difftime % 86400 ) / 3600, \
      ( ( difftime % 86400 ) % 3600 ) / 60, difftime % 60 )
   else:
      string = '%dd%02dh%02dm%02d' % \
      ( difftime / 86400, ( difftime % 86400 ) / 3600, \
      ( ( difftime % 86400 ) % 3600 ) / 60, difftime % 60)

   # For negative diff contact - to front
   if negativeTime:
      string = '-' + string
   return string

class IgmpSnoopingProxySourceList( Model ):
   sources = List( valueType=IpGenericAddress, 
         help='List of sources associated with the multicast group' )
   filterMode = Enum( values=( "Include", "Exclude" ), default="Include", 
         help="Source filter mode" )

   def toString( self ):
      render = self.filterMode
      render += "". join( [  '\n'.ljust( vlanWidth + intfWidth + \
            groupAddressWidth ) + " %s" % src for src in sorted( self.sources ) ] )
      return render

class IgmpSnoopingProxyGroupInfo( Model ):
   sourceList = Submodel( valueType=IgmpSnoopingProxySourceList, optional=True,
         help="Optional Source List" )

class IgmpSnoopingProxyGroups( Model ):
   groups = Dict( keyType=IpGenericAddress, valueType=IgmpSnoopingProxyGroupInfo,
         help='Map Multicast addresses to an optional sourceList' )
         

   def renderHelper( self, vlanId, interface, detail ):
      for group in sorted( self.groups ):
         render = ( "%d" % vlanId ).ljust( vlanWidth ) + \
            ( "%s" % interface ).ljust( intfWidth ) + \
            ( "%s" % group ).ljust( groupAddressWidth )
         if detail :
            render += self.groups[ group ].sourceList.toString()
         yield render


class IgmpSnoopingProxyPorts( Model ):
   portInfo = Dict( keyType=Interface, valueType=IgmpSnoopingProxyGroups,
         help="Dict of Interface to Groups that are reported" )

class IgmpSnoopingProxyInfo( Model ):
   detail = Bool( help='Include detailed attributes', default=False )
   vlanInfo = Dict( keyType=int, valueType=IgmpSnoopingProxyPorts,
         help="Dict of VlanId to per interface proxy related information" )

   def renderInDetail( self ):
      print "Vlan".ljust( vlanWidth ) + \
            "Interface".ljust( intfWidth ) +\
            "Group".ljust( groupAddressWidth ) +\
            "Source/Filter Mode"

      print "-" * 85

      for vlanId in sorted( self.vlanInfo ):
         vlanInfo = self.vlanInfo[ vlanId ]
         for port in sorted( vlanInfo.portInfo ):
            portInfo = vlanInfo.portInfo[ port ]
            for grpRec in portInfo.renderHelper( vlanId, port, self.detail ):
               print grpRec

   def render( self ):
      if self.detail:
         self.renderInDetail()
         return

      if len( self.vlanInfo ) == 0:
         return

      print ( '%-5s %-16s %-8s %s' % (
                 'Vlan', 'Group', 'Type', 'Port-List' ) )
      print "-" * 85

      for vlanId in sorted( self.vlanInfo ):
         vlanInfo = self.vlanInfo[ vlanId ]
         # Map of Group to list of ports
         groupDict = {}
         for port in sorted( vlanInfo.portInfo ):
            portInfo = vlanInfo.portInfo[ port ]
            for group in sorted( portInfo.groups ):
               if group in groupDict:
                  groupDict[ group ].append( port )
               else:
                  groupDict[ group ] = [ port ]
         # render
         for group in sorted( groupDict.keys() ):
            ports = groupDict[ group ]
            def shortName( port ):
               return Tac.Value("Arnet::IntfId", port ).shortName

            portList = ", ".join( [ shortName( port ) for port in ports ] )
            print ( '%-5s %-16s %-8s %s' % (
                 str( vlanId ), group , 'Proxy', portList ) )
         
class IgmpProfileInfo( Model ):
   class ProfileInfo( Model ):
      class Ranges( Model ):
         min = IpGenericAddress( help="Minimum address range" )
         max = IpGenericAddress( help="Maximum address range" )
      action = Enum( help="Filter type of profile permit/deny",
                     values=( 'permit', 'deny' ) )
      ranges = List( help="Multicast groups covered by the profile", 
                         valueType=Ranges )

   profiles = Dict( keyType=str, valueType=ProfileInfo,
                  help="Map IGMP Profile to its filter and list of addresses" )

   def render( self ):
      for name, profile in self.profiles.iteritems():
         print "IGMP Profile", name
         print ' '*4, profile.action
         for r in profile.ranges:
            print ' '*4, "range", r.min, r.max

class IgmpSnoopingVlanInfo( Model ):
   igmpSnoopingState = Enum( help="State of IGMP Snooping",
         values=stateEnumToStr.keys() )
   immediateLeave = Enum( help="Fast-leave processing on the vlan",
                        values=statusEnumToStr.keys() )
   multicastRouterLearningMode = Enum( help="Multicast Router Learning mode",
                        values=( "pim-dvmrp", ) )
   maxGroups = Int( default=65534,
      help="Maximum number of multicast groups that can join the vlan" )
   groupsOverrun = Bool( default=False,
         help="Indicate if there is an attempt to create group more than maxGroups" )
   reportFlooding = Enum( help="Forwarding of membership report messages",
                       values=statusEnumToStr.keys() )
   pruningActive = Bool( help="True if IGMP snooping pruning active",
                         default=False )
   floodingTraffic = Bool( help="Flooding traffic to VLAN",
                           default=True )
   proxyActive = Bool( help="IGMP snooping proxy active", default=False )

class IgmpSnoopingInfo( Model ):
   igmpSnoopingState = Enum( help="Global state of IGMP Snooping",
         values=stateEnumToStr.keys(), optional=True )
   immediateLeave = Enum( help="Fast-leave processing global",
                      values=statusEnumToStr.keys(), optional=True )
   robustness = Int( default=2,
         help="Number of queries sent to age out a port's membership in group" )
   reportFlooding = Enum( help="Forwarding of membership report messages",
                     values=statusEnumToStr.keys(), optional=True )
   reportFloodingSwitchPorts = List( valueType=Interface,
                    help="Interfaces for which report-flooding enabled" )
   vlans = Dict( keyType=int, valueType=IgmpSnoopingVlanInfo,
                    help="Map vlan Id to vlan information" )

   def render( self ):
      if self.igmpSnoopingState is None:
         return
      print "   Global IGMP Snooping configuration:"
      print "-------------------------------------------"
      print "%-30s : %s" % ( 'IGMP snooping', 
            stateEnumToStr[ self.igmpSnoopingState ] )
      print "%-30s : %s" % ( 'IGMPv2 immediate leave', 
            statusEnumToStr [ self.immediateLeave ] )
      print "%-30s : %s" % ( 'Robustness variable', self.robustness )
      print "%-30s : %s" % ( 'Report flooding', 
            statusEnumToStr [ self.reportFlooding ] )
      first_line_desc = 'Report flooding switch-ports '
      for switchPort in self.reportFloodingSwitchPorts:
         print "%-30s  : %s" % ( first_line_desc, switchPort )
         first_line_desc = ''
      print ""
      for vlan, info in sorted( self.vlans.iteritems() ):
         print "Vlan", "%s" % vlan, ":"
         print "----------"
         print "%-30s : %s" % ( 'IGMP snooping', 
               stateEnumToStr[ info.igmpSnoopingState ] )
         print "%-30s : %s" % ( 'IGMPv2 immediate leave', 
               statusEnumToStr [ info.immediateLeave ] )
         print "%-30s : %s" % ( 'Multicast router learning mode',
               info.multicastRouterLearningMode )
         igmpMaxGroupsLimitStr = 'IGMP max group limit'
         if info.maxGroups == 65534:
            print "%-30s : %s " % ( igmpMaxGroupsLimitStr, 'No limit set' )
         else:
            print "%-30s : %u " % ( igmpMaxGroupsLimitStr,
                  info.maxGroups )
         print "%-30s : %s " % ( 'Recent attempt to exceed limit', 
               'Yes' if info.groupsOverrun else 'No' )
         print "%-30s : %s" % ( 'Report flooding',
               statusEnumToStr[ info.reportFlooding ] )
         print "%-30s : %s" % ( 'IGMP snooping pruning active',
               info.pruningActive )
         print "%-30s : %s" % ( 'Flooding traffic to VLAN',
               info.floodingTraffic )
         print "%-30s : %s" % ( 'IGMP snooping proxy active',
               info.proxyActive )

class IgmpSnoopingVlanReportFlooding( Model ):
   reportFloodingOperationalState = Enum( help="State of report flooding in vlan", 
         values=statusEnumToStr.keys() )
   reportFloodingEnabled = Bool( help="True if report flooding enabled in vlan", 
         optional=True )
   switchPorts = Dict( keyType=Interface, valueType=str, help=
         "A mapping of ports forwarding IGMP membership report to port status" )
   sanityCheckFailedPorts = List( valueType=Interface, 
         help="Sanity check for all Switchports")

class IgmpSnoopingReportFlooding( Model ):
   reportFloodingEnabled = Bool( help="True if ReportFlooding enabled", 
         optional=True )
   reportFloodingSwitchPorts = List( valueType=Interface, 
      help="Interfaces for which report-flooding enabled" )
   vlans = Dict( keyType=int, valueType=IgmpSnoopingVlanReportFlooding, 
      help="Maps vlanId to vlan information", optional=True )

   def render( self ):
      if self.reportFloodingEnabled is None:
         return
      print " "
      print "Ip IGMP Snooping report-flooding feature:"
      print "-------------------------------------------"
      print "%-32s: %s" % ( "Report-Flooding Feature Enabled", 
         self.reportFloodingEnabled ) 
      first_line_desc = 'Report flooding switch-ports '
      for switchPort in self.reportFloodingSwitchPorts:
         print "%-30s  : %s" % ( first_line_desc, switchPort )
         first_line_desc = ''
      print " "
      
      for vlanId, value in sorted( self.vlans.iteritems() ):
         print "Vlan %d :" % vlanId
         print "----------"
         print "%-32s: %s" % ( "Vlan feature enabled", 
               statusEnumToStr[ value.reportFloodingOperationalState ] )
         print "%-32s: %s" % ( "Global feature enabled", 
               self.reportFloodingEnabled )
         if value.reportFloodingEnabled:
            print "%-32s: %s" % ( "Runtime feature enabled",
                                 value.reportFloodingEnabled )
         else:
            print "%-32s: Not Exist" % "Vlan Status"
            print "Is IGMP Snooping enabled on this vlan?"
            print " "

         print "Switch port configured: "   

         for port, status in value.switchPorts.iteritems():
            print "%s   %-28s: Status %s" % ( ' ' if status == 'OK' else '!', port,
                                             status )

         print "    Total %d switch-ports configured" % len( value.switchPorts )

         for port in value.sanityCheckFailedPorts:
            print "Error: %s in Status but not Config!" % ( port )
         
         print " "

class IgmpSnoopingVlanQuerier( Model ):
   querierAddress = Ip4Address(
         help="Address of IGMP querier in the Vlan" )
   igmpVersion = Enum( help="Version of IGMP snooping querier in the vlan", 
         values=( 'v1', 'v2', 'v3', 'unknown' ) )
   querierInterface = Interface( help="Interface where the querier is located" )
   queryResponseInterval = Float( help="Effective maximum period a recipient"
   " can wait before responding with a membership report in seconds" )

class IgmpSnoopingQuerier( Model ):
   vlans = Dict( keyType=int, valueType=IgmpSnoopingVlanQuerier,
         help="Mapping vlan Id to the vlan querier info" )
   _vlanSpecific = Bool( default=False,
         help="Display information in vlan specific format if True" )

   def render( self ):
      if len( self.vlans ) == 0:
         return
      elif self._vlanSpecific:
         vlanId = self.vlans.keys()[ 0 ]
         igmpVersion = self.vlans[ vlanId ].igmpVersion
         print '%-20s : %s\n%-20s : %s\n%-20s : %s\n%-20s : %s' % (
               'IP Address', self.vlans[ vlanId ].querierAddress,
               'IGMP Version', '-' if igmpVersion == 'unknown' else igmpVersion,
               'Port', IntfCli.Intf.getShortname(
                  self.vlans[ vlanId ].querierInterface ),
               'Max response time', self.vlans[ vlanId ].queryResponseInterval )
         return
      print "%-5s %-16s %-8s %s" % (
            'Vlan', 'IP Address', 'Version', 'Port' )
      print "-" * 40
      for vlanId, info in sorted( self.vlans.iteritems() ):
         print '%-5s %-16s %-8s %s' % ( vlanId, info.querierAddress, 
               '-' if info.igmpVersion == 'unknown' else info.igmpVersion, 
               IntfCli.Intf.getShortname( info.querierInterface ) )

class IgmpQuerierStatusVlan( Model ):
   adminState = Enum( help="State of IGMP querier in the Vlan",
         values=stateEnumToStr.keys() )
   querierAddress = Ip4Address( help="Ip Address of querier on the vlan" )
   queryInterval = Float( help="Period between IGMP Membership Query messages"
         " in seconds" )
   queryResponseInterval = Float( help="Effective maximum period a recipient"
         " can wait before responding with a Membership Report in seconds" )
   querierExpiryTime = Float( help="Querier's expiry time in seconds" )
   operationalState = Enum( help="Operational state of querier in the Vlan",
         values=operationalStateEnumToStr.keys() )
   version = Enum( help="Version of IGMP querier on the Vlan",
         values=( 'v1', 'v2', 'v3', 'unknown' ) )
   lastMemberQueryInterval = Float( help="Transmission frequency set for"
         " query messages to the group when last member on a particular group" 
         " sends leave message in seconds", optional=True )
   lastMemberQueryCount = Int( help="Number of query messages that are sent before"
         " removing from group state and discontinue multicast transmissions"
         " in the vlan", optional=True )
   startupQueryInterval = Float( help="Period between query messages"
         " that querier in the vlan sends upon startup in seconds", optional=True )
   startupQueryCount = Int( help="Number of query messages that querier in the vlan"
         " sends during the startup query interval", optional=True )

class IgmpQuerierStatus( Model ):
   adminState = Enum( help="State of global IGMP querier",
         values=stateEnumToStr.keys(), optional=True )
   querierAddress = Ip4Address( help="Address of global IGMP querier",
         optional=True )
   queryInterval = Float( help="Period between IGMP Membership Query messages"
         " sent from the global querier in seconds", optional=True )
   robustness = Int( optional=True,
         help="Number of queries sent to age out a port's membership in group" )
   queryResponseInterval = Float(  help="Effective maximum period a recipient"
         " can wait before responding with a Membership Report in seconds",
         optional=True )
   querierExpiryTime = Float( help="Querier's expiry time in seconds", 
         optional=True )
   lastMemberQueryInterval = Float( help="Transmission frequency set for" 
         " query messages to the group when last member"
         " on a particular group sends leave message in seconds", optional=True )
   lastMemberQueryCount = Int( help="Number of query messages that are sent before"
         " removing from group state and discontinue multicast transmissions", 
         optional=True )
   startupQueryInterval = Float( help="Period between query messages"
         " that the querier sends upon startup in seconds", optional=True )
   startupQueryCount = Int( help="Number of query messages that the querier"
         " sends during the startup query interval", optional=True )
   vlans = Dict( keyType=int, valueType=IgmpQuerierStatusVlan,
                   help="Map the vlan id with its querier info" )

   def render( self ):
      if len( self.vlans ) == 0:
         return
      print "   Global IGMP Querier status"
      print "-------------------------------------------------------"

      print "admin state                      :", stateEnumToStr[ self.adminState ]
      print "source IP address                :", self.querierAddress
      print "query-interval (sec)             :", self.queryInterval
      print "max-response-time (sec)          :", self.queryResponseInterval
      print "querier timeout (sec)            :", self.querierExpiryTime
      print "last-member-query-interval (sec) :", self.lastMemberQueryInterval
      print "last-member-query-count          :", ( self.lastMemberQueryCount
            if self.lastMemberQueryCount else
            str( self.robustness ) + ' (robustness)' )
      print "startup-query-interval (sec)     :", ( self.startupQueryInterval
            if self.startupQueryInterval else
            str( self.queryInterval/4 ) + ' (query-interval/4)' )
      print "startup-query-count              :", ( self.startupQueryCount
            if self.startupQueryCount else
            str( self.robustness ) + ' (robustness)' )

      print ""
      if self.vlans:
         print \
         "Vlan Admin    IP              Query    Response Querier Operational Ver"
         print \
         "     State                    Interval Time     Timeout State"
         print \
         "-----------------------------------------------------------------------"

      for vlanId, info in sorted( self.vlans.iteritems() ):
         print "%-4s" % vlanId, \
               "%-8s" % stateEnumToStr[ info.adminState ], \
               "%-15s" % info.querierAddress, \
               "%-8.1f" % info.queryInterval, \
               "%-8.1f" % info.queryResponseInterval, \
               "%-7.1f" % info.querierExpiryTime, \
               "%-11s" % operationalStateEnumToStr[ info.operationalState ], \
               info.version

class GroupSourceInfo( Model ):
   sourceAddress = Ip4Address( help="Source address for multicast group" )
   creationTime = Float( help="UTC time at which this source was created", 
         optional=True )
   expiryTime = Float( help="UTC time at which this source will expire", 
         optional=True)
   expired = Bool( help="True if the source address expires" )

class MulticastGroupInfo( Model ):
   groupAddress = Ip4Address( help="Address of the multicast group" )
   filterMode = Enum( help="Filter mode of multicast group",
         values=filterMode.keys() )
   creationTime = Float( help="UTC time at which the group was created", 
         optional=True )
   expiryTime = Float( help="UTC time at which the group will expire", 
         optional=True )
   lastReporter = Ip4Address( help="The IP Address of the latest host who "
         "joined this group" )
   igmpVersion = Enum( help="Version of IGMP snooping querier",
         values=( 'v1', 'v2', 'v3', 'unknown' ) )
   sources = List( valueType=GroupSourceInfo, optional=True ,
         help="Sources belong to the group and its information" )

class IgmpSnoopingVlanQuerierMembership( Model ):
   querierStatus = Submodel( valueType=IgmpQuerierStatusVlan, 
         help="Querier status information in the vlan" )
   robustness = Int(
         help="Number of queries sent to age out a port's membership in group" )
   lastMemberQueryTime = Float( help="Total time to send last member query" 
         " messages in seconds", optional=True )
   otherQuerierPresentInterval = Float( help="Period for which multicast router" 
         " waits before becoming a querier in seconds", optional=True )
   olderHostPresentInterval = Float( help="Time-out for transitioning a group back" 
         " to IGMPv3 mode once an older version report is received in seconds", 
         optional=True )
   generalQueryExpiryTime = Float( optional=True,
         help="IGMP general query timer expiry in seconds" )
   otherQuerierExpiryTime = Float( optional=True,
         help="Other querier present timer expiry in seconds" )
   addRandomDelayToTimer = Bool( 
         help="True if random delay added to expiry timers" )
   groupMembershipInterval = Float( optional=True,
         help="Period after which multicast group membership expires in seconds" )
   groups = List( valueType=MulticastGroupInfo,
         help="Multicast group information" )

class IgmpSnoopingQuerierMembership( Model ):
   vlans = Dict( keyType=int, valueType=IgmpSnoopingVlanQuerierMembership,
         help="Mapping vlan id with querier membership info" )

   def render( self ):
      if len( self.vlans ) == 0:
         return

      for vlanId, vlanInfo in sorted( self.vlans.iteritems() ):
         print "-" * 73
         print "Vlan: %-4s" % vlanId, \
         "Elected: %-15s" % vlanInfo.querierStatus.querierAddress, \
         "QQI: %-4d" % vlanInfo.querierStatus.queryInterval, \
         "QRV: %-2d" % vlanInfo.robustness, \
         "QRI: %-3d" % vlanInfo.querierStatus.queryResponseInterval, \
         "GMI: %-4d" % vlanInfo.groupMembershipInterval
         print ""
         print "Groups           Mode  Ver  Num of Sources" + " " * 31
         print "-" * 73
         for groupInfo in  vlanInfo.groups:
            numSources = len( groupInfo.sources )
            if numSources == 0 :
               strSources = "[]"
            elif numSources == 1:
               strSources = "[ %s ]" % groupInfo.sources[ 0 ].sourceAddress
            elif numSources == 2:
               sortedAddrs = sorted( [ groupInfo.sources[ 0 ].sourceAddress,
                                       groupInfo.sources[ 1 ].sourceAddress ] )
               strSources = "[ %s, %s ]" % ( sortedAddrs[ 0 ], sortedAddrs[ 1 ] )
            else:
               sortedAddrs = sorted( [ groupInfo.sources[ 0 ].sourceAddress,
                                       groupInfo.sources[ 1 ].sourceAddress,
                                       groupInfo.sources[ 2 ].sourceAddress, ] )
               strSources = "[ %s, %s, ... ]" % ( sortedAddrs[ 0 ],
                                                  sortedAddrs[ 1 ] )
            print "%-16s" % groupInfo.groupAddress, \
                  "%-5s" % filterMode[ groupInfo.filterMode ], \
                  "%-4s" % ( '-' if groupInfo.igmpVersion == 'unknown'
                          else groupInfo.igmpVersion ), \
                  "%d" % numSources, \
                  strSources
         print ""

class IgmpSnoopingVlanQuerierCounters( Model ):
   operationalState = Enum( help="Operational state of querier in the vlan",
           values=operationalStateEnumToStr.keys() )
   querierAddress = Ip4Address(
         help="Address of IGMP querier in the Vlan" )
   igmpVersion = Enum( help="Version of IGMP snooping querier in the vlan",
         values=( 'v1', 'v2', 'v3', 'unknown' ) )
   v1QueriesSent = Int( help="Number of v1 general queries sent" )
   v1QueriesReceived = Int( help="Number of v1 queries received" )
   v1ReportReceived = Int( help="Number of v1 reports received" )
   v2QueriesSent = Int( help="Number of v2 general queries sent" )
   v2QueriesReceived = Int( help="Number of v2 queries received" )
   v2ReportReceived = Int( help="Number of v2 reports received" )
   v2LeaveReceived = Int( help="Number of v2 leaves received" )
   v3QueriesSent = Int( help="Number of v3 general queries sent" )
   v3GSQueriesSent = Int( help="Number of v3 group specific queries sent" )
   v3GSSQueriesSent = Int(
         help="Number of v3 group and source specific queries sent" )
   v3QueriesReceived = Int( help="Number of v3 queries received" )
   v3ReportReceived = Int( help="Number of v3 reports received" )
   totalGeneralQueriesSent = Int( help="Total number of general queries sent" )
   errorPacketReceived = Int( help="Number of error packets received" )
   otherPacketReceived =  Int( help="Number of other packets received" )

class IgmpSnoopingQuerierCounters( Model ):
   vlans = Dict( keyType=int, valueType=IgmpSnoopingVlanQuerierCounters,
                 help="Map Vlan id to Vlan counter info" )

   def render( self ):
      if len( self.vlans ) == 0:
         return
      for vlanId, vlanInfo in sorted( self.vlans.iteritems() ):
         print "-" * 71
         print "Vlan: %-4s" % vlanId, \
         "IP Addr: %-15s" % vlanInfo.querierAddress, \
         "Op State: %-11s" % ( 
                 operationalStateEnumToStr[ vlanInfo.operationalState ] ), \
         "Version: ", ( '-' if vlanInfo.igmpVersion == 'unknown' 
                 else vlanInfo.igmpVersion )

         print ""
         print "v1 General Queries Sent         :%d" % vlanInfo.v1QueriesSent
         print "v1 Queries Received             :%d" % vlanInfo.v1QueriesReceived
         print "v1 Reports Received             :%d" % vlanInfo.v1ReportReceived
         print "v2 General Queries Sent         :%d" % vlanInfo.v2QueriesSent
         print "v2 Queries Received             :%d" % vlanInfo.v2QueriesReceived
         print "v2 Reports Received             :%d" % vlanInfo.v2ReportReceived
         print "v2 Leaves Received              :%d" % vlanInfo.v2LeaveReceived
         print "v3 General Queries Sent         :%d" % vlanInfo.v3QueriesSent
         print "v3 GSQ Queries Sent             :%d" % vlanInfo.v3GSQueriesSent
         print "v3 GSSQ Queries Sent            :%d" % vlanInfo.v3GSSQueriesSent
         print "v3 Queries Received             :%d" % vlanInfo.v3QueriesReceived
         print "v3 Reports Received             :%d" % vlanInfo.v3ReportReceived
         print "Error Packets                   :%d" % vlanInfo.errorPacketReceived
         print "Other Packets                   :%d" % vlanInfo.otherPacketReceived
         print ""

class IgmpQuerierMembershipGroup( Model ):
   sources = List( valueType=GroupSourceInfo, 
         help="Sources belong to the group and its information" )

   def render( self ):
      if self.sources:
         print "Source Ip Addr      Expired"
         print "---------------------------"
         for source in self.sources:
            print "%-15s     %s" % ( source.sourceAddress, source.expired )
         print ""

class IgmpSnoopingMrouterDetail( Model ):
   interfaceName = Interface( help ="Multicast router interface" )
   address = IpGenericAddress( help="Source address of the interface", 
         optional=True )
   firstHeardTime = Float( optional=True, 
         help="UTC time at which this router is first heard" )
   lastHeardTime = Float( optional=True,
         help="UTC time at which this router is last heard" )
   expireTime = Float( help="UTC time at which this router will expire", 
         optional=True )
   routerType = Enum( help="Type of multicast router",
         values=( 'static', 'dynamic', 'pim', 'dvmrp', 'mrd', 'querier', 
            'unknown' ), default='unknown' )

class IgmpSnoopingMrouterVlan( Model ):
   interfaces = List( valueType=IgmpSnoopingMrouterDetail,
         help="Mrouter interface info" )

class IgmpSnoopingMrouter( Model ):
   vlans = Dict( keyType=int, valueType=IgmpSnoopingMrouterVlan,
         help="Maps a VLAN ID to VLAN information" )
   _detailInfo = Bool( help="Display detailed information if True", default=False )

   def renderDetail( self ):
      print '%-6s %-9s %-17s %-12s %-12s %-8s %-9s' % (
            'Vlan', 'Intf', 'Address', 'FirstHeard', 'LastHeard',
            'Expires', 'Type' )
      print "-" * 81

      for vlanId, info in sorted( self.vlans.iteritems() ):
         for intf in info.interfaces:
            now = Tac.utcNow()
            print '%-6s %-9s %-17s %-11s %-11s %-8s %-9s' % (
                  vlanId, IntfCli.Intf.getShortname( intf.interfaceName ),
                  intf.address, '-  ' if intf.firstHeardTime is None 
                  else timeDiff( now, intf.firstHeardTime ),
                  '-  ' if intf.lastHeardTime is None
                  else timeDiff( now, intf.lastHeardTime ),
                  '-  ' if intf.expireTime is None 
                  else timeDiff( intf.expireTime, now, False ),
                  intf.routerType )

   def render( self ):
      if not self.vlans:
         return
      if self._detailInfo:
         self.renderDetail()
         return
      print "Vlan    Interface-ports"
      print "-" *60
      for vlanId, info in sorted( self.vlans.iteritems() ):
         fp = FormattedPrinting( margin=8, startOffset=8, termwidth=75 )
         print "%-7s" % vlanId,
         for intf in info.interfaces:
            fp.add( IntfCli.Intf.getShortname( intf.interfaceName ) +
               '(%s)' % intf.routerType )
         fp.display()

class IgmpSnoopingCountersInterface( Model ):
   igmpV1QueryReceived = Int( help="Number of IGMP v1 queries received",
         optional=True )
   igmpV2QueryReceived = Int( help="Number of IGMP v2 queries received",
         optional=True )
   igmpV3QueryReceived = Int( help="Number of IGMP v3 queries received",
         optional=True )
   igmpV1ReportReceived = Int( help="Number of IGMP v1 reports received",
         optional=True )
   igmpV2ReportReceived = Int( help="Number of IGMP v2 reports received",
         optional=True )
   igmpV3ReportReceived = Int( help="Number of IGMP v3 reports received",
         optional=True )
   igmpV2LeaveReceived = Int( help="Number of IGMP v2 leaves received",
         optional=True )
   dvmrpPacketsReceived = Int( help="Number of DVMRP packets received",
         optional=True )
   mrdPacketsReceived = Int( help="Number of MRD packets received",
         optional=True )
   pimPacketsReceived = Int( help="Number of PIM packets received",
         optional=True )
   otherIgmpPacketsReceived = Int( help="Number of other IGMP packets",
         optional=True )
   shortPacketsReceived = Int( help="Number of packets received"
         " with not enough IP payload" )
   nonIpPacketsReceived = Int( help="Number of non IP packets received" )
   badChecksumIpPacketsReceived = Int( help="Number of packets received"
         " for which IP checksum check failed" )
   unknownIpPacketsReceived = Int( help="Number of packets received"
         " with unknown IP Protocol" )
   badChecksumIgmpPacketsReceived = Int( help="Number of packets received"
         " for which IGMP checksum check failed" )
   badChecksumPimPacketsReceived = Int( help="Number of packets received"
         " for which PIM checksum check failed" )
   igmpQueriesSent = Int( help="Number of IGMP queries sent", optional=True )
   igmpReportSent = Int( help="Number of IGMP reports sent", optional=True )
   igmpLeaveSent = Int( help="Number of IGMP v2 leaves sent", optional=True )
   otherPacketsSent = Int( help="Number of other packets sent", optional=True )

class IgmpSnoopingCounters( Model ):
   interfaces = Dict( keyType=Interface, valueType=IgmpSnoopingCountersInterface,
         help="Map Interface name with its counter details" )
   _errorSpecific = Bool( help="True if only error counters to be displayed" )

   def render( self ):
      if self._errorSpecific:
         errorCounterFormat = '%-10s %-10s %-9s %-9s %-12s %-9s %-9s'
         print errorCounterFormat % ( '', 'Packet',
               '  Packet',  'Bad IP', 'Unknown', 'Bad IGMP', 'Bad PIM' )
         print errorCounterFormat % ( 'Port', 'Too Short',
               '  Not IP',  'Checksum', 'IP Protocol', 'Checksum', 'Checksum' )
         print "-" * 73
         for intf, counters in sorted( self.interfaces.iteritems() ):
            print '%-10s %9s %9s %9s %12s %9s %9s' % (
                  IntfCli.Intf.getShortname( intf ),
                  counters.shortPacketsReceived,
                  counters.nonIpPacketsReceived,
                  counters.badChecksumIpPacketsReceived,
                  counters.unknownIpPacketsReceived,
                  counters.badChecksumIgmpPacketsReceived,
                  counters.badChecksumPimPacketsReceived )
         return
      
      print "{:^46}|{:>18}".format( 'Input', 'Output' )
      print "Port   Queries Reports  Leaves  Others  Errors|" \
            "Queries Reports  Leaves  Others"
      print "-" * 78
      for intf, counters in sorted( self.interfaces.iteritems() ):
         print '%-6s %7d %7d %7d %7d %7d %7d %7d %7d %7d' % ( 
            IntfCli.Intf.getShortname( intf ),
            ( counters.igmpV1QueryReceived + counters.igmpV2QueryReceived + 
               counters.igmpV3QueryReceived ),
            ( counters.igmpV1ReportReceived + counters.igmpV2ReportReceived +
               counters.igmpV3ReportReceived ),
            counters.igmpV2LeaveReceived,
            ( counters.dvmrpPacketsReceived + counters.mrdPacketsReceived +
               counters.otherIgmpPacketsReceived + counters.pimPacketsReceived ),
            ( counters.shortPacketsReceived + counters.nonIpPacketsReceived +
               counters.badChecksumIpPacketsReceived +
               counters.unknownIpPacketsReceived +
               counters.badChecksumIgmpPacketsReceived +
               counters.badChecksumPimPacketsReceived ),
            counters.igmpQueriesSent, 
            counters.igmpReportSent,
            counters.igmpLeaveSent, 
            counters.otherPacketsSent )

class IgmpSnoopingTableEntry( Model ):
   vlanId = Int( help="VLAN ID of MAC address" )
   macAddr = MacAddress( help="MAC address" )
   macType = Enum( values=( "igmp", ), help="Type of MAC address" )
   interfaces = List( valueType=Interface, help="Interfaces of the MAC address" )

class MacAddressTableIgmpSnooping( Model ):
   igmpMacEntries = List( valueType=IgmpSnoopingTableEntry, 
         help="All IGMP Snooping MAC entries" )
   multicastMacTable = Submodel( 
         valueType=BridgingCliModel.MulticastMacAddressTable, 
         help="Multicast MAC entries" )   
   _brief = Bool( help="Display brief information" )

   def render( self ):
      print '{:^66}'.format('IGMP Snooping MAC Address Table')
      print '-' * 66
      print
      fmt = '%-5s %-18s %-5s'
      print ( fmt + ' %s' ) % ( 'Vlan', 'Mac Address', 'Type', 
         ( 'Num of Ports' if self._brief else 'Port-List' ) )
      print '-' * 66
      for entry in self.igmpMacEntries:
         print fmt % ( entry.vlanId, entry.macAddr, entry.macType.upper() ),
         if not self._brief:
            fp = FormattedPrinting( margin=31, startOffset=31 )
            printIntfs = Arnet.sortIntf( IntfCli.Intf.getShortname( i )
                                         for i in entry.interfaces )
            for intf in printIntfs:
               fp.add( intf )
            fp.display()
         else:
            numIntfs = len( entry.interfaces )
            print '%-3d' % numIntfs, '(',
            printIntfs = [ IntfCli.Intf.getShortname( intf ) 
                  for intf in entry.interfaces[ :5 ] ]
            printIntf = ', '.join( printIntfs )
            if numIntfs > 5:
               printIntf += ', ...'
            print printIntf,
            print ')'
      if self.multicastMacTable is not None:
         self.multicastMacTable.render()

class McastTunnelInfo( DeferredModel ):
   tunnelType = Enum( values=( 'pimTunnel', ), help="Type of multicast tunnel",
                      optional=True )

class ReporterInfo( DeferredModel ):
   creationTime = Float( help="The timestamp in UTC when first heard from host",
                         optional=True )
   lastHeard = Float( optional=True,
                      help="The timestamp in UTC when last heard from host" )
   expiryTime = Float( optional=True,
                       help="The timestamp in UTC when membership will expire" )

class MemberInfo( DeferredModel ):
   mcastTunnelInfo = Submodel( valueType=McastTunnelInfo, optional=True,
                               help="Multicast Tunnel Information" )
   reporterInfo = Dict( keyType=IpGenericAddress, valueType=ReporterInfo,
                        optional=True, help='Host information' )

class SourceMembershipDetailInfo( DeferredModel ):
   creationTime = Float( help="The timestamp in UTC when source was created" )

class SourceMembershipInfo( DeferredModel ):
   includeIntf = Dict( keyType=Interface, valueType=MemberInfo,
                       optional=True,
                       help='Filter mode include member ports information' )
   includeVtep = Dict( keyType=IpGenericAddress, valueType=MemberInfo,
                       optional=True,
                       help='Filter mode exclude member VTEPs information' )
   excludeIntf = Dict( keyType=Interface, valueType=MemberInfo,
                       optional=True,
                       help='Filter mode exclude member ports information' )
   excludeVtep = Dict( keyType=IpGenericAddress, valueType=MemberInfo,
                       optional=True,
                       help='Filter mode exclude member VTEPs information' )
   detail = Submodel( valueType=SourceMembershipDetailInfo, optional=True,
                      help="Additional information about the source" )

class GroupMembershipDetailInfo( DeferredModel ):
   sourceMembership = Dict( keyType=IpGenericAddress,
                           valueType=SourceMembershipInfo,
                           help='Map Source address to interested interfaces/VTEPs' )

class VlanMembershipDetailInfo( DeferredModel ):
   groupMembership = Dict( keyType=IpGenericAddress,
                           valueType=GroupMembershipDetailInfo,
                           optional=True,
                           help='Map IGMP group to source membership information' )
   routerIntf = Dict( keyType=Interface, valueType=MemberInfo, optional=True,
                          help='List of router interfaces' )
   routerVtep = Dict( keyType=IpGenericAddress, valueType=MemberInfo, optional=True,
                          help='List of router VTEPs' )

class GmpMembershipDetailInfo( DeferredModel ):
   member = Dict( keyType=int, valueType=VlanMembershipDetailInfo,
                  help='Map vlanId to per VLAN group membership information' )

class GroupMembershipInfo( DeferredModel ):
   interface = Dict( keyType=Interface, valueType=MemberInfo,
                     optional=True, help='Member ports information' )

   vtep = Dict( keyType=IpGenericAddress, valueType=MemberInfo,
                optional=True, help='Member VTEPs information' )

class VlanMembershipInfo( DeferredModel ):
   groupMembership = Dict( keyType=IpGenericAddress, valueType=GroupMembershipInfo,
                           optional=True,
                           help='Map IGMP group to source membership information' )
   routerIntf = Dict( keyType=Interface, valueType=MemberInfo, optional=True,
                      help='List of router interfaces' )
   routerVtep = Dict( keyType=IpGenericAddress, valueType=MemberInfo, optional=True,
                      help='List of router VTEPs' )

class GmpMembershipInfo( DeferredModel ):
   member = Dict( keyType=int, valueType=VlanMembershipInfo,
                  help='Map vlanId to per VLAN group membership information' )

class GmpMembershipCountInfo( DeferredModel ):
   numberOfGroups = Int( help="Number of groups in membership" )
