#!/usr/bin/env python
# Copyright (c) 2008-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Agent, SharedMem, Smash, Tracing, Plugins, weakref, QuickTrace, os
import Cell
import Shark
from Arnet.NsLib import DEFAULT_NS
from Toggles.TacSharkToggleLib import toggleSharkMembershipJoinStatusEnabled

__defaultTraceHandle__ = Tracing.Handle( "IgmpSnooping" )
t0 = Tracing.trace0
t3 = Tracing.trace3
t8 = Tracing.trace8
t6 = Tracing.trace6

Tac.activityManager.useEpoll = True
Client = Tac.Type( "Irb::Multicast::App::Client" )
MembershipJoinStatus = Tac.Type( "Irb::Multicast::Gmp::MembershipJoinStatus" )
MembershipJoinStatusCli = Tac.Type( "Irb::Multicast::Gmp::MembershipJoinStatusCli" )
MembershipRenewStatus = Tac.Type( "Irb::Multicast::Gmp::MembershipRenewStatus" )

class IgmpSnooping ( Agent.Agent ):
   """The IGMP Snooping Agent. This agent implements the IGMP Snooping protocol
   as well as the shim layer that manages the interactions between IGMP
   snooping and the rest of the system.  It reacts to vlan state, topolib state,
   igmp snooping configs, and igmp packets that float by.  It updates IGMP
   Snooping status objects.  The agent is split into two pieces, the shim/glue,
   and the core, although both reside in the same agent."""

   def __init__( self, entityManager, _agentName=None, blocking=False ):
      t0( "Starting IgmpSnooping Agent" )
      self.block = blocking
      self.snoopConfig = None
      self.swForceConfig = None 
      self.snoopCounterConfig = None 
      self.snoopStatus = None 
      self.advSnoopStatus = None
      self.coreConfig = None 
      self.coreStatus = None 
      self.brConfig = None 
      self.brTopoConfig = None 
      self.brTopoStatus = None 
      self.querierConfigDir = None 
      self.querierStatusDir = None 
      self.profileConfig = None 
      self.counterDir = None 
      self.counterCheckpointDir = None 
      self.hwConfig = None 
      self.proxyConfig = None 
      self.proxyIgmpHostDir = None 
      self.eventMonConfig = None 
      self.eventMonStatus = None
      self.kniStatus = None 
      self.root_ = None
      self.standby = False
      self.pluginContext_ = None
      self.redundancyStatusReactor_ = None
      self.allIntfStatusDir = None
      self.allIntfStatusLocalDir = None
      self.ethIntfStatusDir = None 
      self.intfConfigDir = None
      self.bridgingInput = None
      self.joinStatus = None
      self.staticJoinStatus = None
      self.evpnJoinStatus = None
      self.membershipJoinStatusColl = None
      self.mlagJoinStatus = None
      self.cliJoinStatusSysdb = None
      self.cliJoinStatusShark = None
      self.mergedJoinStatus = None
      self.renewStatus = None
      self.evpnRenewStatus = None
      self.mcsJoinStatus = None
      self.evpnStatus = None
      self.mlagStatus = None
      self.mlagProtoStatus = None

      Agent.Agent.__init__( self, entityManager, _agentName )
      qtfile = "%s%s.qt" % ( self.agentName, "-%d" if "QUICKTRACEDIR"
                             not in os.environ else "" )
      QuickTrace.initialize( qtfile, "32,32,32,32,32,32,32,32,32,32" )

   def doInit( self, entityManager ):
      mountGroup = entityManager.mountGroup()
      self.snoopConfig = mountGroup.mount( 'bridging/igmpsnooping/config',
                                      'Bridging::IgmpSnooping::Config', 'r' )
      self.swForceConfig = mountGroup.mount( 'bridging/igmpsnooping/swForceConfig',
                                      'Bridging::IgmpSnooping::SwForceConfig', 'r' )
      self.snoopCounterConfig = mountGroup.mount( 
         'bridging/igmpsnooping/counterConfig',
         'Bridging::IgmpSnooping::CounterConfig', 'r' )
      self.snoopStatus = mountGroup.mount( 'bridging/igmpsnooping/forwarding/status',
                                     'Bridging::IgmpSnooping::Status', 'w' )
      self.advSnoopStatus = mountGroup.mount(
            'bridging/igmpsnooping/advertised/status',
            'Bridging::IgmpSnooping::SyncStatus', 'w' )
      self.coreConfig = mountGroup.mount( 'bridging/igmpsnooping/protocol/config',
                                     'Bridging::IgmpSnooping::IgmpProtocolConfig',
                                     'w')
      self.coreStatus = mountGroup.mount( 'bridging/igmpsnooping/protocol/status',
                                    'Bridging::IgmpSnooping::IgmpProtocolStatus',
                                    'w')
      self.brConfig = mountGroup.mount( 'bridging/config', 'Bridging::Config', 'r' )
      self.brTopoConfig = mountGroup.mount( 'bridging/topology/config',
                                       'Bridging::Topology::Config', 'r' )
      self.querierConfigDir = mountGroup.mount( \
                                           'bridging/igmpsnooping/querier/config',
                                           'Igmp::QuerierConfigDir', 'w' )
      self.querierStatusDir = mountGroup.mount( \
                                           'bridging/igmpsnooping/querier/status',
                                           'Igmp::QuerierStatusDir', 'w' )
      self.profileConfig = mountGroup.mount( 'igmpprofile/profileconfig',
                                        'IgmpProfile::ProfileConfig', 'r' )
      self.counterDir = mountGroup.mount( 'bridging/igmpsnooping/counterDir',
                                     'Bridging::IgmpSnooping::IgmpCounterDir', 'w' )
      self.counterCheckpointDir = mountGroup.mount(
                                  'bridging/igmpsnooping/counterCheckpointDir',
                                  'Bridging::IgmpSnooping::IgmpCounterDir', 'r' )
      self.hwConfig = mountGroup.mount( 'bridging/igmpsnooping/hardware/config',
                                   'Bridging::IgmpSnooping::HwConfig', 'r' )

      self.proxyConfig = mountGroup.mount( 'bridging/igmpsnooping/proxy/config',
                                    'Igmp::Snooping::Proxy::Config', 'r' )
      self.proxyIgmpHostDir = mountGroup.mount( 
                                    'bridging/igmpsnooping/proxy/igmphost/config',
                                    'Igmp::IgmpHostDir', 'w' )
      self.bridgingInput = mountGroup.mount( 'bridging/igmpsnooping/input',
                                             'Tac::Dir', 'w' )

      self.eventMonConfig = mountGroup.mount( 'eventMon/config',
                                 'EventMon::Config', 'r' )

      self.eventMonStatus = mountGroup.mount( 'eventMon/igmpsnooping/status',
                                 'EventMon::TableStatus', 'w' )

      shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
      shmemMg = shmemEm.getMountGroup()
      self.kniStatus = shmemEm.doMount( "kni/ns/%s/status" % DEFAULT_NS,
                                        "KernelNetInfo::Status",
                                        Smash.mountInfo( 'keyshadow' ) )

      self.membershipJoinStatusColl = Tac.newInstance(
         "Bridging::IgmpSnooping::JoinStatusColl" )

      if toggleSharkMembershipJoinStatusEnabled():
         self.joinStatus = shmemMg.doMount(
               MembershipJoinStatus.mountPath( 'ipv4', Client.igmpsnooping ),
               'Irb::Multicast::Gmp::MembershipJoinStatus',
               Shark.mountInfo( 'writer' ) )
         self.staticJoinStatus = shmemMg.doMount(
               MembershipJoinStatus.mountPath( 'ipv4', Client.igmpsnoopingStatic ),
               'Irb::Multicast::Gmp::MembershipJoinStatus',
               Shark.mountInfo( 'writer' ) )
         self.mlagJoinStatus = Tac.newInstance(
               'Irb::Multicast::Gmp::MembershipJoinStatus', 'mlag' )
         self.evpnJoinStatus = shmemMg.doMount(
               MembershipJoinStatus.mountPath( 'ipv4', Client.evpn ),
               'Irb::Multicast::Gmp::MembershipJoinStatus',
               Shark.mountInfo( 'shadow' ) )
         self.mcsJoinStatus = shmemMg.doMount(
               MembershipJoinStatus.mountPath( 'ipv4', Client.mcs ),
               'Irb::Multicast::Gmp::MembershipJoinStatus',
               Shark.mountInfo( 'shadow' ) )
      else:
         self.joinStatus = mountGroup.mount(
               MembershipJoinStatus.mountPath( 'ipv4', Client.igmpsnooping ),
               'Irb::Multicast::Gmp::MembershipJoinStatus', 'w' )
         self.staticJoinStatus = mountGroup.mount(
               MembershipJoinStatus.mountPath( 'ipv4', Client.igmpsnoopingStatic ),
               'Irb::Multicast::Gmp::MembershipJoinStatus', 'w' )
         self.mlagJoinStatus = mountGroup.mount(
               MembershipJoinStatus.mountPath( 'ipv4', Client.mlag ),
               'Irb::Multicast::Gmp::MembershipJoinStatus', 'w' )
         self.evpnJoinStatus = mountGroup.mount(
               MembershipJoinStatus.mountPath( 'ipv4', Client.evpn ),
               'Irb::Multicast::Gmp::MembershipJoinStatus', 'r' )
         self.mcsJoinStatus = mountGroup.mount(
               MembershipJoinStatus.mountPath( 'ipv4', Client.mcs ),
               'Irb::Multicast::Gmp::MembershipJoinStatus', 'r' )

      self.cliJoinStatusSysdb = mountGroup.mount(
            MembershipJoinStatusCli.mountPath( 'ipv4', Client.cli ),
            'Irb::Multicast::Gmp::MembershipJoinStatusCli', 'r' )

      self.renewStatus = mountGroup.mount(
            MembershipRenewStatus.mountPath( 'ipv4', Client.igmpsnooping ),
            'Irb::Multicast::Gmp::MembershipRenewStatus', 'w' )

      # Create a instance for CLI,
      # and convert MembershipJoinStatusCli => MembershipJoinStatus
      # for easy use in JoinStateImportSm
      self.cliJoinStatusShark = Tac.newInstance(
            'Irb::Multicast::Gmp::MembershipJoinStatus', 'cli' )

      self.mergedJoinStatus = Tac.newInstance(
            'Irb::Multicast::Gmp::MembershipJoinStatus', 'igmpsnoopingMerged' )

      self.evpnRenewStatus = mountGroup.mount(
            MembershipRenewStatus.mountPath( 'ipv4', Client.evpn ),
            'Irb::Multicast::Gmp::MembershipRenewStatus', 'r' )

      self.evpnStatus = mountGroup.mount( 'evpn/status', 'Evpn::EvpnStatus', 'r' )

      self.mlagStatus = mountGroup.mount( 'mlag/status', 'Mlag::Status', 'r' )
      self.mlagProtoStatus = mountGroup.mount( 'mlag/proto', 'Mlag::ProtoStatus',
                                               'r' )

      self.createLocalEntity( "TopoStatus",
                              "Bridging::Topology::Status",
                              "bridging/topology/status" )
      self.createLocalEntity( "AllIntfStatusDir",
                              "Interface::AllIntfStatusDir",
                              "interface/status/all" )
      self.createLocalEntity( "AllIntfStatusLocalDir",
                              "Interface::AllIntfStatusLocalDir",
                              Cell.path( 'interface/status/local' ) )
      self.createLocalEntity( "EthIntfStatusDir",
                              "Interface::EthIntfStatusDir",
                              "interface/status/eth/intf" )
      self.createLocalEntity( "EthIntfConfigDir",
                              "Interface::EthIntfConfigDir",
                              "interface/config/eth/intf" )

      self.localEntitiesCreatedIs( True )

      local = entityManager.root().parent
      t0( "Started IgmpSnoopingAgent." )
      self.root_ = local.newEntity( 'IgmpSnoopingAgent::Root',
                                    'IgmpSnoopingAgentRoot' )
      self.root_.agent = None

      self.standby = True
      self.pluginContext_ = None
      self.redundancyStatusReactor_ = None
 
      def _finish():
         t0( "Mounts complete." )
         self.brTopoStatus = \
                     self.entityManager.getLocalEntity( "bridging/topology/status" )
         self.allIntfStatusDir = \
                     self.entityManager.getLocalEntity( "interface/status/all" )
         self.allIntfStatusLocalDir = \
         self.entityManager.getLocalEntity \
            ( Cell.path( 'interface/status/local' ) )
         self.ethIntfStatusDir = \
                     self.entityManager.getLocalEntity( "interface/status/eth/intf" )
         self.intfConfigDir = \
                     self.entityManager.getLocalEntity( "interface/config/eth/intf" )

         self.membershipJoinStatusColl.status.addMember( self.joinStatus )
         self.membershipJoinStatusColl.status.addMember( self.staticJoinStatus )
         self.membershipJoinStatusColl.status.addMember( self.evpnJoinStatus )
         self.membershipJoinStatusColl.status.addMember( self.mlagJoinStatus )
         self.membershipJoinStatusColl.status.addMember( self.cliJoinStatusShark )
         self.membershipJoinStatusColl.status.addMember( self.mcsJoinStatus )
         self.membershipJoinStatusColl.statusCli = self.cliJoinStatusSysdb

         self.redundancyStatusReactor_ = RedundancyStatusReactor( \
            self.redundancyStatus(), self )
      shmemMg.doClose()
      if self.block:
         mountGroup.close( blocking=True )
         _finish()
      else:
         mountGroup.close( _finish )

   def initForActive( self ):
      t0( "initForActive" )

      t0( "dirs: ", self.snoopConfig, self.snoopStatus, self.coreConfig, 
                    self.coreStatus, self.brConfig, self.brTopoStatus, 
                    self.querierConfigDir, self.querierStatusDir )
      self.root_.agent = ( self.entityManager.sysname(),
                           self.snoopConfig, self.swForceConfig, 
                           self.snoopCounterConfig, self.snoopStatus, 
                           self.advSnoopStatus, self.coreConfig, self.coreStatus,
                           self.brConfig, self.allIntfStatusDir, 
                           self.allIntfStatusLocalDir, self.kniStatus,
                           self.ethIntfStatusDir, self.brTopoConfig,
                           self.brTopoStatus, self.querierConfigDir,
                           self.querierStatusDir, self.profileConfig,
                           self.counterDir, self.hwConfig, self.proxyConfig,
                           self.proxyIgmpHostDir, self.agentCmdRequestDirSm(),
                           self.eventMonConfig, self.eventMonStatus,
                           self.membershipJoinStatusColl, self.mergedJoinStatus,
                           self.renewStatus, self.evpnRenewStatus, self.evpnStatus,
                           None,
                           self.mlagStatus, self.mlagProtoStatus,
                           self.agentScheduler() )

      t0( "Loading IgmpSnoopingPlugins" )
      self.pluginContext_ = IgmpSnoopingPluginContext( self )
      Plugins.loadPlugins( 'IgmpSnoopingPlugin', self.pluginContext_ )

   def getDefaultTrace( self ):
      return 'Shark*/01,Smash*/01'

   def __del__( self ):
      if self.root_:
         self.root_.agent = None
         parent = self.root_.parent
         if parent:
            del parent.entry[self.root_.name]
         self.root_ = None
      for base in self.__class__.__bases__:
         if hasattr(base, "__del__"):
            base.__del__(self)

class RedundancyStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Redundancy::RedundancyStatus'

   def __init__( self, redundancyStatus, agent ):
      Tac.Notifiee.__init__( self, redundancyStatus )
      self.redundancyStatus_ = redundancyStatus
      self.agent_ = agent

      self.handleMode()

   @Tac.handler( 'protocol' )
   @Tac.handler( 'mode' )
   def handleMode( self ):
      t0( "handleMode: protocol=%s, mode=%s" % ( self.notifier_.protocol, 
                                                 self.notifier_.mode ) )
      if self.notifier_.protocol == "sso":
         if self.notifier_.mode == "active":
            self.agent_.standby = False
         else:
            self.agent_.standby = True
      else:
         self.agent_.standby = False

      if self.agent_.root_.agent is None and not self.agent_.standby:
         self.agent_.initForActive()

class IgmpSnoopingPluginContext( object ):
   def __init__( self, agent ):
      self.topAgent_ = weakref.proxy( agent )

   def agent( self ):
      return self.topAgent_.root_.agent

   def entityManager( self ):
      return self.topAgent_.entityManager

   def floodControlIs( self, fc ):
      t0( "New FloodControl:", fc )
      self.agent().glueSm.floodControlManager.floodControl.addMember( fc )

   def floodControlDel( self, fcName ):
      t0( "Deleting FloodControl:", fcName )
      del self.agent().glueSm.floodControlManager.floodControl[ fcName ]

   def intfStateControlIs( self, sc ):
      t0( "New IntfStateControl:", sc )
      self.agent().glueSm.intfStateControl.intfStateControl.addMember( sc )

   def intfStateControlDel( self, scName ):
      t0( "Deleting IntfStateControl:", scName )
      del self.agent().glueSm.intfStateControl[ scName ]

   def peerStatusIs( self, ps ):
      t0( "Add peer snooping status :" )
      self.agent().glueSm.peerStatus = ps

   def peerStatusDel( self ):
      t0( "Delete peer snooping status:" )
      self.agent().glueSm.peerStatus = None
   
   def snoopingSyncEnable( self ):
      t0( "Enable snooping sync:" )
      self.agent().glueSm.snoopingSync = True

   def snoopingSyncDisable( self ):
      t0( "Disable snooping sync:" )
      self.agent().glueSm.snoopingSync = False

   def mlagActiveIs( self, active, mi ):
      if active:
         t0( "Mlag is active" )
         if not self.agent().mlagPeerIntfDownSm:
            t0( "Starting mlagPeerIntfDownSm" )
            smArgs = ( mi.mlagStatus,
                       self.agent().protocolStatus,
                       self.agent().softwareForwardStatus,
                       self.agent().topoLibVlanState )
            self.agent().mlagPeerIntfDownSm = smArgs
      else:
         t0( "Mlag is inactive" )
         t0( "Deleting mlagPeerIntfDownSm" )
         self.agent().mlagPeerIntfDownSm = None

   def mlagInfoIs( self, mi ):
      t0( "New Mlag Information:", mi )
      self.agent().glueSm.mlagInfo = mi
      self.agent().agentSyncConfig.mlagInfo = mi

   def mlagInfoDel( self ):
      t0( "Deleting mlag information" )
      self.agent().glueSm.mlagInfo = None
      self.agent().agentSyncConfig.mlagInfo = None

   def handleDisabledIntf( self, intf ):
      t0( "handleDisabledIntf from plugin ctx:", intf )
      for glueVlanSm in self.agent().glueSm.vlanSm.values():
         coreSm = glueVlanSm.coreSm
         if coreSm:
            coreSm.handleDisabledIntf( intf, True )

   def getAgentSyncStatus( self ):
      return self.agent().agentSyncStatus

   def agentSyncEnabledIs( self, enabled ):
      t0( "Enabling Direct agent to agent sync" )
      self.agent().agentSyncConfig.setupDirectSync = enabled
      if enabled:
         mlagInfo = self.agent().agentSyncConfig.mlagInfo
         assert mlagInfo
         agentSyncConfig = self.agent().agentSyncConfig
         agentSyncStatus = self.agent().agentSyncStatus
         forwardingStateSmConfig = self.agent().forwardingStateSmConfig
         t0( 'Instantiate RemoteAgentSyncSm' )
         smArgs = ( agentSyncConfig, agentSyncStatus, forwardingStateSmConfig,
               self.agent().agentname )
         self.agent().agentSyncSm = smArgs

         t0( 'Instantiate MlagLeaveStateExportSm' )
         smArgs = ( self.agent().msgStatus, mlagInfo.mlagStatus,
                    self.agent().agentSyncStatus.localP2pLeaveStatus,
                    self.agent().scheduler )
         self.agent().mlagLeaveStateExportSm = smArgs
      else:
         self.agent().agentSyncSm = None
         self.agent().mlagLeaveStateExportSm = None

   def vxlanInfoIs( self, vx ):
      t0( "Vxlan Information:", vx )
      self.agent().glueSm.vxlanInfo = vx

   def vxlanInfoDel( self ):
      t0( "Deleting mlag information" )
      self.agent().glueSm.vxlanInfo = None

