#!/usr/bin/python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# A simple program that monitors interface link up/down events by listening
# on a netlink socket. Prints "<interface> state <UP/DOWN/UNKNOWN>" per
# line to stdout or a file.
#
# The message processing is not efficient as it copies rest of the message
# into the parsing variable. OK for low number of up/down events.
#
# Must be run as root. 
#

#import Tac
import sys, socket, struct

MaxBufSize = 18432

NETLINK_ROUTE = 0
sock = socket.socket( socket.AF_NETLINK, socket.SOCK_RAW, NETLINK_ROUTE )
sock.setsockopt( socket.SOL_SOCKET, socket.SO_RCVBUF, MaxBufSize )

# rtlink.h
RTMGRP_LINK = 1
RTM_NEWLINK = 16

sock.bind( ( 0, RTMGRP_LINK ) )

# if_link.h
IFLA_IFNAME = 3
IFLA_OPERSTATE = 16

# if.h
IF_OPER_UNKNOWN = 0
IF_OPER_DOWN = 2
IF_OPER_UP = 6

if len( sys.argv ) == 2 :
   fh = open( sys.argv[1], "w" )
   if not fh:
      exit(1)
else:
   fh = sys.stdout

print >> fh, "initialized"
fh.flush()
   
while True:
   data = sock.recv( MaxBufSize )
   if not data:
      print >> sys.stderr, "Error: recv"
      exit(1)

   msgs = []
   while data:
      # Netlink message header is IHHII
      msglen, msgtype, flags, seq, pid = struct.unpack( "IHHII", data[:16] )
      if len( data ) < msglen:
         print >> sys.stderr, \
            "Error: netlink msg len %s:%s" % ( len( data ), msglen )
         exit(1)
      msgs.append( ( msgtype, flags, seq, pid, data[16:msglen] ) )
      data = data[msglen:]

   events = []
   for msg in msgs:
      if msg[0] == RTM_NEWLINK:
         rta = msg[4]
         # struct ifinfomsg
         ifi_family, ifi_type, ifi_index, ifi_flags, ifi_change = \
            struct.unpack( "BHiII", rta[:16] )
         
         rta = rta[16:]
         ifName = None
         ifOperState = None
         
         while rta:
            # struct rtattr
            rtaLen, rtaType = struct.unpack( "HH", rta[:4] )
            #print "rtaLen %s, type %s" % ( rtaLen, rtaType )
            if rtaType == IFLA_IFNAME:
               strLen = rtaLen - 4 - 1 # don't care about null at the end
               ifName, = struct.unpack( "%ds" % strLen, rta[4:rtaLen-1] )
            elif rtaType == IFLA_OPERSTATE:
               if rtaLen != 5:
                  print >> sys.stderr, \
                     "Error: rtaLen for IFLA_OPERSTATE %s" % rtaLen
                  exit(1)
               ifOperState = ord( rta[4] )
            rtaLen = ( rtaLen + 3 ) & ( ~3 ) # alignment
            rta = rta[rtaLen:]
            
         if ifOperState == IF_OPER_UP:
            ifOperState = "state UP"
         elif ifOperState == IF_OPER_DOWN:
            ifOperState = "state DOWN"
         elif ifOperState == IF_OPER_UNKNOWN:
            ifOperState = "state UNKNOWN"
         events.append( ( ifName, ifOperState ) )
         
   for e in events:
      print >> fh, "%s %s" % ( e[0], e[1] )
      fh.flush()
