#!/usr/bin/env python
# Copyright (c) 2008-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
Generate valid ID EEPROM contents against which we can run some Xcvr
breadth tests, and which we can use on a real switch to simulate the
presence of such transceivers.

Templates for the ID EEPROM contents are available in the Sol package,
under manufacturing/XcvrIdprom/...
'''

from __future__ import absolute_import, division, print_function
import Tac
import datetime, random, binascii


idPromLen = 128
# The revision number is defined as an Enum in /Xcvr/HalSff8436.tac
sff8636rev15 = 5
sff8636rev20 = 6


SRv1001SFP = [
        0x03,
        0x04,
        0x07,
        0x10,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x06,
        0x67,
        0x00,
        0x00,
        0x00,
        0x08,
        0x03,
        0x00,
        0x1E,
        'vn', # 'vn' = Vendor name
        'vn',
        'vn',
        'vn',
        'vn',
        'vn',
        'vn',
        'vn',
        'vn',
        'vn',
        'vn',
        'vn',
        'vn',
        'vn',
        'vn',
        'vn',
        0x00,
        0x00,
        0x1C,
        0x73,
        0x53,
        0x46,
        0x50,
        0x2D,
        0x31,
        0x30,
        0x47,
        0x2D,
        0x53,
        0x52,
        0x20,
        0x20,
        0x20,
        0x20,
        0x20,
        0x20,
        0x30,
        0x30,
        0x30,
        0x31,
        0x03,
        0x52,
        0x00,
        'cc', # 'ccBase' ('cc' = 'checksum' (CheckCode?))
        0x00,
        0x1a,
        0x00,
        0x00,
        0x58, # First byte of SN = "X" (0x58)
        'M',
        'M',
        'Y',
        'Y',
        'W',
        'W',
        'Z',
        'Z',
        'Z',
        'Z',
        'Z',
        0x20,
        0x20,
        0x20,
        0x20,
        'Y',  # First byte of Date Code
        'Y',
        'M',
        'M',
        'D',
        'D',
        'L',
        'L',
        0x68, # 'diagsMonitoringType'
        0xf0,
        0x03,
        'cc', # 'ccExt'
        0x10,
        0x01,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0x00,
        0xA2,  # SolRegBase -- typically 0xA278
        0x78,
        0xA2,  # MBase -- typically 0xA280 or 0x0000
        0x80,
        0x00,  # HBase -- varies: 0x0000, 0xA294, 0xA296, ...
        0x00,
        'crc',
        'crc',
        'crc',
        'crc',
]

# SRv1002SFP has both Mbase and HBase
SRv1002SFP = SRv1001SFP[ : ]
SRv1002SFP[ 118 : 124 ] = [ 0xA2, 0x78, 0xA2, 0x80, 0xA2, 0x96 ]

# SRv1003SFP has neither Mbase nor HBase
SRv1003SFP = SRv1001SFP[ : ]
SRv1003SFP[ 118 : 124 ] = [ 0xA2, 0x78, 0x00, 0x00, 0x00, 0x00 ]

# SRv1003SOLSOLREGBASESFP has neither Solregbase nor Mbase nor HBase
SRv1003NOSOLREGBASESFP = SRv1001SFP[ : ]
SRv1003NOSOLREGBASESFP[ 118 : 124 ] = [ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 ]

SRv1001nondomSFP = SRv1001SFP[:]
SRv1001nondomSFP[92] = 0x28 # disable DOM

# SFP+ SRL has length < 30
SRLSFP = SRv1002SFP[ : ]
SRLSFP[ 19 ] = 0x10

# SFP+ AOC is an -SR with the active bit set
AOCSFP = SRv1002SFP[ : ]
AOCSFP[ 8 ] = 0x08
AOCSFP[ 60 : 62 ] = [ 0x0C, 0x00 ] # set cable compliance bits in wavelength field

# Create an eeprom copy of SRv1001SFP above, except with DOM
# externally calibrated.
SRv1001SFPExternal = SRv1001SFP[ : ]
# Set the diagnostic monitoring type to be externally calibrated.
# Externally calibrated: 0x58, internally calibrated: 0x68.
SRv1001SFPExternal[ 92 ] = 0x58
# Set the checksum to the appropriate value. It is 0x10 less than the
# checksum in the SRv1001SFP.
SRv1001SFPExternal[ 95 ] = int( SRv1001SFP[ 95 ], 16 ) - 0x10

SRv1001SFPBadCc = SRv1001SFP[ : ]

SRSFP25G = SRv1001SFP[ : ]
SRSFP25G[ 3 : 11 ] = [ 0x00 ] * 8 # No 10G support
SRSFP25G[ 12 ] = 0xff # 25G speed
SRSFP25G[ 14 : 18 ] = [ 0x00 ] * 4
SRSFP25G[ 18 : 20 ] = [ 0x0a, 0x07 ] # 25G SFP has length < 70m
SRSFP25G[ 36 ] = 0x02
SRSFP25G[ 44 : 46 ] = [ 0x32, 0x35 ] # SFP-25G-SR
SRSFP25G[ 66 ] = 0x67 # 25G speed

XSRSFP25G = SRSFP25G[ : ]
XSRSFP25G[ 18 : 20 ] = [ 0x1e, 0x14 ] # 25G-XSR SFP has length < 300m

MRSRSFP25G = SRv1001SFP[ : ] # dual-rate 10G/25G SFP-25G-MR-SR
MRSRSFP25G[ 3 ] = 0x10  # 10GBASE-SR
MRSRSFP25G[ 4 : 11 ] = [ 0x00 ] * 7
MRSRSFP25G[ 11 ] = 0x06 
MRSRSFP25G[ 12 ] = 0xff # 25G speed
MRSRSFP25G[ 13 : 18 ] = [ 0x00 ] * 5
MRSRSFP25G[ 18 : 20 ] = [ 0x1e, 0x14 ] # SFP-25G-MR-SR has length < 300m
MRSRSFP25G[ 36 ] = 0x02
MRSRSFP25G[ 44 : 54 ] = [ 0x32, 0x35, 0x47, 0x2d, 
                          0x4d, 0x52, 0x2d, 0x58, 
                          0x53, 0x52 ] # vendor PN SFP-25G-MR-SR
MRSRSFP25G[ 66 ] = 0x67 # 25G speed
MRSRSFP25G[ 93 ] = 0xf8 # Enhanced Options
MRSRSFP25G[ 110 ] = 0x08 # Default to 25G speed 

ARSFP25G = SRv1001SFP[ : ]
ARSFP25G[ 2 ] = 0x23 # no separable connector
ARSFP25G[ 3 : 11 ] = [ 0x00 ] * 8 # No infiniband compliance
ARSFP25G[ 12 ] = 0xff # 25G speed
ARSFP25G[ 14 : 18 ] = [ 0x00 ] * 4
ARSFP25G[ 18 : 20 ] = [ 0x03, 0x00 ] # 3 meter AOC
ARSFP25G[ 36 ] = 0x1
ARSFP25G[ 40 : 56 ] = [ 0x41, 0x4f, 0x43, 0x2d, 0x53,
                        0x2d, 0x53, 0x2d, 0x32, 0x35,
                        0x47, 0x2d, 0x33, 0x4d, 0x20,
                        0x20 ] # AOC-S-S-25G-3M
ARSFP25G[ 60 : 62 ] = [ 0x03, 0x52 ] # 850nm
ARSFP25G[ 66 ] = 0x67 # 25G speed

ARSFP25G3RDPTY = ARSFP25G[ : ]
ARSFP25G3RDPTY[ 8 ] = 0x8

ARNSFP25G = ARSFP25G[ : ]
ARNSFP25G[ 36 ] = 0x18

LRSFP25G = SRv1001SFP[ : ]
LRSFP25G[ 3 : 11 ] = [ 0x00 ] * 8 # No 10G support
LRSFP25G[ 12 ] = 0xff # 25G speed
LRSFP25G[ 14 : 16 ] = [ 0x0a, 0x64 ] # 25G SFP LR has length < 10km
LRSFP25G[ 16 : 20 ] = [ 0x00 ] * 4
LRSFP25G[ 36 ] = 0x03
LRSFP25G[ 44 : 46 ] = [ 0x32, 0x35 ]
LRSFP25G[ 48 : 50 ] = [ 0x4c, 0x52 ] # SFP-25G-LR
LRSFP25G[ 60 : 62 ] = [ 0x05, 0x1e ] # Wavelength 1310nm
LRSFP25G[ 66 ] = 0x67 # 25G speed

LRLSFP25G = SRv1001SFP[ : ]
LRLSFP25G[ 3 : 11 ] = [ 0x00 ] * 8 # No 10G support
LRLSFP25G[ 12 ] = 0xff # 25G speed
LRLSFP25G[ 14 : 16 ] = [ 0x02, 0x14 ] # 25G SFP LR has length < 2km
LRLSFP25G[ 16 : 20 ] = [ 0x00 ] * 4
LRLSFP25G[ 36 ] = 0x03
LRLSFP25G[ 44 : 46 ] = [ 0x32, 0x35 ]
LRLSFP25G[ 48 : 51 ] = [ 0x4c, 0x52, 0x4c ] # SFP-25G-LRL
LRLSFP25G[ 60 : 62 ] = [ 0x05, 0x1e ] # Wavelength 1310nm
LRLSFP25G[ 66 ] = 0x67 # 25G speed

MRLRSFP25G =SRv1001SFP[ : ]
MRLRSFP25G[ 3 ] = 0x20 # 10GBASE-LR
MRLRSFP25G[ 4 : 11 ] = [ 0x00 ] * 7
MRLRSFP25G[ 12 ] = 0xff # 25G speed
MRLRSFP25G[ 14 : 16 ] = [ 0x0a, 0x64 ] # 25G SFP MR LR has length < 10km
MRLRSFP25G[ 16 : 20 ] = [ 0x00 ] * 4
MRLRSFP25G[ 36 ] = 0x03
MRLRSFP25G[ 44 : 54 ] = [ 0x32, 0x35, 0x47, 0x2d, 0x4d,
                          0x52, 0x2d, 0x4c, 0x52, 0x20 ] # Vendor PN SFP-25G-MR-LR
MRLRSFP25G[ 60: 62 ] = [ 0x05, 0x1e ] # Wavelength 1310nm
MRLRSFP25G[ 66 ] = 0x67 # 25G speed
MRLRSFP25G[ 93 ] = 0xf8 # Enhanced Options
MRLRSFP25G[ 110 ] = 0x08 # Default to 25G speed

UNKNOWNSFP25G = SRSFP25G[ : ]
UNKNOWNSFP25G[ 36 ] = 0

# BX-U SFP
BXUSFP = [
   0x03, 0x04, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x01, 0x0d, 0x00, 0x0a, 0x64,
   0x00, 0x00, 0x00, 0x00, 0x4f, 0x45, 0x4d, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x90, 0x65,
   0x53, 0x46, 0x50, 0x2d, 0x42, 0x58, 0x2d, 0x55,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x31, 0x31, 0x2e, 0x30, 0x05, 0x1e, 0x00, 0x15,
   0x00, 0x1a, 0x00, 0x00, 0x53, 0x31, 0x33, 0x31,
   0x30, 0x30, 0x36, 0x37, 0x31, 0x36, 0x33, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x31, 0x33, 0x31, 0x31,
   0x31, 0x34, 0x20, 0x20, 0x68, 0x80, 0x01, 0x5d,
   0x2d, 0x00, 0x11, 0x15, 0x82, 0x40, 0xd0, 0xf7,
   0x0d, 0xe8, 0xea, 0x13, 0x54, 0x45, 0x19, 0x1f,
   0xe2, 0x14, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x61, 0xb7, 0xbb, 0x7f
]

# BX-D SFP
BXDSFP = BXUSFP[ : ]
BXDSFP[ 60 : 62 ] = [ 0x05, 0xD2 ]

LRv1001SFP = SRv1001SFP[:]
LRv1001SFP[3] = 0x20
LRv1001SFP[14] = 0x10
LRv1001SFP[40] = 'S'
LRv1001SFP[41] = 'F'
LRv1001SFP[42] = 'P'
LRv1001SFP[43] = '-'
LRv1001SFP[44] = '1'
LRv1001SFP[45] = '0'
LRv1001SFP[46] = 'G'
LRv1001SFP[47] = '-'
LRv1001SFP[48] = 'L'
LRv1001SFP[49] = 'R'
LRv1001SFP[50] = ' '
LRv1001SFP[51] = ' '
LRv1001SFP[60] = 0x05
LRv1001SFP[61] = 0x1E

# LRL has length less than 10
LRLSFP = LRv1001SFP[:]
LRLSFP[14] = 0x05

# ER has length less than 40
ERSFP = LRv1001SFP[:]
ERSFP[3] = 0x80
ERSFP[14] = 0x24

# ERBDU ( BiDi upstream ) has transmit wavelength of 1270nm
ERBDUSFP = ERSFP[:]
ERBDUSFP[60] = 0x04
ERBDUSFP[61] = 0xF6
# ERBDD ( BiDi Downstream ) has transmit wavelength of 1330nm
ERBDDSFP = ERSFP[:]
ERBDDSFP[60] = 0x05
ERBDDSFP[61] = 0x32

# ERL's have length less than 30
ERLSFP = LRv1001SFP[:]
ERLSFP[3] = 0x80
ERLSFP[14] = 0x1D

# ERBDU ( BiDi upstream ) has transmit wavelength of 1270nm
ERLBDUSFP = ERLSFP[:]
ERLBDUSFP[60] = 0x04
ERLBDUSFP[61] = 0xF6
# ERBDD ( BiDi Downstream ) has transmit wavelength of 1330nm
ERLBDDSFP = ERLSFP[:]
ERLBDDSFP[60] = 0x05
ERLBDDSFP[61] = 0x32


# ZR has length over 40
ZRSFP = ERSFP[:]
ZRSFP[14] = 0x30

LRMv1001SFP = SRv1001SFP[:]
LRMv1001SFP[3] = 0x40
LRMv1001SFP[40] = 'S'
LRMv1001SFP[41] = 'F'
LRMv1001SFP[42] = 'P'
LRMv1001SFP[43] = '-'
LRMv1001SFP[44] = '1'
LRMv1001SFP[45] = '0'
LRMv1001SFP[46] = 'G'
LRMv1001SFP[47] = '-'
LRMv1001SFP[48] = 'L'
LRMv1001SFP[49] = 'R'
LRMv1001SFP[50] = 'M'
LRMv1001SFP[51] = ' '
LRMv1001SFP[60] = 0x05
LRMv1001SFP[61] = 0x1E

CRSFP = SRv1001SFP[:]
CRSFP[3] = 0x1
CRSFP[40] = 'S'
CRSFP[41] = 'F'
CRSFP[42] = 'P'
CRSFP[43] = '-'
CRSFP[44] = '1'
CRSFP[45] = '0'
CRSFP[46] = 'G'
CRSFP[47] = '-'
CRSFP[48] = 'C'
CRSFP[49] = 'R'
CRSFP[50] = ' '
CRSFP[51] = ' '

# Note that the only difference between CRSFP and CRSFPP is that
# CRSFPP explicitly has the passive bit (8.2) set, but CRSFP doesnt
# and is considered passive by default

CRSFPP = CRSFP[:]
CRSFPP[8] = 0x4

CRSFPA = CRSFP[:]
CRSFPA[8] = 0x8

LXv1001SFP = SRv1001SFP[:]
LXv1001SFP[3] = 0x00
LXv1001SFP[4] = 0x00
LXv1001SFP[5] = 0x00
LXv1001SFP[6] = 0x02
LXv1001SFP[11] = 0x01
LXv1001SFP[12] = 0x0C
LXv1001SFP[14] = 0x0A
LXv1001SFP[15] = 0x64
LXv1001SFP[16] = 0x37
LXv1001SFP[17] = 0x37
LXv1001SFP[19] = 0x00
LXv1001SFP[40] = 'S'
LXv1001SFP[41] = 'F'
LXv1001SFP[42] = 'P'
LXv1001SFP[43] = '-'
LXv1001SFP[44] = '1'
LXv1001SFP[45] = 'G'
LXv1001SFP[46] = '-'
LXv1001SFP[47] = 'L'
LXv1001SFP[48] = 'X'
LXv1001SFP[49] = ' '
LXv1001SFP[50] = ' '
LXv1001SFP[51] = ' '
# XXX_LWR: This is incomplete -- wavelength, etc., aren't overridden from SR

SXSFP = LXv1001SFP[:]
SXSFP[6] = 1

LX10SFP = LXv1001SFP[:]
LX10SFP[6] = 0x10

LXv1001nondomSFP = LXv1001SFP[:]
LXv1001nondomSFP[92] = 0x28 # disable DOM

# Third-party dual-speed 1G/10G SFPs
SRDualSpeed = SRv1001SFP[ : ]
SRDualSpeed[6] = 0x01 # Add 1000BASE-SX support

LRDualSpeed = LRv1001SFP[ : ]
LRDualSpeed[6] = 0x02 # Add 1000BASE-LX support

# XXX_RMO Adding this case to test the important part of the HP Twin Ax cable.
# For now, this is only used in test test/t001.py just to make sure we continue
# to support HP TwinAx.
HPTwinAx = SRv1001SFP[:]
HPTwinAx[3] = 0x81

# XXX_RMO Adding this for testing DWDM optics. Only changing the fields that
# enable us to detect the DWDM optic and report wavelength.
SFPDWDM = SRv1001SFP[:]
SFPDWDM[0] = 0x0B
SFPDWDM[1] = 0x24
SFPDWDM[3] = 0x80
SFPDWDM[13] = 0x00
SFPDWDM[14] = 0x28
SFPDWDM[60] = 0x06
SFPDWDM[61] = 0x13
SFPDWDM[62] = 0x09

# XXX_RMO Adding the new case for ZR which differ from ER by one field.
SFPDWDMZR = SFPDWDM[:]
SFPDWDMZR[14] = 0x50

SFPDWDMZT = SRv1001SFP[:]
SFPDWDMZT[0] = 0x0B
SFPDWDMZT[1] = 0x04
SFPDWDMZT[3] = 0x80
SFPDWDMZT[13] = 0x00
SFPDWDMZT[14] = 0x50
# The wavelength field (byte 60-61) is not supported for tunable SFP+.
SFPDWDMZT[60] = 0x00
SFPDWDMZT[61] = 0x00
SFPDWDMZT[62] = 0x00
SFPDWDMZT[65] = 0x5A

CRSFP25G = CRSFP[:]
CRSFP25G[12] = 0xff
CRSFP25G[36] = 0x0B
CRSFP25G[44] = '2'
CRSFP25G[45] = '5'
CRSFP25G[66] = 0x67

CRSSFP25G = CRSFP25G[:]
CRSSFP25G[36] = 0x0C

CRNSFP25G = CRSFP25G[:]
CRNSFP25G[36] = 0x0D

CRSFP50G = CRSFP25G[:]
CRSFP50G[36] = 0x40

# Note that the only difference between CRSFP25G and CRSFPP25G is that
# CRSFPP25G explicitly has the passive bit (8.2) set, but CRSFP25G doesnt
# and is considered passive by default

CRSFPP25G = CRSFP25G[:]
CRSFPP25G[8] = 0x4 # Passive copper
CRSFPP25G[36] = 0xff # Extended compliance to invalid value

CRSFPP50G = CRSFP50G[:]
CRSFPP50G[8] = 0x4 # Passive copper
CRSFPP50G[36] = 0xff # Extended compliance to invalid value
CRSFPP50G[66] = 0xCE # BR max to > 200

# 1000BASE-T
Tv2SFP = [
   0x03,  # 0
   0x04,
   0x22,
   0x00,
   0x00,
   0x00,  # 5
   0x08,
   0x00,
   0x00,
   0x00,
   0x00,  # 10
   0x01,
   0x0C,
   0x00,
   0x00,
   0x00,  # 15
   0x00,
   0x00,
   0x64,
   0x00,
   0x41,  # 20
   0x72,
   0x69,
   0x73,
   0x74,
   0x61,  # 25
   0x20,
   0x4E,
   0x65,
   0x74,
   0x77,  # 30
   0x6F,
   0x72,
   0x6B,
   0x73,
   0x20,  # 35
   0x00,
   0x00,
   0x1C,
   0x73,
   0x53,  # 40
   0x46,
   0x50,
   0x2D,
   0x31,
   0x47,  # 45
   0x2D,
   0x54,
   0x20,
   0x20,
   0x20,  # 50
   0x20,
   0x20,
   0x20,
   0x20,
   0x20,  # 55
   0x30,
   0x30,
   0x30,
   0x32,
   0x00,  # 60
   0x00,
   0x00,
   'cc',
   'ms',
   0X02 ,  # 65, RX_LOS[1] implemented and TX_FAULT[3] not implemented
   0x00,
   0x00,
   0x58,
   'M',
   'M',  # 70
   'Y',
   'Y',
   'W',
   'W',
   'Z',  # 75
   'Z',
   'Z',
   'Z',
   'Z',
   0x20,  # 80
   0x20,
   0x20,
   0x20,
   'Y',
   'Y',  # 85
   'M',
   'M',
   'D',
   'D',
   'L',  # 90
   'L',
   0x00,
   'ms',
   0x00,
   'cc',  # 95
   0x01,
   'aa',
   'csum',
   'csum',
   'csum',  # 100
   'csum',
   'csum',
   'csum',
   'csum',
   'csum',  # 105
   'csum',
   'csum',
   'csum',
   'csum',
   'csum',  # 110
   'csum',
   'csum',
   'csum',
   'csum',
   'csum',  # 115
   'csum',
   'csum',
   0x00,
   0x00,
   0x00,  # 120
   0x00,
   0x00,
   0x00,
   'crc',
   'crc',  # 125
   'crc',
   'crc',
]

# 3rd party 1000Base-T SFP may have non-zero extended compliance code
TSFP3RDPTY = [
   0x03,  # 0
   0x04,
   0x00,
   0x00,
   0x00,
   0x00,  # 5
   0x08,
   0x00,
   0x00,
   0x00,
   0x00,  # 10
   0x01,
   0x0D,
   0x00,
   0x00,
   0x00,  # 15
   0x00,
   0x00,
   0x64,
   0x00,
   0x41,  # 20
   0x72,
   0x69,
   0x73,
   0x74,
   0x61,  # 25
   0x20,
   0x4E,
   0x65,
   0x74,
   0x77,  # 30
   0x6F,
   0x72,
   0x6B,
   0x73,
   0x20,  # 35
   0x01,
   0x00,
   0x1C,
   0x73,
   0x53,  # 40
   0x46,
   0x50,
   0x2D,
   0x31,
   0x47,  # 45
   0x2D,
   0x54,
   0x20,
   0x20,
   0x20,  # 50
   0x20,
   0x20,
   0x20,
   0x20,
   0x20,  # 55
   0x30,
   0x30,
   0x30,
   0x32,
   0x41,  # 60
   0x0C,
   0xC1,
   'cc',
   0x00,
   0x01,  # 65
   0x00,
   0x00,
   0x58,
   'M',
   'M',  # 70
   'Y',
   'Y',
   'W',
   'W',
   'Z',  # 75
   'Z',
   'Z',
   'Z',
   'Z',
   0x20,  # 80
   0x20,
   0x20,
   0x20,
   'Y',
   'Y',  # 85
   'M',
   'M',
   'D',
   'D',
   'L',  # 90
   'L',
   0x00,
   0x00,
   0x00,
   'cc',  # 95
   0x01,
   'aa',
   'csum',
   'csum',
   'csum',  # 100
   'csum',
   'csum',
   'csum',
   'csum',
   'csum',  # 105
   'csum',
   'csum',
   'csum',
   'csum',
   'csum',  # 110
   'csum',
   'csum',
   'csum',
   'csum',
   'csum',  # 115
   'csum',
   'csum',
   0x00,
   0x00,
   0x00,  # 120
   0x00,
   0x00,
   0x00,
   'crc',
   'crc',  # 125
   'crc',
   'crc',
]

TSFP10G = [
   0x3 ,  # 0
   0x4 ,
   0x22 ,
   0x0 ,
   0x0 ,
   0x0 ,  # 5
   0x0 ,
   0x0 ,
   0x0 ,
   0x0 ,
   0x0 ,  # 10
   0x6 ,
   0x67 ,
   0x0 ,
   0x0 ,
   0x0 ,  # 15
   0x0 ,
   0x0 ,
   0x1e ,
   0x0 ,
   0x41 ,  # 20
   0x72 ,
   0x69 ,
   0x73 ,
   0x74 ,
   0x61 ,  # 25
   0x20 ,
   0x4e ,
   0x65 ,
   0x74 ,
   0x77 ,  # 30
   0x6f ,
   0x72 ,
   0x6b ,
   0x73 ,
   0x20 ,  # 35
   0x16 ,
   0x00 ,
   0x1c ,
   0x73 ,
   0x53 ,  # 40
   0x46 ,
   0x50 ,
   0x2d ,
   0x31 ,
   0x47 ,  # 45
   0x2d ,
   0x54 ,
   0x20 ,
   0x20 ,
   0x20 ,  # 50
   0x20 ,
   0x20 ,
   0x20 ,
   0x20 ,
   0x20 ,  # 55
   0x30 ,
   0x30 ,
   0x30 ,
   0x31 ,
   0x0 ,  # 60
   0x0 ,
   0x0 ,
   0 ,
   0 ,
   0X0A ,  # 65, RX_LOS[1] and TX_FAULT[3] implemented
   0x0 ,
   0x0 ,
   0x58 ,
   0 ,
   0 ,  # 70
   0 ,
   0 ,
   0 ,
   0 ,
   0 ,  # 75
   0 ,
   0 ,
   0 ,
   0 ,
   0x20 ,  # 80
   0x20 ,
   0x20 ,
   0x20 ,
   0 ,
   0 ,  # 85
   0 ,
   0 ,
   0 ,
   0 ,
   0 ,  # 90
   0 ,
   0x0 ,
   0 ,
   0x8 ,
   0 ,  # 95
   0x0 ,
   0 ,
   0 ,
   0 ,
   0 ,  # 100
   0 ,
   0 ,
   0 ,
   0 ,
   0 ,  # 105
   0 ,
   0 ,
   0 ,
   0 ,
   0 ,  # 110
   0 ,
   0 ,
   0 ,
   0 ,
   0 ,  # 115
   0 ,
   0 ,
   0x0,
   0x0,
   0x0,  # 120
   0x0,
   0x0,
   0x0,
   0 ,
   0 ,  # 125
   0 ,
   0 ,
   # 128-255 reserved 
]

AQS107_TSFP10GSR = [
   0x3, # 0
   0x4,
   0x22,
   0x0,
   0x0,
   0x0, # 5
   0x0,
   0x0,
   0x0,
   0x0,
   0x0, # 10
   0x0,
   0x64,
   0x0,
   0x0,
   0x0, # 15
   0x0,
   0x0,
   0x1e,
   0x0,
   0x41, # 20
   0x51,
   0x55,
   0x41,
   0x44,
   0x54, # 25
   0x49,
   0x41,
   0x20,
   0x20,
   0x20, # 30
   0x20,
   0x20,
   0x20,
   0x20,
   0x20, # 35
   0x1c,
   0x00,
   0x17,
   0xb6,
   0x41, # 40
   0x51,
   0x53,
   0x2d,
   0x31,
   0x30, # 45
   0x37,
   0x2d,
   0x43,
   0x58,
   0x2d, # 50
   0x31,
   0x20,
   0x20,
   0x20,
   0x20, # 55
   0x30,
   0x31,
   0x20,
   0x20,
   0x0, # 60
   0x0,
   0x0,
   0xd9,
   0x28,
   0x0, # 65
   0x0,
   0x0,
   0x31,
   0x39,
   0x32, # 70
   0x30,
   0x30,
   0x30,
   0x30,
   0x36, # 75
   0x32,
   0x32,
   0x20,
   0x20,
   0x20, # 80
   0x20,
   0x20,
   0x20,
   0x31,
   0x39, # 85
   0x30,
   0x35,
   0x31,
   0x33,
   0x20, # 90
   0x20,
   0x0,
   0x70,
   0x0,
   0xc1, # 95
   0x0,
   0,
   0,
   0,
   0, # 100
   0,
   0,
   0,
   0,
   0, # 105
   0,
   0,
   0,
   0,
   0, # 110
   0,
   0,
   0,
   0,
   0, # 115
   0,
   0,
   0x0,
   0x0,
   0x0, # 120
   0x0,
   0x0,
   0x0,
   0,
   0, # 125
   0,
   0,
   # 128-255 reserved
]

Molex_74765_0902 = [
   0x3 , # 0
   0x4 ,
   0x80 ,
   0x1 ,
   0x0 ,
   0x0 , # 5
   0x4 ,
   0x41 ,
   0x84 ,
   0x80 ,
   0xd5 , # 10
   0x0 ,
   0x78 ,
   0x0 ,
   0x0 ,
   0x0 , # 15
   0x0 ,
   0x0 ,
   0x0 ,
   0x0 ,
   0x4d , # 20
   0x6f ,
   0x6c ,
   0x65 ,
   0x78 ,
   0x20 , # 25
   0x49 ,
   0x6e ,
   0x63 ,
   0x2e ,
   0x20 , # 30
   0x20 ,
   0x20 ,
   0x20 ,
   0x20 ,
   0x20 , # 35
   0x0 ,
   0x0 ,
   0x9 ,
   0x3a ,
   0x37 , # 40
   0x34 ,
   0x37 ,
   0x36 ,
   0x35 ,
   0x2d , # 45
   0x30 ,
   0x39 ,
   0x30 ,
   0x32 ,
   0x20 , # 50
   0x20 ,
   0x20 ,
   0x20 ,
   0x20 ,
   0x20 , # 55
   0x41 ,
   0x20 ,
   0x20 ,
   0x20 ,
   0x0 , # 60
   0x0 ,
   0x0 ,
   'cc',
   'ms',
   'ms',  # 65
   0x0 ,
   0x0 ,
   0x38 ,
   'M',
   'M',  # 70
   'Y',
   'Y',
   'W',
   'W',
   'Z',  # 75
   'Z',
   'Z',
   'Z',
   'Z',
   0x20 , # 80
   0x20 ,
   0x20 ,
   0x20 ,
   'Y',
   'Y',  # 85
   'M',
   'M',
   'D',
   'D',
   'L',  # 90
   'L',
   0x0 ,
   'ms' ,
   0x0 ,
   'cc' , # 95
   0x0 ,
   'aa' ,
   'csum',
   'csum',
   'csum',  # 100
   'csum',
   'csum',
   'csum',
   'csum',
   'csum',  # 105
   'csum',
   'csum',
   'csum',
   'csum',
   'csum',  # 110
   'csum',
   'csum',
   'csum',
   'csum',
   'csum',  # 115
   'csum',
   'csum',
   0x0 ,
   0x0 ,
   0x0 , # 120
   0x0 ,
   0x0 ,
   0x0 ,
   'crc' ,
   'crc' , # 125
   'crc' ,
   'crc' 
]


# This is for QSFP+.  This page reflects only the page 00 upper page
# (address 128-255) as this is the only part we look at for basic
# id eeprom information. The lower page is control and status and 
# the upper page, provided here, is the static information.
qsfpVendorNameOffset = 148 - 128
qsfpVendorNameLength = 16
qsfpVendorPartNumOffset = 168 - 128
qsfpVendorPartNumLength = 16
qsfpVendorRevNumOffset = 184 - 128
qsfpVendorRevNumLength = 2
qsfpSerialNumOffset = 196 - 128
qsfpSerialNumLength = 16
qsfpDateCodeOffset = 212 - 128
qsfpDateCodeLength = 8
qsfpBaseChecksumOffset = 191 - 128
qsfpExtChecksumOffset = 223 - 128

sff8436VendorInfoDefs = { 
   'vendorName' : ( qsfpVendorNameOffset, qsfpVendorNameLength ),
   'vendorPN' : ( qsfpVendorPartNumOffset, qsfpVendorPartNumLength ),
   'vendorRev' : ( qsfpVendorRevNumOffset, qsfpVendorRevNumLength ),
   'vendorSN' : ( qsfpSerialNumOffset, qsfpSerialNumLength ),
   'vendorDateCode' : ( qsfpDateCodeOffset, qsfpDateCodeLength ) }

# Upper page 0
SRQSFP = [
   0x0D, # -128-
   0x00,
   0x00,
   0x04,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00, # -136-
   0x00,
   0x00,
   0x05,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00, # -144-
   0x00,
   0x00,
   0x00,
   'vn',
   'vn',
   'vn',
   'vn',
   'vn', # -152-
   'vn',
   'vn',
   'vn',
   'vn',
   'vn',
   'vn',
   'vn',
   'vn', # -160-
   'vn',
   'vn',
   'vn',
   0x00,
   0x00,
   0x1C,
   0x73,
   0x53, # -168-
   0x46,
   0x50,
   0x2D,
   0x34,
   0x30,
   0x47,
   0x2D,
   0x53, # -176-
   0x52,
   0x20,
   0x20,
   0x20,
   0x20,
   0x20,
   0x20,
   0x30, # -184-
   0x31,
   0x42,
   0x68,
   0x00,
   0x00,
   0x00,
   'cc',
   0x00, # -192-
   0x00,
   0x00,
   0x00,
   0x58,
   'M', 
   'M', 
   'Y', 
   'Y', # -200-
   'W', 
   'W', 
   'Z', 
   'Z', 
   'Z', 
   'Z', 
   'Z', 
   0x20, # -208-
   0x20,
   0x20,
   0x20,
   'y',
   'y',
   'm',
   'm',
   'd', # -216-
   'd',
   'lc',
   'lc',
   0x00, # -220-
   0x00,
   0x00,
   'cc',
   0x00, # -224-
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00, # -232-
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00, # -240-
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,
   0x02, # SolRegBase -- typically 0x02f8
   0xf8,
   0x02, # -248- MBase -- typically 0x02CE or 0x0000
   0xce,
   0x00, # HBase -- typically 0x02E4, 0x0000
   0x00,
   0x00,
   0x00,
   0x00,
   0x00
]

# SRQSFP1002 has both Mbase and HBase
SRQSFP1002 = SRQSFP[:]
SRQSFP1002[ 118 : 124 ] = [ 0x02, 0xf8, 0x02, 0xce, 0x02, 0xe4 ]

# SRQSFP1003 has neither Mbase nor HBase
SRQSFP1003 = SRQSFP[:]
SRQSFP1003[ 118 : 124 ] = [ 0x02, 0xf8, 0x00, 0x00, 0x00, 0x00 ]

# SRQSFP1004 supports Tx power measurement
SRQSFP1004 = SRQSFP[:]
SRQSFP1004[ 92 ] = 0x4 # Enable Tx power measurement (page 0 byte 220)

# SRQSFPNOSOLREGBASE1003 has neither SolRegBase nor Mbase nor HBase
SRQSFPNOSOLREGBASE1003 = SRQSFP[:]
SRQSFPNOSOLREGBASE1003[ 118 : 124 ] = [ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 ]

LR4QSFP = SRQSFP[:]
LR4QSFP[3] = 0x02 #Set compliance10And40G 
LR4QSFP[2] = 0x07 #Set connector type to LC
LR4QSFP[14] = 0x0A #Set lengthSmf = 10km
LR4QSFP[15] = 0x0 #Set OM3 length 
LR4QSFP[16] = 0x0 #Set OM2 length
LR4QSFP[58] = 0x66 #set wavelength to 1310
LR4QSFP[59] = 0x58

# LR4QSFP1002 supports Tx power measurement
LR4QSFP1002 = LR4QSFP[:]
LR4QSFP1002[ 92 ] = 0x4 # Enable Tx power measurement (page 0 byte 220)

LRL4QSFP = LR4QSFP[:]
LRL4QSFP[14] = 0x01 #Set lengthSmf = 1km

PLR4QSFP = LR4QSFP[:]
PLR4QSFP[2] = 0x0C #Set connector as MPO

PLRL4QSFP = PLR4QSFP[:]
PLRL4QSFP[14] = 0x01 #Set lengthSmf

PLR4QSFP3PTY = PLR4QSFP[:]
PLR4QSFP3PTY[3] = 0x20 #Set compliance10And40G 

PSM4QSFP = PLRL4QSFP[:]
PSM4QSFP[3] = 0x80    # Bit7 = 1 for PSM4

XSR4QSFP = SRQSFP1004[:]
XSR4QSFP[15] = 0x96 #Set lengthOM3 

SR4DQSFP = SRQSFP[:]
SR4DQSFP[2] = 0x07 #Set connector type to LC
SR4DQSFP[3] = 0x04 #Set compliance10And40G
SR4DQSFP[15] = 0x78 #Set OM3 length

SRBDQSFP = SRQSFP[:]
SRBDQSFP[1] = 0x0C #Set CDR in TX and RX path
SRBDQSFP[3] = 0x00 #Set compliance10And40G, no code specified
SRBDQSFP[12] = 0xCE #Set BitRate

SRBDQSFP3RDPTY = SRBDQSFP[:]
SRBDQSFP3RDPTY[1] = 0xDC #Set other bits apart from CDR in TX and RX path

CR4QSFP = SRQSFP[:]
CR4QSFP[2] = 0x23 #No separate connector
CR4QSFP[3] = 0x08 #Set compliance10And40G 

# Alternate QSFP passive copper identifier
CR4QSFP3RDPTY = CR4QSFP[:]
CR4QSFP3RDPTY[3] = 0x00 #Unset compliance10And40G 
CR4QSFP3RDPTY[11] = 0x67 #Bit-rate 10.3G
CR4QSFP3RDPTY[19] = 0xA0 #Cu cable, unequalized

AR4QSFP = CR4QSFP[:]
AR4QSFP[3] = 0x01 #Set compliance10And40G 

UNIVQSFP = LR4QSFP[:]
UNIVQSFP[14] = 0x01 #Set lengthSmf = 1km
UNIVQSFP[15] = 0x4B #Set OM3 length 

ER4QSFP = LR4QSFP[:]
ER4QSFP[3] = 0x80    # Bit7 = Extended type
ER4QSFP[64] = 0x10    # Extended type QSFP+ ER4

UNKNOWNQSFP = SRQSFP[:]
UNKNOWNQSFP[3] = 0x00    # Set compliance10And40G

# 100GBASE-SR4 QSFP is the same as 40G SR4 with minor changes.
SR4QSFP100G = SRQSFP[:]
SR4QSFP100G[0] = 0x11
SR4QSFP100G[3] = 0x80    # Set the compliance to point to extended table.
SR4QSFP100G[64] = 2
SR4QSFP100G[65] = 7      # Set optical tuning supported

# 100GBASE-XSR4
XSR4QSFP100G = SR4QSFP100G[:]
XSR4QSFP100G[15] = 170//2 # Length OM3
XSR4QSFP100G[18] = 300//2 # Length OM4

UNKNOWNQSFP100G = UNKNOWNQSFP[:]
UNKNOWNQSFP100G[0] = 0x11

# UNKNOWN QSFP100 optical modules with device technology 
# shouldn't be identified as copper.
UNKNOWNQSFP100GOPTICS = UNKNOWNQSFP100G[:]
UNKNOWNQSFP100GOPTICS[19] = 0x60 # not passive copper

# 3rd party modules that identifies both 40G and 100G SR4.
SR4QSFP100G40G3RDPTY = SR4QSFP100G[:]
SR4QSFP100G40G3RDPTY[3] = 0x84 # Set compliance10And40G.

LR4QSFP100G = SR4QSFP100G[:]
LR4QSFP100G[64] = 3
LR4QSFP100G[2] = 0x07 #Set connector type to LC
LR4QSFP100G[14] = 0x0A #Set lengthSmf = 10km
LR4QSFP100G[15] = 0x0 #Set OM3 length 
LR4QSFP100G[16] = 0x0 #Set OM2 length
LR4QSFP100G[58] = 0x66 #set wavelength to 1310
LR4QSFP100G[59] = 0x58
# We assume initComplete ready when timeout. BUG161309
LR4QSFP100G[93] = 0x10  # Set initialization complete flag implemented

LRL4QSFP100G = LR4QSFP100G[:]
LRL4QSFP100G[14] = 0x02 #Set lengthSmf = 2km

# 3rd party modules that identifies both 40G and 100G LR4.
LR4QSFP100G40G3RDPTY = LR4QSFP100G[:]
LR4QSFP100G40G3RDPTY[3] = 0x82   # Set compliance10And40G.
LR4QSFP100G40G3RDPTY[92] = 0x0C  # Set transmitter power measurement

# 100GBASE-DR
DRQSFP100G = LR4QSFP100G[:]
DRQSFP100G[64] = 0x25

# 100GBASE-FR
FRQSFP100G = LR4QSFP100G[:]
FRQSFP100G[64] = 0x26

# 100GBASE-LR
LRQSFP100G = LR4QSFP100G[:]
LRQSFP100G[64] = 0X27

ER4QSFP100G = LR4QSFP100G[:]
ER4QSFP100G[64] = 4
ER4QSFP100G[14] = 0x28 #Set lengthSmf = 10km

AR4QSFP100G = SR4QSFP100G[:]
AR4QSFP100G[2] = 0x23
AR4QSFP100G[64] = 0x01

AR4NQSFP100G = SR4QSFP100G[:]
AR4NQSFP100G[2] = 0x23
AR4NQSFP100G[64] = 0x18

# 3rd party modules that identifies both 40G and 100G AR4.
AR4QSFP100G40G3RDPTY = AR4QSFP100G[:]
AR4QSFP100G40G3RDPTY[3] = 0x81   # Set compliance10And40G.

AR4NQSFP100G40G3RDPTY = AR4NQSFP100G[:]
AR4NQSFP100G40G3RDPTY[3] = 0x81   # Set compliance10And40G.

CR4QSFP100G = SR4QSFP100G[:]
CR4QSFP100G[0] = 0x11 #QSFP100
CR4QSFP100G[2] = 0x23 #No separate connector
CR4QSFP100G[64] = 0x0B # Extended type 100GBASE-CR4 or 25GBASE-CR CA-L

CR2LQSFP50G = CR4QSFP100G[:] # 50GBASE-CR2 CA-L

CR4SQSFP100G = CR4QSFP100G[:]
CR4SQSFP100G[64] = 0x0C # 25GBASE-CR CA-S

CR2SQSFP50G = CR4SQSFP100G[:] # 50GBASE-CR2 CA-S

CR4NQSFP100G = CR4QSFP100G[:]
CR4NQSFP100G[64] = 0x0D # 25GBASE-CR CA-N

CRA4QSFP100G = CR4QSFP100G[:]
CRA4QSFP100G[64] = 0x19 # 100G ACC

CR2NQSFP50G = CR4NQSFP100G[:] # 50GBASE-CR2 CA-N

# Alternate 100G QSFP passive copper identifier
CR4QSFP100G3RDPTY = CR4QSFP100G[:]
CR4QSFP100G3RDPTY[3] = 0x00 #Unset compliance10And40G 
CR4QSFP100G3RDPTY[19] = 0xA0 #Cu cable, unequalized
CR4QSFP100G3RDPTY[64] = 0x00 # Unset extended type

# 3rd party modules that identifies both 40G and 100G CR4.
CR4QSFP100G40G3RDPTY = CR4QSFP100G[:]
CR4QSFP100G40G3RDPTY[3] = 0x88 # Set compliance10And40G.

PSM4QSFP100G = SR4QSFP100G[:]
PSM4QSFP100G[2] = 0x0C #Set connector type to MPO 1x12
PSM4QSFP100G[64] = 0x07 #Extended type for 100GBASE-PSM4

CLR4QSFP100G = SR4QSFP100G[:]
CLR4QSFP100G[2] = 0x07 #Set connector type to LC
CLR4QSFP100G[64] = 0x17 #Extended type for 100GBASE-CLR4

# 3rd party modules that identifies both 40G and 100G CR4.
CR4QSFP100G40G3RDPTY = CR4QSFP100G[:]
CR4QSFP100G40G3RDPTY[3] = 0x88 # Set compliance10And40G.

CWDM4QSFP100G = SR4QSFP100G[:]
CWDM4QSFP100G[64] = 0x06 #Extended type for 100GBASE-CWDM4

# 2-wavelength DWDM module
DWDM2QSFP100G = SR4QSFP100G[:]
DWDM2QSFP100G[2] = 0x07 #Set connector type to LC
DWDM2QSFP100G[14] = 0x28 #Set lengthSmf = 40km
DWDM2QSFP100G[58] = 0x79 #Set wavelength to 1559.8nm
DWDM2QSFP100G[59] = 0xDC #Set wavelength to 1559.8nm
DWDM2QSFP100G[64] = 0x1A #Extended type for 100GE-DWDM2

SWDM4QSFP100G = SR4QSFP100G[:]
SWDM4QSFP100G[2] = 0x07 # Set connector type to LC
SWDM4QSFP100G[64] = 0x20 # Extended type for 100GBASE-SWDM4

SRBDQSFP100G = SR4QSFP100G[:]
SRBDQSFP100G[2] = 0x07 # Set connector type to LC
SRBDQSFP100G[64] = 0x21 # Extended type for 100GBASE-SRBD

ERL4QSFP100G = LR4QSFP100G[:]
ERL4QSFP100G[64] = 4
ERL4QSFP100G[14] = 0x1E # Set lengthSmf = 30km
ERL4QSFP100G[2] = 0x07 # Set connector type to LC

XCWDM4QSFP100G = CWDM4QSFP100G[:]
XCWDM4QSFP100G[14] = 0x0A #Set lengthSmf = 10km

Qsfp400g2QqsfpSide = [
   0x11, 0x00, 0x23, 0x80, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x08, 0xff, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x02, 0xa0, 0x41, 0x72, 0x69, 0x73,
   0x74, 0x61, 0x20, 0x4e, 0x65, 0x74, 0x77, 0x6f,
   0x72, 0x6b, 0x73, 0x20, 0x00, 0x00, 0x1c, 0x73,
   0x43, 0x41, 0x42, 0x2d, 0x44, 0x2d, 0x32, 0x51,
   0x2d, 0x34, 0x30, 0x30, 0x2d, 0x32, 0x2e, 0x35,
   0x32, 0x30, 0x05, 0x07, 0x09, 0x0f, 0x55, 0x32,
   0x40, 0x00, 0x00, 0x00, 0x41, 0x50, 0x46, 0x31,
   0x39, 0x32, 0x35, 0x30, 0x30, 0x35, 0x30, 0x52,
   0x41, 0x57, 0x20, 0x20, 0x31, 0x39, 0x30, 0x38,
   0x31, 0x34, 0x20, 0x20, 0x00, 0x00, 0xd4, 0x22,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x5e, 0x8f, 0x34, 0x08,
]

Qsfp400g2QqsfpSide40G = Qsfp400g2QqsfpSide[:]
Qsfp400g2QqsfpSide40G[3] = 0x88 # Set compliance code to 40GBASE-CR4 + extended

Qsfp400g2QqsfpSideDepop = Qsfp400g2QqsfpSide[:]

# ---------------------------------------------------------
# MBO channel to lane mapping
# Channel to Lane map configuration for the MBO ports
# Table of ( U8, U8, U8 ) channelId that reflects what channel of the optics
# corresponds to what lane of an Xcvr's MBO port.
# For a port identified by the mboPortId,
# laneToChannelMapping[ mboPortId ] returns a list of ( U8, U8, U8 ) which
# corresponds to its ( SupPortId, channelIdTx, channelIdRx ).
Finisar300mLaneChannelMap = {
   0 : [ ( 0, 23, 23 ),
         ( 1, 22, 22 ),
         ( 2, 21, 21 ),
         ( 3, 20, 20 ),
         ( 4, 19, 19 ),
         ( 5, 18, 18 ),
         ( 6, 17, 17 ),
         ( 7, 16, 16 ),
         ( 8, 15, 15 ),
         ( 9, 14, 14 ),
         ( 10, 13, 13 ),
         ( 11, 12, 12 ),
       ],
   1 : [ ( 0, 11, 11 ),
         ( 1, 10, 10 ),
         ( 2, 9, 9 ),
         ( 3, 8, 8 ),
         ( 4, 7, 7 ),
         ( 5, 6, 6 ),
         ( 6, 5, 5 ),
         ( 7, 4, 4 ),
         ( 8, 3, 3 ),
         ( 9, 2, 2 ),
         ( 10, 1, 1 ),
         ( 11, 0, 0 )
       ]
}

AvagoLaneChannelMap = {
   0 : [ ( 0, 0, 0 ),
         ( 1, 1, 1 ),
         ( 2, 2, 2 ),
         ( 3, 3, 3 ),
         ( 4, 4, 4 ),
         ( 5, 5, 5 ),
         ( 6, 6, 6 ),
         ( 7, 7, 7 ),
         ( 8, 8, 8 ),
         ( 9, 9, 9 ),
         ( 10, 10, 10 ),
         ( 11, 11, 11 )
       ],
   1 : [ ( 0, 0, 0 ),
         ( 1, 1, 1 ),
         ( 2, 2, 2 ),
         ( 3, 3, 3 ),
         ( 4, 4, 4 ),
         ( 5, 5, 5 ),
         ( 6, 6, 6 ),
         ( 7, 7, 7 ),
         ( 8, 8, 8 ),
         ( 9, 9, 9 ),
         ( 10, 10, 10 ),
         ( 11, 11, 11 )
       ]
}

# ---------------------------------------------------------
# MBO device register definitions 

AFBR81RxyZChecksumOffset = 223 - 128

SRMBOAFBR81RXYZ = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 0 - 7 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 8 - 15
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 16 - 23
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 24 - 31
   0x80, 0xe8, 0x61, 0xa8, 0x00, 0x00, 0x00, 0x00, # 32 - 39
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 40 - 47
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 48 - 55
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 56 - 63
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 64 - 71
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 72 - 79
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 80 - 87
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 88 - 95
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 96 - 103
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 104 - 111
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 112 - 119
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00  # 120 - 127
]

SRMBOAFBR81RXYZ0 = [
   0x00, 0x42, 0xc0, 0x46, 0x19, 0x67, 0x42, 0x04, # 128 - 135
   0x0b, 0xbb, 0xc9, 0xc5, 0x60, 0xa2, 0x0b, 0x00, # 136 - 143
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 144 - 151
   0x41, 0x56, 0x41, 0x47, 0x4f, 0x20, 0x20, 0x20, # 152 - 159
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, # 160 - 167
   0x00, 0x17, 0x6a, 0x41, 0x46, 0x42, 0x52, 0x2d, # 168 - 175
   0x38, 0x31, 0x31, 0x46, 0x4e, 0x31, 0x5a, 0x20, # 176 - 183
   0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, # 184 - 191
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 192 - 199
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 200 - 207
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 208 - 215
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 216 - 223
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 224 - 231
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 232 - 239
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 240 - 247
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00  # 248 - 255
]

SRMBOAFBR81RXYZ1 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 128 - 135
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 136 - 143
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 144 - 151
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 152 - 159
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 160 - 167
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 168 - 175
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 176 - 183
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 184 - 191
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 192 - 199
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 200 - 207
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 208 - 215
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 216 - 223
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 224 - 231
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 232 - 239
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 240 - 247
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00  # 248 - 255
]

SRMBOAFBR82RXYZ = [
   0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, # 0 - 7 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 8 - 15
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 16 - 23
   0x00, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00, # 24 - 31
   0x80, 0xe8, 0x61, 0xa8, 0x00, 0x00, 0x00, 0x00, # 32 - 39
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 40 - 47
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 48 - 55
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 56 - 63
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 64 - 71
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 72 - 79
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 80 - 87
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 88 - 95
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 96 - 103
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 104 - 111
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 112 - 119
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00  # 120 - 127
]

SRMBOAFBR82RXYZ0 = [
   0x00, 0x00, 0xc0, 0x46, 0x19, 0xc7, 0x00, 0x00, # 128 - 135
   0x00, 0x00, 0x28, 0x35, 0x60, 0xa8, 0xa3, 0x00, # 136 - 143
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 144 - 151
   0x41, 0x56, 0x41, 0x47, 0x4F, 0x20, 0x20, 0x20, # 152 - 159
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, # 160 - 167
   0x00, 0x17, 0x6A, 0x41, 0x46, 0x42, 0x52, 0x2D, # 168 - 175
   0x38, 0x31, 0x31, 0x46, 0x4E, 0x31, 0x5A, 0x20, # 176 - 183
   0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, # 184 - 191
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 192 - 199
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 200 - 207
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 208 - 215
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 216 - 223
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 224 - 231
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 232 - 239
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 240 - 247
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00  # 248 - 255
]

SRMBOAFBR82RXYZ1 = [
   0x00, 0x42, 0xC0, 0x46, 0x19, 0x67, 0x42, 0x04, # 128 - 135
   0x0B, 0xBB, 0xC9, 0xC5, 0x60, 0xA2, 0x0B, 0x00, # 136 - 143
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 144 - 151
   0x41, 0x56, 0x41, 0x47, 0x4F, 0x20, 0x20, 0x20, # 152 - 159
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, # 160 - 167
   0x00, 0x17, 0x6A, 0x41, 0x46, 0x42, 0x52, 0x2D, # 168 - 175
   0x38, 0x31, 0x31, 0x46, 0x4E, 0x31, 0x5A, 0x20, # 176 - 183
   0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, # 184 - 191
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 192 - 199
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 200 - 207
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 208 - 215
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 216 - 223
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 224 - 231
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 232 - 239
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 240 - 247
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00  # 248 - 255
]

SRMBOFINISAR300MTX = [
   0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 0 - 7
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 8 - 15
   0x00, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, # 16 - 23
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 24 - 31
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 32 - 39
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 40 - 47
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 48 - 55
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 56 - 63
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 64 - 71
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 72 - 79
   0x00, 0x1D, 0x00, 0x00, 0x2A, 0x7D, 0x00, 0x00, # 80 - 87
   0x81, 0xEA, 0x5F, 0x50, 0x00, 0x00, 0x00, 0x00, # 88 - 95
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 96 - 103
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 104 - 111
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, # 112 - 119
   0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x01 # 120 - 127
]

SRMBOFINISAR300MTX0 = [
   0x0D, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 128 - 135
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 136 - 143
   0x00, 0x00, 0x00, 0x00, 0x46, 0x49, 0x4E, 0x49, # 144 - 151
   0x53, 0x41, 0x52, 0x20, 0x43, 0x4F, 0x52, 0x50, # 152 - 159
   0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x90, 0x65, # 160 - 167
   0x46, 0x42, 0x54, 0x54, 0x50, 0x31, 0x30, 0x53, # 168 - 175
   0x56, 0x31, 0x43, 0x30, 0x30, 0x20, 0x20, 0x20, # 176 - 183
   0x41, 0x30, 0x42, 0x68, 0x07, 0xD0, 0x46, 0x98, # 184 - 191
   0x00, 0x00, 0x00, 0x00, 0x59, 0x51, 0x47, 0x30, # 192 - 199
   0x44, 0x4E, 0x44, 0x20, 0x20, 0x20, 0x20, 0x20, # 200 - 207
   0x20, 0x20, 0x20, 0x20, 0x31, 0x33, 0x31, 0x32, # 208 - 215
   0x31, 0x32, 0x20, 0x20, 0x00, 0x00, 0x00, 0x81, # 216 - 223
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 224 - 231
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 232 - 239
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 240 - 247
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 # 248 - 255
]

SRMBOFINISAR300MTX1 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 128 - 135
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 136 - 143
   0x11, 0x09, 0x11, 0x46, 0x11, 0x2F, 0x11, 0x2D, # 144 - 151
   0x10, 0xDB, 0x10, 0xCA, 0x11, 0xC5, 0x0F, 0x96, # 152 - 159
   0x10, 0xE3, 0x10, 0x9E, 0x0F, 0xAD, 0x10, 0x9C, # 160 - 167
   0x11, 0x23, 0x11, 0x62, 0x10, 0x3B, 0x10, 0x92, # 168 - 175
   0x10, 0xE3, 0x10, 0x5F, 0x10, 0x24, 0x11, 0x83, # 176 - 183
   0x11, 0x1B, 0x11, 0x13, 0x10, 0xBD, 0x11, 0x16, # 184 - 191
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 192 - 199
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 200 - 207
   0x24, 0x30, 0x22, 0xCD, 0x25, 0x00, 0x23, 0xFC, # 208 - 215
   0x23, 0x00, 0x22, 0x3B, 0x1E, 0x68, 0x24, 0x47, # 216 - 223
   0x22, 0x95, 0x23, 0xB2, 0x23, 0xE5, 0x24, 0x3D, # 224 - 231
   0x20, 0xB8, 0x24, 0xA9, 0x1F, 0x37, 0x1D, 0x80, # 232 - 239
   0x20, 0x92, 0x1F, 0xB3, 0x1D, 0x1F, 0x19, 0x25, # 240 - 247
   0x20, 0x04, 0x1E, 0x36, 0x1E, 0xA4, 0x20, 0x5E # 248 - 255
]

SRMBOFINISAR300MTX2 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 128 - 135
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 136 - 143
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 144 - 151
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 152 - 159
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 160 - 167
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 168 - 175
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 176 - 183
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 184 - 191
   0x4B, 0x00, 0xFB, 0x00, 0x46, 0x00, 0x00, 0x00, # 192 - 199
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 200 - 207
   0x8C, 0xA0, 0x75, 0x30, 0x88, 0xB8, 0x79, 0x18, # 208 - 215
   0x6D, 0x60, 0x55, 0xF0, 0x69, 0x78, 0x59, 0xD8, # 216 - 223
   0x1D, 0x4C, 0x01, 0xF4, 0x17, 0x70, 0x03, 0xE8, # 224 - 231
   0x43, 0xE2, 0x04, 0xEB, 0x31, 0x2D, 0x06, 0xCA, # 232 - 239
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 240 - 247
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 # 248 - 255
]

SRMBOFINISAR300MTX3 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 128 - 135
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 136 - 143
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 144 - 151
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 152 - 159
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 160 - 167
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 168 - 175
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 176 - 183
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 184 - 191
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 192 - 199
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 200 - 207
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 208 - 215
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 216 - 223
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 224 - 231
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 232 - 239
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 240 - 247
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 # 248 - 255
]

SRMBOFINISAR300MTX4 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 128 - 135
   0x00, 0xFF, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, # 136 - 143
   0x00, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, # 144 - 151
   0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 152 - 159
   0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, # 160 - 167
   0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, # 168 - 175
   0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, # 176 - 183
   0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, # 184 - 191
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 192 - 199
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 200 - 207
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 208 - 215
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 216 - 223
   0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, # 224 - 231
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 232 - 239
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 240 - 247
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 # 248 - 255
]

SRMBOFINISAR300MRX = [
   0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, # 0 - 7
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 8 - 15
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 16 - 23
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 24 - 31
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 32 - 39
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 40 - 47
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 48 - 55
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 56 - 63
   0x00, 0x00, 0x00, 0x00, 0x55, 0x55, 0x55, 0x55, # 64 - 71
   0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, # 72 - 79
   0x00, 0x20, 0x00, 0x00, 0x2A, 0x23, 0x00, 0x00, # 80 - 87
   0x81, 0x34, 0x60, 0xE7, 0x00, 0x00, 0x00, 0x00, # 88 - 95
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 96 - 103
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 104 - 111
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, # 112 - 119
   0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x04 # 120 - 127
]

SRMBOFINISAR300MRX0 = [
   0x0D, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 128 - 135
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 136 - 143
   0x00, 0x00, 0x00, 0x00, 0x46, 0x49, 0x4E, 0x49, # 144 - 151
   0x53, 0x41, 0x52, 0x20, 0x43, 0x4F, 0x52, 0x50, # 152 - 159
   0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x90, 0x65, # 160 - 167
   0x46, 0x42, 0x52, 0x54, 0x50, 0x31, 0x30, 0x53, # 168 - 175
   0x56, 0x31, 0x43, 0x30, 0x30, 0x20, 0x20, 0x20, # 176 - 183
   0x41, 0x30, 0x42, 0x68, 0x07, 0xD0, 0x46, 0x96, # 184 - 191
   0x00, 0x00, 0x00, 0x00, 0x5A, 0x51, 0x31, 0x30, # 192 - 199
   0x30, 0x34, 0x59, 0x20, 0x20, 0x20, 0x20, 0x20, # 200 - 207
   0x20, 0x20, 0x20, 0x20, 0x31, 0x33, 0x31, 0x32, # 208 - 215
   0x31, 0x34, 0x20, 0x20, 0x00, 0x00, 0x00, 0x55, # 216 - 223
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 224 - 231
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 232 - 239
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 240 - 247
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 # 248 - 255
]

SRMBOFINISAR300MRX1 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 128 - 135
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 136 - 143
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 144 - 151
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 152 - 159
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 160 - 167
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 168 - 175
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 176 - 183
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 184 - 191
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 192 - 199
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 200 - 207
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 208 - 215
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 216 - 223
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 224 - 231
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 232 - 239
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 240 - 247
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 # 248 - 255
]

SRMBOFINISAR300MRX2 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 128 - 135
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 136 - 143
   0x1B, 0xE2, 0x1C, 0xEF, 0x20, 0xFA, 0x1D, 0x5B, # 144 - 151
   0x1C, 0x2E, 0x1B, 0x85, 0x1D, 0xF8, 0x1E, 0x39, # 152 - 159
   0x1D, 0x41, 0x17, 0x91, 0x1F, 0x6A, 0x1C, 0xB9, # 160 - 167
   0x23, 0x6A, 0x20, 0x57, 0x21, 0x34, 0x20, 0x03, # 168 - 175
   0x22, 0x44, 0x21, 0x53, 0x23, 0x60, 0x15, 0x57, # 176 - 183
   0x21, 0xE0, 0x20, 0x53, 0x1B, 0x86, 0x1D, 0xA2, # 184 - 191
   0x4B, 0x00, 0xFB, 0x00, 0x46, 0x00, 0x00, 0x00, # 192 - 199
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 200 - 207
   0x8C, 0xA0, 0x75, 0x30, 0x88, 0xB8, 0x79, 0x18, # 208 - 215
   0x6D, 0x60, 0x55, 0xF0, 0x69, 0x78, 0x59, 0xD8, # 216 - 223
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 224 - 231
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 232 - 239
   0x3D, 0xE9, 0x02, 0xC4, 0x31, 0x2D, 0x04, 0x62, # 240 - 247
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 # 248 - 255
]

SRMBOFINISAR300MRX3 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 128 - 135
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 136 - 143
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 144 - 151
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 152 - 159
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 160 - 167
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 168 - 175
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 176 - 183
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 184 - 191
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 192 - 199
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 200 - 207
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 208 - 215
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 216 - 223
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 224 - 231
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 232 - 239
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 240 - 247
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 # 248 - 255
]

SRMBOFINISAR300MRX4 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 128 - 135
   0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 136 - 143
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 144 - 151
   0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 152 - 159
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 160 - 167
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 168 - 175
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 176 - 183
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 184 - 191
   0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, # 192 - 199
   0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, # 200 - 207
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 208 - 215
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 216 - 223
   0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, # 224 - 231
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 232 - 239
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 240 - 247
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 # 248 - 255
]

FBOxD1xSL1LxxChecksumOffset = 180 - 128

SRMBOFBOXD1XSL1LXXTX = [
   0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, # 0 - 7 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 8 - 15
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0xC5, # 16 - 23
   0x22, 0xC5, 0x7E, 0xBA, 0x00, 0x00, 0x00, 0x00, # 24 - 31
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 32 - 39
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 40 - 47
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 48 - 55
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 56 - 63
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 64 - 71
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 72 - 79
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 80 - 87
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 88 - 95
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 96 - 103
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 104 - 111
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 112 - 119
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00  # 120 - 127
]

SRMBOFBOXD1XSL1LXXTX0 = [
   0x00, 0x80, 0x33, 0x80, 0x3C, 0x0A, 0x69, 0x42, # 128 - 135
   0x68, 0x07, 0xD0, 0xD4, 0x14, 0x60, 0xA3, 0x00, # 136 - 143
   0xA0, 0x08, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, # 144 - 151
   0x46, 0x49, 0x4E, 0x49, 0x53, 0x41, 0x52, 0x20, # 152 - 159
   0x43, 0x4F, 0x52, 0x50, 0x20, 0x20, 0x20, 0x20, # 160 - 167
   0x00, 0x90, 0x65, 0x46, 0x42, 0x4F, 0x00, 0x44, # 168 - 175
   0x31, 0x00, 0x53, 0x4C, 0x31, 0x4C, 0x00, 0x00, # 176 - 183
   0x20, 0x20, 0x20, 0x41, 0x20, 0x00, 0x00, 0x00, # 184 - 191
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 192 - 199
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 200 - 207
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 208 - 215
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 216 - 223
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 224 - 231
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 232 - 239
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 240 - 247
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00  # 248 - 255
]

SRMBOFBOXD1XSL1LXXTX1 = [
   0x4B, 0x00, 0xFB, 0x00, 0x4B, 0x00, 0xFB, 0x00, # 128 - 135
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 136 - 143
   0x94, 0x70, 0x6E, 0xF0, 0x00, 0x00, 0x00, 0x00, # 144 - 151
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 152 - 159
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 160 - 167
   0x1D, 0x4C, 0x03, 0xE8, 0x4D, 0xF1, 0x08, 0xBF, # 168 - 175
   0x00, 0x00, 0x00, 0x00, 0x07, 0x1B, 0x13, 0x6C, # 176 - 183
   0x13, 0x6C, 0x13, 0x6C, 0x13, 0x6C, 0x13, 0x6C, # 184 - 191
   0x13, 0x6C, 0x13, 0x6C, 0x13, 0x6C, 0x13, 0x6C, # 192 - 199
   0x13, 0x6C, 0x13, 0x6C, 0x13, 0x6C, 0x18, 0x6F, # 200 - 207
   0x18, 0x6F, 0x18, 0x6F, 0x18, 0x6F, 0x18, 0x6F, # 208 - 215
   0x18, 0x6F, 0x18, 0x6F, 0x18, 0x6F, 0x18, 0x6F, # 216 - 223
   0x18, 0x6F, 0x18, 0x6F, 0x18, 0x6F, 0xFF, 0xFF, # 224 - 231
   0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, # 232 - 239
   0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, # 240 - 247
   0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF  # 248 - 255
]

SRMBOFBOXD1XSL1LXXRX = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 0 - 7 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 8 - 15
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 16 - 23
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 24 - 31
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 32 - 39
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 40 - 47
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 48 - 55
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x05, # 56 - 63
   0x05, 0x05, 0x05, 0x05, 0x03, 0x03, 0x03, 0x03, # 64 - 71
   0x03, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 72 - 79
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 80 - 87
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 88 - 95
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 96 - 103
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 104 - 111
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 112 - 119
   0x00, 0x00, 0x00, 0xD1, 0xC4, 0xD3, 0xC5, 0x00  # 120 - 127
]

SRMBOFBOXD1XSL1LXXRX0 = [
   0x00, 0x80, 0x33, 0x80, 0x3C, 0x0A, 0x69, 0x42, # 128 - 135
   0x68, 0x07, 0xD0, 0xD4, 0x14, 0x60, 0xA3, 0x00, # 136 - 143
   0xA0, 0x08, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, # 144 - 151
   0x46, 0x49, 0x4E, 0x49, 0x53, 0x41, 0x52, 0x20, # 152 - 159
   0x43, 0x4F, 0x52, 0x50, 0x20, 0x20, 0x20, 0x20, # 160 - 167
   0x00, 0x90, 0x65, 0x46, 0x42, 0x4F, 0x00, 0x44, # 168 - 175
   0x31, 0x00, 0x53, 0x4C, 0x31, 0x4C, 0x00, 0x00, # 176 - 183
   0x20, 0x20, 0x20, 0x41, 0x20, 0x00, 0x00, 0x00, # 184 - 191
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 192 - 199
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 200 - 207
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 208 - 215
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 216 - 223
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 224 - 231
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 232 - 239
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 240 - 247
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00  # 248 - 255
]

SRMBOFBOXD1XSL1LXXRX1 = [
   0x4B, 0x00, 0xFB, 0x00, 0x4B, 0x00, 0xFB, 0x00, # 128 - 135
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 136 - 143
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 144 - 151
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 152 - 159
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 160 - 167
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 168 - 175
   0x00, 0x00, 0x00, 0x00, 0x91, 0xA6, 0x00, 0x00, # 176 - 183
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 184 - 191
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 192 - 199
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 200 - 207
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 208 - 215
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 216 - 223
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, # 224 - 231
   0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, # 232 - 239
   0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, # 240 - 247
   0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF  # 248 - 255
]

# 100GBASE-LR4 CFP2 module
# All tables have 128 entries. Tables A000 and on
# do not contain reserved entries, as XcvrCtrl
# does not read them.
#################################################
# Table 8000 has data width of 8 bits.
Cfp2MsaMis100GLR4Table8000 = [
   0x11, 0xe4, 0x07, 0x01, 0x00, 0x00, 0x00, 0x00,
   0x08, 0x44, 0x11, 0x81, 0x81, 0x0a, 0x00, 0x00,
   0x01, 0x04, 0xca, 0x6e, 0xcc, 0x8d, 0x00, 0x64,
   0x21, 0x44, 0x40, 0x70, 0x1c, 0x3c, 0x64, 0x46,
   0x00, 0x41, 0x72, 0x69, 0x73, 0x74, 0x61, 0x20,
   0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b, 0x73,
   0x20, 0x00, 0x1c, 0x73, 0x43, 0x46, 0x50, 0x32,
   0x2d, 0x31, 0x30, 0x30, 0x47, 0x2d, 0x4c, 0x52,
   0x34, 0x20, 0x20, 0x20, 0x58, 0x41, 0x50, 0x31,
   0x33, 0x34, 0x39, 0x4a, 0x38, 0x30, 0x30, 0x35,
   0x20, 0x20, 0x20, 0x20, 0x31, 0x33, 0x31, 0x32,
   0x31, 0x32, 0x30, 0x30, 0x30, 0x30, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x00, 0x16, 0x01, 0x00, 0x01, 0x00, 0x0c, 0x03,
   0x0f, 0x68, 0x0a, 0x01, 0x0c, 0x00, 0x01, 0x01,
   0x20, 0x07, 0x07, 0x00, 0x00, 0x00, 0x20, 0xf6
]

# Table 8080 has data width of 8 bits.
Cfp2MsaMis100GLR4Table8080 = [
   0x41, 0x00, 0x3c, 0x00, 0x28, 0x00, 0x23, 0x00, # Temp thresholds
   0x8c, 0xa0, 0x88, 0xb8, 0x79, 0x18, 0x75, 0x30,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xaf, 0xc8, 0x92, 0x7c, 0x44, 0x5c, 0x30, 0xd4,
   0x8a, 0x99, 0x7b, 0x86, 0x0c, 0xef, 0x0b, 0x87,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x8a, 0x99, 0x7b, 0x86, 0x02, 0xe5, 0x02, 0x4d,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x01, 0x3c, 0x01, 0x3c, 0x01, 0x3c, 0x01, 0x3c,
   0x8a, 0x99, 0x8a, 0x99, 0x8a, 0x99, 0x8a, 0x99,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e
]

# Table 8420 has data width of 8 bits.
Cfp2MsaMis100GLR4Table8420 = [ 0x00 ] * 128 # Version 0 = 01

# Table A000 has data width of 16 bits.
Cfp2MsaMis100GLR4TableA000 = [
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
   0x03, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x0e, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0xf0, 0x20, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x6a, 0x00, 0xf0, 0xa0, 0x62, 0x00, 0xf0, 0x0f,
   0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0xff, 0x7f, # Temp A02f
   0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # Voltage A030
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x99, 0x8a, 0x86, 0x7b, 0xe5, 0x02, 0x4d, 0x02
]

# Table A200 has data width of 16 bits.
Cfp2MsaMis100GLR4TableA200 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xd8, 0x80, 0xd8, 0x80, 0xd8, 0x80, 0xd8, 0x80,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

# Table A280 has data width of 16 bits.
Cfp2MsaMis100GLR4TableA280 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, # txBias A2A0 up
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, # txPower A2B0 up
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, # rxPower A2D0 up
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

# Table A400 has data width of 16 bits.
Cfp2MsaMis100GLR4TableA400 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

   0x07, 0x00, 0x07, 0x00, 0x07, 0x00, 0x07, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

#################################################

# Bobo module
# This module is only used for the XcvrUnlockTest
# It does not match any real transceiver
#
# All tables have 128 entries.
#################################################
Cfp2MsaMisBobo100GLR4Table8000 = [
   0x11, 0xe4, 0x07, 0x01, 0x00, 0x00, 0x00, 0x00,
   0x08, 0x44, 0x11, 0x81, 0x81, 0x0a, 0x00, 0x00,
   0x01, 0x04, 0xca, 0x6e, 0xcc, 0x8d, 0x00, 0x64,
   0x21, 0x44, 0x40, 0x70, 0x1c, 0x3c, 0x64, 0x46,
   0x00, 0x42, 0x6f, 0x62, 0x6f, 0x43, 0x6f, 0x72,
   0x70, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x20, 0x00, 0x1c, 0x73, 0x43, 0x46, 0x50, 0x32,
   0x2d, 0x31, 0x30, 0x30, 0x47, 0x2d, 0x4c, 0x52,
   0x34, 0x20, 0x20, 0x20, 0x58, 0x41, 0x50, 0x31,
   0x33, 0x34, 0x39, 0x4a, 0x38, 0x30, 0x30, 0x35,
   0x20, 0x20, 0x20, 0x20, 0x31, 0x33, 0x31, 0x32,
   0x31, 0x32, 0x30, 0x30, 0x30, 0x30, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x00, 0x16, 0x01, 0x00, 0x01, 0x00, 0x0c, 0x03,
   0x0f, 0x68, 0x0a, 0x01, 0x0c, 0x00, 0x01, 0x01,
   0x20, 0x07, 0x07, 0x00, 0x00, 0x00, 0x20, 0x0b
]

Cfp2MsaMisBobo100GLR4TableA000 = [
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
   0x03, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x0e, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0xf0, 0x20, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x6a, 0x00, 0xf0, 0xa0, 0x62, 0x00, 0xf0, 0x0f,
   0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0xff, 0x0f, # Temp A02f
   0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # Voltage A030
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x99, 0x8a, 0x86, 0x7b, 0xe5, 0x02, 0x4d, 0x02
]

Cfp2MsaMisBobo100GLR4Table8080 = Cfp2MsaMis100GLR4Table8080
Cfp2MsaMisBobo100GLR4Table8420 = Cfp2MsaMis100GLR4Table8420
Cfp2MsaMisBobo100GLR4TableA200 = Cfp2MsaMis100GLR4TableA200
Cfp2MsaMisBobo100GLR4TableA280 = Cfp2MsaMis100GLR4TableA280
Cfp2MsaMisBobo100GLR4TableA400 = Cfp2MsaMis100GLR4TableA400
#################################################

# 100GBASE-SR10 CFP2 module
# This module is only supported for module
# detection in simulation at this time.
#
# All tables have 128 entries. Tables A000 and on
# do not contain reserved entries, as XcvrCtrl
# does not read them.
#################################################
# Table 8000 has data width of 8 bits.
Cfp2MsaMis100GSR10Table8000 = [
   0x11, 0xa0, 0x09, 0x03, 0x00, 0x00, 0x00, 0x00,
   0x00, 0xaa, 0x4a, 0x39, 0x39, 0x00, 0x0a, 0x00,

   0x0A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x04, 0x40, 0x11, 0x11, 0x2d, 0x64, 0x46,
   0x00, 0x41, 0x72, 0x69, 0x73, 0x74, 0x61, 0x20,
   0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b, 0x73,
   0x20, 0x00, 0x1c, 0x73, 0x43, 0x46, 0x50, 0x32,
   0x2d, 0x31, 0x30, 0x30, 0x47, 0x2d, 0x4c, 0x52,
   0x34, 0x20, 0x20, 0x20, 0x58, 0x41, 0x50, 0x31,
   0x33, 0x34, 0x39, 0x4a, 0x38, 0x30, 0x30, 0x35,
   0x20, 0x20, 0x20, 0x20, 0x31, 0x33, 0x31, 0x32,
   0x31, 0x32, 0x30, 0x30, 0x30, 0x30, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x00, 0x16, 0x01, 0x00, 0x01, 0x00, 0x0c, 0x03,
   0x0f, 0x68, 0x0a, 0x01, 0x0c, 0x00, 0x01, 0x01,
   0x20, 0x07, 0x07, 0x00, 0x00, 0x00, 0x20, 0xf3
]

Cfp2MsaMis100GSR10Table8080 = Cfp2MsaMis100GLR4Table8080
Cfp2MsaMis100GSR10Table8420 = Cfp2MsaMis100GLR4Table8420
Cfp2MsaMis100GSR10TableA000 = Cfp2MsaMis100GLR4TableA000
Cfp2MsaMis100GSR10TableA200 = Cfp2MsaMis100GLR4TableA200
Cfp2MsaMis100GSR10TableA280 = Cfp2MsaMis100GLR4TableA280
Cfp2MsaMis100GSR10TableA400 = Cfp2MsaMis100GLR4TableA400
#################################################

# 100GBASE-ER4 CFP2 module
# This module is only supported for module
# detection in simulation at this time.
#################################################
# Table 8000 has data width of 8 bits.
#Cfp2MsaMis100GER4Table8000 = [
#   Cfp2MsaMis100GLR4Table8000[ 0:2 ] +
#   [ 0x02 ] +                           # 8003 = EthAppCode ER4
#   Cfp2MsaMis100GLR4Table8000[ 4:0xC ] +
#   [ 0x28 ] +                           # 800D = SMF length 40km
#   Cfp2MsaMis100GLR4Table8000[ 0xE:0x7F ]
#]
Cfp2MsaMis100GER4Table8000 = Cfp2MsaMis100GLR4Table8000[:]
Cfp2MsaMis100GER4Table8000[ 3 ] = 0x02   # 8003 = EthAppCode ER4
Cfp2MsaMis100GER4Table8000[ 0xD ] = 0x28 # 800D = SMF length 40km
Cfp2MsaMis100GER4Table8000[ 0x7f ] = 0x15 # 807F = Checksum

Cfp2MsaMis100GER4Table8080 = Cfp2MsaMis100GLR4Table8080
Cfp2MsaMis100GER4Table8420 = Cfp2MsaMis100GLR4Table8420
Cfp2MsaMis100GER4TableA000 = Cfp2MsaMis100GLR4TableA000
Cfp2MsaMis100GER4TableA200 = Cfp2MsaMis100GLR4TableA200
Cfp2MsaMis100GER4TableA280 = Cfp2MsaMis100GLR4TableA280
Cfp2MsaMis100GER4TableA400 = Cfp2MsaMis100GLR4TableA400

# 100GDwdmCoherent CFP2 module
#################################################
# Table 8000 has data width of 8 bits.
Cfp2MsaMis100GDwdmCoherentTable8000 = [
   0x14, 0xca, 0x07, 0x11, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x11, 0x01, 0x00, 0xa0, 0xff, 0x00, 0x00,
   0x01, 0x01, 0xee, 0xde, 0xf5, 0x1e, 0x01, 0x90,
   0x32, 0xf4, 0x14, 0x19, 0x00, 0x3c, 0x64, 0x46,
   0xfb, 0x41, 0x72, 0x69, 0x73, 0x74, 0x61, 0x20,
   0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b, 0x73,
   0x20, 0x00, 0x1c, 0x73, 0x43, 0x46, 0x50, 0x58,
   0x2d, 0x31, 0x30, 0x30, 0x47, 0x2d, 0x44, 0x57,
   0x44, 0x4d, 0x20, 0x20, 0x58, 0x41, 0x50, 0x31,
   0x35, 0x31, 0x30, 0x30, 0x30, 0x31, 0x32, 0x36,
   0x20, 0x20, 0x20, 0x20, 0x32, 0x30, 0x31, 0x35,
   0x30, 0x33, 0x30, 0x35, 0x00, 0x00, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x0a, 0x16, 0x01, 0x01, 0x34, 0x01, 0x0c, 0x03,
   0x0e, 0x00, 0x3c, 0x3c, 0x00, 0x00, 0x64, 0x05,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfa
]

Cfp2MsaMis100GDwdmCoherentEV2Table8000 = Cfp2MsaMis100GDwdmCoherentTable8000[:]
Cfp2MsaMis100GDwdmCoherentEV2Table8000[ 0x45 ] = int( binascii.b2a_hex('T'), 16 )
Cfp2MsaMis100GDwdmCoherentEV2Table8000[ 0x46 ] = int( binascii.b2a_hex('C'), 16 )
Cfp2MsaMis100GOclaroDwdmCoherentETable8000 = Cfp2MsaMis100GDwdmCoherentTable8000[:]
vendor = "OCLARO          "
vendorPn = "TRB100BR-01     "
vendorSn = "SMD5407.1       "
for idx in range( 16 ):
   Cfp2MsaMis100GOclaroDwdmCoherentETable8000[ 0x21 + idx ] = \
         int( binascii.b2a_hex( vendor[ idx ] ), 16 )
   Cfp2MsaMis100GOclaroDwdmCoherentETable8000[ 0x34 + idx ] = \
         int( binascii.b2a_hex( vendorPn[ idx ] ), 16 )
   Cfp2MsaMis100GOclaroDwdmCoherentETable8000[ 0x44 + idx ] = \
         int( binascii.b2a_hex( vendorSn[ idx ] ), 16 )
Cfp2MsaMis100GAcaciaDwdmCoherentETable8000 = Cfp2MsaMis100GDwdmCoherentTable8000[:]
vendor = "Acacia Comm Inc."
vendorPn = "AC200-A20-00E   "
vendorSn = "163107143       "
for idx in range( 16 ):
   Cfp2MsaMis100GAcaciaDwdmCoherentETable8000[ 0x21 + idx ] = \
         int( binascii.b2a_hex( vendor[ idx ] ), 16 )
   Cfp2MsaMis100GAcaciaDwdmCoherentETable8000[ 0x34 + idx ] = \
         int( binascii.b2a_hex( vendorPn[ idx ] ), 16 )
   Cfp2MsaMis100GAcaciaDwdmCoherentETable8000[ 0x44 + idx ] = \
         int( binascii.b2a_hex( vendorSn[ idx ] ), 16 )

# Table 8080 has data width of 8 bits.
Cfp2MsaMis100GDwdmCoherentTable8080 = [
   0x46, 0x00, 0x41, 0x00, 0x00, 0x00, 0xfb, 0x00, # Temp thresholds
   0x8c, 0xa0, 0x88, 0xb8, 0x79, 0x18, 0x75, 0x30,
   0xfd, 0xe8, 0xfd, 0xe8, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xfd, 0xe8, 0xfd, 0xe8, 0x00, 0x00, 0x00, 0x00,
   0x01, 0x90, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x94,
   0x32, 0x00, 0x31, 0xcc, 0x28, 0x33, 0x28, 0x00,
   0x62, 0x1f, 0x4d, 0xf1, 0x00, 0x20, 0x00, 0x14,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7a
]

# Table 8180 has data width of 16 bits. ( 0x8180 and 0x8181 are 8-bits width)
Cfp2MsaMis100GDwdmCoherentTable8180 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0xbf, 0x17, 0x70, 0x00, 0xc4,
   0x07, 0xd0, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, # Tuning capabilities
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xed
]

# Table 8420 has data width of 8 bits.
Cfp2MsaMis100GDwdmCoherentTable8420 = Cfp2MsaMis100GLR4Table8420

# Table B000 has data width of 16 bits.
Cfp2MsaMis100GDwdmCoherentTableB000 = [
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, # Password is not readable
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
   0x03, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x02, 0x00, 0x02, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, # Module state
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # Module FAWs
   0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x6a, 0x00, 0xf8, 0xa7, 0x62, 0x00, 0xff, 0x0f,
   0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x7f, # Temp B02f
   0x99, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # Voltage B030
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x90, 0x01, 0x2c, 0x01, 0x3c, 0xf6, 0x74, 0xf5,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # MDIO write ready bit
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # MDIO write ready bit enabled
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

# Table B180 has data width of 16 bits.
Cfp2MsaMis100GDwdmCoherentTableB180 = [
   # 0xB180 Network lane alarm and warning 1
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB190 Network lane alarm and warning 2
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB1A0 Network lane fualt and status
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB1B0
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB1C0
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB1D0
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB1E0
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   # 0xB1F0
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff
]

# Table B300 has data width of 16 bits.
Cfp2MsaMis100GDwdmCoherentTableB300 = [
   # 0xB300
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB310
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB320 Network Lane TX Laser Bias Current
   0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB330 Network Lane TX Laser Output Power
   0xff, 0x7f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB340 Network Lane TX Laser Temp
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB350 Network Lane Receiver Input Power
   0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB360
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB370
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

Cfp2MsaMis100GDwdmCoherentTableB400 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]
Cfp2MsaMis100GDwdmCoherentTableB480 = Cfp2MsaMis100GDwdmCoherentTableB400[:]
Cfp2MsaMis100GDwdmCoherentTableB480[ 64 ] = 0xff
Cfp2MsaMis100GDwdmCoherentTableB480[ 65 ] = 0x7f
Cfp2MsaMis100GDwdmCoherentTableB480[ 192 ] = 0xb3
Cfp2MsaMis100GDwdmCoherentTableB480[ 193 ] = 0x15
Cfp2MsaMis100GDwdmCoherentTableBA80 = Cfp2MsaMis100GDwdmCoherentTableB400
Cfp2MsaMis100GDwdmCoherentTableBB00 = Cfp2MsaMis100GDwdmCoherentTableB400[:]
Cfp2MsaMis100GDwdmCoherentTableBB00[ 1 ] = 0x02
Cfp2MsaMis100GDwdmCoherentTableBB00[ 0x55*2+1 ] = 0xF0
Cfp2MsaMis100GDwdmCoherentETable8180 = Cfp2MsaMis100GDwdmCoherentTable8180[:]
Cfp2MsaMis100GDwdmCoherentETable8180[ 18 ] = 0x27
Cfp2MsaMis100GDwdmCoherentETable8180[ 19 ] = 0x10
Cfp2MsaMis100GDwdmCoherentETable8180[ 20 ] = 0x27
Cfp2MsaMis100GDwdmCoherentETable8180[ 21 ] = 0x10
Cfp2MsaMis100GDwdmCoherentETable8180[ 127 ] = 0xcb
Cfp2MsaMis100GDwdmCoherentETableBB00 = Cfp2MsaMis100GDwdmCoherentTableB400[:]
Cfp2MsaMis100GDwdmCoherentETableBB00[ 0x9 ] = 0x40
Cfp2MsaMis100GDwdmCoherentETableBB00[ 1 ] = 0x02
Cfp2MsaMis100GDwdmCoherentETableBB00[ 0x55*2+1 ] = 0xF0
Cfp2MsaMis100GDwdmCoherentTableBB80 = Cfp2MsaMis100GDwdmCoherentTableB400[:]
Cfp2MsaMis100GDwdmCoherentTableBB80[ 21 ] = 0x02
Cfp2MsaMis100GDwdmCoherentTableBB80[ 42 ] = 0x40
Cfp2MsaMis100GDwdmCoherentTableBB80[ 43 ] = 0x06

#################################################
#
# Bobo DWDM Coherent module
# This module is only used for the XcvrUnlockTest
# It does not match any real transceiver
#
#################################################

Cfp2MsaMis100GBoboDwdmCoherentTable8000 = [
   0x14, 0xe4, 0x07, 0x01, 0x00, 0x00, 0x00, 0x00,
   0x08, 0x44, 0x11, 0x81, 0x81, 0x0a, 0x00, 0x00,
   0x01, 0x04, 0xca, 0x6e, 0xcc, 0x8d, 0x00, 0x64,
   0x21, 0x44, 0x40, 0x70, 0x1c, 0x3c, 0x64, 0x46,
   0x00, 0x42, 0x6f, 0x62, 0x6f, 0x43, 0x6f, 0x72,
   0x70, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x20, 0x00, 0x1c, 0x73, 0x43, 0x46, 0x50, 0x32,
   0x2d, 0x31, 0x30, 0x30, 0x47, 0x2d, 0x4c, 0x52,
   0x34, 0x20, 0x20, 0x20, 0x58, 0x41, 0x50, 0x31,
   0x33, 0x34, 0x39, 0x4a, 0x38, 0x30, 0x30, 0x35,
   0x20, 0x20, 0x20, 0x20, 0x31, 0x33, 0x31, 0x32,
   0x31, 0x32, 0x30, 0x30, 0x30, 0x30, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x00, 0x16, 0x01, 0x00, 0x01, 0x00, 0x0c, 0x03,
   0x0f, 0x68, 0x0a, 0x01, 0x0c, 0x00, 0x01, 0x01,
   0x20, 0x07, 0x07, 0x00, 0x00, 0x00, 0x20, 0x0e
]

# Table B000 has data width of 16 bits.
Cfp2MsaMis100GBoboDwdmCoherentTableB000 = [
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, # Password is not readable
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
   0x03, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x02, 0x00, 0x02, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, # Module state
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # Module FAWs
   0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x6a, 0x00, 0xf8, 0xa7, 0x62, 0x00, 0xff, 0x0f,
   0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x0f, # Temp B02f
   0x99, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # Voltage B030
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x90, 0x01, 0x2c, 0x01, 0x3c, 0xf6, 0x74, 0xf5,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # MDIO write ready bit
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # MDIO write ready bit enabled
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

Cfp2MsaMis100GBoboDwdmCoherentTable8080 = Cfp2MsaMis100GDwdmCoherentTable8080 
Cfp2MsaMis100GBoboDwdmCoherentTable8180 = Cfp2MsaMis100GDwdmCoherentTable8180
Cfp2MsaMis100GBoboDwdmCoherentTable8420 = Cfp2MsaMis100GDwdmCoherentTable8420
Cfp2MsaMis100GBoboDwdmCoherentTableB180 = Cfp2MsaMis100GDwdmCoherentTableB180
Cfp2MsaMis100GBoboDwdmCoherentTableB300 = Cfp2MsaMis100GDwdmCoherentTableB300
Cfp2MsaMis100GBoboDwdmCoherentTableB400 = Cfp2MsaMis100GDwdmCoherentTableB400
Cfp2MsaMis100GBoboDwdmCoherentTableB480 = Cfp2MsaMis100GDwdmCoherentTableB480
Cfp2MsaMis100GBoboDwdmCoherentTableBA80 = Cfp2MsaMis100GDwdmCoherentTableBA80
Cfp2MsaMis100GBoboDwdmCoherentTableBB00 = Cfp2MsaMis100GDwdmCoherentTableBB00
Cfp2MsaMis100GBoboDwdmCoherentTableBB80 = Cfp2MsaMis100GDwdmCoherentTableBB80

#################################################

# 100GBASE-XSR10 CFP2 module
# This module is only supported for module
# detection in simulation at this time.
#
# All tables have 128 entries. Tables A000 and on
# do not contain reserved entries, as XcvrCtrl
# does not read them.
#################################################
# Table 8000 has data width of 8 bits.
Cfp2MsaMis100GXSR10Table8000 = Cfp2MsaMis100GSR10Table8000[:]
# EthAppCode 0x8003 remains SR10
Cfp2MsaMis100GXSR10Table8000[ 0xe ] = 0x1e # 800E = MMF length 300m
Cfp2MsaMis100GXSR10Table8000[ 0x7f ] = 0x07 # 807F = Checksum

Cfp2MsaMis100GXSR10Table8080 = Cfp2MsaMis100GLR4Table8080
Cfp2MsaMis100GXSR10Table8420 = Cfp2MsaMis100GLR4Table8420
Cfp2MsaMis100GXSR10TableA000 = Cfp2MsaMis100GLR4TableA000
Cfp2MsaMis100GXSR10TableA200 = Cfp2MsaMis100GLR4TableA200
Cfp2MsaMis100GXSR10TableA280 = Cfp2MsaMis100GLR4TableA280
Cfp2MsaMis100GXSR10TableA400 = Cfp2MsaMis100GLR4TableA400
#################################################



# AC200-DCFP2 CFP2-DCO module
# All tables have 128 entries.
#################################################
# Page 8000 has data width of 8 bits.
Cfp2MsaMisCfp2DwdmDcoTable8000 = [
   0x19, 0xca, 0x07, 0x11, 0x00, 0x00, 0x00, 0x00,
   # 0x8009 - num of host/network lanes supported
   0x50, 0x18, 0x01, 0x00, 0x8c, 0x00, 0x00, 0x00,
   0x01, 0x01, 0xee, 0xdf, 0xf4, 0xee, 0x01, 0x90,
   # 0x801D - max power consumption
   0x35, 0xe2, 0x6c, 0x0a, 0x64, 0x00, 0x64, 0x46,
   # 0x8021 - 0x8030 Vendor Name
   0xfb, 0x41, 0x72, 0x69, 0x73, 0x74, 0x61, 0x20,
   0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b, 0x73,
   0x20, 0x00, 0x1c, 0x73, 0x43, 0x46, 0x50, 0x32,
   0x2d, 0x32, 0x30, 0x30, 0x47, 0x2d, 0x44, 0x43,
   0x4f, 0x2d, 0x41, 0x20, 0x58, 0x54, 0x43, 0x31,
   0x37, 0x33, 0x30, 0x35, 0x33, 0x33, 0x32, 0x30,
   0x20, 0x20, 0x20, 0x20, 0x32, 0x30, 0x31, 0x37,
   0x30, 0x38, 0x30, 0x34, 0xff, 0xff, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   # 0x806a-806d - hardware and firmware versions
   0x0e, 0x18, 0x30, 0x31, 0x25, 0x00, 0x0c, 0x03,
   0x0d, 0xf8, 0xb4, 0x01, 0xff, 0x00, 0x0a, 0x02,
   0x25, 0x21, 0x21, 0x25, 0x00, 0x14, 0xe0, 0x05
]

# CFP NVR2 Alarm/Warning Threshold Registers
Cfp2MsaMisCfp2DwdmDcoTable8080 = [
   0x5a, 0x00, 0x55, 0x00, 0x00, 0x00, 0xfd, 0x80,
   0x84, 0xd0, 0x83, 0xa4, 0x7d, 0x00, 0x7a, 0xa8,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0x9b, 0x82, 0x00, 0x3f, 0x00, 0x32,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xf5, 0x54, 0xee, 0xee, 0x0a, 0xaa, 0x04, 0x44,
   0x03, 0x1a, 0x03, 0x08, 0x00, 0x0a, 0x00, 0x01,
   0xf6, 0x78, 0xc3, 0xc3, 0x9b, 0x80, 0x7b, 0x87,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x37,
]

# CFP NVR4: MSA-100GLH Extended Identifiers
Cfp2MsaMisCfp2DwdmDcoTable8180 = [
   0xa0, 0x00, 0x00, 0x00, 0x09, 0xce, 0x07, 0xd0,
   0x00, 0x00, 0x00, 0xbf, 0x13, 0x88, 0x00, 0xc4,
   0x07, 0xd0, 0x17, 0x70, 0x17, 0x70, 0xc8, 0x62,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x08,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xea,
]

# CFP VR1:
Cfp2MsaMisCfp2DwdmDcoTableB000 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x03, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x0e, 0x10, 0x01, 0x10, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x10, 0x00,
   0xe1, 0xa0, 0x00, 0x00, 0x00, 0x01, 0x00, 0xff,
   0x00, 0x00, 0x00, 0x46, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x01, 0xbe, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x6a, 0xa6, 0xfc, 0x01, 0xe2, 0x0f, 0xf0,
   0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x7f,
   0x96, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
   0x00, 0x80, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x80, 0x00, 0x25, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]

Cfp2MsaMisCfp2DwdmDcoTableB480 = Cfp2MsaMis100GDwdmCoherentTableB480[ : ]
Cfp2MsaMisCfp2DwdmDcoTableB480[ 64 ] = 0xfd
Cfp2MsaMisCfp2DwdmDcoTableB480[ 65 ] = 0x44
Cfp2MsaMisCfp2DwdmDcoTableB480[ 192 ] = 0x00
Cfp2MsaMisCfp2DwdmDcoTableB480[ 193 ] = 0xf0


# 3rd party CFP2-DCO module
# Ethernet Application code (0x8003) is not populated
Cfp2MsaMisCfp2DwdmDco3rdPartyTable8000 = [
   0x19, 0x0a, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x50, 0x12, 0x01, 0x00, 0x8c, 0x00, 0x00, 0x00,
   0x01, 0x01, 0xee, 0xdf, 0xf4, 0xee, 0x01, 0x90,
   0x35, 0xe2, 0x2c, 0x0a, 0x64, 0x00, 0x64, 0x46,
   0xfb, 0x41, 0x63, 0x61, 0x63, 0x69, 0x61, 0x20,
   0x43, 0x6f, 0x6d, 0x6d, 0x20, 0x49, 0x6e, 0x63,
   0x2e, 0x00, 0x00, 0x00, 0x41, 0x43, 0x32, 0x30,
   0x30, 0x2d, 0x44, 0x32, 0x33, 0x2d, 0x30, 0x30,
   0x35, 0x20, 0x20, 0x20, 0x31, 0x38, 0x31, 0x38,
   0x35, 0x33, 0x34, 0x38, 0x38, 0x20, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x32, 0x30, 0x31, 0x38,
   0x30, 0x35, 0x30, 0x38, 0xff, 0xff, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x0e, 0x18, 0x42, 0x20, 0x26, 0x02, 0x0c, 0x03,
   0x0d, 0xf8, 0x3c, 0x01, 0xff, 0x00, 0x0a, 0x02,
   0x25, 0x21, 0x21, 0x8a, 0x69, 0x14, 0xe0, 0xb3
]

Cfp2MsaMisCfp2DwdmDco3rdPartyTable8080 = [
   0x4b, 0x00, 0x46, 0x00, 0x00, 0x00, 0xfb, 0x00,
   0x84, 0xd0, 0x83, 0xa4, 0x7e, 0x2c, 0x7d, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0x9b, 0x82, 0x00, 0x3f, 0x00, 0x32,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xf5, 0x54, 0xee, 0xee, 0x0a, 0xaa, 0x04, 0x44,
   0x03, 0x1a, 0x03, 0x08, 0x00, 0x0a, 0x00, 0x01,
   0xf6, 0x78, 0xc3, 0xc3, 0x9b, 0x80, 0x7b, 0x87,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f,
]

Cfp2MsaMisCfp2DwdmDco3rdPartyTable8180 = Cfp2MsaMisCfp2DwdmDcoTable8180[ : ]
Cfp2MsaMisCfp2DwdmDco3rdPartyTableB000 = Cfp2MsaMisCfp2DwdmDcoTableB000[ : ]
Cfp2MsaMisCfp2DwdmDco3rdPartyTableB480 = Cfp2MsaMisCfp2DwdmDcoTableB480[ : ]

##################################################
#
# OSFP Id EEPROM definitions
#
# CAB-O-O-400G-1M
#   - CMIS 3.0 compliant
#   - supports 400kHz TWI access
#
##################################################

Osfp400gCr8LowerPage00 = [
   0x19, 0x30, 0x80, 0x03, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x1D, 0x01,
   0x88, 0x01, 0x1C, 0x01, 0x44, 0x11, 0x1B, 0x01,
   0x22, 0x55, 0x18, 0x01, 0x11, 0xFF, 0x1A, 0x01,
   0x44, 0x11, 0x16, 0x01, 0x11, 0xFF, 0x17, 0x01,
   0x44, 0x11, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]

Osfp400gCr8UpperPage00 = [
   0x19, 'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn',
   'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn',
   'vn', 0x00, 0x1C, 0x73, 0x43, 0x41, 0x42, 0x2D,
   0x4F, 0x2D, 0x4F, 0x2D, 0x34, 0x30, 0x30, 0x47,
   0x2D, 0x31, 0x4D, 0x20, 'sn', 'sn', 'sn', 'sn',
   'sn', 'sn', 'sn', 'sn', 'sn', 'sn', 'sn', 'sn',
   'sn', 'sn', 'sn', 'sn', 'dc', 'dc', 'dc', 'dc',
   'dc', 'dc', 'dc', 'dc', 0x20, 0x20, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x41, 0x23, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x02, 0x0A, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

##################################################
#
# OSFP Id EEPROM definitions
#
# CAB-O-O-400G-1M
#   - CMIS 4.0 compliant
#   - supports 400kHz TWI access
#
##################################################

Cmis4Osfp400gCr8LowerPage00 = [
   0x19, 0x40, 0x80, 0x03, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x1D, 0x01,
   0x88, 0x01, 0x1C, 0x01, 0x44, 0x11, 0x1B, 0x01,
   0x22, 0x55, 0x18, 0x01, 0x11, 0xFF, 0x1A, 0x01,
   0x44, 0x11, 0x16, 0x01, 0x11, 0xFF, 0x17, 0x01,
   0x44, 0x11, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]              
               
Cmis4Osfp400gCr8UpperPage00 = [
   0x19, 'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn',
   'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn',
   'vn', 0x00, 0x1C, 0x73, 0x43, 0x41, 0x42, 0x2D,
   0x4F, 0x2D, 0x4F, 0x2D, 0x34, 0x30, 0x30, 0x47,
   0x2D, 0x31, 0x4D, 0x20, 'sn', 'sn', 'sn', 'sn',
   'sn', 'sn', 'sn', 'sn', 'sn', 'sn', 'sn', 'sn',
   'sn', 'sn', 'sn', 'sn', 'dc', 'dc', 'dc', 'dc',
   'dc', 'dc', 'dc', 'dc', 0x20, 0x20, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x41, 0x23, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x02, 0x0A, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]              


##################################################
#
# OSFP 400gSr8
# INNOLIGHT T-OS8FNS-H00
# Note: EEPROM is edited to our needs.
##################################################

Osfp400gSr8LowerPage00 = [
   0x19, 0x40, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x26, 0xf6,
   0x84, 0x8a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x11, 0x10,
   0x88, 0x01, 0x0b, 0x09, 0x44, 0x11, 0x0a, 0x07,
   0x11, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

Osfp400gSr8UpperPage00 = [
   0x19, 'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn',
   'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn',
   'vn', 0x44, 0x7c, 0x7f, 0x54, 0x2d, 0x4f, 0x53,
   0x38, 0x46, 0x4e, 0x53, 0x2d, 0x48, 0x30, 0x30,
   0x20, 0x20, 0x20, 0x20, 0x31, 0x41, 'sn', 'sn',
   'sn', 'sn', 'sn', 'sn', 'sn', 'sn', 'sn', 'sn',
   'sn', 'sn', 'sn', 'sn', 'sn', 'sn', 'dc', 'dc',
   'dc', 'dc', 'dc', 'dc', 'dc', 'dc', 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xc0, 0x38, 0x00, 0x6a, 0x00, 0x46, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x37, 0x72,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]

Osfp400gSr8UpperPage01 = [
   0xff, 0xff, 0x33, 0x41, 0x00, 0x00, 0x32, 0x23,
   0x00, 0xff, 0x42, 0x68, 0x07, 0xd0, 0x35, 0x00,
   0xff, 0x00, 0x3c, 0x14, 0xff, 0xff, 0x9b, 0x78,
   0x6e, 0xf9, 0x32, 0x12, 0x00, 0x06, 0x06, 0x03,
   0x07, 0x1f, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x01, 0x11, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xef,
]

Osfp400gSr8UpperPage02 = [
   0x50, 0x00, 0xf6, 0x00, 0x4b, 0x00, 0xfb, 0x00,
   0x8e, 0x00, 0x74, 0x00, 0x87, 0x70, 0x7a, 0x48,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x50, 0x00, 0xf6, 0x00, 0x4b, 0x00, 0xfb, 0x00,
   0x50, 0x00, 0xf6, 0x00, 0x4b, 0x00, 0xfb, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x8a, 0x99, 0x11, 0x72, 0x6e, 0x17, 0x15, 0xf7,
   0x1d, 0x4a, 0x08, 0xc8, 0x05, 0xdd, 0x19, 0x63,
   0x7b, 0x86, 0x02, 0x77, 0x62, 0x1e, 0x03, 0xe8,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
]

Osfp400gSr8UpperPage10Bank0 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

Osfp400gSr8UpperPage11Bank0 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00,
   0x00, 0x21, 0x43, 0x65, 0x87, 0x00, 0x00, 0x10,
   0x32, 0x54, 0x76, 0x67, 0x45, 0x23, 0x01, 0x10,
   0x10, 0x10, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

Osfp400gSr8Cmis3LowerPage00 = [
   0x19, 0x30, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x26, 0xf6,
   0x84, 0x8a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x11, 0x10,
   0x88, 0x01, 0x0b, 0x09, 0x44, 0x11, 0x0a, 0x07,
   0x11, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

##################################################
#
# OSFP 400gAr8
# INNOLIGHT        T-OC8FNT-H00
#
# Note: EEPROM is edited to our needs.
# Upper pages are not defined here. We will use
# 400gSr8 upper pages for now.
##################################################

Osfp400gAr8LowerPage00 = [
   0x19, 0x06, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2c, 0xb4,
   0x7c, 0xb7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x11, 0x01,
   0x88, 0x01, 0x0f, 0x01, 0x44, 0x11, 0xff, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

Osfp400gAr8UpperPage00 = [
   0x11, 'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn',
   'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn',
   'vn', 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0xff, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x3f, 0xb9, 0x30, 0x31, 'sn', 'sn',
   'sn', 'sn', 'sn', 'sn', 'sn', 'sn', 'sn', 'sn',
   'sn', 'sn', 'sn', 'sn', 'sn', 'sn', 'dc', 'dc',
   'dc', 'dc', 'dc', 'dc', 'dc', 'dc', 0x0e, 0xc3,
   0x0e, 0xc3, 0x0e, 0xc3, 0x4d, 0x73, 0x4a, 0x25,
   0x4f, 0x03, 0x4b, 0x7a, 0x47, 0x87, 0x4b, 0xfa,
   0x4b, 0x5d, 0x48, 0x46, 0x00, 0x00, 0x00, 0x00,
   0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
   0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]

Osfp400gAr8UpperPage01 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
]

Osfp400gAr8UpperPage02 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

# Cra8 changes mediaInterfaceTech
Osfp400gCra8UpperPage00 = Osfp400gAr8UpperPage00[ : ]
Osfp400gCra8UpperPage00[ 84 ] = 0xA

##################################################
#
# OSFP 400g2Fr4
# Arista Networks  OSFP-400G-2FR4   INIAI4850208
##################################################

Osfp400g2Fr4LowerPage00 = [
   0x19, 0x40, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
   0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x29, 0xaa,
   0x84, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x0f, 0x18,
   0x44, 0x11, 0x0b, 0x10, 0x44, 0x11, 0x06, 0x09,
   0x44, 0x11, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

Osfp400g2Fr4UpperPage00 = [
   0x19, 0x41, 0x72, 0x69, 0x73, 0x74, 0x61, 0x20,
   0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b, 0x73,
   0x20, 0x00, 0x1c, 0x73, 0x4f, 0x53, 0x46, 0x50,
   0x2d, 0x34, 0x30, 0x30, 0x47, 0x2d, 0x32, 0x46,
   0x52, 0x34, 0x20, 0x20, 0x32, 0x30, 0x49, 0x4e,
   0x49, 0x41, 0x4f, 0x34, 0x38, 0x35, 0x30, 0x32,
   0x33, 0x35, 0x20, 0x20, 0x20, 0x20, 0x31, 0x38,
   0x30, 0x34, 0x30, 0x33, 0x20, 0x20, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xa0, 0x30, 0x00, 0x25, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x43, 0xc6, 0x9d, 0x88
]

Osfp400g2Fr4UpperPage01 = [
   0x00, 0x00, 0x35, 0x41, 0x42, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x65, 0xa4, 0x05, 0x14, 0x24, 0x04,
   0x44, 0x20, 0x3c, 0x14, 0x00, 0x00, 0x9d, 0x78,
   0x6e, 0x00, 0x00, 0x02, 0x00, 0x06, 0x06, 0x03,
   0x07, 0x09, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x11, 0x11, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8e
]

Osfp400g2Fr4UpperPage02 = [
   0x46, 0x00, 0x0a, 0x00, 0x41, 0x00, 0x0f, 0x00,
   0x8c, 0xa0, 0x75, 0x30, 0x87, 0x55, 0x7a, 0x75,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x57, 0x88, 0x0c, 0x5a, 0x4e, 0x01, 0x0d, 0xdc,
   0xff, 0xdc, 0x44, 0x5a, 0xfb, 0xf4, 0x48, 0x45,
   0x7b, 0x86, 0x02, 0x77, 0x62, 0x1e, 0x03, 0xe8,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x93
]

Osfp400g2Fr4UpperPage10Bank0 = [
   0xff, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x11, 0x11, 0x11, 0x11, 0x19, 0x19, 0x19,
   0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x10,
   0x28, 0x28, 0x28, 0x28, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

Osfp400g2Fr4UpperPage11Bank0 = [
   0x44, 0x44, 0x44, 0x44, 0x00, 0x00, 0xff, 0x00,
   0xff, 0xfb, 0x00, 0x00, 0xff, 0x00, 0xff, 0x00,
   0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0x00,
   0xff, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
   0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
   0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
   0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
   0x00, 0x01, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
   0x11, 0x11, 0x19, 0x19, 0x19, 0x19, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x11, 0x21, 0x31, 0x41, 0x12, 0x22, 0x32, 0x42,
   0x11, 0x21, 0x31, 0x41, 0x12, 0x22, 0x32, 0x42
]

##################################################
#
# OSFP POLS AmpZr
# II-VI            60P010001        P01000004
##################################################
OsfpAmpZrLowerPage00 = [
   0x19, 0x40, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0xed,
   0xff, 0xff, 0x00, 0x00, 0x22, 0x1d, 0x21, 0xe9,
   0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0xff, 0xff, 0x02, 0x00, 0xC2,
   0x02, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

# TODO: Fake upper page data for POLS
#       Will update them once we have a sample
OsfpAmpZrUpperPage00 = [
   0x19, 'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn',
   'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn',
   'vn', 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0xff, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x3f, 0xb9, 0x30, 0x31, 'sn', 'sn',
   'sn', 'sn', 'sn', 'sn', 'sn', 'sn', 'sn', 'sn',
   'sn', 'sn', 'sn', 'sn', 'sn', 'sn', 'dc', 'dc',
   'dc', 'dc', 'dc', 'dc', 'dc', 'dc', 0x0e, 0xc3,
   0x0e, 0xc3, 0x0e, 0xc3, 0x4d, 0x73, 0x4a, 0x25,
   0x4f, 0x03, 0x4b, 0x7a, 0x47, 0x87, 0x4b, 0xfa,
   0x4b, 0x5d, 0x48, 0x46, 0x00, 0x00, 0x00, 0x00,
   0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x03,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
   0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]

# Currently, TX bias thresholds are arbitrary numbers because II-VI has not
# implemented current monitoring yet. Aux2 thresholds are inflated by 1C
# to differentiate with Aux3 thresholds in btest.
OsfpAmpZrUpperPage02 = [
   0x46, 0x00, 0xfb, 0x00, 0x41, 0x00, 0x00, 0x00,
   0x87, 0x5a, 0x7a, 0x76, 0x83, 0x7c, 0x7e, 0x54,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x4c, 0x00, 0x01, 0x00, 0x47, 0x00, 0x06, 0x00,
   0x4b, 0x00, 0x00, 0x00, 0x46, 0x00, 0x05, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xf4, 0x24, 0x01, 0x38, 0xdb, 0xba, 0x04, 0xe2,
   0x0b, 0xb8, 0x00, 0xfa, 0x07, 0xd0, 0x01, 0xf4,
   0x93, 0x76, 0x00, 0x08, 0x75, 0x30, 0x00, 0x0f,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf4, 0x24,
   0x01, 0x38, 0xdb, 0xba, 0x04, 0xe2, 0x75, 0x30,
   0x09, 0xc4, 0x4e, 0x20, 0x13, 0x88, 0x00, 0xee,
   0x00, 0x05, 0x00, 0xbc, 0x00, 0x0a, 0x02, 0x70,
]

# Set some monitors to arbitrary values for use in breadth tests. These values
# were taken from an operating module.
OsfpAmpZrUpperPage11Bank0 = [ 0 ] * 128
OsfpAmpZrUpperPage11Bank0[ 0 ] = 0x44
OsfpAmpZrUpperPage11Bank0[ 26:28 ] = [ 0x7b, 0x0b ]
OsfpAmpZrUpperPage11Bank0[ 28:30 ] = [ 0x79, 0x70 ]
OsfpAmpZrUpperPage11Bank0[ 58:60 ] = [ 0x01, 0x0d ]
OsfpAmpZrUpperPage11Bank0[ 60:62 ] = [ 0x00, 0x97 ]
OsfpAmpZrUpperPage11Bank0[ 78:80 ] = [ 0x10, 0x10 ]

OsfpAmpZrUpperPageC0 = [ 0 ] * 128

OsfpAmpZrUpperPageC1 = [ 0 ] * 128

OsfpAmpZrUpperPageD0 = [ 0 ] * 128

OsfpAmpZrUpperPageD1 = [ 0 ] * 128

OsfpAmpZrUpperPageD8 = [ 0 ] * 128

OsfpAmpZrUpperPageD9 = [ 0 ] * 128

##################################################
#
# Arista Networks  OSFP-400G-DR4    XKT193800012
# OSFP 400G DR4
##################################################

Osfp400gDr4LowerPage00 = [
   0x19, 0x40, 0x00, 0x06, 0x08, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2a, 0x1a,
   0x7f, 0xa7, 0x00, 0x00, 0x37, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x11, 0x1c,
   0x84, 0x01, 0x0d, 0x14, 0x21, 0x55, 0xff, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]

Osfp400gDr4UpperPage00 = [
   0x19, 0x41, 0x72, 0x69, 0x73, 0x74, 0x61, 0x20,
   0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b, 0x73,
   0x20, 0x00, 0x1c, 0x73, 0x4f, 0x53, 0x46, 0x50,
   0x2d, 0x34, 0x30, 0x30, 0x47, 0x2d, 0x44, 0x52,
   0x34, 0x20, 0x20, 0x20, 0x32, 0x30, 0x58, 0x4b,
   0x54, 0x31, 0x39, 0x33, 0x38, 0x30, 0x30, 0x30,
   0x31, 0x32, 0x20, 0x20, 0x20, 0x20, 0x31, 0x39,
   0x30, 0x39, 0x32, 0x31, 0x20, 0x20, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x80, 0x28, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0xf0, 0x00, 0x06, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0xd5, 0xd8, 0x27, 0xdd,
]

Osfp400gDr4UpperPage01 = [
   0x01, 0x00, 0x01, 0x00, 0x05, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x66, 0x6c, 0x05, 0x14, 0x04, 0xdf,
   0x77, 0xc0, 0x46, 0x00, 0x00, 0x00, 0x9d, 0x18,
   0x00, 0xf0, 0x77, 0x2f, 0x03, 0x0f, 0x06, 0x0b,
   0x07, 0x09, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x01, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc,
]

Osfp400gDr4UpperPage02 = [
   0x4b, 0x00, 0xfb, 0x00, 0x46, 0x00, 0x00, 0x00,
   0x8d, 0xcc, 0x77, 0x24, 0x87, 0x5a, 0x7a, 0x76,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x48, 0x00, 0x1c, 0x00, 0x46, 0x00, 0x1e, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x8a, 0x99, 0x0f, 0xea, 0x6e, 0x18, 0x14, 0x09,
   0xfd, 0xe8, 0x13, 0x88, 0xf4, 0x24, 0x1d, 0x4c,
   0x8a, 0x99, 0x05, 0x08, 0x6e, 0x18, 0x07, 0xfa,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x90,
]

Osfp400gDr4UpperPage10Bank0 = [
   0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x21, 0x21, 0x25, 0x25, 0x29, 0x29, 0x2d,
   0x2d, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0x55, 0x55, 0x55, 0x55, 0x00, 0x00,
   0x00, 0x00, 0x22, 0x22, 0x22, 0x22, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x10,
   0x10, 0x10, 0x10, 0x10, 0xff, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0xff, 0xff, 0x55, 0x55, 0x55,
   0x55, 0x00, 0x00, 0x00, 0x00, 0x22, 0x22, 0x22,
   0x22, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]

Osfp400gDr4UpperPage11Bank0 = [
   0x44, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x3d, 0x01, 0x3c, 0x75, 0x2a, 0x62,
   0x2d, 0x4e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x93, 0x44, 0x9c, 0xd6, 0xa6, 0xcc,
   0xa6, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x5b, 0xf2, 0x3a, 0xce, 0x33, 0x02,
   0x4d, 0xda, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x11, 0x11, 0x11, 0x11, 0x21, 0x21,
   0x25, 0x25, 0x29, 0x29, 0x2d, 0x2d, 0xff, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x55,
   0x55, 0x55, 0x55, 0x00, 0x00, 0x00, 0x00, 0x22,
   0x22, 0x22, 0x22, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x11, 0x12, 0x13, 0x14, 0x00, 0x00, 0x00, 0x00,
   0x11, 0x12, 0x13, 0x14, 0x00, 0x00, 0x00, 0x00,
]

Osfp400gXdr4UpperPage01 = Osfp400gDr4UpperPage01[ : ]
# Set link length (SMF) for Xdr4 to 2 km
Osfp400gXdr4UpperPage01[ 4 ] = 0x42
# Update checksum
Osfp400gXdr4UpperPage01[ 127 ] = 0x39

# Fr4
Osfp400gFr4LowerPage00 = Osfp400gDr4LowerPage00[ : ]
# Change media interface code for ApSel 1 to 400GBASE-FR4
Osfp400gFr4LowerPage00[ 87 ] = 0x1d
Osfp400gFr4LowerPage00[ 90 ] = 0xff

##################################################
# QSFP-DD
# Id EEPROM taken from Amphenol 1m 30AWG module
##################################################

QsfpDd400gCr8LowerPage00 = [
   0x18, 0x27, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x1D, 0x01,
   0x88, 0x01, 0x1C, 0x01, 0x44, 0x11, 0x1B, 0x01,
   0x22, 0x55, 0x18, 0x01, 0x11, 0xFF, 0x1A, 0x01,
   0x44, 0x11, 0x16, 0x01, 0x11, 0xFF, 0x17, 0x01,
   0x44, 0x11, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]

QsfpDd400gCr8UpperPage00 = [
   0x18, 'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn',
   'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn', 'vn',
   'vn', 0x78, 0xa7, 0x14, 0x4e, 0x44, 0x59, 0x59,
   0x59, 0x46, 0x2d, 0x30, 0x30, 0x30, 0x31, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x30, 0x31, 'sn', 'sn',
   'sn', 'sn', 'sn', 'sn', 'sn', 'sn', 'sn', 'sn',
   'sn', 'sn', 'sn', 'sn', 'sn', 'sn', 'dc', 'dc',
   'dc', 'dc', 'dc', 'dc', 'dc', 'dc', 0x20, 0x20,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x01, 0x0a, 0x23, 0x05, 0x06,
   0x0b, 0x00, 0x00, 0x00, 0x00, 0x02, 0x0a, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

##################################################
# QSFP-CMIS 200GBASE-Cr4
# Sufficient to change identifier to QSFP-CMIS
##################################################

QsfpCmis200gCr4LowerPage00 = QsfpDd400gCr8LowerPage00[:]
QsfpCmis200gCr4LowerPage00[ 0 ] = 0x1E # identifier QSFP-CMIS

QsfpCmis200gCr4UpperPage00 = QsfpDd400gCr8UpperPage00[:]
QsfpCmis200gCr4UpperPage00[ 0 ] = 0x1E # identifier QSFP-CMIS

##################################################
# QSFP-DD and OSFP 400GBASE-ZR
# Id EEPROM taken from Acacia module
##################################################

QsfpDd400gZrLowerPage00 = [
   0x18, 0x40, 0x04, 0x02, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x11, 0x3e,
   0x81, 0x01, 0x0f, 0x00, 0x41, 0x11, 0x0d, 0x00,
   0x21, 0x55, 0x11, 0xc0, 0x81, 0x01, 0x11, 0xc1,
   0x81, 0x01, 0x11, 0xc2, 0x81, 0x01, 0x0f, 0xc0,
   0x41, 0x11, 0x0f, 0xc2, 0x41, 0x11, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]
Osfp400gZrLowerPage00 = QsfpDd400gZrLowerPage00[ : ]
Osfp400gZrLowerPage00[ 0 ] = 0x19

QsfpDd400gZrUpperPage00 = [
   0x18, 0x41, 0x72, 0x69, 0x73, 0x74, 0x61, 0x20,
   0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b, 0x73,
   0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xe0, 0x50, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]
Osfp400gZrUpperPage00 = QsfpDd400gZrUpperPage00[ : ]
Osfp400gZrUpperPage00[ 0 ] = 0x19

QsfpDd400gZrUpperPage01 = Osfp400g2Fr4UpperPage01[ : ]
# Set Media Lane Assignment for ApSel 1 to 0x1
QsfpDd400gZrUpperPage01[ 48 ] = 0x01
# Set support of tunable wavelength
QsfpDd400gZrUpperPage01[ 27 ] |= 0x40
# Set VDM support
QsfpDd400gZrUpperPage01[ 14 ] |= 0x40
# Update checksum
QsfpDd400gZrUpperPage01[ 127 ] = 0xfe

QsfpDd400gZrUpperPage04 = [
   0xa0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xe8,
   0x00, 0x23, 0x00, 0x00, 0x00, 0x00, 0xff, 0x9f,
   0x00, 0x8f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]

QsfpDd400gZrUpperPage12Bank0 = [ 0 ] * 128

# 400G-ZR VDM Pages
QsfpDd400gZrUpperPage2FBank0 = [ 0 ] * 128
# Set number of supported VDM groups to 4
QsfpDd400gZrUpperPage2FBank0[ 0 ] = 0x3

QsfpDd400gZrUpperPage20Bank0 = [ 0 ] * 128
QsfpDd400gZrUpperPage20Bank0[ 2:4 ] = [ 0x10, 0x02 ]
QsfpDd400gZrUpperPage20Bank0[ 6:8 ] = [ 0x30, 0x04 ]
QsfpDd400gZrUpperPage20Bank0[ 16:18 ] = [ 0x80, 0x09 ]
QsfpDd400gZrUpperPage20Bank0[ 18:20 ] = [ 0x90, 0x0a ]
QsfpDd400gZrUpperPage20Bank0[ 22:24 ] = [ 0xd0, 0x0e ]

QsfpDd400gZrUpperPage28Bank0 = [ 0 ] * 128
QsfpDd400gZrUpperPage28Bank0[ 24:32 ] = [ 0x64, 0x00, 0xe7, 0x00,
                                          0x5f, 0x00, 0xec, 0x00 ]

##################################################
# OSFP Google
# Id EEPROM taken from Google's supplied module
##################################################

GoogleOM8FNTUpperPage00 = [
   0x19, 0x49, 0x4e, 0x4e, 0x4f, 0x4c, 0x49, 0x47,
   0x48, 0x54, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x20, 0x44, 0x7c, 0x7f, 0x54, 0x2d, 0x4f, 0x4d,
   0x38, 0x46, 0x4e, 0x54, 0x2d, 0x48, 0x30, 0x31,
   0x20, 0x20, 0x20, 0x20, 0x31, 0x42, 0x49, 0x4e,
   0x49, 0x42, 0x4b, 0x38, 0x31, 0x36, 0x33, 0x30,
   0x37, 0x30, 0x20, 0x20, 0x20, 0x20, 0x31, 0x38,
   0x30, 0x39, 0x32, 0x36, 0x30, 0x36, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0xa0, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf6, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]


##################################################
# QSFP56 200GBASE-FR4
# Finisar
##################################################
QsfpCmis200gFr4LowerPage00 = [
   0x1e, 0x40, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x7d,
   0x80, 0xac, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xca,
   0xa7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x0f, 0x18,
   0x44, 0x01, 0x0b, 0x10, 0x44, 0x01, 0xff, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]

QsfpCmis200gFr4UpperPage00 = [
   0x1e, 0x46, 0x49, 0x4e, 0x49, 0x53, 0x41, 0x52,
   0x20, 0x43, 0x4f, 0x52, 0x50, 0x2e, 0x20, 0x20,
   0x20, 0x00, 0x90, 0x65, 0x46, 0x54, 0x43, 0x43,
   0x31, 0x31, 0x31, 0x32, 0x45, 0x31, 0x50, 0x4c,
   0x4c, 0x2d, 0x46, 0x42, 0x20, 0x20, 0x58, 0x31,
   0x50, 0x41, 0x48, 0x54, 0x44, 0x20, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x31, 0x39,
   0x30, 0x38, 0x31, 0x34, 0x00, 0x00, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x40, 0x1a, 0xde, 0x07, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]

QsfpCmis200gFr4UpperPage01 = [
   0x00, 0x00, 0x0a, 0x01, 0x1e, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x65, 0xa4, 0x1b, 0xee, 0x20, 0x04,
   0x57, 0x00, 0x41, 0x0f, 0x00, 0x00, 0x9d, 0x18,
   0x00, 0x20, 0x00, 0x2f, 0x03, 0x07, 0x06, 0x03,
   0x07, 0x09, 0x21, 0x60, 0x0f, 0x9f, 0x80, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xde,
   #0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf,
]

QsfpCmis200gFr4UpperPage02 = [
   0x46, 0x00, 0x0a, 0x00, 0x46, 0x00, 0x0a, 0x00,
   0x8b, 0x42, 0x76, 0x8e, 0x8b, 0x42, 0x76, 0x8e,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xb6, 0xb6, 0x09, 0x5f, 0xb6, 0xb6, 0x09, 0x5f,
   0x92, 0x7c, 0x1d, 0x4c, 0x92, 0x7c, 0x1d, 0x4c,
   0xb6, 0xb6, 0x00, 0x01, 0xb6, 0xb6, 0x00, 0x01,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb2,
]

QsfpCmis200gFr4UpperPage10 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x11, 0x11, 0x11, 0x11, 0x18, 0x18, 0x18,
   0x18, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x10,
   0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]

QsfpCmis200gFr4UpperPage11 = [
   0x44, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x52, 0xf6, 0x69, 0xae, 0x59, 0x79,
   0x59, 0xd6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x46, 0x7e, 0x48, 0x58, 0x4d, 0xd0,
   0x49, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x56, 0x90, 0x5f, 0xa9, 0x58, 0x58,
   0x5e, 0x3e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x55, 0x55, 0x55, 0x55, 0x10, 0x10,
   0x10, 0x10, 0x18, 0x18, 0x18, 0x18, 0xff, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]

##################################################
# QSFP-DD 400GBASE-Sr8 (CMIS 4.0)
# Replace the OSFP identifier with QSFP-DD
##################################################

QsfpDd400gSr8LowerPage00 = Osfp400gSr8LowerPage00[ : ]
QsfpDd400gSr8LowerPage00[ 0 ] = 0x18

QsfpDd400gSr8UpperPage00 = Osfp400gSr8UpperPage00[ : ]
QsfpDd400gSr8UpperPage00[ 0 ] = 0x18

##################################################
# QSFP-DD 400GBASE-Sr8 (CMIS 3.0)
# Replace the 4.0 revision with 3.0 revision
##################################################

QsfpDd400gSr8Cmis3LowerPage00 = QsfpDd400gSr8LowerPage00[ : ]
QsfpDd400gSr8Cmis3LowerPage00[ 1 ] = 0x30

##################################################

emptyEprom = [ 0x0 ] * idPromLen

# trailing spaces in these names are required to meet the 16 byte 
# field size.
eepromTemplates = {
   "empty" : ("sfp", emptyEprom, None),
   "BoboSRv1001"    : ("sfp", SRv1001SFP, "BoboCorp        "),
   "BoboSRv1001nondom" : ("sfp", SRv1001nondomSFP, "BoboCorp        "),
   "ArastraSRv1001" : ("sfp", SRv1001SFP, "Arastra, Inc.   "),
   "AristaSRv1001"  : ("sfp", SRv1001SFP, "Arista Networks "),
   "AristaSRv1002"  : ("sfp", SRv1002SFP, "Arista Networks "),
   "AristaSRv1003"  : ("sfp", SRv1003SFP, "Arista Networks "),
   "AristaSRNSRBv1003"  : ("sfp", SRv1003NOSOLREGBASESFP, "Arista Networks "),
   "AristaSRv1001nondom"  : ("sfp", SRv1001nondomSFP, "Arista Networks "),
   "AristaSRv1001External"  : ("sfp", SRv1001SFPExternal, "Arista Networks "),
   "AristaSRv1001BadCc"  : ("sfp", SRv1001SFPBadCc, "Arista Networks "),
   "OemBXD" : ("sfp", BXDSFP, "OEM             "),
   "OemBXU" : ("sfp", BXUSFP, "OEM             "),
   "HPTwinAx"  : ("sfp", HPTwinAx, "Hewlett Packard "),
   "AristaLRv1001"  : ("sfp", LRv1001SFP, "Arista Networks "),
   "AristaLRMv1001"  : ("sfp", LRMv1001SFP, "Arista Networks "),
   "AristaCR"  : ("sfp", CRSFP, "Arista Networks "),
   "AristaCRA"  : ("sfp", CRSFPA, "Arista Networks "),
   "AristaCRP"  : ("sfp", CRSFPP, "Arista Networks "),
   "AristaCR25G"  : ("sfp", CRSFP25G, "Arista Networks "),
   "AristaCR25GS"  : ("sfp", CRSSFP25G, "Arista Networks "),
   "AristaCR25GN"  : ("sfp", CRNSFP25G, "Arista Networks "),
   "AristaCRP25G"  : ("sfp", CRSFPP25G, "Arista Networks "),
   "AristaCR50G"  : ("sfp", CRSFP50G, "Arista Networks "),
   "AristaCRP50G"  : ("sfp", CRSFPP50G, "Arista Networks "),
   "AristaLXv1001"  : ("sfp", LXv1001SFP, "Arista Networks "),
   "AristaLXv1001nondom"  : ("sfp", LXv1001nondomSFP, "Arista Networks "),
   "AristaLX10"  : ("sfp", LX10SFP, "Arista Networks "),
   "BoboSRDualSpeed"  : ("sfp", SRDualSpeed, "BoboCorp        "),
   "BoboLRDualSpeed"  : ("sfp", LRDualSpeed, "BoboCorp        "),
   "AristaTv2" : ("sfp", Tv2SFP, "Arista Networks "), 
   "Molex_Tan" : ("sfp", Molex_74765_0902 , "Molex Inc.      " ),
   "AristaDWDMER"  : ("sfp", SFPDWDM, "Arista Networks "),
   "AristaDWDMZR"  : ("sfp", SFPDWDMZR, "Arista Networks "),
   "AristaDWDMZT"  : ("sfp", SFPDWDMZT, "Arista Networks "),
   "AristaSRAOC"  : ("sfp", AOCSFP, "Arista Networks "),
   "AristaSRL"  : ("sfp", SRLSFP, "Arista Networks "),
   "AristaLRL"  : ("sfp", LRLSFP, "Arista Networks "),
   "AristaER"  : ("sfp", ERSFP, "Arista Networks "),
   "AristaERBDU"  : ("sfp", ERBDUSFP, "Arista Networks "),
   "AristaERBDD"  : ("sfp", ERBDDSFP, "Arista Networks "),
   "AristaERLBDU"  : ("sfp", ERLBDUSFP, "Arista Networks "),
   "AristaERLBDD"  : ("sfp", ERLBDDSFP, "Arista Networks "),
   "AristaZR"  : ("sfp", ZRSFP, "Arista Networks "),
   "AristaSR25G"  : ("sfp", SRSFP25G, "Arista Networks "),
   "AristaXSR25G" : ("sfp", XSRSFP25G, "Arista Networks "),
   "AristaAR25G"  : ("sfp", ARSFP25G, "Arista Networks "),
   "3rdPartyAR25G"  : ("sfp", ARSFP25G3RDPTY, "Arista Networks "),
   "AristaAR25GN"  : ("sfp", ARNSFP25G, "Arista Networks "),
   "AristaLR25G"  : ("sfp", LRSFP25G, "Arista Networks "),
   "AristaLRL25G"  : ("sfp", LRLSFP25G, "Arista Networks "),
   "Unknown25GSfp" : ("sfp", UNKNOWNSFP25G, "Unknown Corp "),
   "AristaT10G" : ("sfp", TSFP10G, "Arista Networks "),
   "3rdPartyT10GSR" : ( "sfp", AQS107_TSFP10GSR, "AQUANTIA" ),
   "AristaMRSR25G" : ("sfp", MRSRSFP25G, "Arista Networks"),
   "AristaMRLR25G" : ("sfp", MRLRSFP25G, "Arista Networks"),
   # Useful shorthands for the command-line 'xcvr' tool
   "CR"  : ("sfp", CRSFP, "Arista Networks "),
   "SR"  : ("sfp", SRv1001SFP, "Arista Networks "),
   "SRnondom"  : ("sfp", SRv1001nondomSFP, "Arista Networks "),
   "LR"  : ("sfp", LRv1001SFP, "Arista Networks "),
   "LRM"  : ("sfp", LRMv1001SFP, "Arista Networks "),
   "SX"  : ("sfp", SXSFP, "Arista Networks "),
   "LX"  : ("sfp", LXv1001SFP, "Arista Networks "),
   "LXnondom"  : ("sfp", LXv1001nondomSFP, "Arista Networks "),
   "T"  : ("sfp", Tv2SFP, "Arista Networks "),
   "3rdPartyT" : ("sfp", TSFP3RDPTY, "Arista Networks "), 

   # Qsfp types.
   "AristaSR4Qsfp" : ("qsfp", SRQSFP, "Arista Networks "),
   "AristaSR4Qsfp1002" : ("qsfp", SRQSFP1002, "Arista Networks "),
   "AristaSR4Qsfp1003" : ("qsfp", SRQSFP1003, "Arista Networks "),
   "AristaSR4Qsfp1004" : ("qsfp", SRQSFP1004, "Arista Networks "),
   "AristaSR4QsfpNSRB1003" : ("qsfp", SRQSFPNOSOLREGBASE1003, "Arista Networks "),
   "AristaXSR4Qsfp" : ("qsfp", XSR4QSFP, "Arista Networks "),
   "AristaSR4DQsfp" : ("qsfp", SR4DQSFP, "Arista Networks "),
   "AristaSRBDQsfp" : ("qsfp", SRBDQSFP, "Arista Networks "),
   "3rdPartySRBDQsfp" : ("qsfp", SRBDQSFP3RDPTY, "BoboCorp        "),
   "BoboSR4Qsfp" : ("qsfp", SRQSFP, "BoboCorp        "),
   "AristaAR4Qsfp" : ("qsfp", AR4QSFP, "Arista Networks "),
   "AristaCR4Qsfp" : ("qsfp", CR4QSFP, "Arista Networks "), 
   "3rdPartyCR4Qsfp" : ("qsfp", CR4QSFP3RDPTY, "BoboCorp        "),
   "AristaLR4Qsfp" : ("qsfp", LR4QSFP, "Arista Networks "),
   "AristaLR4Qsfp1002" : ("qsfp", LR4QSFP1002, "Arista Networks "),
   "AristaLRL4Qsfp" : ("qsfp", LRL4QSFP, "Arista Networks "),
   "AristaPLR4Qsfp" : ("qsfp", PLR4QSFP, "Arista Networks "),
   "AristaPLRL4Qsfp" : ("qsfp", PLRL4QSFP, "Arista Networks "),
   "3rdPartyPLR4Qsfp" : ("qsfp", PLR4QSFP3PTY, "BoboCorp        "),
   "Psm4Qsfp" : ("qsfp", PSM4QSFP, "Psm4 Corp      "),
   "AristaUNIVQsfp" : ("qsfp", UNIVQSFP, "Arista Networks "),
   "AristaER4Qsfp" : ("qsfp", ER4QSFP, "Arista Networks "), 
   "Arista100GSR4Qsfp" : ("qsfp", SR4QSFP100G, "Arista Networks "),
   "Arista100GXSR4Qsfp" : ("qsfp", XSR4QSFP100G, "Arista Networks "),
   "3rdParty100G40GSR4Qsfp" : ("qsfp", SR4QSFP100G40G3RDPTY, "BoboCorp        "),
   "Arista100GLR4Qsfp" : ("qsfp", LR4QSFP100G, "Arista Networks "),
   "Arista100GLRL4Qsfp" : ("qsfp", LRL4QSFP100G, "Arista Networks "),
   "3rdParty100G40GLR4Qsfp" : ("qsfp", LR4QSFP100G40G3RDPTY, "BoboCorp        "),
   "Arista100GER4Qsfp" : ("qsfp", ER4QSFP100G, "Arista Networks "),
   "Arista100GAR4Qsfp" : ("qsfp", AR4QSFP100G, "Arista Networks "),
   "3rdParty100G40GAR4Qsfp" : ("qsfp", AR4QSFP100G40G3RDPTY, "BoboCorp        "),
   "Arista100GAR4NQsfp" : ("qsfp", AR4NQSFP100G, "Arista Networks "),
   "3rdParty100G40GAR4NQsfp" : ("qsfp", AR4NQSFP100G40G3RDPTY, "BoboCorp        "),
   "Arista100GCR4Qsfp" : ("qsfp", CR4QSFP100G, "Arista Networks "), 
   "Arista100GCR4SQsfp" : ("qsfp", CR4SQSFP100G, "Arista Networks "), 
   "Arista100GCR4NQsfp" : ("qsfp", CR4NQSFP100G, "Arista Networks "), 
   "Arista100GCRA4Qsfp" : ("qsfp", CRA4QSFP100G, "Arista Networks "), 
   "Arista50GCR2LQsfp" : ("qsfp", CR2LQSFP50G, "Arista Networks "), 
   "Arista50GCR2SQsfp" : ("qsfp", CR2SQSFP50G, "Arista Networks "), 
   "Arista50GCR2NQsfp" : ("qsfp", CR2NQSFP50G, "Arista Networks "), 
   "3rdParty100GCR4Qsfp" : ("qsfp", CR4QSFP100G3RDPTY, "BoboCorp        "),
   "3rdParty100G40GCR4Qsfp" : ("qsfp", CR4QSFP100G40G3RDPTY, "BoboCorp        "),
   "Arista100GPSM4Qsfp" : ("qsfp", PSM4QSFP100G, "Arista Networks "), 
   "Arista100GCLR4Qsfp" : ("qsfp", CLR4QSFP100G, "Arista Networks "), 
   "Arista100GCWDM4Qsfp" : ("qsfp", CWDM4QSFP100G, "Arista Networks "), 
   "Arista100GEDWDM2Qsfp" : ("qsfp", DWDM2QSFP100G, "Arista Networks "), 
   "Arista100GSWDM4Qsfp" : ("qsfp", SWDM4QSFP100G, "Arista Networks "),
   "Arista100GSRBDQsfp" : ("qsfp", SRBDQSFP100G, "Arista Networks "),
   "Arista100GERL4Qsfp" : ("qsfp", ERL4QSFP100G, "Arista Networks "), 
   "Arista100GDRQsfp" : ("qsfp", DRQSFP100G, "Arista Networks "),
   "Arista100GFRQsfp" : ("qsfp", FRQSFP100G, "Arista Networks "),
   "Arista100GLRQsfp" : ("qsfp", LRQSFP100G, "Arista Networks "),
   "Arista100GXCWDM4Qsfp" : ("qsfp", XCWDM4QSFP100G, "Arista Networks "), 
   "AristaTxPowerQsfp" : ("qsfp", SRQSFP, "Arista Networks "),
   "Arista400g2QqsfpSide" : ( "qsfp", Qsfp400g2QqsfpSide, "Arista Networks " ),
   "Arista400g2QqsfpSide40G" : ( "qsfp", Qsfp400g2QqsfpSide40G, "Arista Networks " ),
   "Arista400g2QqsfpSideDepop" : ( "qsfp", Qsfp400g2QqsfpSideDepop,
                                   "Arista Networks " ),
   "UnknownQsfp" : ("qsfp", UNKNOWNQSFP, "Unknown Corp "),
   "Unknown100GQsfp" : ("qsfp", UNKNOWNQSFP100G, "Unknown Corp "),
   "Unknown100GQsfpOptics" : ("qsfp", UNKNOWNQSFP100GOPTICS, "Unknown Corp "),

   # Mbo types (base + individual page templates)
   # AFBR81RxyZ Avago Minipod
   "AvagoAFBRXXRxyZ" : ( "mbo", SRMBOAFBR81RXYZ, "Arista Networks " ),
   "AFBR81RxyZ_Lower" : ( "mbo", SRMBOAFBR81RXYZ, "Arista Networks " ),
   "AFBR81RxyZ_Upper0" : ( "mbo", SRMBOAFBR81RXYZ0, "Arista Networks " ),
   "AFBR81RxyZ_Upper1" : ( "mbo", SRMBOAFBR81RXYZ1, "Arista Networks " ),
   # AFBR82RxyZ Avago Minipod
   "AFBR82RxyZ_Lower" : ( "mbo", SRMBOAFBR82RXYZ, "Arista Networks " ),
   "AFBR82RxyZ_Upper0" : ( "mbo", SRMBOAFBR82RXYZ0, "Arista Networks " ),
   "AFBR82RxyZ_Upper1" : ( "mbo", SRMBOAFBR82RXYZ1, "Arista Networks " ),
   # Finisar300m Finisar AOM4
   "Finisar300m" : ( "mbo", SRMBOFINISAR300MTX, "Arista Networks " ),
   "Finisar300m_TxLower" : ( "mbo", SRMBOFINISAR300MTX, "Arista Networks " ),
   "Finisar300m_TxUpper0" : ( "mbo", SRMBOFINISAR300MTX0, "Arista Networks " ),
   "Finisar300m_TxUpper1" : ( "mbo", SRMBOFINISAR300MTX1, "Arista Networks " ),
   "Finisar300m_TxUpper2" : ( "mbo", SRMBOFINISAR300MTX2, "Arista Networks " ),
   "Finisar300m_TxUpper3" : ( "mbo", SRMBOFINISAR300MTX3, "Arista Networks " ),
   "Finisar300m_TxUpper4" : ( "mbo", SRMBOFINISAR300MTX4, "Arista Networks " ),
   "Finisar300m_RxLower" : ( "mbo", SRMBOFINISAR300MRX, "Arista Networks " ),
   "Finisar300m_RxUpper0" : ( "mbo", SRMBOFINISAR300MRX0, "Arista Networks " ),
   "Finisar300m_RxUpper1" : ( "mbo", SRMBOFINISAR300MRX1, "Arista Networks " ),
   "Finisar300m_RxUpper2" : ( "mbo", SRMBOFINISAR300MRX2, "Arista Networks " ),
   "Finisar300m_RxUpper3" : ( "mbo", SRMBOFINISAR300MRX3, "Arista Networks " ),
   "Finisar300m_RxUpper4" : ( "mbo", SRMBOFINISAR300MRX4, "Arista Networks " ),
   # FBOxD1xSL1Lxx Finisar
   "FBOxD1xSL1Lxx" : ( "mbo", SRMBOFBOXD1XSL1LXXTX, "Arista Networks " ),
   "FBOxD1xSL1Lxx_TxLower" : ( "mbo", SRMBOFBOXD1XSL1LXXTX, "Arista Networks " ),
   "FBOxD1xSL1Lxx_TxUpper0" : ( "mbo", SRMBOFBOXD1XSL1LXXTX0, "Arista Networks " ),
   "FBOxD1xSL1Lxx_TxUpper1" : ( "mbo", SRMBOFBOXD1XSL1LXXTX1, "Arista Networks " ),
   "FBOxD1xSL1Lxx_RxLower" : ( "mbo", SRMBOFBOXD1XSL1LXXRX, "Arista Networks " ),
   "FBOxD1xSL1Lxx_RxUpper0" : ( "mbo", SRMBOFBOXD1XSL1LXXRX0, "Arista Networks " ),
   "FBOxD1xSL1Lxx_RxUpper1" : ( "mbo", SRMBOFBOXD1XSL1LXXRX1, "Arista Networks " ),

   # CFP2 LR4 ( base entry + EEPROM tables )
   "AristaLR4Cfp2" : ( 'cfp2', None, "Arista Networks " ),
   "AristaLR4Cfp2_table8000" : ( 'cfp2', Cfp2MsaMis100GLR4Table8000,
                                 "Arista Networks " ),
   "AristaLR4Cfp2_table8080" : ( 'cfp2', Cfp2MsaMis100GLR4Table8080,
                                 "Arista Networks " ),
   "AristaLR4Cfp2_table8420" : ( 'cfp2', Cfp2MsaMis100GLR4Table8420,
                                 "Arista Networks " ),
   "AristaLR4Cfp2_tableA000" : ( 'cfp2', Cfp2MsaMis100GLR4TableA000,
                                 "Arista Networks " ),
   "AristaLR4Cfp2_tableA200" : ( 'cfp2', Cfp2MsaMis100GLR4TableA200,
                                 "Arista Networks " ),
   "AristaLR4Cfp2_tableA280" : ( 'cfp2', Cfp2MsaMis100GLR4TableA280,
                                 "Arista Networks " ),
   "AristaLR4Cfp2_tableA400" : ( 'cfp2', Cfp2MsaMis100GLR4TableA400,
                                 "Arista Networks " ),
   # 3rd party CFP2 LR4 ( base entry + EEprom tables )
   "BoboLR4Cfp2" : ( 'cfp2', None, "BoboCorp        " ),
   "BoboLR4Cfp2_table8000" : ( 'cfp2', Cfp2MsaMisBobo100GLR4Table8000,
                               "BoboCorp        " ),
   "BoboLR4Cfp2_table8080" : ( 'cfp2', Cfp2MsaMisBobo100GLR4Table8080,
                               "BoboCorp        " ),
   "BoboLR4Cfp2_table8420" : ( 'cfp2', Cfp2MsaMisBobo100GLR4Table8420,
                               "BoboCorp        " ),
   "BoboLR4Cfp2_tableA000" : ( 'cfp2', Cfp2MsaMisBobo100GLR4TableA000,
                               "BoboCorp        " ),
   "BoboLR4Cfp2_tableA200" : ( 'cfp2', Cfp2MsaMisBobo100GLR4TableA200,
                               "BoboCorp        " ),
   "BoboLR4Cfp2_tableA280" : ( 'cfp2', Cfp2MsaMisBobo100GLR4TableA280,
                               "BoboCorp        " ),
   "BoboLR4Cfp2_tableA400" : ( 'cfp2', Cfp2MsaMisBobo100GLR4TableA400,
                               "BoboCorp        " ),
   # CFP2 SR10 ( base entry + EEPROM tables )
   "AristaSR10Cfp2" : ( 'cfp2', None, "Arista Networks " ),
   "AristaSR10Cfp2_table8000" : ( 'cfp2', Cfp2MsaMis100GSR10Table8000,
                                  "Arista Networks " ),
   "AristaSR10Cfp2_table8080" : ( 'cfp2', Cfp2MsaMis100GSR10Table8080,
                                  "Arista Networks " ),
   "AristaSR10Cfp2_table8420" : ( 'cfp2', Cfp2MsaMis100GSR10Table8420,
                                  "Arista Networks " ),
   "AristaSR10Cfp2_tableA000" : ( 'cfp2', Cfp2MsaMis100GSR10TableA000,
                                  "Arista Networks " ),
   "AristaSR10Cfp2_tableA200" : ( 'cfp2', Cfp2MsaMis100GSR10TableA200,
                                  "Arista Networks " ),
   "AristaSR10Cfp2_tableA280" : ( 'cfp2', Cfp2MsaMis100GSR10TableA280,
                                  "Arista Networks " ),
   "AristaSR10Cfp2_tableA400" : ( 'cfp2', Cfp2MsaMis100GSR10TableA400,
                                  "Arista Networks " ),
   # CFP2 ER4 ( base entry + EEPROM tables )
   "AristaER4Cfp2" : ( 'cfp2', None, "Arista Networks " ),
   "AristaER4Cfp2_table8000" : ( 'cfp2', Cfp2MsaMis100GER4Table8000,
                                  "Arista Networks " ),
   "AristaER4Cfp2_table8080" : ( 'cfp2', Cfp2MsaMis100GER4Table8080,
                                  "Arista Networks " ),
   "AristaER4Cfp2_table8420" : ( 'cfp2', Cfp2MsaMis100GER4Table8420,
                                  "Arista Networks " ),
   "AristaER4Cfp2_tableA000" : ( 'cfp2', Cfp2MsaMis100GER4TableA000,
                                  "Arista Networks " ),
   "AristaER4Cfp2_tableA200" : ( 'cfp2', Cfp2MsaMis100GER4TableA200,
                                  "Arista Networks " ),
   "AristaER4Cfp2_tableA280" : ( 'cfp2', Cfp2MsaMis100GER4TableA280,
                                  "Arista Networks " ),
   "AristaER4Cfp2_tableA400" : ( 'cfp2', Cfp2MsaMis100GER4TableA400,
                                  "Arista Networks " ),
   # 3rd party CFPX ( base entry + EEprom tables )
   "BoboDwdmCoherentCfp2" : ( 'cfp2', None, "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_table8000" : ( 'cfp2',
                     Cfp2MsaMis100GBoboDwdmCoherentTable8000, "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_table8080" : ( 'cfp2',
                     Cfp2MsaMis100GBoboDwdmCoherentTable8080, "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_table8180" : ( 'cfp2',
                     Cfp2MsaMis100GBoboDwdmCoherentTable8180, "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_table8420" : ( 'cfp2', 
                     Cfp2MsaMis100GBoboDwdmCoherentTable8420, "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_tableB000" : ( 'cfp2',
                     Cfp2MsaMis100GBoboDwdmCoherentTableB000, "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_tableB180" : ( 'cfp2',
                     Cfp2MsaMis100GBoboDwdmCoherentTableB180, "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_tableB300" : ( 'cfp2',
                     Cfp2MsaMis100GBoboDwdmCoherentTableB300, "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_tableB400" : ( 'cfp2',
                     Cfp2MsaMis100GBoboDwdmCoherentTableB400, "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_tableB480" : ( 'cfp2',
                     Cfp2MsaMis100GBoboDwdmCoherentTableB480, "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_tableBA80" : ( 'cfp2',
                     Cfp2MsaMis100GBoboDwdmCoherentTableBA80, "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_tableBB00" : ( 'cfp2',
                     Cfp2MsaMis100GBoboDwdmCoherentTableBB00, "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_tableBB80" : ( 'cfp2',
                     Cfp2MsaMis100GBoboDwdmCoherentTableBB80, "BoboCorp        " ),
   # CFP2 Dwdm Coherent limiting module ( base entry + EEPROM tables )
   "AristaDwdmCoherentCfp2" : ( 'cfp2', None, "Arista Networks " ),
   "AristaDwdmCoherentCfp2_table8000" : ( 'cfp2',
                        Cfp2MsaMis100GDwdmCoherentTable8000, "Arista Networks " ),
   "AristaDwdmCoherentCfp2_table8080" : ( 'cfp2',
                        Cfp2MsaMis100GDwdmCoherentTable8080, "Arista Networks " ),
   "AristaDwdmCoherentCfp2_table8180" : ( 'cfp2',
                        Cfp2MsaMis100GDwdmCoherentTable8180, "Arista Networks " ),
   "AristaDwdmCoherentCfp2_table8420" : ( 'cfp2', 
                        Cfp2MsaMis100GDwdmCoherentTable8420, "AristaNetworks " ),
   "AristaDwdmCoherentCfp2_tableB000" : ( 'cfp2',
                        Cfp2MsaMis100GDwdmCoherentTableB000, "Arista Networks " ),
   "AristaDwdmCoherentCfp2_tableB180" : ( 'cfp2',
                        Cfp2MsaMis100GDwdmCoherentTableB180, "Arista Networks " ),
   "AristaDwdmCoherentCfp2_tableB300" : ( 'cfp2',
                        Cfp2MsaMis100GDwdmCoherentTableB300, "Arista Networks " ),
   "AristaDwdmCoherentCfp2_tableB400" : ( 'cfp2',
                        Cfp2MsaMis100GDwdmCoherentTableB400, "Arista Networks " ),
   "AristaDwdmCoherentCfp2_tableB480" : ( 'cfp2',
                        Cfp2MsaMis100GDwdmCoherentTableB480, "Arista Networks " ),
   "AristaDwdmCoherentCfp2_tableBA80" : ( 'cfp2',
                        Cfp2MsaMis100GDwdmCoherentTableBA80, "Arista Networks " ),
   "AristaDwdmCoherentCfp2_tableBB00" : ( 'cfp2',
                        Cfp2MsaMis100GDwdmCoherentTableBB00, "Arista Networks " ),
   "AristaDwdmCoherentCfp2_tableBB80" : ( 'cfp2',
                        Cfp2MsaMis100GDwdmCoherentTableBB80, "Arista Networks " ),
   # CFP2 Dwdm Coherent linear module ( base entry + EEPROM tables ) Oclaro
   "AristaDwdmCoherentECfp2" : ( 'cfp2', None, "Arista Networks " ),
   "AristaDwdmCoherentECfp2_table8180" : ( 'cfp2',
                        Cfp2MsaMis100GDwdmCoherentETable8180, "Arista Networks " ),
   "AristaDwdmCoherentECfp2_tableBB00" : ( 'cfp2',
                        Cfp2MsaMis100GDwdmCoherentETableBB00, "Arista Networks " ),
   # CFP2 Dwdm Coherent linear module V2 ( base entry + EEPROM tables ) Acacia
   "AristaDwdmCoherentECfp2V2" : ( 'cfp2', None, "Arista Networks " ),
   "AristaDwdmCoherentECfp2V2_table8000" : ( 'cfp2',
                        Cfp2MsaMis100GDwdmCoherentEV2Table8000, "Arista Networks " ),
   "OclaroDwdmCoherentECfp2" : ( 'cfp2', None, "OCLARO          " ),
   "OclaroDwdmCoherentECfp2_table8000" : ( 'cfp2',
               Cfp2MsaMis100GOclaroDwdmCoherentETable8000     , "OCLARO          " ),
   "AcaciaDwdmCoherentECfp2" : ( 'cfp2', None, "Acacia Comm Inc." ),
   "AcaciaDwdmCoherentECfp2_table8000" : ( 'cfp2',
               Cfp2MsaMis100GAcaciaDwdmCoherentETable8000     , "Acacia Comm Inc." ),
   # CFP2 DCO module ( base entry + EEPROM tables )
   "AristaCfp2DwdmDco" : ( 'cfp2', None, "Arista Networks " ),
   "AristaCfp2DwdmDco_table8000" : ( 'cfp2',
                        Cfp2MsaMisCfp2DwdmDcoTable8000, "Arista Networks " ),
   "AristaCfp2DwdmDco_table8080" : ( 'cfp2',
                        Cfp2MsaMisCfp2DwdmDcoTable8080, "Arista Networks " ),
   "AristaCfp2DwdmDco_table8180" : ( 'cfp2',
                        Cfp2MsaMisCfp2DwdmDcoTable8180, "Arista Networks " ),
   "AristaCfp2DwdmDco_tableB000" : ( 'cfp2',
                        Cfp2MsaMisCfp2DwdmDcoTableB000, "Arista Networks " ),
   "AristaCfp2DwdmDco_tableB480" : ( 'cfp2',
                        Cfp2MsaMisCfp2DwdmDcoTableB480, "Arista Networks " ),
   # CFP2 DCO module 3rd party ( base entry + EEPROM tables ) Acacia
   "AcaciaCfp2DwdmDco" : ( 'cfp2', None, "Acacia Comm Inc " ),
   "AcaciaCfp2DwdmDco_table8000" : ( 'cfp2',
                        Cfp2MsaMisCfp2DwdmDco3rdPartyTable8000, "Acacia Comm Inc " ),
   "AcaciaCfp2DwdmDco_table8080" : ( 'cfp2',
                        Cfp2MsaMisCfp2DwdmDco3rdPartyTable8080, "Acacia Comm Inc " ),
   "AcaciaCfp2DwdmDco_table8180" : ( 'cfp2',
                        Cfp2MsaMisCfp2DwdmDco3rdPartyTable8180, "Acacia Comm Inc " ),
   "AcaciaCfp2DwdmDco_tableB000" : ( 'cfp2',
                        Cfp2MsaMisCfp2DwdmDco3rdPartyTableB000, "Acacia Comm Inc " ),
   "AcaciaCfp2DwdmDco_tableB480" : ( 'cfp2',
                        Cfp2MsaMisCfp2DwdmDco3rdPartyTableB480, "Acacia Comm Inc " ),

   # CFP2 XSR10 ( base entry + EEPROM tables )
   "AristaXSR10Cfp2" : ( 'cfp2', None, "Arista Networks " ),
   "AristaXSR10Cfp2_table8000" : ( 'cfp2', Cfp2MsaMis100GXSR10Table8000,
                                  "Arista Networks " ),
   "AristaXSR10Cfp2_table8080" : ( 'cfp2', Cfp2MsaMis100GXSR10Table8080,
                                  "Arista Networks " ),
   "AristaXSR10Cfp2_table8420" : ( 'cfp2', Cfp2MsaMis100GXSR10Table8420,
                                  "Arista Networks " ),
   "AristaXSR10Cfp2_tableA000" : ( 'cfp2', Cfp2MsaMis100GXSR10TableA000,
                                  "Arista Networks " ),
   "AristaXSR10Cfp2_tableA200" : ( 'cfp2', Cfp2MsaMis100GXSR10TableA200,
                                  "Arista Networks " ),
   "AristaXSR10Cfp2_tableA280" : ( 'cfp2', Cfp2MsaMis100GXSR10TableA280,
                                  "Arista Networks " ),
   "AristaXSR10Cfp2_tableA400" : ( 'cfp2', Cfp2MsaMis100GXSR10TableA400,
                                  "Arista Networks " ),

   # OSFP 400G-CR8 ( base entry + Id EEPROM pages lowerPage00 and upperPage00 )
   "AristaOsfp400gCr8" : ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gCr8Lower00" : ( 'osfp', Osfp400gCr8LowerPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gCr8Upper00" : ( 'osfp', Osfp400gCr8UpperPage00,
                                  "Arista Networks " ),
   "AristaCmis4Osfp400gCr8" : ( 'osfp', None, "Arista Networks " ),
   "AristaCmis4Osfp400gCr8Lower00" : ( 'osfp', Cmis4Osfp400gCr8LowerPage00,
                                  "Arista Networks " ),
   "AristaCmis4Osfp400gCr8Upper00" : ( 'osfp', Cmis4Osfp400gCr8UpperPage00,
                                  "Arista Networks " ),
   "VeridianOsfp400gCr8" : ( 'osfp', None, "VeridianDynamics" ),
   "AristaQsfpDd400gCr8" : ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd400gCr8Lower00" : ( 'qsfpDd', QsfpDd400gCr8LowerPage00,
                                    "Arista Networks " ),
   "AristaQsfpDd400gCr8Upper00" : ( 'qsfpDd', QsfpDd400gCr8UpperPage00,
                                    "Arista Networks " ),
   "VeridianQsfpDd400gCr8" : ( 'qsfpDd', None, "VeridianDynamics" ),
   "AristaQsfpCmis200gCr4" : ( 'qsfpCmis', None, "Arista Networks " ),
   "AristaQsfpCmis200gCr4Lower00" : ( 'qsfpCmis', QsfpCmis200gCr4LowerPage00,
                                    "Arista Networks " ),
   "AristaQsfpCmis200gCr4Upper00" : ( 'qsfpCmis', QsfpCmis200gCr4UpperPage00,
                                    "Arista Networks " ),
   "VeridianQsfpCmis200gCr4" : ( 'qsfpCmis', None, "VeridianDynamics" ),
   "AristaOsfp400gSr8" : ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gSr8Lower00" : ( 'osfp', Osfp400gSr8LowerPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gSr8Upper00" : ( 'osfp', Osfp400gSr8UpperPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gSr8Upper01" : ( 'osfp', Osfp400gSr8UpperPage01,
                                  "Arista Networks " ),
   "AristaOsfp400gSr8Upper02" : ( 'osfp', Osfp400gSr8UpperPage02,
                                  "Arista Networks " ),
   "AristaOsfp400gSr8Upper10Bank0" : ( 'osfp', Osfp400gSr8UpperPage10Bank0,
                                       "Arista Networks " ),
   "AristaOsfp400gSr8Upper11Bank0" : ( 'osfp', Osfp400gSr8UpperPage11Bank0,
                                       "Arista Networks " ),
   "AristaOsfp400gSr8Cmis3" : ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gSr8Cmis3Lower00" : ( 'osfp', Osfp400gSr8Cmis3LowerPage00,
                                       "Arista Networks " ),
   # Dr4
   "AristaOsfp400gDr4" : ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gDr4Lower00" : ( 'osfp', Osfp400gDr4LowerPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gDr4Upper00" : ( 'osfp', Osfp400gDr4UpperPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gDr4Upper01" : ( 'osfp', Osfp400gDr4UpperPage01,
                                  "Arista Networks " ),
   "AristaOsfp400gDr4Upper02" : ( 'osfp', Osfp400gDr4UpperPage02,
                                  "Arista Networks " ),
   "AristaOsfp400gDr4Upper10Bank0" : ( 'osfp', Osfp400gDr4UpperPage10Bank0,
                                       "Arista Networks " ),
   "AristaOsfp400gDr4Upper11Bank0" : ( 'osfp', Osfp400gDr4UpperPage11Bank0,
                                       "Arista Networks " ),
   "AristaOsfp400gXdr4" : ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gXdr4Upper01" : ( 'osfp', Osfp400gXdr4UpperPage01,
                                   "Arista Networks " ),
    # Fr4
   "AristaOsfp400gFr4" : ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gFr4Lower00" : ( 'osfp', Osfp400gFr4LowerPage00,
                                  "Arista Networks " ),
   # 2Fr4
   "AristaOsfp400g2Fr4" : ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400g2Fr4Lower00" : ( 'osfp', Osfp400g2Fr4LowerPage00,
                                    "Arista Networks " ),
   "AristaOsfp400g2Fr4Upper00" : ( 'osfp', Osfp400g2Fr4UpperPage00,
                                    "Arista Networks " ),
   "AristaOsfp400g2Fr4Upper01" : ( 'osfp', Osfp400g2Fr4UpperPage01,
                                    "Arista Networks " ),
   "AristaOsfp400g2Fr4Upper02" : ( 'osfp', Osfp400g2Fr4UpperPage02,
                                    "Arista Networks " ),
   "AristaOsfp400g2Fr4Upper10Bank0" : ( 'osfp', Osfp400g2Fr4UpperPage10Bank0,
                                    "Arista Networks " ),
   "AristaOsfp400g2Fr4Upper11Bank0" : ( 'osfp', Osfp400g2Fr4UpperPage11Bank0,
                                    "Arista Networks " ),
   "VeridianOsfp400gSr8" : ( 'osfp', None, "VeridianDynamics" ),
   # OsfpAr8 borrows Sr8's 10h and 11h pages
   "AristaOsfp400gAr8" : ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gAr8Lower00" : ( 'osfp', Osfp400gAr8LowerPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gAr8Upper00" : ( 'osfp', Osfp400gAr8UpperPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gAr8Upper01" : ( 'osfp', Osfp400gAr8UpperPage01,
                                  "Arista Networks " ),
   "AristaOsfp400gAr8Upper02" : ( 'osfp', Osfp400gAr8UpperPage02,
                                  "Arista Networks " ),
   "AristaOsfp400gAr8Upper10Bank0" : ( 'osfp', Osfp400gSr8UpperPage10Bank0,
                                       "Arista Networks " ),
   "AristaOsfp400gAr8Upper11Bank0" : ( 'osfp', Osfp400gSr8UpperPage11Bank0,
                                       "Arista Networks " ),
   "VeridianOsfp400gAr8" : ( 'osfp', None, "VeridianDynamics" ),
   "AristaOsfp400gCra8" : ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gCra8Upper00" : ( 'osfp', Osfp400gCra8UpperPage00,
                                  "Arista Networks " ),
   "AristaOsfpAmpZr" : ( 'osfp', None, "Arista Networks " ),
   "AristaOsfpAmpZrLower00" : ( 'osfp', OsfpAmpZrLowerPage00, "Arista Networks " ),
   "AristaOsfpAmpZrUpper00" : ( 'osfp', OsfpAmpZrUpperPage00, "Arista Networks " ),
   "AristaOsfpAmpZrUpper02" : ( 'osfp', OsfpAmpZrUpperPage02, "Arista Networks " ),
   "AristaOsfpAmpZrUpper11Bank0" : ( 'osfp', OsfpAmpZrUpperPage11Bank0,
                                     "Arista Networks " ),
   "AristaOsfpAmpZrUpperC0" : ( 'osfp', OsfpAmpZrUpperPageC0, "Arista Networks " ),
   "AristaOsfpAmpZrUpperC1" : ( 'osfp', OsfpAmpZrUpperPageC1, "Arista Networks " ),
   "AristaOsfpAmpZrUpperD0" : ( 'osfp', OsfpAmpZrUpperPageD0, "Arista Networks " ),
   "AristaOsfpAmpZrUpperD1" : ( 'osfp', OsfpAmpZrUpperPageD1, "Arista Networks " ),
   "AristaOsfpAmpZrUpperD8" : ( 'osfp', OsfpAmpZrUpperPageD8, "Arista Networks " ),
   "AristaOsfpAmpZrUpperD9" : ( 'osfp', OsfpAmpZrUpperPageD9, "Arista Networks " ),
   'AristaQsfpDd400gZr' : ( 'qsfpDd', None, "Arista Networks " ),
   'AcaciaQsfpDd400gZr' : ( 'qsfpDd', None, "Acacia Networks " ),
   'AristaQsfpDd400gZrLower00' : ( 'qsfpDd', QsfpDd400gZrLowerPage00,
                                   "Arista Networks " ),
   'AristaQsfpDd400gZrUpper00' : ( 'qsfpDd', QsfpDd400gZrUpperPage00,
                                   "Arista Networks " ),
   'AristaQsfpDd400gZrUpper01' : ( 'qsfpDd', QsfpDd400gZrUpperPage01,
                                   "Arista Networks " ),
   'AristaQsfpDd400gZrUpper04' : ( 'qsfpDd', QsfpDd400gZrUpperPage04,
                                   "Arista Networks " ),
   "AristaQsfpDd400gZrUpper12Bank0" : ( 'qsfpDd', QsfpDd400gZrUpperPage12Bank0,
                                        "Arista Networks " ),
   "AristaQsfpDd400gZrUpper20Bank0" : ( 'qsfpDd', QsfpDd400gZrUpperPage20Bank0,
                                        "Arista Networks " ),
   "AristaQsfpDd400gZrUpper28Bank0" : ( 'qsfpDd', QsfpDd400gZrUpperPage28Bank0,
                                        "Arista Networks " ),
   "AristaQsfpDd400gZrUpper2FBank0" : ( 'qsfpDd', QsfpDd400gZrUpperPage2FBank0,
                                        "Arista Networks " ),

   'AristaOsfp400gZr' : ( 'osfp', None, "Arista Networks " ),
   'AristaOsfp400gZrLower00' : ( 'osfp', Osfp400gZrLowerPage00,
                                   "Arista Networks " ),
   'AristaOsfp400gZrUpper00' : ( 'osfp', Osfp400gZrUpperPage00,
                                   "Arista Networks " ),

   "GoogleOM8FNT" : ( 'osfp', None, "Google" ),
   "GoogleOM8FNTUpper00" : ( 'osfp', GoogleOM8FNTUpperPage00, "Google" ),
   "AristaQsfpDd400gSr8" : ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd400gSr8Lower00" : ( 'qsfpDd', QsfpDd400gSr8LowerPage00,
                                  "Arista Networks " ),
   "AristaQsfpDd400gSr8Upper00" : ( 'qsfpDd', QsfpDd400gSr8UpperPage00,
                                  "Arista Networks " ),
   "AristaQsfpCmis200gFr4" : ( 'qsfpCmis', None, "Arista Networks " ),
   "AristaQsfpCmis200gFr4Lower00" : ( 'qsfpCmis', QsfpCmis200gFr4LowerPage00,
                                       "Arista Networks " ),
   "AristaQsfpCmis200gFr4Upper00" : ( 'qsfpCmis', QsfpCmis200gFr4UpperPage00,
                                       "Arista Networks " ),
   "AristaQsfpCmis200gFr4Upper01" : ( 'qsfpCmis', QsfpCmis200gFr4UpperPage01,
                                       "Arista Networks " ),
   "AristaQsfpCmis200gFr4Upper02" : ( 'qsfpCmis', QsfpCmis200gFr4UpperPage02,
                                       "Arista Networks " ),
   "AristaQsfpCmis200gFr4Upper10" : ( 'qsfpCmis', QsfpCmis200gFr4UpperPage10,
                                       "Arista Networks " ),
   "AristaQsfpCmis200gFr4Upper11" : ( 'qsfpCmis', QsfpCmis200gFr4UpperPage11,
                                       "Arista Networks " ),
   "AristaQsfpDd400gSr8Cmis3" : ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd400gSr8Cmis3Lower00" : ( 'qsfpDd', QsfpDd400gSr8Cmis3LowerPage00,
                                         "Arista Networks " ),
   }

qsfpEepromRevisions = {
   "AristaSR4Qsfp" : 0,
   "AristaSR4Qsfp1002" : 0,
   "AristaSR4Qsfp1003" : 0,
   "AristaSR4Qsfp1004" : 0,
   "AristaSR4QsfpNSRB1003" : 0,
   "AristaSR4DQsfp" : 0,
   "AristaSRBDQsfp" : 0,
   "3rdPartySRBDQsfp" : 0,
   "AristaXSR4Qsfp" : 0,
   "BoboSR4Qsfp" : 0,
   "AristaAR4Qsfp" : 0,
   "AristaCR4Qsfp" : 0, 
   "3rdPartyCR4Qsfp" : 0, 
   "AristaLR4Qsfp" : 0,
   "AristaLR4Qsfp1002" : 0,
   "AristaLRL4Qsfp" : 0,
   "AristaPLR4Qsfp" : 0,
   "AristaPLRL4Qsfp" : 0,
   "3rdPartyPLR4Qsfp" : 0,
   "Psm4Qsfp" : 0,
   "AristaUNIVQsfp" : sff8636rev15,
   "Arista100GSR4Qsfp" : sff8636rev15,
   "Arista100GXSR4Qsfp" : sff8636rev15,
   "3rdParty100G40GSR4Qsfp" : sff8636rev15,
   "Arista100GLR4Qsfp" : sff8636rev15,
   "Arista100GLRL4Qsfp" : sff8636rev15,
   "3rdParty100G40GLR4Qsfp" : sff8636rev20,
   "Arista100GER4Qsfp" : sff8636rev15,
   "AristaER4Qsfp" : sff8636rev15,
   "Arista100GAR4Qsfp" : sff8636rev15,
   "Arista100GAR4NQsfp" : sff8636rev15,
   "3rdParty100G40GAR4Qsfp" : sff8636rev15,
   "3rdParty100G40GAR4NQsfp" : sff8636rev15,
   "Arista100GCR4Qsfp" : sff8636rev15,
   "Arista100GCR4SQsfp" : sff8636rev15,
   "Arista100GCR4NQsfp" : sff8636rev15,
   "Arista100GCRA4Qsfp" : sff8636rev15,
   "Arista50GCR2LQsfp" : sff8636rev15,
   "Arista50GCR2SQsfp" : sff8636rev15,
   "Arista50GCR2NQsfp" : sff8636rev15,
   "3rdParty100GCR4Qsfp" : sff8636rev15,
   "3rdParty100G40GCR4Qsfp" : sff8636rev15,
   "Arista100GPSM4Qsfp" : sff8636rev15,
   "Arista100GCLR4Qsfp" : sff8636rev15,
   "Arista100GCWDM4Qsfp" : sff8636rev15,
   "Arista100GEDWDM2Qsfp" : sff8636rev15,
   "Arista100GSWDM4Qsfp" : sff8636rev15,
   "Arista100GSRBDQsfp" : sff8636rev15,
   "Arista100GERL4Qsfp" : sff8636rev15,
   "Arista100GDRQsfp" : sff8636rev15,
   "Arista100GFRQsfp" : sff8636rev15,
   "Arista100GLRQsfp" : sff8636rev15,
   "Arista100GXCWDM4Qsfp" : sff8636rev15,
   "AristaTxPowerQsfp" : 0,
   "Arista400g2QqsfpSide" : sff8636rev15,
   "Arista400g2QqsfpSide40G" : sff8636rev15,
   "Arista400g2QqsfpSideDepop" : sff8636rev15,
   "UnknownQsfp" : 0,
   "Unknown100GQsfp" : 0,
   "Unknown100GQsfpOptics" : 0,
   }

def getXcvrType( mediaType ):
   ( xcvrType, _templBytes, _vendorName ) = eepromTemplates[ mediaType ]
   return xcvrType

def getVendorName( mediaType ):
   ( _xcvrType, _templBytes, vendorName ) = eepromTemplates[ mediaType ]
   return vendorName

def getBytes( mediaType ):
   ( _xcvrType, templBytes, _vendorName ) = eepromTemplates[ mediaType ]
   return templBytes

def convertBytesToBinStr( bytez ):
   s = ''.join( chr( b ) for b in bytez )
   return s

def zeroOutStrEntries( byteList ):
   bytez = []
   for byte in byteList:
      i = byte
      if isinstance( i, str ):
         i = 0
      bytez.append( i )
   return bytez

def convertSff8472EepromTemplateToBytes( mediaType ):
   idEeprom = Tac.Value( "Xcvr::XcvrSff8472IdEepromContents" )
   templBytes = getBytes( mediaType )
   assert len( templBytes ) == 128, ( 'EEPROM template %s has %d '
          'bytes instead of the expected 128' % ( mediaType, len( templBytes ) ) )
   bytez = zeroOutStrEntries( templBytes )
   idEeprom.buf = convertBytesToBinStr( bytez )
   vendorName = getVendorName( mediaType )
   if vendorName:
      idEeprom.vendorName = vendorName

   return idEeprom

def convertSff8436EepromTemplateToBytes( mediaType ):
   idEeprom = Tac.Value( "Xcvr::Sff8436IdEepromContents" )
   templBytes = getBytes( mediaType )
   assert len( templBytes ) == 128, ( 'EEPROM template %s has %d '
          'bytes instead of the expected 128' % ( mediaType, len( templBytes ) ) )
   bytez = zeroOutStrEntries( templBytes )
   idEeprom.eepromPage00UpperData = convertBytesToBinStr( bytez )

   return idEeprom

def pages( mediaType ):
   pageId = Tac.Value( "Xcvr::MboPageId" )
   if mediaType == "AvagoAFBRXXRxyZ":
      pagez = [ ( 0x50, pageId.pageNotInUse, 'AFBR81RxyZ_Lower' ),
                ( 0x50, 0x00, 'AFBR81RxyZ_Upper0' ),
                ( 0x50, 0x01, 'AFBR81RxyZ_Upper1' ),
                ( 0x60, pageId.pageNotInUse, 'AFBR82RxyZ_Lower' ),
                ( 0x60, 0x00, 'AFBR82RxyZ_Upper0' ),
                ( 0x60, 0x01, 'AFBR82RxyZ_Upper1' ) ]
   elif mediaType == "Finisar300m":
      pagez = [ ( 0x50, pageId.pageNotInUse, 'Finisar300m_TxLower' ),
                ( 0x50, 0x00, 'Finisar300m_TxUpper0' ),
                ( 0x50, 0x01, 'Finisar300m_TxUpper1' ),
                ( 0x50, 0x02, 'Finisar300m_TxUpper2' ),
                ( 0x50, 0x03, 'Finisar300m_TxUpper3' ),
                ( 0x50, 0x04, 'Finisar300m_TxUpper4' ),
                ( 0x54, pageId.pageNotInUse, 'Finisar300m_RxLower' ),
                ( 0x54, 0x00, 'Finisar300m_RxUpper0' ),
                ( 0x54, 0x01, 'Finisar300m_RxUpper1' ),
                ( 0x54, 0x02, 'Finisar300m_RxUpper2' ),
                ( 0x54, 0x03, 'Finisar300m_RxUpper3' ),
                ( 0x54, 0x04, 'Finisar300m_RxUpper4' ) ]
   elif mediaType == "FBOxD1xSL1Lxx":
      pagez = [ ( 0x50, pageId.pageNotInUse, 'FBOxD1xSL1Lxx_TxLower' ),
                ( 0x50, 0x00, 'FBOxD1xSL1Lxx_TxUpper0' ),
                ( 0x50, 0x01, 'FBOxD1xSL1Lxx_TxUpper1' ),
                ( 0x54, pageId.pageNotInUse, 'FBOxD1xSL1Lxx_RxLower' ),
                ( 0x54, 0x00, 'FBOxD1xSL1Lxx_RxUpper0' ),
                ( 0x54, 0x01, 'FBOxD1xSL1Lxx_RxUpper1' ) ]
   else:
      pagez = []
   return pagez

def convertMboEepromTemplateToBytes( mediaType ):
   if mediaType == "AvagoAFBRXXRxyZ":
      idEeprom = Tac.newInstance( "Xcvr::AvagoAFBRXXRxyZ", mediaType, 0x50, 0x60 )
   elif mediaType == "Finisar300m":
      idEeprom = Tac.newInstance( "Xcvr::Finisar300m", mediaType, 0x50, 0x54 )
   elif mediaType == "FBOxD1xSL1Lxx":
      idEeprom = Tac.newInstance( "Xcvr::FBOxD1xSL1Lxx", mediaType, 0x50, 0x54 )
   # Call setup function to initialize MboEepromContents object
   idEeprom.setup()
   pageId = Tac.Value( "Xcvr::MboPageId" )
   for ( smbusAddr, upper, pageStr ) in pages( mediaType ):
      templBytes = getBytes( pageStr )
      bytez = zeroOutStrEntries( templBytes )
      pageId.smbusAddr = smbusAddr
      pageId.upper = upper
      idEeprom.memory[ pageId ].contents = convertBytesToBinStr( bytez )

   return idEeprom

def cfp2Tables( mediaType ):
   if mediaType == 'AristaLR4Cfp2':
      return [ ( 'table8000', 0x8000, 8, 'AristaLR4Cfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaLR4Cfp2_table8080' ),
               ( 'table8420', 0x8420, 8, 'AristaLR4Cfp2_table8420' ),
               ( 'tableA000', 0xA000, 16, 'AristaLR4Cfp2_tableA000' ),
               ( 'tableA200', 0xA200, 16, 'AristaLR4Cfp2_tableA200' ),
               ( 'tableA280', 0xA280, 16, 'AristaLR4Cfp2_tableA280' ),
               ( 'tableA400', 0xA400, 16, 'AristaLR4Cfp2_tableA400' ),
             ]
   elif mediaType == 'BoboLR4Cfp2':
      return [ ( 'table8000', 0x8000, 8, 'BoboLR4Cfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'BoboLR4Cfp2_table8080' ),
               ( 'table8420', 0x8420, 8, 'BoboLR4Cfp2_table8420' ),
               ( 'tableA000', 0xA000, 16, 'BoboLR4Cfp2_tableA000' ),
               ( 'tableA200', 0xA200, 16, 'BoboLR4Cfp2_tableA200' ),
               ( 'tableA280', 0xA280, 16, 'BoboLR4Cfp2_tableA280' ),
               ( 'tableA400', 0xA400, 16, 'BoboLR4Cfp2_tableA400' ),
             ]
   elif mediaType == 'AristaSR10Cfp2':
      return [ ( 'table8000', 0x8000, 8, 'AristaSR10Cfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaSR10Cfp2_table8080' ),
               ( 'table8420', 0x8420, 8, 'AristaSR10Cfp2_table8420' ),
               ( 'tableA000', 0xA000, 16, 'AristaSR10Cfp2_tableA000' ),
               ( 'tableA200', 0xA200, 16, 'AristaSR10Cfp2_tableA200' ),
               ( 'tableA280', 0xA280, 16, 'AristaSR10Cfp2_tableA280' ),
               ( 'tableA400', 0xA400, 16, 'AristaSR10Cfp2_tableA400' ),
             ]
   elif mediaType == 'AristaXSR10Cfp2':
      return [ ( 'table8000', 0x8000, 8, 'AristaXSR10Cfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaXSR10Cfp2_table8080' ),
               ( 'table8420', 0x8420, 8, 'AristaXSR10Cfp2_table8420' ),
               ( 'tableA000', 0xA000, 16, 'AristaXSR10Cfp2_tableA000' ),
               ( 'tableA200', 0xA200, 16, 'AristaXSR10Cfp2_tableA200' ),
               ( 'tableA280', 0xA280, 16, 'AristaXSR10Cfp2_tableA280' ),
               ( 'tableA400', 0xA400, 16, 'AristaXSR10Cfp2_tableA400' ),
             ]
   elif mediaType == 'AristaER4Cfp2':
      return [ ( 'table8000', 0x8000, 8, 'AristaER4Cfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaER4Cfp2_table8080' ),
               ( 'table8420', 0x8420, 8, 'AristaER4Cfp2_table8420' ),
               ( 'tableA000', 0xA000, 16, 'AristaER4Cfp2_tableA000' ),
               ( 'tableA200', 0xA200, 16, 'AristaER4Cfp2_tableA200' ),
               ( 'tableA280', 0xA280, 16, 'AristaER4Cfp2_tableA280' ),
               ( 'tableA400', 0xA400, 16, 'AristaER4Cfp2_tableA400' ),
             ]
   elif mediaType == 'BoboDwdmCoherentCfp2':
      return [ ( 'table8000', 0x8000, 8, 'BoboDwdmCoherentCfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'BoboDwdmCoherentCfp2_table8080' ),
               ( 'table8180', 0x8180, 8, 'BoboDwdmCoherentCfp2_table8180' ),
               ( 'table8420', 0x8420, 8, 'BoboDwdmCoherentCfp2_table8420' ),
               ( 'tableB000', 0xB000, 16, 'BoboDwdmCoherentCfp2_tableB000' ),
               ( 'tableB180', 0xB180, 16, 'BoboDwdmCoherentCfp2_tableB180' ),
               ( 'tableB300', 0xB300, 16, 'BoboDwdmCoherentCfp2_tableB300' ),
               ( 'tableB400', 0xB400, 16, 'BoboDwdmCoherentCfp2_tableB400' ),
               ( 'tableB480', 0xB480, 16, 'BoboDwdmCoherentCfp2_tableB480' ),
               ( 'tableBA80', 0xBA80, 16, 'BoboDwdmCoherentCfp2_tableBA80' ),
               ( 'tableBB00', 0xBB00, 16, 'BoboDwdmCoherentCfp2_tableBB00' ),
               ( 'tableBB80', 0xBB80, 16, 'BoboDwdmCoherentCfp2_tableBB80' ),
             ]
   elif mediaType == 'AristaDwdmCoherentCfp2':
      return [ ( 'table8000', 0x8000, 8, 'AristaDwdmCoherentCfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaDwdmCoherentCfp2_table8080' ),
               ( 'table8180', 0x8180, 8, 'AristaDwdmCoherentCfp2_table8180' ),
               ( 'table8420', 0x8420, 8, 'AristaDwdmCoherentCfp2_table8420' ),
               ( 'tableB000', 0xB000, 16, 'AristaDwdmCoherentCfp2_tableB000' ),
               ( 'tableB180', 0xB180, 16, 'AristaDwdmCoherentCfp2_tableB180' ),
               ( 'tableB300', 0xB300, 16, 'AristaDwdmCoherentCfp2_tableB300' ),
               ( 'tableB400', 0xB400, 16, 'AristaDwdmCoherentCfp2_tableB400' ),
               ( 'tableB480', 0xB480, 16, 'AristaDwdmCoherentCfp2_tableB480' ),
               ( 'tableBA80', 0xBA80, 16, 'AristaDwdmCoherentCfp2_tableBA80' ),
               ( 'tableBB00', 0xBB00, 16, 'AristaDwdmCoherentCfp2_tableBB00' ),
               ( 'tableBB80', 0xBB80, 16, 'AristaDwdmCoherentCfp2_tableBB80' ),
             ]
   elif mediaType == 'AristaDwdmCoherentECfp2': # Oclaro
      return [ ( 'table8000', 0x8000, 8, 'AristaDwdmCoherentCfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaDwdmCoherentCfp2_table8080' ),
               ( 'table8180', 0x8180, 8, 'AristaDwdmCoherentCfp2_table8180' ),
               ( 'table8420', 0x8420, 8, 'AristaDwdmCoherentCfp2_table8420' ),
               ( 'tableB000', 0xB000, 16, 'AristaDwdmCoherentCfp2_tableB000' ),
               ( 'tableB180', 0xB180, 16, 'AristaDwdmCoherentECfp2_table8180' ),
               ( 'tableB300', 0xB300, 16, 'AristaDwdmCoherentCfp2_tableB300' ),
               ( 'tableB400', 0xB400, 16, 'AristaDwdmCoherentCfp2_tableB400' ),
               ( 'tableB480', 0xB480, 16, 'AristaDwdmCoherentCfp2_tableB480' ),
               ( 'tableBA80', 0xBA80, 16, 'AristaDwdmCoherentCfp2_tableBA80' ),
               ( 'tableBB00', 0xBB00, 16, 'AristaDwdmCoherentECfp2_tableBB00' ),
               ( 'tableBB80', 0xBB80, 16, 'AristaDwdmCoherentCfp2_tableBB80' ),
             ]
   elif mediaType == 'AristaDwdmCoherentECfp2V2': # Acacia
      return [ ( 'table8000', 0x8000, 8, 'AristaDwdmCoherentECfp2V2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaDwdmCoherentCfp2_table8080' ),
               ( 'table8180', 0x8180, 8, 'AristaDwdmCoherentCfp2_table8180' ),
               ( 'table8420', 0x8420, 8, 'AristaDwdmCoherentCfp2_table8420' ),
               ( 'tableB000', 0xB000, 16, 'AristaDwdmCoherentCfp2_tableB000' ),
               ( 'tableB180', 0xB180, 16, 'AristaDwdmCoherentECfp2_table8180' ),
               ( 'tableB300', 0xB300, 16, 'AristaDwdmCoherentCfp2_tableB300' ),
               ( 'tableB400', 0xB400, 16, 'AristaDwdmCoherentCfp2_tableB400' ),
               ( 'tableB480', 0xB480, 16, 'AristaDwdmCoherentCfp2_tableB480' ),
               ( 'tableBA80', 0xBA80, 16, 'AristaDwdmCoherentCfp2_tableBA80' ),
               ( 'tableBB00', 0xBB00, 16, 'AristaDwdmCoherentECfp2_tableBB00' ),
               ( 'tableBB80', 0xBB80, 16, 'AristaDwdmCoherentCfp2_tableBB80' ),
             ]
   elif mediaType == 'OclaroDwdmCoherentECfp2':
      return [ ( 'table8000', 0x8000, 8, 'OclaroDwdmCoherentECfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaDwdmCoherentCfp2_table8080' ),
               ( 'table8180', 0x8180, 8, 'AristaDwdmCoherentCfp2_table8180' ),
               ( 'table8420', 0x8420, 8, 'AristaDwdmCoherentCfp2_table8420' ),
               ( 'tableB000', 0xB000, 16, 'AristaDwdmCoherentCfp2_tableB000' ),
               ( 'tableB180', 0xB180, 16, 'AristaDwdmCoherentECfp2_table8180' ),
               ( 'tableB300', 0xB300, 16, 'AristaDwdmCoherentCfp2_tableB300' ),
               ( 'tableB400', 0xB400, 16, 'AristaDwdmCoherentCfp2_tableB400' ),
               ( 'tableB480', 0xB480, 16, 'AristaDwdmCoherentCfp2_tableB480' ),
               ( 'tableBA80', 0xBA80, 16, 'AristaDwdmCoherentCfp2_tableBA80' ),
               ( 'tableBB00', 0xBB00, 16, 'AristaDwdmCoherentECfp2_tableBB00' ),
               ( 'tableBB80', 0xBB80, 16, 'AristaDwdmCoherentCfp2_tableBB80' ),
             ]
   elif mediaType == 'AcaciaDwdmCoherentECfp2':
      return [ ( 'table8000', 0x8000, 8, 'AcaciaDwdmCoherentECfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaDwdmCoherentCfp2_table8080' ),
               ( 'table8180', 0x8180, 8, 'AristaDwdmCoherentCfp2_table8180' ),
               ( 'table8420', 0x8420, 8, 'AristaDwdmCoherentCfp2_table8420' ),
               ( 'tableB000', 0xB000, 16, 'AristaDwdmCoherentCfp2_tableB000' ),
               ( 'tableB180', 0xB180, 16, 'AristaDwdmCoherentECfp2_table8180' ),
               ( 'tableB300', 0xB300, 16, 'AristaDwdmCoherentCfp2_tableB300' ),
               ( 'tableB400', 0xB400, 16, 'AristaDwdmCoherentCfp2_tableB400' ),
               ( 'tableB480', 0xB480, 16, 'AristaDwdmCoherentCfp2_tableB480' ),
               ( 'tableBA80', 0xBA80, 16, 'AristaDwdmCoherentCfp2_tableBA80' ),
               ( 'tableBB00', 0xBB00, 16, 'AristaDwdmCoherentECfp2_tableBB00' ),
               ( 'tableBB80', 0xBB80, 16, 'AristaDwdmCoherentCfp2_tableBB80' ),
             ]
   elif mediaType == 'AristaCfp2DwdmDco':
      return [ ( 'table8000', 0x8000, 8, 'AristaCfp2DwdmDco_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaCfp2DwdmDco_table8080' ),
               ( 'table8180', 0x8180, 8, 'AristaCfp2DwdmDco_table8180' ),
               ( 'tableB000', 0xB000, 16, 'AristaCfp2DwdmDco_tableB000' ),
               ( 'tableB480', 0xB480, 16, 'AristaCfp2DwdmDco_tableB480' ),
             ]
   elif mediaType == 'AcaciaCfp2DwdmDco':
      return [ ( 'table8000', 0x8000, 8, 'AcaciaCfp2DwdmDco_table8000' ),
               ( 'table8080', 0x8080, 8, 'AcaciaCfp2DwdmDco_table8080' ),
               ( 'table8180', 0x8180, 8, 'AcaciaCfp2DwdmDco_table8180' ),
               ( 'tableB000', 0xB000, 16, 'AcaciaCfp2DwdmDco_tableB000' ),
               ( 'tableB480', 0xB480, 16, 'AcaciaCfp2DwdmDco_tableB480' ),
             ]
   else:
      assert 0, "New mediaType is not implemented in cfp2Tables()"
      return None

def convertCfp2EepromTemplateToBytes( mediaType ):
   cfp2 = Tac.newInstance( "Xcvr::CfpMsaMisIdEepromContents", "idprom" )
   for ( tableId, addr, dataWidth, templateStr ) in cfp2Tables( mediaType ):
      templBytes = getBytes( templateStr )
      dataBytes = zeroOutStrEntries( templBytes )
      memory = cfp2.newMemory( tableId, addr, dataWidth )
      memory.contents = convertBytesToBinStr( dataBytes )
   return cfp2

# Given a media type, return a list of tuples of the form
# ( OsfpBankAndPage, pageTemplateString ) such that pageTemplateString
# indexes into the eepromTemplates table.  Accepts CMIS transceiver
# types i.e. OSFP and QSFP-DD
def cmisPages( mediaType ):
   if getXcvrType( mediaType ) not in ( "osfp", "qsfpDd", "qsfpCmis" ):
      assert False, "%s does not map to a CMIS xcvr type" % mediaType

   osfpBank = Tac.Type( "Xcvr::OsfpBank" )
   osfpPage = Tac.Type( "Xcvr::OsfpPage" )
   lowerPage00 = Tac.Value( "Xcvr::OsfpBankAndPage",
                            osfpBank.bankNa,
                            osfpPage.lowerPage00 )
   upperPage00 = Tac.Value( "Xcvr::OsfpBankAndPage",
                            osfpBank.bankNa,
                            osfpPage.upperPage00 )
   upperPage01 = Tac.Value( "Xcvr::OsfpBankAndPage",
                            osfpBank.bankNa,
                            osfpPage.upperPage01 )
   upperPage02 = Tac.Value( "Xcvr::OsfpBankAndPage",
                            osfpBank.bankNa,
                            osfpPage.upperPage02 )
   upperPage04 = Tac.Value( "Xcvr::OsfpBankAndPage",
                            osfpBank.bankNa,
                            osfpPage.upperPage04 )
   upperPage10Bank0 = Tac.Value( "Xcvr::OsfpBankAndPage",
                                 osfpBank.bank0,
                                 osfpPage.upperPage10 )
   upperPage11Bank0 = Tac.Value( "Xcvr::OsfpBankAndPage",
                                 osfpBank.bank0,
                                 osfpPage.upperPage11 )
   upperPage12Bank0 = Tac.Value( "Xcvr::OsfpBankAndPage",
                                 osfpBank.bank0,
                                 osfpPage.upperPage12 )
   upperPage20Bank0 = Tac.Value( "Xcvr::OsfpBankAndPage",
                                 osfpBank.bank0,
                                 osfpPage.upperPage20 )
   upperPage28Bank0 = Tac.Value( "Xcvr::OsfpBankAndPage",
                                 osfpBank.bank0,
                                 osfpPage.upperPage28 )
   upperPage2FBank0 = Tac.Value( "Xcvr::OsfpBankAndPage",
                                 osfpBank.bank0,
                                 osfpPage.upperPage2F )
   upperPageC0 = Tac.Value( "Xcvr::OsfpBankAndPage",
                                 osfpBank.bankNa,
                                 osfpPage.upperPageC0 )
   upperPageC1 = Tac.Value( "Xcvr::OsfpBankAndPage",
                                 osfpBank.bankNa,
                                 osfpPage.upperPageC1 )
   upperPageD0 = Tac.Value( "Xcvr::OsfpBankAndPage",
                                 osfpBank.bankNa,
                                 osfpPage.upperPageD0 )
   upperPageD1 = Tac.Value( "Xcvr::OsfpBankAndPage",
                                 osfpBank.bankNa,
                                 osfpPage.upperPageD1 )
   upperPageD8 = Tac.Value( "Xcvr::OsfpBankAndPage",
                                 osfpBank.bankNa,
                                 osfpPage.upperPageD8 )
   upperPageD9 = Tac.Value( "Xcvr::OsfpBankAndPage",
                                 osfpBank.bankNa,
                                 osfpPage.upperPageD9 )

   if mediaType in ( 'AristaOsfp400gCr8', 'VeridianOsfp400gCr8', ):
      return [ ( lowerPage00, 'AristaOsfp400gCr8Lower00' ),
               ( upperPage00, 'AristaOsfp400gCr8Upper00' ) ]
   elif mediaType in ( 'AristaCmis4Osfp400gCr8', ):
      return [ ( lowerPage00, 'AristaCmis4Osfp400gCr8Lower00' ),
               ( upperPage00, 'AristaCmis4Osfp400gCr8Upper00' ) ]
   elif mediaType in ( 'AristaQsfpDd400gCr8', 'VeridianQsfpDd400gCr8', ):
      return [ ( lowerPage00, 'AristaQsfpDd400gCr8Lower00' ),
               ( upperPage00, 'AristaQsfpDd400gCr8Upper00' ) ]
   elif mediaType in ( 'AristaQsfpCmis200gCr4', 'VeridianQsfpCmis200gCr4', ):
      return [ ( lowerPage00, 'AristaQsfpCmis200gCr4Lower00' ),
               ( upperPage00, 'AristaQsfpCmis200gCr4Upper00' ) ]
   elif mediaType in ( 'AristaOsfp400gSr8', 'VeridianOsfp400gSr8', ):
      return [ ( lowerPage00, 'AristaOsfp400gSr8Lower00' ),
               ( upperPage00, 'AristaOsfp400gSr8Upper00' ),
               ( upperPage01, 'AristaOsfp400gSr8Upper01' ),
               ( upperPage02, 'AristaOsfp400gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400gSr8Cmis3', ):
      return [ ( lowerPage00, 'AristaOsfp400gSr8Cmis3Lower00' ),
               ( upperPage00, 'AristaOsfp400gSr8Upper00' ),
               ( upperPage01, 'AristaOsfp400gSr8Upper01' ),
               ( upperPage02, 'AristaOsfp400gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400gAr8', 'VeridianOsfp400gAr8', ):
      return [ ( lowerPage00, 'AristaOsfp400gAr8Lower00' ),
               ( upperPage00, 'AristaOsfp400gAr8Upper00' ),
               ( upperPage01, 'AristaOsfp400gAr8Upper01' ),
               ( upperPage02, 'AristaOsfp400gAr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gAr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gAr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400gCra8', ):
      return [ ( lowerPage00, 'AristaOsfp400gAr8Lower00' ),
               ( upperPage00, 'AristaOsfp400gCra8Upper00' ),
               ( upperPage01, 'AristaOsfp400gAr8Upper01' ),
               ( upperPage02, 'AristaOsfp400gAr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gAr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gAr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400g2Fr4', ):
      return [ ( lowerPage00, 'AristaOsfp400g2Fr4Lower00' ),
               ( upperPage00, 'AristaOsfp400g2Fr4Upper00' ),
               ( upperPage01, 'AristaOsfp400g2Fr4Upper01' ),
               ( upperPage02, 'AristaOsfp400g2Fr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400g2Fr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400g2Fr4Upper11Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDd400gZr', 'AcaciaQsfpDd400gZr' ):
      return [ ( lowerPage00, 'AristaQsfpDd400gZrLower00' ),
               ( upperPage00, 'AristaQsfpDd400gZrUpper00' ),
               ( upperPage01, 'AristaQsfpDd400gZrUpper01' ),
               ( upperPage02, 'AristaOsfp400g2Fr4Upper02' ),
               ( upperPage04, 'AristaQsfpDd400gZrUpper04' ),
               ( upperPage10Bank0, 'AristaOsfp400g2Fr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400g2Fr4Upper11Bank0' ),
               ( upperPage12Bank0, 'AristaQsfpDd400gZrUpper12Bank0' ),
               ( upperPage20Bank0, 'AristaQsfpDd400gZrUpper20Bank0' ),
               ( upperPage28Bank0, 'AristaQsfpDd400gZrUpper28Bank0' ),
               ( upperPage2FBank0, 'AristaQsfpDd400gZrUpper2FBank0' ) ]
   elif mediaType in ( 'AristaOsfp400gZr', ):
      return [ ( lowerPage00, 'AristaOsfp400gZrLower00' ),
               ( upperPage00, 'AristaOsfp400gZrUpper00' ),
               ( upperPage01, 'AristaQsfpDd400gZrUpper01' ),
               ( upperPage02, 'AristaOsfp400g2Fr4Upper02' ),
               ( upperPage04, 'AristaQsfpDd400gZrUpper04' ),
               ( upperPage10Bank0, 'AristaOsfp400g2Fr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400g2Fr4Upper11Bank0' ),
               ( upperPage12Bank0, 'AristaQsfpDd400gZrUpper12Bank0' ),
               ( upperPage20Bank0, 'AristaQsfpDd400gZrUpper20Bank0' ),
               ( upperPage28Bank0, 'AristaQsfpDd400gZrUpper28Bank0' ),
               ( upperPage2FBank0, 'AristaQsfpDd400gZrUpper2FBank0' ) ]
   elif mediaType in ( 'AristaOsfpAmpZr', ):
      return [ ( lowerPage00, 'AristaOsfpAmpZrLower00' ),
               ( upperPage00, 'AristaOsfpAmpZrUpper00' ),
               ( upperPage01, 'AristaOsfp400g2Fr4Upper01' ),
               ( upperPage02, 'AristaOsfpAmpZrUpper02' ),
               ( upperPage10Bank0, 'AristaOsfp400g2Fr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfpAmpZrUpper11Bank0' ),
               ( upperPageC0, 'AristaOsfpAmpZrUpperC0' ),
               ( upperPageC1, 'AristaOsfpAmpZrUpperC1' ),
               ( upperPageD0, 'AristaOsfpAmpZrUpperD0' ),
               ( upperPageD1, 'AristaOsfpAmpZrUpperD1' ),
               ( upperPageD8, 'AristaOsfpAmpZrUpperD8' ),
               ( upperPageD9, 'AristaOsfpAmpZrUpperD9' ) ]
   elif mediaType in ( 'GoogleOM8FNT', ):
      return [ ( lowerPage00, 'AristaOsfp400gSr8Lower00' ),
               ( upperPage00, 'GoogleOM8FNTUpper00' ),
               ( upperPage01, 'AristaOsfp400gSr8Upper01' ),
               ( upperPage02, 'AristaOsfp400gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDd400gSr8', ):
      return [ ( lowerPage00, 'AristaQsfpDd400gSr8Lower00' ),
               ( upperPage00, 'AristaQsfpDd400gSr8Upper00' ),
               ( upperPage01, 'AristaOsfp400gSr8Upper01' ),
               ( upperPage02, 'AristaOsfp400gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDd400gSr8Cmis3', ):
      return [ ( lowerPage00, 'AristaQsfpDd400gSr8Cmis3Lower00' ),
               ( upperPage00, 'AristaQsfpDd400gSr8Upper00' ),
               ( upperPage01, 'AristaOsfp400gSr8Upper01' ),
               ( upperPage02, 'AristaOsfp400gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400gDr4', ):
      return [ ( lowerPage00, 'AristaOsfp400gDr4Lower00' ),
               ( upperPage00, 'AristaOsfp400gDr4Upper00' ),
               ( upperPage01, 'AristaOsfp400gDr4Upper01' ),
               ( upperPage02, 'AristaOsfp400gDr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gDr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gDr4Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400gXdr4', ):
      return [ ( lowerPage00, 'AristaOsfp400gDr4Lower00' ),
               ( upperPage00, 'AristaOsfp400gDr4Upper00' ),
               ( upperPage01, 'AristaOsfp400gXdr4Upper01' ),
               ( upperPage02, 'AristaOsfp400gDr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gDr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gDr4Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400gFr4', ):
      return [ ( lowerPage00, 'AristaOsfp400gFr4Lower00' ),
               ( upperPage00, 'AristaOsfp400gDr4Upper00' ),
               ( upperPage01, 'AristaOsfp400gDr4Upper01' ),
               ( upperPage02, 'AristaOsfp400gDr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gDr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gDr4Upper11Bank0' ) ]
   elif mediaType in ( 'AristaQsfpCmis200gFr4', ):
      return [ ( lowerPage00, 'AristaQsfpCmis200gFr4Lower00' ),
               ( upperPage00, 'AristaQsfpCmis200gFr4Upper00' ),
               ( upperPage01, 'AristaQsfpCmis200gFr4Upper01' ),
               ( upperPage02, 'AristaQsfpCmis200gFr4Upper02' ),
               ( upperPage10Bank0, 'AristaQsfpCmis200gFr4Upper10' ),
               ( upperPage11Bank0, 'AristaQsfpCmis200gFr4Upper11' ) ]
   else:
      assert 0, "CMIS mediaType %s does not exist in IdpromLib" % mediaType
      return None


def cmisDynamicPageFactory( bank, page ):
   return Tac.Value( "Xcvr::OsfpBankAndPage", bank, page )

def getCmisDynamicPages( mediaType ):
   if getXcvrType( mediaType ) not in ( "osfp", "qsfpDd", "qsfpCmis" ):
      assert False, "%s does not map to a CMIS xcvr type" % mediaType

   osfpBank = Tac.Type( "Xcvr::OsfpBank" )
   osfpPage = Tac.Type( "Xcvr::OsfpPage" )
   dynamicPages = []
   if mediaType in ( 'AristaQsfpDd400gZr', 'AcaciaQsfpDd400gZr',
                     'AristaOsfp400gZr' ):
      dynamicPages += \
            [ cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage21 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage22 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage23 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage24 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage25 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage26 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage27 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage29 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage2A ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage2B ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage2C ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage2D ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage30 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage31 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage32 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage33 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage34 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage35 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage38 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage3A ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage3B ),
              cmisDynamicPageFactory( osfpBank.bankNa, osfpPage.upperPage41 ),
              cmisDynamicPageFactory( osfpBank.bankNa, osfpPage.upperPage42 ),
              cmisDynamicPageFactory( osfpBank.bankNa, osfpPage.upperPage43 ) ]
      if mediaType in [ 'AcaciaQsfpDd400gZr' ]:
         dynamicPages += \
            [ cmisDynamicPageFactory( osfpBank.bankNa, osfpPage.upperPageB8 ) ]
      return dynamicPages
   else:
      return []

def convertCmisEepromTemplateToBytes( mediaType ):
   cmisRaw = Tac.newInstance( "Xcvr::OsfpRawEepromContents", "idprom" )
   for ( bankAndPage, templateStr ) in cmisPages( mediaType ):
      templBytes = getBytes( templateStr )
      assert len( templBytes ) == cmisRaw.eepromPageSize, ( 'CMIS page %s has %d '
             'bytes instead of the expected %d' % ( templateStr, len( templBytes ),
             cmisRaw.eepromPageSize ) )
      bytez = zeroOutStrEntries( templBytes )
      pageContent = convertBytesToBinStr( bytez )
      cmisRaw.memory[ bankAndPage ] = pageContent

   if mediaType in ( 'AristaQsfpDd400gZr', 'AcaciaQsfpDd400gZr',
                     'AristaOsfp400gZr' ):
      for bankAndPage in getCmisDynamicPages( mediaType ):
         bytez = [ 0 ] * 128
         pageContent = convertBytesToBinStr( bytez )
         cmisRaw.memory[ bankAndPage ] = pageContent
   return cmisRaw

# Utility function to edit a single register of a raw CMIS EEPROM object.
# Parameters:
#  idprom - a OsfpRawEepromContents object
#  reg - a OsfpRegister object
#  bytez - a list of characters
def editCmisEeprom( idprom, reg, bytez ):
   length = len( bytez )
   assert length == reg.size, "Number of new bytes must equal register size"

   bankAndPage = Tac.Value( "Xcvr::OsfpBankAndPage", reg.bank, reg.page )
   if reg.bank == Tac.Type( "Xcvr::OsfpPage" ).lowerPage00:
      regOffset = reg.addr
   else:
      regOffset = reg.addr - idprom.eepromUpperPageOffset

   pageStr = list( idprom.memory[ bankAndPage ] )
   pageStr[ regOffset : regOffset + reg.size ] = bytez
   idprom.memory[ bankAndPage ] = ''.join( pageStr )
   assert len( idprom.memory[ bankAndPage ] ) == idprom.eepromPageSize

# Function to recompute the checksum value of a raw CMIS EEPROM page
def recomputeCmisEepromPageChecksum( idprom, page ):
   osfpReg = Tac.Type( "Xcvr::OsfpRegister" )
   osfpPage = Tac.Type( "Xcvr::OsfpPage" )
   pageToReg = { osfpPage.upperPage00 : osfpReg.checksumUpPg00,
                 osfpPage.upperPage01 : osfpReg.checksumUpPg01,
                 osfpPage.upperPage02 : osfpReg.checksumUpPg02 }

   checksumReg = pageToReg.get( page )
   if checksumReg:
      byteList = list( idprom.memory[ Tac.Value( "Xcvr::OsfpBankAndPage",
                                                 checksumReg.bank,
                                                 checksumReg.page ) ] )

      # UpperPage01 is weird. Its checksum range starts at byte 130.
      up01hStart = 130 - idprom.eepromUpperPageOffset
      offsetStart = up01hStart if page == osfpPage.upperPage01 else 0
      offsetEnd = checksumReg.addr - idprom.eepromUpperPageOffset

      checksum = 0
      for indx in xrange( offsetStart, offsetEnd ):
         checksum += ord( byteList[ indx ] )
      checksum &= 0xff
      editCmisEeprom( idprom, checksumReg, [ chr( checksum ) ] )
   else:
      assert False, "Page: %s does not have a checksum" % page

# Inputs are Xcvr::OsfpRawEepromContents and Xcvr::VendorInfo.
# This function replaces the vendor info bytes in raw EEPROM
# with the data in vendorInfo, recomputes the checksum, and
# re-writes the checksum byte
def overwriteCmisEepromVendorInfo( idprom, vendorInfo ):
   assert vendorInfo.vendorName
   assert vendorInfo.vendorSn
   assert vendorInfo.vendorDateCode

   # Helper function which turns a vendor info string into a list of
   # characters while padding the list with whitespace up to 'size' length.
   def padVendorInfo( size, infoStr ):
      charList = list( infoStr )
      for _ in xrange( len( charList ), size ):
         charList.append( ' ' )
      return charList

   osfpReg = Tac.Type( "Xcvr::OsfpRegister" )
   # Map from register to the vendor info string stored in that register
   regToInfo = { osfpReg.vendorName : vendorInfo.vendorName,
                 osfpReg.vendorSn : vendorInfo.vendorSn,
                 osfpReg.dateCode : vendorInfo.vendorDateCode }
   if vendorInfo.vendorPn:
      regToInfo[ osfpReg.vendorPn ] = vendorInfo.vendorPn
   if vendorInfo.vendorRev:
      regToInfo[ osfpReg.vendorRev ] = vendorInfo.vendorRev

   # Loop over our mapping of vendor info registers, editing our raw EEPROM to
   # contain the new vendor info.
   for reg, info in regToInfo.iteritems():
      paddedInfo = padVendorInfo( reg.size, info )
      editCmisEeprom( idprom, reg, paddedInfo )

   # Since we just overwrote EEPROM, we must recompute the checksum
   # for upperPage00 and overwrite the previous one.
   recomputeCmisEepromPageChecksum( idprom, osfpReg.vendorName.page )

def decodeCmisIdprom( idEeprom ):
   # Invoke the OsfpEepromParser to build a decoded EEPROM object
   decodedIdEeprom = Tac.newInstance( "Xcvr::OsfpDecodedEepromContents",
                                      "idprom" )
   parser = Tac.newInstance( "Xcvr::OsfpEepromParser",
                             idEeprom,
                             decodedIdEeprom )
   parser.parseLowerPage()
   parser.parseUpperPage00()

   if not decodedIdEeprom.flatMem:
      # keep parsing banked and paged memory
      parser.parseUpperPage01()

   return decodedIdEeprom

def addSff8436VendorInfoField( vendorInfoField, vendorInfoString, eList ):
   # The trick to adding a vendor name to the Sff8436 data is to
   # have the string put in the right place and have it be the
   # correct length.
   # Begin with an empty string of the right length.
   ( fieldOffset, fieldLength ) = sff8436VendorInfoDefs[ vendorInfoField ]
   newVI = ' ' * fieldLength
   # Make sure the length isn't too big
   length = len( vendorInfoString )
   if length > fieldLength:
      length = fieldLength

   # Use slices to insert the vendor name into a list.
   viList = list( vendorInfoString )
   newVIList = list( newVI )
   newVIList[ : length ] = viList[ : length ]

   # Use slices to insert the string into our eList data.
   eList[ fieldOffset : fieldOffset + fieldLength ] = newVIList

   return eList


def genIdprom( mediaType, vendorInfo=None ):
   # Start by figuring out which type of xcvr we are dealing with.
   # The specifics of the creation depend on the type.
   xcvrType = getXcvrType( mediaType )
   if xcvrType == "sfp":
      idEeprom = convertSff8472EepromTemplateToBytes( mediaType )
   elif xcvrType == "qsfp":
      idEeprom = convertSff8436EepromTemplateToBytes( mediaType )
   elif xcvrType == "mbo":
      idEeprom = convertMboEepromTemplateToBytes( mediaType )
   elif xcvrType == "cfp2":
      idEeprom = convertCfp2EepromTemplateToBytes( mediaType )
   elif xcvrType in ( "osfp", "qsfpDd", "qsfpCmis" ):
      idEeprom = convertCmisEepromTemplateToBytes( mediaType )
   else:
      assert False, "Unknown Xcvr type"

   if mediaType == "empty":
      return idEeprom

   # Get some date/time info for IdEeprom fields
   # The values we create here are used directly in sfp.
   vn = None
   pn = None
   sn = None
   rev = None
   dc = None

   if vendorInfo:
      (vn, pn, sn, rev, dc) = ( vendorInfo.vendorName,
                                vendorInfo.vendorPn,
                                vendorInfo.vendorSn,
                                vendorInfo.vendorRev,
                                vendorInfo.vendorDateCode )
      assert vn and pn and sn and rev and dc

   else:
      vn = getVendorName( mediaType )
      today = datetime.datetime.today()
      dy = today.day
      mo = today.month
      # A bit of a brutal hack to just get the last two digits of the
      # year, i.e., '08' for '2008'.
      yr = str( today.year )[-2:]
      ( _d1, wk, _d2 ) = today.isocalendar()
      # Serial nos start with "X", then date (MMYYWW), and 5-digit number
      sn = "X" + "%02d%2s%02d%05d" % ( mo, yr, wk, random.randint( 0, 1E5 ) )
      # Datecode = date (YYMMDD) + lot code (LL)
      dc = "%02s%02d%02d%02d" % ( yr, mo, dy, random.randint( 0, 99 ) )

   if xcvrType == "sfp":
      # Now doctor up the bits that aren't filled in in the template
      # Make a copy of the string as a list.
      if vn:
         idEeprom.vendorName = vn

      if pn:
         idEeprom.vendorPn = pn

      assert sn
      idEeprom.vendorSn = sn

      if rev:
         idEeprom.vendorRev = rev

      assert dc
      idEeprom.dateCode = dc

      idEeprom.storedCcBase = idEeprom.computedCcBase
      idEeprom.storedCcExt = idEeprom.computedCcExt

      # Corrupt the checksum for this mediaType ( used in a test )
      if mediaType == 'AristaSRv1001BadCc':
         idEeprom.storedCcBase = idEeprom.storedCcBase + 1

      # NOTE: we are *NOT* updating the CRC (bytes 124-127)!!!
      #
      # There's code to do that in Sol/manufacturing/xcvrIdprom.py

   elif xcvrType == "qsfp":
      # Now doctor up the bits that aren't filled in in the template. Since the
      # thing we are modifying is a string and they are hard to mutate in Python
      # we convert the eeprom data to a list of bytes.
      eList = list( idEeprom.eepromPage00UpperData )

      # Add the Vendor name.
      if not vn:
         vn = getVendorName( mediaType )

      eList = addSff8436VendorInfoField( 'vendorName', vn, eList )


      # Vendor SN
      eList = addSff8436VendorInfoField( 'vendorSN', sn, eList )

      # Vendor Datecode
      eList = addSff8436VendorInfoField( 'vendorDateCode', dc, eList )

      # Vendor PN
      if pn:
         eList = addSff8436VendorInfoField( 'vendorPN', pn, eList )

      # Vendor Rev
      if rev:
         eList = addSff8436VendorInfoField( 'vendorRev', rev, eList )

      # Put the modified data back into the idEeprom so the CRC
      # calculations will be on the corrected data.
      idEeprom.eepromPage00UpperData = ''.join( eList )

      # Re-list the data so we can add the CRC fields.
      eList = list( idEeprom.eepromPage00UpperData )

      eList[ qsfpBaseChecksumOffset ] = \
         chr( idEeprom.baseChecksumCalculated[ 1 ] & 0xff )

      eList[ qsfpExtChecksumOffset ] = \
         chr( idEeprom.extendedChecksumCalculated & 0xff )

      # Revision Compliance
      idEeprom.sff8636Revision = qsfpEepromRevisions[ mediaType ]

      # Put the completed list back into the eeprom data.
      idEeprom.eepromPage00UpperData = ''.join( eList )
      # Populate lower page A0 with DOM data.
      idEeprom.eepromPage00LowerData = genSff8436Page00LowerData( mediaType )
      # Populate the upper page2 for security data
      idEeprom.eepromPage02UpperData = genSff8436Page02UpperData()
      # And lastly, populate the upper page 3 for DOM thresholds
      idEeprom.eepromPage03UpperData = genSff8436Page03UpperData( mediaType )

      if mediaType == 'Arista100GEDWDM2Qsfp':
         idEeprom.eepromPage20hUpperData = genSff8436Page20hUpperData( )
         idEeprom.eepromPage21hUpperData = genSff8436Page21hUpperData( )

   elif xcvrType == "mbo":
      # Update the checksums
      if mediaType == "AvagoAFBRXXRxyZ":
         pageId = [ Tac.Value( "Xcvr::MboPageId", 0x50, 0xff ),
                    Tac.Value( "Xcvr::MboPageId", 0x50, 0 ),
                    Tac.Value( "Xcvr::MboPageId", 0x50, 1 ),
                    Tac.Value( "Xcvr::MboPageId", 0x60, 0xff ),
                    Tac.Value( "Xcvr::MboPageId", 0x60, 0 ),
                    Tac.Value( "Xcvr::MboPageId", 0x60, 1 ) ]
         for page in pageId:
            elemList = list( idEeprom.memory[ page ].contents )
            elemList[ AFBR81RxyZChecksumOffset ] = \
                      chr( idEeprom.checksumCalculated( page ) )
            idEeprom.memory[ page ].contents = ''.join( elemList )
      elif mediaType == "Finisar300m":
         pass
      elif mediaType == "FBOxD1xSL1Lxx":
         pageId = [ Tac.Value( "Xcvr::MboPageId", 0x50, 0xff ),
                    Tac.Value( "Xcvr::MboPageId", 0x50, 0 ),
                    Tac.Value( "Xcvr::MboPageId", 0x50, 1 ),
                    Tac.Value( "Xcvr::MboPageId", 0x54, 0xff ),
                    Tac.Value( "Xcvr::MboPageId", 0x54, 0 ),
                    Tac.Value( "Xcvr::MboPageId", 0x54, 1 ) ]
         for page in pageId:
            elemList = list( idEeprom.memory[ page ].contents )
            elemList[ FBOxD1xSL1LxxChecksumOffset ] = \
                      chr( idEeprom.checksumCalculated( page ) )
            idEeprom.memory[ page ].contents = ''.join( elemList )
   elif xcvrType == "cfp2":
      pass
   elif xcvrType in ( "osfp", "qsfpDd", "qsfpCmis" ):
      # Rewrite vendor information into the raw EEPROM object
      if vendorInfo:
         vi = vendorInfo
      else:
         vi = Tac.Value( "Xcvr::VendorInfo" )
         vi.vendorName = vn
         vi.vendorSn = sn
         vi.vendorDateCode = dc

      overwriteCmisEepromVendorInfo( idEeprom, vi )
   else:
      assert False, "Unknown xcvr type"

   return idEeprom

def changeQsfpPowerClass( idEeprom, powerClass, powerDissipated=None ):
   newEeprom = idEeprom

   eUpperList = list( newEeprom.eepromPage00UpperData )

   if powerClass >= 1 and powerClass <= 4:
      eUpperList[1] = chr( ( powerClass - 1 ) << 6 )
   elif powerClass >= 5 and powerClass <= 7:
      eUpperList[1] = chr( ( powerClass - 4 ) )
   elif powerClass == 8:
      assert powerDissipated is not None, \
             "powerDissipated should be provided to set powerClass8"
      eUpperList[1] = chr( 0x20 )
   else:
      assert False, "Unknown Power Class"

   newEeprom.eepromPage00UpperData = ''.join( eUpperList )

   eUpperList[ qsfpBaseChecksumOffset ] = \
      chr( newEeprom.baseChecksumCalculated[ 1 ] & 0xff )

   eUpperList[ qsfpExtChecksumOffset ] = \
      chr( newEeprom.extendedChecksumCalculated & 0xff )

   # Put the completed list back into the eeprom data.
   newEeprom.eepromPage00UpperData = ''.join( eUpperList )

   if powerClass == 8:
      powerDissipated = int( powerDissipated * 4 )
      eLowerList = bytearray( newEeprom.eepromPage00LowerData )
      eLowerList[107] = powerDissipated
      eLowerList[93] = 0x08
      newEeprom.eepromPage00LowerData = str( eLowerList )
   return newEeprom

def changeSfpPowerClass( idEeprom, powerClass ):
   newEeprom = idEeprom
   eList = list( newEeprom.buf )
   optionsValueBaseAddress = 64
   sfpExtChecksumOffset = 95
   existingFlag = int( eList[ optionsValueBaseAddress ].encode( 'hex' ), 16 ) 
   if powerClass == 3:
      eList[ optionsValueBaseAddress ] = chr( ( 1 << 5 ) | existingFlag )
   else:
      eList[ optionsValueBaseAddress ] = chr( ~( 1 << 5 ) & existingFlag )
      existingFlag = int( eList[ optionsValueBaseAddress ].encode( 'hex' ), 16 ) 
      if powerClass == 1:
         eList[ optionsValueBaseAddress ] = chr( ~( 1 << 1 ) & existingFlag )
      elif powerClass == 2:
         eList[ optionsValueBaseAddress ] = chr( ( 1 << 1 ) | existingFlag )

   newEeprom.buf = ''.join( eList )
   eList[ sfpExtChecksumOffset ] = chr( newEeprom.computedCcExt )
   
   newEeprom.buf = ''.join( eList )
   return newEeprom

# Make tuning parameters supported or not
def changeTuningSupport( idEeprom, enable ):
   newEeprom = idEeprom
   eList = list( newEeprom.eepromPage00UpperData )

   suppByte = 0
   if enable:
      suppByte |= 0x07

   eList[ 65 ] = chr( suppByte )
   newEeprom.eepromPage00UpperData = ''.join( eList )
   eList[ qsfpBaseChecksumOffset ] = \
      chr( newEeprom.baseChecksumCalculated[ 1 ] & 0xff )
   eList[ qsfpExtChecksumOffset ] = \
      chr( newEeprom.extendedChecksumCalculated & 0xff )
   # Put the completed list back into the eeprom data.
   newEeprom.eepromPage00UpperData = ''.join( eList )
   return newEeprom

def getMboLaneChannelMap( mediaType ):
   if mediaType == 'AvagoAFBRXXRxyZ' or mediaType == 'FBOxD1xSL1Lxx':
      return AvagoLaneChannelMap
   elif mediaType == 'Finisar300m':
      return Finisar300mLaneChannelMap
   else:
      assert "Unknown mediaType {}".format( mediaType )
      return None

# Generating the Sff-8436 A0 lower page. Primarily for DOM data.

sff8436Page00LowerData = [
   0x0D, 0x00 ] + [ # QSFP+ Identifier
   0 for _i in range( 2, 22 ) ] + [
   # Module monitors:
   0x7F, # Byte 22 Temperature MSB
   0xFF, # Temperature LSB
   0x00, # Reserved
   0x00, # Reserved
   0xFF, # Supply Voltage MSB
   0xFF, # Supply Voltage LSB
   0x00, # Reserved
   0x00, # Reserved
   0x00, # Byte 30 Vendor Specific
   0x00, # Vendor Specific
   0x00, # Vendor Specific
   0x00, # Vendor Specific
   # Channel Monitors:
   0xFF, # Byte 34 Rx1 Power MSB
   0xFF, # Rx1 Power LSB
   0xFF, # Rx2 Power MSB
   0xFF, # Rx2 Power LSB
   0xFF, # Rx3 Power MSB
   0xFF, # Rx3 Power LSB
   0xFF, # Byte 40 Rx4 Power MSB
   0xFF, # Rx4 Power LSB
   0xFF, # Tx1 Bias MSB
   0xFF, # Tx1 Bias LSB
   0xFF, # Tx2 Bias MSB
   0xFF, # Tx2 Bias LSB
   0xFF, # Tx3 Bias MSB
   0xFF, # Tx3 Bias LSB
   0xFF, # Tx4 Bias MSB
   0xFF ] + [ # Tx4 Bias LSB
   # Bytes 50-127 not applicable to test, all zeroed
   0 for _i in range( 50, 128 ) ]

# Generating the Sff-8636 A0 lower page for data updated after Revision 1.5

sff8636Page00LowerDataAfterRev15 = sff8436Page00LowerData[ 0:50 ] + [
   0xFF, # Byte 50 Tx1 PWR MSB
   0xFF, # Tx1 PWR LSB
   0xFF, # Tx2 PWR MSB
   0xFF, # Tx2 PWR LSB
   0xFF, # Tx3 PWR MSB
   0xFF, # Tx3 PWR LSB
   0xFF, # Tx4 PWR MSB
   0xFF  # Tx4 PWR LSB
   ] + sff8436Page00LowerData[ 58:128 ]

def genSff8436Page00LowerData( mediaType ):
   revision = qsfpEepromRevisions[ mediaType ]
   # The TX power is only defined after Revision 1.5 in SFF8636 or PSM4
   if mediaType == 'Psm4Qsfp' or revision >= sff8636rev15 or \
          mediaType == 'AristaTxPowerQsfp':
      thisLowerPage = sff8636Page00LowerDataAfterRev15[:]
      if mediaType in ( 'Arista50GCR2LQsfp', 'Arista50GCR2SQsfp',
            'Arista50GCR2NQsfp', 'Arista400g2QqsfpSideDepop' ):
         # Set only channels 0 and 1 implemented in the near-end config
         thisLowerPage[ 113 ] = 0xC
      if mediaType == 'Arista100GAR4Qsfp':
         thisLowerPage[ 50 : 57 ] = [0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00]
   else:
      thisLowerPage = sff8436Page00LowerData[:]
   # Mark the revision compliance byte
   thisLowerPage[ 1 ] = revision
   # Update the identifier to match upper page
   thisLowerPage[ 0 ] = getBytes( mediaType )[ 0 ]
   return convertBytesToBinStr( thisLowerPage )

# Generating the SFF-8436 upper page 20h and 21h. Primarily for Enhanced DOM.

Sff8436Page20hUpper = [
   0x01, 0x00, 0x00, 0x20, 0x11, 0x00, 0x02, 0xa2,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  
   0x13, 0x64, 0x12, 0x59, 0xff, 0xfe, 0xff, 0xfe,   
   0x00, 0x4b, 0x01, 0xa7, 0x8b, 0x13, 0x00, 0x00,  
   0xfb, 0x3d, 0xfd, 0x1a, 0x00, 0x18, 0x00, 0x18,  
   0x00, 0x00, 0x8b, 0x11, 0x90, 0x92, 0x8b, 0x13,  
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8b, 0x13,  
   0x00, 0x36, 0x80, 0x05, 0x00, 0x00, 0x00, 0x00,  
   0x04, 0x01, 0x05, 0x01, 0x14, 0x02, 0x15, 0x02,  
   0x24, 0x03, 0x25, 0x03, 0x30, 0x04, 0x40, 0x05,  
   0x54, 0x06, 0x55, 0x06, 0x64, 0x07, 0x65, 0x07,  
   0x80, 0x0b, 0x90, 0x09, 0xa0, 0x0a, 0xb0, 0x0c,  
   0xc0, 0x0f, 0xd0, 0x0d, 0xe0, 0x0e, 0xf0, 0x10, 
   0x74, 0x08, 0x75, 0x08, 0x00, 0x00, 0x00, 0x00,
   0x01, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 ]

Sff8436Page21hUpper = [
   0x28, 0x00, 0x12, 0x00, 0x27, 0x00, 0x13, 0x00,
   0x10, 0x00, 0xf0, 0x00, 0x05, 0x00, 0xfb, 0x00, 
   0xe0, 0x00, 0x00, 0x00, 0xd0, 0x00, 0x00, 0x00,
   0x90, 0x64, 0x00, 0x00, 0x88, 0x64, 0x00, 0x00, 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x10, 0x00, 0xf0, 0x00, 0x05, 0x00, 0xfb, 0x00, 
   0x02, 0x00, 0xfe, 0x00, 0x01, 0x00, 0xff, 0x00,
   0x00, 0x40, 0xff, 0xc0, 0x00, 0x33, 0xff, 0xcd, 
   0x90, 0x64, 0x00, 0x00, 0x88, 0x64, 0x00, 0x00,
   0x90, 0x64, 0x00, 0x00, 0x88, 0x64, 0x00, 0x00, 
   0x90, 0x64, 0x00, 0x00, 0x88, 0x64, 0x00, 0x00,
   0x90, 0x64, 0x00, 0x00, 0x88, 0x64, 0x00, 0x00, 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 ]

# Generating the SFF-8436 upper page 201. Primarily for Enhanced DOM thresholds.
# Generating the SFF-8436 upper page 3. Primarily for DOM thresholds.

Sff8436Page03Upper = [
   0x01, # Temp high alarm MSB -128-
   0x02, # Temp high alarm LSB
   0x03, # Temp low alarm MSB
   0x04, # Temp low alarm LSB
   0x05, # Temp high warn MSB
   0x06, # Temp high warn LSB
   0x07, # Temp low warn MSB
   0x08 ] + [ # Temp low warn LSB
   # Reserved 136 to 143
   0x00 for _i in range( 8 ) ] + [
   0x09, # Vcc high alarm MSB
   0x0A, # Vcc high alarm LSB
   0x0B, # Vcc low alarm MSB
   0x0C, # Vcc low alarm LSB
   0x0D, # Vcc high warn MSB -148-
   0x0E, # Vcc high warn LSB
   0x0F, # Vcc low warn MSB
   0x10 ] + [ # Vcc low warn LSB
   # Reserved 152-168, Vendor-specific 160-175
   0x00 for _i in range( 24 ) ] + [
   0x21, # Rx power high alarm MSB 
   0x22, # Rx power high alarm LSB 
   0x23, # Rx power low alarm MSB -178-
   0x24, # Rx power low alarm LSB
   0x25, # Rx power high warn MSB 
   0x26, # Rx power high warn LSB 
   0x27, # Rx power low warn MSB
   0x28, # Rx power low warn LSB
   0x11, # Tx bias high alarm MSB 
   0x12, # Tx bias high alarm LSB 
   0x13, # Tx bias low alarm MSB 
   0x14, # Tx bias low alarm LSB
   0x15, # Tx bias high warn MSB -188-
   0x16, # Tx bias high warn LSB 
   0x17, # Tx bias low warn MSB
   0x18 ] + [ # Tx bias low warn LSB
   # We are not interested in the rest of this page
   0x00 for _i in range( 64 ) ]

# The following values are for testing tuning values
offset = 128
# values from sff8436 specification
# Sff8436InputEqualization
Sff8436Page03Upper[ 234 - offset ] = Sff8436Page03Upper[ 235 - offset ] = 0x11
# Sff8436OutputEmphasisControl
Sff8436Page03Upper[ 236 - offset ] = Sff8436Page03Upper[ 237 - offset ] = 0x22
# Sff8436OutputDiffAmpControl
Sff8436Page03Upper[ 238 - offset ] = Sff8436Page03Upper[ 239 - offset ] = 0x33

# Generating the SFF-8636 upper page 3 for data updated after Revision 1.5

Sff8436Page02Upper = [ 0x00 ] * 128
Sff8436Page02Upper[ 0xce - 128 : ( 0xce - 128 + 22 ) ] = range( 22 )

def  genSff8436Page02UpperData():
   return convertBytesToBinStr( Sff8436Page02Upper )

Sff8636Page03UpperAfterRev15 = Sff8436Page03Upper[ 0:64 ] + [
   0x31, # Byte 192 Tx PWR High Alarm MSB
   0x32, # Tx PWR High Alarm LSB
   0x33, # Tx PWR Low Alarm MSB
   0x34, # Tx PWR Low Alarm LSB
   0x35, # Tx PWR High Warning MSB
   0x36, # Tx PWR High Warning LSB
   0x37, # Tx PWR Low Warning MSB
   0x38  # Tx PWR Low Warning LSB
   ] + Sff8436Page03Upper[ 72:128 ]

def genSff8436Page03UpperData( mediaType ):
   revision = qsfpEepromRevisions[ mediaType ]
   # The TX power is only defined after Revision 1.5 in SFF8636 or PSM4
   if mediaType == 'Psm4Qsfp' or revision >= sff8636rev15:
      thisUpperPage = Sff8636Page03UpperAfterRev15[:]
   else:
      thisUpperPage = Sff8436Page03Upper[:]
   return convertBytesToBinStr( thisUpperPage )

def genSff8436Page20hUpperData( ):
   thisUpperPage = Sff8436Page20hUpper[:]
   return convertBytesToBinStr( thisUpperPage )

def genSff8436Page21hUpperData( ):
   thisUpperPage = Sff8436Page21hUpper[:]
   return convertBytesToBinStr( thisUpperPage )

# Generating the Sff-8472 A2 page. Primarily for DOM data.

sff8472A2Page = [
   0x01,
   0x02,
   0x03,
   0x04,
   0x05,
   0x06,
   0x07,
   0x08,
   0x09,
   0x0A,
   0x0B, # Byte 10
   0x0C,
   0x0D,
   0x0E,
   0x0F,
   0x10,
   0x11,
   0x12,
   0x13,
   0x14,
   0x15, # Byte 20
   0x16,
   0x17,
   0x18,
   0x19,
   0x1A,
   0x1B,
   0x1C,
   0x1D,
   0x1E,
   0x1F, # Byte 30
   0x20,
   0x21,
   0x22,
   0x23,
   0x24,
   0x25,
   0x26,
   0x27,
   0x28,
   0x29, # Byte 40
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00, # Byte 50
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,
   0x3F, # Byte 56. Beginning of the DOM external calibration
         # constants. Rx_PWR(4).
   0x80,
   0x00,
   0x00,
   0x3F, # Byte 60
   0x80,
   0x00,
   0x00,
   0x3F,
   0x80,
   0x00,
   0x00,
   0x3F,
   0x80,
   0x00, # Byte 70
   0x00,
   0x3F,
   0x80,
   0x00,
   0x00,
   0x01,
   0x00,
   0x7F,
   0xFF,
   0x01, # Byte 80
   0x00,
   0x00,
   0xFF,
   0x01,
   0x00,
   0x01,
   0x00,
   0x01,
   0x00,
   0x27, # Byte 90
   0x10,
   0x00,
   0x00,
   0x00,
   0x00,
   0x7F, # Byte 96. Beginning of the DOM internal A/D data.
         # Temperature MSB.
   0xFF,
   0xFF,
   0xFF,
   0xFF,
   0xFF, # Byte 100
   0xFF,
   0xFF,
   0xFF,
   0xFF,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00, # Byte 109

   0xFF,
   0xFF,
   0xFF,
   0xFF,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00, # 118

   0xFF,
   0xFF,
   0xFF,
   0xFF,
   0x00,
   0x00,
   0x00,
   0x00,
   0x01, # 128

   0x02,
   0x03,
   0x04,
   0x00, # Simulate tunable SFP+ to support frequency range 191.1-196.15 THz
   0xbf, # (byte 132-141)
   0x03,
   0xe8,
   0x00,
   0xc4, # 136

   0x05,
   0xdc,
   0x01,
   0xf4,
   0x0F,
   0x10,
   0x00, # Simulate tunable SFP+ having nvram with initial values
   0x01, # (byte 144-145)
   0x13, # 145

   0x14,
   0x15,
   0x16,
   0x01,
   0x02,
   0x03,
   0x04,
   0x05, # 154

   0x06,
   0x07,
   0x08,
   0x09,
   0x0A,
   0x0B,
   0x0C,
   0x0D,
   0x0E,

   0x0F,
   0x10,
   0x11,
   0x12,
   0x13,
   0x14,
   0x00,
   0x00,
   0x00,
   0x00,

   0xFF,
   0xFF,
   0xFF,
   0xFF,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,

   0xFF,
   0xFF,
   0xFF,
   0xFF,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,

   0xFF,
   0xFF,
   0xFF,
   0xFF,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,

   0xFF,
   0xFF,
   0xFF,
   0xFF,
   0x00,
   0x00,
   0x00,
   0x00,
   0x00,
]

def genSff8472A2Page():
   return convertBytesToBinStr( sff8472A2Page )

# Mappings - fieldName : ( offset, sizeInBytes ) 
# Note: this mapping is according to the specs, some Arista specific information
# might be missing or incomplete!
sfpIdpromPageA0Mapping = {
   'identifier' : ( 0, 1 ),
   'identifierExtended' : ( 1, 1 ),
   'connector' : ( 2, 1 ),
   'complianceEth10GAndInfiniBand' : ( 3, 1 ),
   'complianceEsconAndSonet' : ( 4, 1 ),
   'complianceSonet' : ( 5, 1 ),
   'complianceEth' : ( 6, 1 ),
   'complianceFibreChannelLinkLengthAndTech' : ( 7, 1 ),
   'complianceFibreChannelTech' : ( 8, 1 ),
   'complianceFibreChannelTxMedia' : ( 9, 1 ),
   'complianceFibreChannelSpeed' : ( 10, 1 ),
   'encoding' : ( 11, 1 ),
   'bitrateNominal' : ( 12, 1 ),
   'rateIdentifier' : ( 13, 1 ),
   'lengthSmfKm' : ( 14, 1 ),
   'lengthSmf100m' : ( 15, 1 ),
   'lengthOm2' : ( 16, 1 ),
   'lengthOm1' : ( 17, 1 ),
   'lengthDirectAttach' : ( 18, 1 ),
   'lengthOm3' : ( 19, 1 ),
   'vendorName' : ( 20 , 16 ),
   'vendorOui' : ( 37, 3 ),
   'vendorPn' : ( 40, 16 ),
   'vendorRev' : ( 56, 4 ),
   'wavelength' : ( 60, 2 ),
   'wavelengthFraction' : ( 62, 1 ),
   'ccBase' : ( 63, 1 ),
   'options' : ( 64, 2 ),
   'bitrateMax' : ( 66, 1 ),
   'bitrateMin' : ( 67, 1 ),
   'vendorSn' : ( 68, 16 ),
   'dateCode' : ( 84, 8 ),
   'diagsMonitoringType' : ( 92, 1 ),
   'enhancedOptions' : ( 93, 1 ),
   'complianceSff8472' : ( 94, 1 ),
   'ccExt' : ( 95, 1 ),
   'vendorSpecific' : ( 96, 21 ),
   'arastraAttnValue' : ( 118, 1 )
}

qsfpIdpromPage00UpperMapping = {
   'identifier' : ( 0, 1 ),
   'identifierExtended' : ( 1, 1 ),
   'connector' : ( 2, 1 ),
   'complianceEth10And40G' : ( 3, 1 ),
   'complianceSonet' : ( 4, 1 ),
   'complianceSasSata' : ( 5, 1 ),
   'complianceEth1G' : ( 6, 1 ),
   'complianceFibreChannelLinkLengthAndTxTech' : ( 7, 2 ),
   'complianceFibreChannelTxMedia' : ( 9, 1 ),
   'complianceFibreChannelSpeed' : ( 10, 1 ),
   'encoding' : ( 11, 1 ),
   'bitrateNominal' : ( 12, 1 ),
   'extendedRateSelectCompliance' : ( 13, 1 ),
   'lengthSmf' : ( 14, 1 ),
   'lengthOm3' : ( 15, 1 ),
   'lengthOm2' : ( 16, 1 ),
   'lengthOm1' : ( 17, 1 ),
   'lengthCableOrOm4' : ( 18, 1 ),
   'deviceTech' : ( 19, 1 ), 
   'vendorName' : ( 20, 16 ),
   'vendorOui' : ( 37, 3 ),
   'vendorPn' : ( 40, 16 ),
   'vendorRev' : ( 56, 2 ),
   'wavelength' : ( 58, 2 ),
   'wavelengthTolerance' : ( 60, 2 ),
   'maxCaseTemp' : ( 62, 1 ),
   'ccBase' : ( 63, 1 ),
   'extendedCompliance' : ( 64, 1 ),
   'vendorSn' : ( 68, 16 ),
   'dateCode' : ( 84, 8 ),
   'diagsMonitoringType' : ( 92, 1 ),
   'enhancedOptions' : ( 93, 1 ),
   'ccExt' : ( 95, 1 )
}

QSA28 = [
   0xff, 0x51, 0x53, 0x41, 0x00, 0x01, 0x00, 0x00,
   0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0x01, 0x01, 0x01, 0x19, 0x46, 0x00, 0x00, 0x00,
   0x03, 0x03, 0x06, 0x07, 0xff, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x02, 0xc9, 0x00, 0x4d, 0x65, 0x6c, 0x6c,
   0x61, 0x6e, 0x6f, 0x78, 0x20, 0x20, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x4d, 0x54, 0x31, 0x37,
   0x33, 0x35, 0x55, 0x4b, 0x30, 0x30, 0x32, 0x31,
   0x33, 0x20, 0x20, 0x20, 0x4d, 0x41, 0x4d, 0x31,
   0x51, 0x30, 0x30, 0x41, 0x2d, 0x51, 0x53, 0x41,
   0x32, 0x38, 0x20, 0x20, 0x41, 0x34, 0x31, 0x37,
   0x30, 0x38, 0x32, 0x39, 0xff, 0xff, 0x00, 0x32 ]

def genQsaEepromData( ):
   return convertBytesToBinStr( QSA28 )
