# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Pci, sys

IPROC_SUBWIN_MAX = 8
IPROC_SUBWIN_SIZE = 4096
IPROC_SUBWIN_MASK = 0xfff
IPROC_PAXB_SUBWIN = 2
IPROC_PAXB_IMAP0_ADDR = 0xc00
IPROC_BAR0_SIZE = 32 * 1024

PCI_VENDOR_BROADCOM = 0x14e4

PCI_EXT_CAP_START = 0x100
PCI_EXT_CAP_ID_VNDR = 0x0b

def PCI_EXT_CAP_ID(hdr):
   return hdr & 0x0000ffff

def PCI_EXT_CAP_NEXT(hdr):
   return (hdr >> 20) & 0xffc

def error( s ):
   print >> sys.stderr, s
   sys.exit( 1 )

class IProc( object ):
   def __init__( self, pcidev, subwin=3 ):
      self.pcidev = pcidev
      self.subwin = subwin
      self.page = None

      if ( subwin < 2 ) or ( subwin > 7 ):
         raise ValueError( "invalid subwin %u" % subwin )

      self.res0 = Pci.Device( pcidev ).resource( 0 )
      if self.res0 is None:
         raise ValueError( "device not found: %s" % pcidev )

      if not self.is_iproc_device():
         raise ValueError( "%s is not a Broadcom iProc device" % pcidev )

   def is_iproc_device( self ):
      config = Pci.Device( self.pcidev ).config( )
      if not config:
         return False

      # vendor must be Broadcom
      vendor = config.read32( 0 ) & 0xffff
      if vendor != PCI_VENDOR_BROADCOM:
         return False

      # Copied from Broadcom SDK linux-kernel-bde.c
      # Look for PCIe vendor-specific extended capability (VSEC)
      cap_base = PCI_EXT_CAP_START
      while cap_base:
         cap_val = config.read32( cap_base )
         if PCI_EXT_CAP_ID( cap_val ) == PCI_EXT_CAP_ID_VNDR:
            break
         cap_base = PCI_EXT_CAP_NEXT( cap_val )

      if cap_base:
         # VSEC layout:
         #
         # 0x00: PCI Express Extended Capability Header
         # 0x04: Vendor-Specific Header
         # 0x08: Vendor-Specific Register 1
         # 0x0c: Vendor-Specific Register 2
         #     ...
         # 0x24: Vendor-Specific Register 8
         val = config.read32( cap_base + 8 )
         if val == 0x100:
            return True # iproc
      return False

   def iproc_map( self, addr ):
      if addr == self.page:
         return
      offset = ( IPROC_PAXB_SUBWIN * IPROC_SUBWIN_SIZE ) + IPROC_PAXB_IMAP0_ADDR + \
         ( self.subwin * 4 )
      self.res0.write32( offset, addr | 1 )
      self.page = addr

   def read32( self, addr ):
      if addr % 4:
         raise ValueError( 'address must be 4byte aligned' )
      self.iproc_map( addr & ~IPROC_SUBWIN_MASK )
      offset = ( self.subwin * IPROC_SUBWIN_SIZE ) + ( addr & IPROC_SUBWIN_MASK )
      return self.res0.read32( offset )

   def write32( self, addr, value ):
      if addr % 4:
         raise ValueError( 'address must be 4byte aligned' )
      self.iproc_map( addr & ~IPROC_SUBWIN_MASK )
      offset = ( self.subwin * IPROC_SUBWIN_SIZE ) + ( addr & IPROC_SUBWIN_MASK )
      self.res0.write32( offset, value )
