# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines all types, enums, etc... related with defining tuning
information on HW L1 components.
'''

from __future__ import absolute_import, division, print_function

from abc import ABCMeta, abstractmethod, abstractproperty
from collections import namedtuple

import Tac

SerdesSpeed = Tac.Type( 'Hardware::Phy::SerdesSpeed' )
SPEED_ANY = SerdesSpeed.serdesSpeedAny
SPEED_10G = SerdesSpeed.serdesSpeed10Point3125Gbps
SPEED_20G = SerdesSpeed.serdesSpeed20Point625Gbps
SPEED_25G = SerdesSpeed.serdesSpeed25Point78125Gbps
SPEED_50G = SerdesSpeed.serdesSpeed53Point125Gbps

SerdesMedium = Tac.Type( 'Hardware::Phy::SerdesMedium' )
MEDIUM_ANY = SerdesMedium.mediumAny
MEDIUM_CR = SerdesMedium.copper
MEDIUM_FR = SerdesMedium.fiber

TapIndex = Tac.Type( 'Hardware::Phy::SerdesTuningData::TapIndex' )
TAP1 = TapIndex.tap
TAP2 = TapIndex.tap2
TAP3 = TapIndex.tap3

class Taps( object ):
   __metaclass__ = ABCMeta

   @abstractproperty
   def parameters( self ):
      """
      The parameters that should be used for this Taps class. This should be a
      namedtuple class.

      Note that this will define the order and names of the column headers we expect
      in the csv, not including the required ComponentId and SerdesId columns.
      """
      raise NotImplementedError

   @classmethod
   @abstractmethod
   def addTuning( cls, serdesTuningData, tuningParams ):
      """
      Applies the passed in tuple of tuning params to the serdesTuningData.
      Resolves which tuple value goes where.

      Parameters
      -------
      serdesTuningData: Hardware::Phy::SerdesTuningData; the tuning object to apply
         the tuning values to
      tuningParams: tuple( int, ... ); The tuple of tuning values to put into the
         serdesTuningData. This tuple will be of the namedtuple type defined by the
         parameters property.
      """
      raise NotImplementedError

class NoTuning( Taps ):
   parameters = namedtuple( "parameters", [] )

   @classmethod
   def addTuning( cls, serdesTuningData, tuningParams ):
      # Leave everything at their default values
      pass

class Amp3Taps( Taps ):
   parameters = namedtuple( "parameters", [
      "Amplitude",
      "Pre1Tap",
      "MainTap",
      "Post1Tap",
   ] )

   @classmethod
   def addTuning( cls, serdesTuningData, tuningParams ):
      serdesTuningData.amplitude = tuningParams.Amplitude
      serdesTuningData.preTap[ TAP1 ] = tuningParams.Pre1Tap
      serdesTuningData.mainTap = tuningParams.MainTap
      serdesTuningData.postTap[ TAP1 ] = tuningParams.Post1Tap
