# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines HW L1 Topology Component Lib related errors. These should be
the exceptions that are raised whenever an error in the lib or in processing
components is detected.

Note:
   All errors defined in this file should inherit from HwL1ComponentLibraryError.
'''

from __future__ import absolute_import, division, print_function

import ArPyUtils.Types

class HwL1ComponentLibraryInternalError( ArPyUtils.Types.ArException ):
   '''Denotes an internal error in the L1 Topology Component Library.'''
   def __init__( self, message, **kwargs ):
      message = ( 'Internal Library Error: {}\n'
                  'Please contract the library maintainers!\n' ).format( message )
      super( HwL1ComponentLibraryInternalError, self ).__init__( message, **kwargs )

class HwL1ComponentLibraryError( ArPyUtils.Types.ArException ):
   '''Denotes an error in the L1 Topology Component Library whose cause can be traced
   to user input errors.
   '''
   def __init__( self, message, **kwargs ):
      message = ( '{}\nThis is most likely a result of a '
                  'logical error in the definition of the L1 Topology component. '
                  'Please check double check the FDL arguments with L1 Topology '
                  'component library definitions.\n' ).format( message )
      super( HwL1ComponentLibraryError, self ).__init__( message, **kwargs )

class HwL1ComponentError( HwL1ComponentLibraryError ):
   '''Denotes a logical error in the definition or processing of components from the
   L1 Topology Component Library.
   '''
   def __init__( self, component, message, **kwargs ):
      message = 'Error in component {} - {}'.format( component, message )
      super( HwL1ComponentError, self ).__init__( message, **kwargs )

class HwL1ComponentCoreError( HwL1ComponentLibraryError ):
   '''Denotes a logical error in the definition or processing of a specific core from
   a component.
   '''
   def __init__( self, component, coreId, message, **kwargs ):
      message = 'core {} - {}'.format( coreId, message )
      super( HwL1ComponentCoreError, self ).__init__( component, message, **kwargs )

class SerdesLogicalMappingConsistencyError( HwL1ComponentCoreError ):
   '''Denotes an error with the definition of the logical SerDes mappings for a
   component. e.g. The number of lanes and the SerDes group mode not matching up.
   '''
   def __init__( self, component, coreId, side, **kwargs ):
      message = ( 'The {} side of a SerDes mapping is not '
                  'consistent.'.format( side ) )
      super( SerdesLogicalMappingConsistencyError, self ).__init__( component,
                                                                    coreId,
                                                                    message,
                                                                    **kwargs )
