# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from abc import ABCMeta, abstractmethod, abstractproperty

from .Errors import (
   HwL1ComponentError,
)
from .Serdes import (
   MappingDesc,
   G400_8_RS544,
   G200_4_RS544,
   G100_2_RS544,
   G100_4_RS528,
   G100_4_NOFEC,
   G50_1_RS544,
   G50_2_RS528,
   G50_2_FCFEC,
   G50_2_NOFEC,
   G40_2_NOFEC,
   G40_4_NOFEC,
   G25_1_RS528,
   G25_1_FCFEC,
   G25_1_NOFEC,
   G10_1_NOFEC,
   G5_1_NOFEC,
   G2P5_1_NOFEC,
   G1_1_NOFEC,
   M100_1_NOFEC,
   M10_1_NOFEC,
)
from .Tuning import (
   Amp3Taps,
   NoTuning,
   SPEED_10G,
   SPEED_25G,
   SPEED_50G,
)

class Core( object ):
   """A base class for all cores. The behaviour of the cores are defined
   by the abstract methods and attributes below.

   Note:
      If the core is initialized in one of the special modes listed in
      HIDDEN_MODES, the mode will not be appended to the class name in
      the getComponentType output.
   """
   __metaclass__ = ABCMeta

   # We consider these modes to be special "class defining" basic modes and thus
   # don't append them to the component type string, see AID7567 for more details
   # TODO: See BUG500705; Remove these
   HIDDEN_MODES = { "Default", "Asic", "Crosspoint" }

   def __init__( self, mode ):
      if not isinstance( self.supportedModes, set ):
         raise HwL1ComponentError( self, 'supportedModes must be a set.',
                                   supportedModes=self.supportedModes )
      if mode not in self.supportedModes:
         raise HwL1ComponentError( self, "Invalid core mode.",
                                   mode=mode, supportedModes=self.supportedModes )
      self.mode = mode

   @property
   def name( self ):
      "The name of the class to output for things like strings."
      return self.__class__.__name__

   def __str__( self ):
      """
      A human-readable output for what the instance of the Core represents.

      If the core is initialized in no mode ( if supported ), we will output
      only '<core name>'.
      """
      return self.name if self.mode else "%s-%s" % ( self.name, self.mode )

   def getComponentType( self ):
      """
      Returns
      -------
      String; a name for l1 topology state machines to use for the core on the chip.
      By convention this is the string conversion of the component.
      
      """
      if self.mode in self.HIDDEN_MODES:
         return self.name
      return str( self )

   @abstractproperty
   def supportedModes( self ):
      "A set describing the modes that this core supports being configured in."
      raise NotImplementedError

   @abstractmethod
   def getNumSerdes( self ):
      """
      Returns
      -------
      int; The number of serdes within the core
      """
      raise NotImplementedError

   @abstractmethod
   def getPhysicalMapping( self ):
      """
      Parameters
      -------
      baseSerdesId: int; the base serdes of the core. All other serdes
         are numbered consecutively from here

      Returns
      -------
      Physical mapping descriptor object
      """
      raise NotImplementedError

   @abstractmethod
   def getLogicalSerdesMapping( self ):
      """
      Retrieves the logical SerDes mappings for this core.

      Returns
      -------
      An iterable which produces items of type: Serdes.MappingDesc
      The Serdes.MappingDesc iteslf has the format:
         ( Hardware::Phy::SerdesGroupMode: System Group Mode,
           list( int ): System Lanes,
           Hardware::Phy::SerdesGroupMode: Line Group Mode,
           list( int ): Line lanes )

      Group Modes are defined in the SerDes module.

      Note
      -----
      The Group Mode and the number of lanes must be consistent.
      e.g. if System Group Mode is G400_8_RS544, System Lanes must be a type that
      supports indexing with 8 items.
      """
      raise NotImplementedError

   @abstractmethod
   def getTapGroups( self ):
      """
      Retrieves the supported Taps for each speed for this core.

      Returns
      -------
      A dict of SerdesSpeed to Taps class, where the Taps class has the addTuning
      method defined on it. See the Tuning module for more information/examples.
      """
      raise NotImplementedError

class AsicCore( Core ):
   # TODO: See BUG500705; Remove Asic
   supportedModes = { None, "Asic" }

   # Asics do not have any mappings, just pass here
   def getPhysicalMapping( self ):
      pass

   @abstractmethod
   def getNumSerdes( self ):
      raise NotImplementedError

   @abstractmethod
   def getLogicalSerdesMapping( self ):
      raise NotImplementedError

   @abstractmethod
   def getTapGroups( self ):
      raise NotImplementedError

class Blackhawk( AsicCore ):
   def getNumSerdes( self ):
      return 8

   def getLogicalSerdesMapping( self ):
      mappings = []
      # single lane modes
      for x in range( 8 ):
         mappings.extend( [
            MappingDesc( None, [], G50_1_RS544, [ x ] ),
            MappingDesc( None, [], G25_1_RS528, [ x ] ),
            MappingDesc( None, [], G25_1_FCFEC, [ x ] ),
            MappingDesc( None, [], G25_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G10_1_NOFEC, [ x ] ),
         ] )
      # two lane modes
      for x in range( 0, 8, 2 ):
         mappings.extend( [
            MappingDesc( None, [], G100_2_RS544, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_RS528, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_NOFEC, [ x, x + 1 ] ),
            MappingDesc( None, [], G40_2_NOFEC, [ x, x + 1 ] ),
         ] )
      # four lane modes
      for x in ( range( 0, 4 ), range( 4, 8 ) ): 
         mappings.extend( [
            MappingDesc( None, [], G200_4_RS544, x ),
            MappingDesc( None, [], G100_4_RS528, x ),
            MappingDesc( None, [], G100_4_NOFEC, x ),
            MappingDesc( None, [], G40_4_NOFEC, x ),
         ] )
      # eight lane modes
      mappings.extend( [
         MappingDesc( None, [], G400_8_RS544, range( 8 ) ),
         ] )
      return mappings

   def getTapGroups( self ):
      """
      TODO
      ----
         Actually claim support for the tuning classes.
      """
      return { SPEED_10G: NoTuning, SPEED_25G: NoTuning, SPEED_50G: NoTuning }

# BlackhawkGen3 is the implementation of the Blackhawk serdes core in the Tomahawk4
# chip. It has only a maximum of 4 ports and a single PLL, while still having 8
# serdes lanes. Because this has software implications, we will differentiate it
# from the base Blackhawk type. 
class BlackhawkGen3( Blackhawk ):
   pass

class Falcon( AsicCore ):
   def getNumSerdes( self ):
      return 4

   def getLogicalSerdesMapping( self ):
      return [
         MappingDesc( None, [], G100_4_NOFEC, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], G100_4_RS528, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], G50_2_FCFEC, [ 0, 1 ] ),
         MappingDesc( None, [], G50_2_FCFEC, [ 2, 3 ] ),
         MappingDesc( None, [], G50_2_NOFEC, [ 0, 1 ] ),
         MappingDesc( None, [], G50_2_NOFEC, [ 2, 3 ] ),
         MappingDesc( None, [], G50_2_RS528, [ 0, 1 ] ),
         MappingDesc( None, [], G50_2_RS528, [ 2, 3 ] ),
         MappingDesc( None, [], G40_4_NOFEC, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], G25_1_FCFEC, [ 0 ] ),
         MappingDesc( None, [], G25_1_FCFEC, [ 1 ] ),
         MappingDesc( None, [], G25_1_FCFEC, [ 2 ] ),
         MappingDesc( None, [], G25_1_FCFEC, [ 3 ] ),
         MappingDesc( None, [], G25_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G25_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G25_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G25_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G25_1_RS528, [ 0 ] ),
         MappingDesc( None, [], G25_1_RS528, [ 1 ] ),
         MappingDesc( None, [], G25_1_RS528, [ 2 ] ),
         MappingDesc( None, [], G25_1_RS528, [ 3 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( G100_4_NOFEC, [ 0, 1, 2, 3 ], None, [] ),
         MappingDesc( G100_4_RS528, [ 0, 1, 2, 3 ], None, [] ),
         MappingDesc( G50_2_FCFEC, [ 0, 1 ], None, [] ),
         MappingDesc( G50_2_FCFEC, [ 2, 3 ], None, [] ),
         MappingDesc( G50_2_NOFEC, [ 0, 1 ], None, [] ),
         MappingDesc( G50_2_NOFEC, [ 2, 3 ], None, [] ),
         MappingDesc( G50_2_RS528, [ 0, 1 ], None, [] ),
         MappingDesc( G50_2_RS528, [ 2, 3 ], None, [] ),
         MappingDesc( G40_4_NOFEC, [ 0, 1, 2, 3 ], None, [] ),
         MappingDesc( G25_1_FCFEC, [ 0 ], None, [] ),
         MappingDesc( G25_1_FCFEC, [ 1 ], None, [] ),
         MappingDesc( G25_1_FCFEC, [ 2 ], None, [] ),
         MappingDesc( G25_1_FCFEC, [ 3 ], None, [] ),
         MappingDesc( G25_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G25_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G25_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G25_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G25_1_RS528, [ 0 ], None, [] ),
         MappingDesc( G25_1_RS528, [ 1 ], None, [] ),
         MappingDesc( G25_1_RS528, [ 2 ], None, [] ),
         MappingDesc( G25_1_RS528, [ 3 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], None, [] ),
      ]

   def getTapGroups( self ):
      return { SPEED_10G: Amp3Taps, SPEED_25G: Amp3Taps }

class GPhy( AsicCore ):
   def getNumSerdes( self ):
      return 4

   def getLogicalSerdesMapping( self ):
      return [
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], M10_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], M10_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], M10_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], M10_1_NOFEC, [ 3 ] ),
      ]

   def getTapGroups( self ):
      """
      TODO
      ----
         Actually claim support for the tuning classes.
      """
      return { SPEED_10G: NoTuning }

class Merlin( AsicCore ):
   def getNumSerdes( self ):
      return 4

   def getLogicalSerdesMapping( self ):
      return [
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 3 ] ),
      ]

   def getTapGroups( self ):
      """
      TODO
      ----
         Actually claim support for the tuning classes.
      """
      return { SPEED_10G: NoTuning }

class MerlinQ( Merlin ):
   # MerlinCore-Q supports the same serdes rates as a normal merlin core, but can
   # also break each serdes up into 4 physical ports each running at 2.5G to support
   # multi-gig ports on the front-panel. We are handling this breakup at the external
   # phy instead, however, as it makes the traversal easier to reason about from a
   # hardware perspective (still only 1 trace between the chips, just time division
   # duplexed). Thus those external phys will declare the serdesMapping between the
   # 5G/10G serdes rate from the asic to the data/serdes rates on the FPP
   supportedModes = { None, "Asic", "Q" }

   # TODO: See BUG478138, remove once we can specify per-core modes
   def __init__( self, mode ):
      super( MerlinQ, self ).__init__( "Q" )

   # This will make a MerlinCore-Q simply appear as if it were a normal MerlinCore
   # with an additional supported mode
   name = "Merlin"

   # TODO: See BUG478135, remove in favor of using the <core type>-<mode> convention
   def getComponentType( self ):
      if self.mode == "Q":
         return "MerlinQ"
      return "Merlin"

   # TODO: See BUG458080; also specifying supported data rates for now.
   # Return the 2 serdes rates used for USXGMII and QSGMII when running in Q mode,
   # otherwise just return the usual merlin core mappings.
   def getLogicalSerdesMapping( self ):
      if self.mode == "Q":
         return [
            MappingDesc( None, [], G10_1_NOFEC, [ 0 ] ),
            MappingDesc( None, [], G10_1_NOFEC, [ 1 ] ),
            MappingDesc( None, [], G10_1_NOFEC, [ 2 ] ),
            MappingDesc( None, [], G10_1_NOFEC, [ 3 ] ),
            MappingDesc( None, [], G5_1_NOFEC, [ 0 ] ),
            MappingDesc( None, [], G5_1_NOFEC, [ 1 ] ),
            MappingDesc( None, [], G5_1_NOFEC, [ 2 ] ),
            MappingDesc( None, [], G5_1_NOFEC, [ 3 ] ),
         # TODO: See BUG458080; Remove the mappings that are specifying data rate
         #       rather than the serdes rate once channels are supported.
         #       Though may need something similar to specify these data rate support
         #       in the end as well; final design is still TBD
         ] + [
            MappingDesc( None, [], G2P5_1_NOFEC, [ 0 ] ),
            MappingDesc( None, [], G2P5_1_NOFEC, [ 1 ] ),
            MappingDesc( None, [], G2P5_1_NOFEC, [ 2 ] ),
            MappingDesc( None, [], G2P5_1_NOFEC, [ 3 ] ),
            MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
            MappingDesc( None, [], G1_1_NOFEC, [ 1 ] ),
            MappingDesc( None, [], G1_1_NOFEC, [ 2 ] ),
            MappingDesc( None, [], G1_1_NOFEC, [ 3 ] ),
            MappingDesc( None, [], M100_1_NOFEC, [ 0 ] ),
            MappingDesc( None, [], M100_1_NOFEC, [ 1 ] ),
            MappingDesc( None, [], M100_1_NOFEC, [ 2 ] ),
            MappingDesc( None, [], M100_1_NOFEC, [ 3 ] ),
            MappingDesc( None, [], M10_1_NOFEC, [ 0 ] ),
            MappingDesc( None, [], M10_1_NOFEC, [ 1 ] ),
            MappingDesc( None, [], M10_1_NOFEC, [ 2 ] ),
            MappingDesc( None, [], M10_1_NOFEC, [ 3 ] ),
         ]
      return super( MerlinQ, self ).getLogicalSerdesMapping()

class BabbageCore( Core ):
   #               +-------------------+
   #               |      Babbage      |
   #     serdes    | +---------------+ |    serdes
   #       0   ----| |0             0| |----  0
   #       1   ----| |1             1| |----  1
   #       2   ----| |2             2| |----  2
   #       3   ----| |3             3| |----  3
   #       4   ----| |4   Core 1    4| |----  4
   #       5   ----| |5             5| |----  5
   #       6   ----| |6             6| |----  6
   #       7   ----| |7             7| |----  7
   #               | +---------------+ |
   #               | +---------------+ |
   #       8   ----| |8             8| |----  8
   #       9   ----| |9             9| |----  9
   #       10  ----| |10           10| |----  10
   #       11  ----| |11           11| |----  11
   #       12  ----| |12  Core 2   12| |----  12
   #       13  ----| |13           13| |----  13
   #       14  ----| |14           14| |----  14
   #       15  ----| |15           15| |----  15
   #               | +---------------+ |
   #               +-------------------+
   supportedModes = { "Gearbox", "Retimer" }

   def getComponentType( self ):
      # TODO change these names to match <component>-<core>
      if self.mode == "Gearbox":
         return "crt50216gb"
      elif self.mode == "Retimer":
         return "crt50216rt"
      assert False, "invalid mode %s" % self.mode
      return "unknown"

   def getNumSerdes( self ):
      return 8

   def getPhysicalMapping( self ):
      # The format for the mapping is a dictionary of tuples mapping
      # the system serdes to the line serdes. In most cases there will
      # be only one system serdes
      # { ( system serdes, ... ) : ( line serdes, ... ), ... }
      mapping = None
      if self.mode == "Gearbox":
         # Map will be:
         # A0 : B0, B1
         # A1 : B2, B3
         # A2 : B4, B5
         # A3 : B6, B7
         mapping = { ( 0, ) : ( 0, 1 ),
                     ( 1, ) : ( 2, 3 ),
                     ( 2, ) : ( 4, 5 ),
                     ( 3, ) : ( 6, 7 ) }
      elif self.mode == "Retimer":
         # Map will be:
         # A[ 0-7 ]: B[ 0-7 ]
         mapping = { ( x, ) : ( x, ) for x in range( 8 ) }
      else:
         assert False, "invalid mode %s" % self.mode
      return mapping

   def getLogicalSerdesMapping( self ):
      return []

   def getTapGroups( self ):
      # XXX FIX ME
      return { SPEED_10G: NoTuning }

class Tofino2Core( AsicCore ):
   # Tofino2 is one chip controlling 16 serdes groups, each serdes group control
   # 8 serdes lanes capable for PAM4 and NRZ.
   # Although in HW perspective there is no concept as cores, we still define
   # one serdes group as "Tofino2Core", based on the two assumptions:
   # 1. We will never use serdes cross two serdes groups together
   # 2. We can not lane swap serdes between two serdes groups
   def getNumSerdes( self ):
      return 8

   def getLogicalSerdesMapping( self ):
      mappings = []
      # single lane modes
      for x in range( 8 ):
         mappings.extend( [
            MappingDesc( None, [], G50_1_RS544, [ x ] ),
            MappingDesc( None, [], G25_1_RS528, [ x ] ),
            MappingDesc( None, [], G25_1_FCFEC, [ x ] ),
            MappingDesc( None, [], G25_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G10_1_NOFEC, [ x ] ),
         ] )
      # two lane modes
      for x in range( 0, 8, 2 ):
         mappings.extend( [
            MappingDesc( None, [], G100_2_RS544, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_RS528, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_FCFEC, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_NOFEC, [ x, x + 1 ] ),
            MappingDesc( None, [], G40_2_NOFEC, [ x, x + 1 ] ),
         ] )
      # four lane modes
      for x in ( range( 0, 4 ), range( 4, 8 ) ): 
         mappings.extend( [
            MappingDesc( None, [], G200_4_RS544, x ),
            MappingDesc( None, [], G100_4_RS528, x ),
            MappingDesc( None, [], G100_4_NOFEC, x ),
            MappingDesc( None, [], G40_4_NOFEC, x ),
         ] )
      # eight lane modes
      mappings.extend( [
         MappingDesc( None, [], G400_8_RS544, range( 8 ) ),
         ] )
      return mappings

   def getTapGroups( self ):
      """
      TODO
      ----
         Actually claim support for the tuning classes.
      """
      return { SPEED_10G: NoTuning, SPEED_25G: NoTuning, SPEED_50G: NoTuning }


