# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from abc import ABCMeta, abstractmethod, abstractproperty

import Tac

from .Errors import (
   HwL1ComponentError,
   HwL1ComponentLibraryError,
   HwL1ComponentLibraryInternalError,
)
from .Serdes import (
   MappingDesc,
   G400_8_RS544,
   G200_4_RS544,
   G100_2_RS544,
   G100_4_RS528,
   G100_4_NOFEC,
   G50_1_RS544,
   G50_2_RS528,
   G50_2_FCFEC,
   G50_2_NOFEC,
   G40_2_NOFEC,
   G40_4_NOFEC,
   G25_1_RS528,
   G25_1_FCFEC,
   G25_1_NOFEC,
   G10_1_NOFEC,
   G5_1_NOFEC,
   G2P5_1_NOFEC,
   G1_1_NOFEC,
   M100_1_NOFEC,
   M10_1_NOFEC,
)
from .Cores import (
   Blackhawk,
   BlackhawkGen3,
   Falcon,
   GPhy,
   Merlin,
   MerlinQ,
   BabbageCore,
   Tofino2Core,
)

# Each component has a mapping of serdes id to ( coreId, phySide, serdesId, txBool )
class Component( object ):
   """A base class for all components. The behaviour of the components are defined
   by the abstract methods and attributes below.

   Note:
      If the component is initialized in one of the special modes listed in
      HIDDEN_MODES, the mode will not be appended to the class name in
      the getComponentType output.
   """

   __metaclass__ = ABCMeta

   # We consider these modes to be special "class defining" basic modes and thus
   # don't append them to the component type string, see AID7567 for more details
   # TODO: See BUG500705; Remove these
   HIDDEN_MODES = { "Default", "Asic", "Crosspoint" }

   def __init__( self, mode, coreModes=None ):
      self.mode = mode
      if not isinstance( self.supportedModes, set ):
         raise HwL1ComponentError( self, 'supportedModes must be a set.',
                                   supportedModes=self.supportedModes )
      if mode not in self.supportedModes:
         raise HwL1ComponentError( self, 'Invalid component mode.',
                                   mode=mode, supportedModes=self.supportedModes )
      if coreModes:
         raise HwL1ComponentError( self, "Core modes unsupported on basic component",
                                   mode=mode, supportedModes=self.supportedModes )

   @property
   def name( self ):
      "A convienent wrapper for class name"
      return self.__class__.__name__

   def __str__( self ):
      """
      A human-readable output for what the instance of the Component represents.

      If the component is initialized in no mode ( if supported ), we will output
      the type as only '<component name>'.
      """
      return "%s-%s" % ( self.name, self.mode ) if self.mode else self.name

   def getComponentType( self, coreId ):
      """
      Parameters
      -------
      coreId: int
      The core for which to get the type for

      Returns
      -------
      String; a name for l1 topology state machines to use for the core on the chip.
      By convention this is the string conversion of the component.
      
      """
      if self.mode in self.HIDDEN_MODES:
         return self.name
      return str( self )

   def laneRemapCapable( self ):
      """
      Returns if the component can handle lane remaps.
      By default, this will return True
      """
      return True

   @abstractproperty
   def supportedModes( self ):
      "A set describing the modes that this component supports being configured in."
      raise NotImplementedError

   @abstractmethod
   def getSerdes( self, serdesId ):
      """
      Parameters
      -------
      serdesId: int
      The serdes to get information about

      Returns
      -------
      tuple( int, int ). First int is the id of the core on which the input
      serdes is. Second int is the id of the lane for that serdes.
      """
      raise NotImplementedError

   @abstractmethod
   def getNumCores( self ):
      """
      Returns
      -------
      Int; the number of cores in the component
      """
      raise NotImplementedError

   @abstractmethod
   # TODO bug434791: remove default of None
   def getNumSerdesPerCore( self, coreId=None ):
      """
      Parameters
      -------
      coreId: int
      The core for which to return the number of serdes
      During conversion, a default of None is used so each function
      can be converted independently. 

      Returns
      -------
      Int; the number of serdes on the core
      """
      raise NotImplementedError

   # TODO bug434793: convert this to "getPhysicalMapping"
   # Not all classes have a physical mapping. Asics will not override
   # this method
   def addOtherSideSerdes( self, phySide, serdesTopology, coreId ):
      pass

   def getLogicalSerdesMappings( self, coreId ):
      """
      Retrieves the logical SerDes mappings for a specific core.

      Returns
      -------
      An iterable which produces items of type: Serdes.MappingDesc
      The Serdes.MappingDesc iteslf has the format:
         ( Hardware::Phy::SerdesGroupMode: System Group Mode,
           list( int ): System Lanes,
           Hardware::Phy::SerdesGroupMode: Line Group Mode,
           list( int ): Line lanes )

      Group Modes are defined in the SerDes module.

      Note
      ----
      The Group Mode and the number of lanes must be consistent.
      e.g. if System Group Mode is G400_8_RS544, System Lanes must be a type that
      supports indexing with 8 items.

      TODO
      ----
         Make into an abstractmethod once all components are converted to use this
         and BUG429103 is resolved.
      """
      return []

   def getTapGroups( self, coreId ):
      """
      Retrieves the supported Taps for each speed for a specific core.

      Returns
      -------
      A dict of SerdesSpeed to Taps class, where the Taps class has the addTuning
      method defined on it. See the Tuning module for more information/examples.

      TODO
      ----
         Make into an abstractmethod once all components are converted to specify
         this and BUG489453 is resolved.
      """
      return {}

class CoreBasedComponent( Component ):
   """A base class for all core based components. The behaviour of these components
   is derived from the cores they are comprised of.
   """

   CORE_MODE = "Core"

   @abstractproperty
   def cores( self ):
      """A dict from core type to an iterable of core IDs. The core id must be unique
      per component. e.g. { Falcon: [ 0, 3 ], Blackhawk: [ 1, 2 ] }. The component's
      core IDs must start from 0 and be contiguous.
      """
      raise NotImplementedError

   @property
   def supportedModes( self ):
      """
      Defines the supportedModes for a CoreBasedComponent to be the intersection of
      all supportedModes of the component cores, plus the special mode "Core".
      """
      coreModes = set.intersection( *( core.supportedModes for core in self.cores ) )
      return coreModes | { self.CORE_MODE }

   # We will never use this since we override getComponentType in the
   # CoreBasedComponent class to just use the individual core values. The core mode
   # will always be supported from the supportedModes output, so use that to make the
   # parent's __init__ call sane in all cases.
   baseMode = CORE_MODE

   def __init__( self, mode, coreModes=None ):
      """Takes in the mode to initialize the component in. This mode is then applied
      to all the cores defined in the component. If the special mode "Core" is
      specified, however, it also takes a dict of coreId to coreMode that is used to
      initialize each core instead.
      """
      super( CoreBasedComponent, self ).__init__( mode )

      if mode == self.CORE_MODE and coreModes is None:
         raise HwL1ComponentError( self,
                                   'Core mode given but have no per-core modes.',
                                   mode=mode, coreModes=coreModes )
      elif mode != self.CORE_MODE and coreModes is not None:
         raise HwL1ComponentError( self,
                                   'Core mode not given but have per-core modes.',
                                   mode=mode, coreModes=coreModes )

      self.coreMap = {}
      for coreType, coreIds in self.cores.items():
         for coreId in coreIds:
            if coreId in self.coreMap:
               raise HwL1ComponentError( self, 'Duplicate core ID detected.',
                                         coreId=coreId, cores=self.cores )
            coreMode = mode if mode != self.CORE_MODE else coreModes.get( coreId )
            self.coreMap[ coreId ] = coreType( coreMode )

      # Due to the way SerDes ID numberings are generated, core IDs are expected
      # to always start from 0 and be contiguous. In order for this restriction to be
      # relaxed, `serdesInfo` generation would have to be modified to allow the user
      # to specify a SerDes range for each core.
      if ( sorted( self.coreMap.keys() ) !=
           range( max( self.coreMap.keys() ) + 1 ) ):
         raise HwL1ComponentError( self, 'Core ID definitions did not start at 0 or '
                                         'were not contiguous.',
                                         cores=self.cores,
                                         coreIds=sorted( self.coreMap.keys() ) )

      # The serdesInfo is a cached map that allows the component to easily convert
      # from a component indexed SerDes ID to the core and physical lane tuple that
      # it maps to.
      self.serdesInfo = {}
      componentSerdesId = 0
      for coreId, core in sorted( self.coreMap.items() ):
         for coreSerdesId in range( core.getNumSerdes() ):
            if componentSerdesId in self.serdesInfo:
               raise HwL1ComponentLibraryInternalError(
                  'Duplicate SerDes ID detected', component=self, coreId=coreId,
                  coreSerdesId=coreSerdesId, componentSerdesId=componentSerdesId,
                  cores=self.cores, serdesInfo=self.serdesInfo )

            self.serdesInfo[ componentSerdesId ] = ( coreId, coreSerdesId )
            componentSerdesId += 1

   def getSerdes( self, serdesId ):
      return self.serdesInfo[ serdesId ]

   def getNumCores( self ):
      return len( self.coreMap )

   # TODO remove pylint disable once bug434791 is fixed
   def getNumSerdesPerCore( self, coreId ): # pylint: disable=signature-differs
      return self.coreMap[ coreId ].getNumSerdes()

   def getComponentType( self, coreId ):
      return self.coreMap[ coreId ].getComponentType()

   def addOtherSideSerdes( self, phySide, serdesTopology, coreId ):
      # Cores return a mapping. We would like to just return the complete
      # mapping here and let the fru plugin deal with it. For now,
      # lets grab the mapping we need from the core and create it here.

      # The format for the mapping is a dictionary of tuples mapping
      # the system serdes to the line serdes. In most cases there will
      # be only one system serdes
      # { ( system serdes, ... ) : ( line serdes, ... ), ... }
      lane = serdesTopology.physicalLane
      coreMapping = self.coreMap[ coreId ].getPhysicalMapping()

      # Not all components need a mapping across, if no mapping is available
      # then there is nothing to do
      if coreMapping:

         # Get the serdes we map to on the other side
         otherSideLaneMapping = None
         for systemLanes, lineLanes in coreMapping.iteritems():
            if phySide == Tac.Type( "PhyEee::PhySide" ).phySideSystem:
               if lane in systemLanes:
                  otherSideLaneMapping = lineLanes
            else:
               if lane in lineLanes:
                  otherSideLaneMapping = systemLanes
         assert otherSideLaneMapping

         # Add each lane into the other side serdes
         for otherLane in otherSideLaneMapping:
            serdesTopology.otherSideSerdes[ otherLane ] = True

   def getLogicalSerdesMappings( self, coreId ):
      return self.coreMap[ coreId ].getLogicalSerdesMapping()

   def getTapGroups( self, coreId ):
      return self.coreMap[ coreId ].getTapGroups()

class Midplane( Component ):
   supportedModes = { None }

   def getSerdes( self, serdesId ):
      return ( 0, serdesId )

   def getNumCores( self ):
      return 1

   @abstractmethod
   # TODO bug434791: remove default of None
   def getNumSerdesPerCore( self, coreId=None ):
      raise NotImplementedError

   def addOtherSideSerdes( self, phySide, serdesTopology, coreId ):
      # This makes no sense for a midplane, need to investigate restructuring
      pass

   def getLogicalSerdesMappings( self, coreId ):
      # This makes no sense for a midplane, need to investigate restructuring
      pass

   def getTapGroups( self, coreId ):
      # This makes no sense for a midplane, need to investigate restructuring
      pass

class ExternalPhy( Component ):
   # TODO: See BUG500705; Remove Default
   supportedModes = { None, "Default" }

   @abstractmethod
   def getSerdes( self, serdesId ):
      raise NotImplementedError

   @abstractmethod
   def getNumCores( self ):
      raise NotImplementedError

   @abstractmethod
   # TODO bug434791: remove default of None
   def getNumSerdesPerCore( self, coreId=None ):
      raise NotImplementedError

# Component Registration System
registeredComponents = {}

def registerHwL1Component( componentCls ):
   """A utility function used to register components so that the Hw L1 Topology FRU
   plugin ( as well as other external consumers ) can be made aware of them.

   Args
   ----
      componentCls:  The component class to register.

   Returns
   -------
      The componentCls.

   Raises
   ------
      HwL1ComponentLibraryError if the componentCls does not subclass Component.
   """
   name = componentCls.__name__

   if not issubclass( componentCls, Component ):
      raise HwL1ComponentLibraryError( 'Attempted to register a component that does '
                                       'not subclass from the base Component class.',
                                       offendingComponentCls=componentCls )
   if name in registeredComponents:
      raise HwL1ComponentLibraryError( 'Attempted to register multiple component '
                                       'classes with the same name.',
                                       name=name,
                                       existingComp=registeredComponents[ name ],
                                       offendingComponentCls=componentCls)

   registeredComponents[ name ] = componentCls
   return componentCls

@registerHwL1Component
class Sliver64( Midplane ):
   # TODO bug434791: remove default of None
   def getNumSerdesPerCore( self, coreId=None ):
      return 64

@registerHwL1Component
class Jericho2( CoreBasedComponent ):
   cores = {
      Blackhawk : range( 0, 12 ),
   }

@registerHwL1Component
class Jericho2C( CoreBasedComponent ):
   cores = {
      Blackhawk : range( 0, 4 ),
      Falcon : range( 4, 28 ),
   }

@registerHwL1Component
class Qumran2A( CoreBasedComponent ):
   cores = {
         Blackhawk : range( 0, 2 ),
         Falcon : range( 2, 11 )
   }

@registerHwL1Component
class Tofino2( CoreBasedComponent ):
   cores = {
      Tofino2Core : range( 0, 16 ),
   }

@registerHwL1Component
class Ds280df810( ExternalPhy ):
   # Note that RetimerA, RetimerB, etc. may physically be any of the 8 retimers
   # on the chip.  This is determined by board routing and will be resolved during
   # L1 topo lane mapping
   #
   #               +-------------------+
   #     serdes    |    Ds280df810     |    serdes
   #               |---- RetimerA ---->|----  0
   #       0   ----|<--- RetimerB -----|
   #               |---- RetimerC ---->|----  1
   #       1   ----|<--- RetimerD -----|
   #               |---- RetimerE ---->|----  2
   #       2   ----|<--- RetimerF -----|
   #               |---- RetimerG ---->|----  3
   #       3   ----|<--- RetimerH -----|
   #               +-------------------+

   def getSerdes( self, serdesId ):
      return ( 0, serdesId )

   def getNumCores( self ):
      return 1

   def getNumSerdesPerCore( self, coreId=None ):
      return 4

   def addOtherSideSerdes( self, phySide, serdesTopology, coreId ):
      lane = serdesTopology.physicalLane
      serdesTopology.otherSideSerdes[ lane ] = True

   def getLogicalSerdesMappings( self, coreId ):
      return [
         MappingDesc( G10_1_NOFEC, [ 0 ], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], G10_1_NOFEC, [ 1 ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], G10_1_NOFEC, [ 2 ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], G10_1_NOFEC, [ 3 ] ),
         MappingDesc( G25_1_NOFEC, [ 0 ], G25_1_NOFEC, [ 0 ] ),
         MappingDesc( G25_1_NOFEC, [ 1 ], G25_1_NOFEC, [ 1 ] ),
         MappingDesc( G25_1_NOFEC, [ 2 ], G25_1_NOFEC, [ 2 ] ),
         MappingDesc( G25_1_NOFEC, [ 3 ], G25_1_NOFEC, [ 3 ] ),
         MappingDesc( G25_1_FCFEC, [ 0 ], G25_1_FCFEC, [ 0 ] ),
         MappingDesc( G25_1_FCFEC, [ 1 ], G25_1_FCFEC, [ 1 ] ),
         MappingDesc( G25_1_FCFEC, [ 2 ], G25_1_FCFEC, [ 2 ] ),
         MappingDesc( G25_1_FCFEC, [ 3 ], G25_1_FCFEC, [ 3 ] ),
         MappingDesc( G25_1_RS528, [ 0 ], G25_1_RS528, [ 0 ] ),
         MappingDesc( G25_1_RS528, [ 1 ], G25_1_RS528, [ 1 ] ),
         MappingDesc( G25_1_RS528, [ 2 ], G25_1_RS528, [ 2 ] ),
         MappingDesc( G25_1_RS528, [ 3 ], G25_1_RS528, [ 3 ] ),
      ]

@registerHwL1Component
class BabbageMux( Component ):
   #               +-------------------+
   #               |      Babbage      |
   #     serdes    | +---------------+ |    serdes
   #       0   ----| |0             0| |----  0
   #       1   ----| |1             1| |----  1
   #       2   ----| |2             2| |----  2
   #       3   ----| |3             3| |----  3
   #       4   ----| |4   Core 1    4| |----  4
   #       5   ----| |5             5| |----  5
   #       6   ----| |6             6| |----  6
   #       7   ----| |7             7| |----  7
   #               | +---------------+ |
   #               | +---------------+ |
   #       8   ----| |8             8| |----  8
   #       9   ----| |9             9| |----  9
   #       10  ----| |10           10| |----  10
   #       11  ----| |11           11| |----  11
   #       12  ----| |12  Core 2   12| |----  12
   #       13  ----| |13           13| |----  13
   #       14  ----| |14           14| |----  14
   #       15  ----| |15           15| |----  15
   #               | +---------------+ |
   #               +-------------------+

   supportedModes = { "Default" }
   # This version of the babbage component is for cases where both
   # cores need to be considered together and can not be split up
   # independently the way the standard babbage class does things.
   # Currently this is only used for Tambo, which is not shipping.

   def getSerdes( self, serdesId ):
      # For Babbage Mux, we will assemble one babbage core in
      # retimer mode from the two Mux cores by putting all
      # 8 serdes on each side onto one core.
      #
      # For Mux mode the serdes are connected across the
      # chip as follows
      # SIDE A         SIDE B
      # 0 - 3   <--->  4 - 7
      # 8 - 11  <--->  12 - 15
      # We can just pretend the serdes on each side are numbered 0 - 7
      # Since this maintains the connections between the sides of the chip
      # and we do not care about the serdes groups used on Babbage this
      # will be ok.
      physicalLaneId = None
      if serdesId in range( 0, 4 ):
         physicalLaneId = serdesId
      elif serdesId in range( 4, 8 ):
         physicalLaneId = serdesId - 4
      elif serdesId in range( 8, 12 ):
         physicalLaneId = serdesId - 4
      elif serdesId in range( 12, 16 ):
         physicalLaneId = serdesId - 8
      else:
         assert False, "Invalid serdesId %s" % serdesId

      return ( 0, physicalLaneId )

   def getNumCores( self ):
      return 2

   def getComponentType( self, coreId ):
      # Temporarily Mux mode will spoof retimer mode by setting up
      # the serdes to match a retimer mode babbage
      return "crt50216rt"

   def getNumSerdesPerCore( self, coreId=None ):
      return 8

   def addOtherSideSerdes( self, phySide, serdesTopology, coreId ):
      lane = serdesTopology.physicalLane
      # Map will be:
      # A[ 0-7 ]: B[ 0-7 ]
      serdesTopology.otherSideSerdes[ lane ] = True

@registerHwL1Component
class Babbage( CoreBasedComponent ):

   #               +-------------------+
   #               |      Babbage      |
   #     serdes    | +---------------+ |    serdes
   #       0   ----| |0             0| |----  0
   #       1   ----| |1             1| |----  1
   #       2   ----| |2             2| |----  2
   #       3   ----| |3             3| |----  3
   #       4   ----| |4   Core 1    4| |----  4
   #       5   ----| |5             5| |----  5
   #       6   ----| |6             6| |----  6
   #       7   ----| |7             7| |----  7
   #               | +---------------+ |
   #               | +---------------+ |
   #       8   ----| |8             8| |----  8
   #       9   ----| |9             9| |----  9
   #       10  ----| |10           10| |----  10
   #       11  ----| |11           11| |----  11
   #       12  ----| |12  Core 2   12| |----  12
   #       13  ----| |13           13| |----  13
   #       14  ----| |14           14| |----  14
   #       15  ----| |15           15| |----  15
   #               | +---------------+ |
   #               +-------------------+

   cores = {
      BabbageCore: range( 0, 2 ),
   }

# TODO all this is wrong, but here for proof of concept
@registerHwL1Component
class B52( ExternalPhy ):

   #               +-------------------+
   #               |        B52        |
   #     serdes    | +---------------+ |    serdes
   #               | |             12| |----  12
   #               | |             13| |----  13
   #               | |             14| |----  14
   #               | |             15| |----  15
   #        0  ----| |0            16| |----  16
   #        1  ----| |1            17| |----  17
   #        2  ----| |2            18| |----  18
   #        3  ----| |3            19| |----  19
   #               | |     Die 0     | |        
   #        4  ----| |4            20| |----  20
   #        5  ----| |5            21| |----  21
   #        6  ----| |6            22| |----  22
   #        7  ----| |7            23| |----  23
   #               | |              8| |----  8
   #               | |              9| |----  9
   #               | |             10| |----  10
   #               | |             11| |----  11
   #               | +---------------+ |
   #               +-------------------+

   supportedModes = { "Gearbox", "Retimer" }

   def getSerdes( self, serdesId ):
      physicalLaneId = serdesId
      return ( 0, physicalLaneId ) # ( dieId, laneId )

   def getNumCores( self ):
      return 1

   def getComponentType( self, coreId ):
      # TODO change these names to match <component>-<core>
      if self.mode == "Gearbox":
         return "cms42550gb"
      elif self.mode == "Retimer":
         return "cms42550rt"
      assert False, "invalid mode %s" % self.mode
      return "unknown"

   def getNumSerdesPerCore( self, coreId=None ):
      return 8

   def addOtherSideSerdes( self, phySide, serdesTopology, coreId ):
      if self.mode == "Gearbox":
         # Map will be:
         # A0 : B12, B13
         # A1 : B14, B15
         # A2 : B16, B17
         # A3 : B18, B19
         # A4 : B20, B21
         # A5 : B22, B23
         # A6 : A8,  A9
         # A7 : A10, A11
         systemToLineLaneMap = {
               0 : ( 12, 13 ),
               1 : ( 14, 15 ),
               2 : ( 16, 17 ),
               3 : ( 18, 19 ),
               4 : ( 20, 21 ),
               5 : ( 22, 23 ),
               6 : ( 8, 9 ),
               7 : ( 10, 11 ) }
         lineToSystemLaneMap = {
               12 : 0, 13 : 0,
               14 : 1, 15 : 1,
               16 : 2, 17 : 2,
               18 : 3, 19 : 3,
               20 : 4, 21 : 4,
               22 : 5, 23 : 5,
               8 : 6, 9 : 6,
               10 : 7, 11 : 7 }
         lane = serdesTopology.physicalLane
         if phySide == Tac.Type( "PhyEee::PhySide" ).phySideSystem:
            otherSideLanes = systemToLineLaneMap[ lane ]
            serdesTopology.otherSideSerdes[ otherSideLanes[ 0 ] ] = True
            serdesTopology.otherSideSerdes[ otherSideLanes[ 1 ] ] = True
         else:
            serdesTopology.otherSideSerdes[ lineToSystemLaneMap[ lane ] ] = True
      elif self.mode == "Retimer":
         # Map will be:
         # A0 : B12
         # A1 : B13
         # A2 : B14
         # A3 : B15
         # A4 : B16
         # A5 : B17
         # A6 : B18
         # A7 : B19
         lane = serdesTopology.physicalLane
         otherLane = lane - 12 # Line to System, aka B to A
         if phySide == Tac.Type( "PhyEee::PhySide" ).phySideSystem:
            otherLane = lane + 12 # System to Line, aka A to B
         serdesTopology.otherSideSerdes[ otherLane ] = True
      else:
         assert False, "invalid mode %s" % self.mode

   def getLogicalSerdesMappings( self, coreId ):
      mappings = []
      if self.mode == "Gearbox":
         # four lane gearbox modes
         for x in range( 0, 8, 2 ):
            sysIntfs = [ x, x + 1 ]
            lineIntfs = range( 2 * x, 2 * x + 4 )
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysIntfs, G100_4_RS528, lineIntfs ),
               MappingDesc( G100_2_RS544, sysIntfs, G100_4_NOFEC, lineIntfs ),
               MappingDesc( G40_2_NOFEC, sysIntfs, G40_4_NOFEC, lineIntfs ),
            ] )
         # two lane gearbox modes
         for x in range( 0, 8 ):
            lineIntfs = [ x * 2, x * 2 + 1 ]
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ x ], G50_2_RS528, lineIntfs ),
               MappingDesc( G50_1_RS544, [ x ], G50_2_FCFEC, lineIntfs ),
               MappingDesc( G50_1_RS544, [ x ], G50_2_NOFEC, lineIntfs ),
            ] )
         # single lane modes
         for x in range( 8 ):
            lineSerdes = x if x < 4 else x + 4
            mappings.extend( [
               MappingDesc( G25_1_NOFEC, [ x ], G25_1_RS528, [ lineSerdes ] ),
               MappingDesc( G25_1_NOFEC, [ x ], G25_1_FCFEC, [ lineSerdes ] ),
               MappingDesc( G25_1_NOFEC, [ x ], G25_1_NOFEC, [ lineSerdes ] ),
               MappingDesc( G10_1_NOFEC, [ x ], G10_1_NOFEC, [ lineSerdes ] ),
            ] )
      elif self.mode == "Retimer":
         # eight lane modes
         mappings.extend( [
            MappingDesc( G400_8_RS544, range( 8 ), G400_8_RS544, range( 8 ) ),
            ] )
         # four lane modes
         for x in ( range( 0, 4 ), range( 4, 8 ) ): 
            mappings.extend( [
               MappingDesc( G200_4_RS544, x, G200_4_RS544, x ),
               MappingDesc( G100_4_NOFEC, x, G100_4_NOFEC, x ),
               MappingDesc( G100_4_NOFEC, x, G100_4_RS528, x ),
               MappingDesc( G40_4_NOFEC, x, G40_4_NOFEC, x ),
            ] )
         # two lane modes
         for x in range( 0, 8, 2 ):
            intfs = [ x, x + 1 ]
            mappings.extend( [
               MappingDesc( G100_2_RS544, intfs, G100_2_RS544, intfs ),
               MappingDesc( G50_2_NOFEC, intfs, G50_2_RS528, intfs ),
               MappingDesc( G50_2_NOFEC, intfs, G50_2_FCFEC, intfs ),
               MappingDesc( G50_2_NOFEC, intfs, G50_2_NOFEC, intfs ),
            ] )
         # single lane modes
         for x in range( 8 ):
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ x ], G50_1_RS544, [ x ] ),
               MappingDesc( G25_1_NOFEC, [ x ], G25_1_RS528, [ x ] ),
               MappingDesc( G25_1_NOFEC, [ x ], G25_1_FCFEC, [ x ] ),
               MappingDesc( G25_1_NOFEC, [ x ], G25_1_NOFEC, [ x ] ),
               MappingDesc( G10_1_NOFEC, [ x ], G10_1_NOFEC, [ x ] ),
            ] )
      else:
         assert False, "invalid mode %s" % self.mode
      return mappings

@registerHwL1Component
class Enigma( ExternalPhy ):

   def getSerdes( self, serdesId ):
      physicalLaneId = serdesId % self.getNumSerdesPerCore()
      serdesCoreId = serdesId // self.getNumSerdesPerCore()
      return ( serdesCoreId, physicalLaneId )

   def addOtherSideSerdes( self, phySide, serdesTopology, coreId ):
      # Map will be:
      # A[ 0-7 ]: B[ 0-7 ]
      lane = serdesTopology.physicalLane
      serdesTopology.otherSideSerdes[ lane ] = True

   def getNumCores( self ):
      return 2

   def getComponentType( self, coreId ):
      return "cmx42550"

   def getNumSerdesPerCore( self, coreId=None ):
      return 4

   def getLogicalSerdesMappings( self, coreId ):
      mappings = []
      # four lane modes
      intfs = range( 4 )
      mappings.extend( [
         MappingDesc( G100_4_NOFEC, intfs, G100_4_RS528, intfs ),
         MappingDesc( G100_4_NOFEC, intfs, G100_4_NOFEC, intfs ),
         MappingDesc( G40_4_NOFEC, intfs, G40_4_NOFEC, intfs ),
      ] )
      return mappings

@registerHwL1Component
class Evora( ExternalPhy ):

   def getSerdes( self, serdesId ):
      physicalLaneId = serdesId % self.getNumSerdesPerCore()
      serdesCoreId = serdesId // self.getNumSerdesPerCore()
      return ( serdesCoreId, physicalLaneId )

   def addOtherSideSerdes( self, phySide, serdesTopology, coreId ):
      lane = serdesTopology.physicalLane
      serdesTopology.otherSideSerdes[ lane ] = True

   def getNumCores( self ):
      return 2

   def getComponentType( self, coreId ):
      return "bcm82391"

   def getNumSerdesPerCore( self, coreId=None ):
      return 4

@registerHwL1Component
class Millenio( Component ):

   #               +-------------------+
   #               |      Millenio     |
   #     serdes    | +---------------+ |    serdes
   #       0   ----| |0             0| |----  0
   #       1   ----| |1             1| |----  1
   #       2   ----| |2             2| |----  2
   #       3   ----| |3             3| |----  3
   #       4   ----| |4             4| |----  4
   #       5   ----| |5             5| |----  5
   #       6   ----| |6             6| |----  6
   #       7   ----| |7             7| |----  7
   #               | |     Core 0    | |
   #       8   ----| |8             8| |----  8
   #       9   ----| |9             9| |----  9
   #       10  ----| |10           10| |----  10
   #       11  ----| |11           11| |----  11
   #       12  ----| |12           12| |----  12
   #       13  ----| |13           13| |----  13
   #       14  ----| |14           14| |----  14
   #       15  ----| |15           15| |----  15
   #               | +---------------+ |
   #               +-------------------+

   supportedModes = { "Gearbox", "Mux" }

   def getComponentType( self, coreId ):
      if self.mode == "Gearbox":
         return "bcm81356-gb"
      elif self.mode == "Mux":
         return "bcm81356-mux"
      assert False, "Invalid mode: %s" % self.mode
      return "unknown"

   def getSerdes( self, serdesId ):
      # Millenio has a single "core" represented by a unique SDK phy_addr
      # Note: this shouldn't be confused with physical cores which Millenio has four
      coreId = serdesId // self.getNumSerdesPerCore()
      assert coreId == 0, "Invalid coreId: %d" % coreId
      physicalLaneId = serdesId % self.getNumSerdesPerCore()
      return ( coreId, physicalLaneId )

   def getNumSerdesPerCore( self, coreId=None ):
      return 16

   def getNumCores( self ):
      return 1

   def getLogicalSerdesMappings( self, coreId ):
      mappings = []
      if self.mode == "Gearbox":
         # four lane modes
         for ( sysLanes, lineLanes ) in (
            ( ( 0, 1 ), range( 0, 4 ) ),
            ( ( 2, 3 ), range( 4, 8 ) ),
            ( ( 4, 5 ), range( 8, 12 ) ),
            ( ( 6, 7 ), range( 12, 16 ) ), ):
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysLanes, G100_4_RS528, lineLanes ),
               MappingDesc( G100_2_RS544, sysLanes, G100_4_NOFEC, lineLanes ),
               MappingDesc( G40_2_NOFEC, sysLanes, G40_4_NOFEC, lineLanes ),
            ] )
         # two lane modes
         for sysLane in range( 8 ):
            lineLanes = [ sysLane * 2, sysLane * 2 + 1 ]
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ sysLane ], G50_2_RS528, lineLanes ),
               MappingDesc( G50_1_RS544, [ sysLane ], G50_2_FCFEC, lineLanes ),
               MappingDesc( G50_1_RS544, [ sysLane ], G50_2_NOFEC, lineLanes ),
            ] )
         # single lane modes
         for lane in range( 4 ):
            mappings.extend( [
               MappingDesc( G25_1_RS528, [ lane ], G25_1_RS528, [ lane ] ),
               MappingDesc( G25_1_FCFEC, [ lane ], G25_1_FCFEC, [ lane ] ),
               MappingDesc( G25_1_NOFEC, [ lane ], G25_1_NOFEC, [ lane ] ),
               MappingDesc( G10_1_NOFEC, [ lane ], G10_1_NOFEC, [ lane ] ),
            ] )
         for ( sysLane, lineLane ) in zip( range( 4, 8 ), range( 8, 12 ) ):
            mappings.extend( [
               MappingDesc( G25_1_RS528, [ sysLane ], G25_1_RS528, [ lineLane ] ),
               MappingDesc( G25_1_FCFEC, [ sysLane ], G25_1_FCFEC, [ lineLane ] ),
               MappingDesc( G25_1_NOFEC, [ sysLane ], G25_1_NOFEC, [ lineLane ] ),
               MappingDesc( G10_1_NOFEC, [ sysLane ], G10_1_NOFEC, [ lineLane ] ),
            ] )
      elif self.mode == "Mux":
         # Mux mode is a mode with 4 line serdes unused, 4 line serdes straight
         # through and 8 line serdes in a mux such that only the first or latter 4
         # can be active.
         # Mux mode on Millenio is unique in that there are multiple mappings for
         # groups of line side serdes to the same group of system side serdes. This
         # is a result of treating each Millenio chip as a single core and
         # "crossing" normal 8 serdes core boundaries. This also implies that
         # performing l1 topology traversal from the system side would be impossible
         # since multiple front panel interfaces can potential map to the same set
         # of system side serdes for a particular speed.

         # eight lane modes
         mappings.extend( [
            MappingDesc( G400_8_RS544, range( 8 ), G400_8_RS544, range( 4, 12 ) ),
         ] )
         # four lane PAM4 modes
         mappings.extend( [
            MappingDesc( G200_4_RS544, range( 4, 8 ), G200_4_RS544, range( 8, 12 ) ),
         ] )
         # four lane NRZ modes
         mappings.extend( [
            MappingDesc( G100_4_RS528, range( 4 ), G100_4_RS528, range( 4 ) ),
            MappingDesc( G100_4_RS528, range( 4 ), G100_4_RS528, range( 4, 8 ) ),
            MappingDesc( G100_4_RS528, range( 4, 8 ), G100_4_RS528, range( 8, 12 ) ),
         ] )
         # two lane modes
         for lane in range( 0, 4, 2 ):
            sysLanes = [ lane, lane + 1 ]
            lineLanes = sysLanes
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysLanes, G100_2_RS544, lineLanes ),
               MappingDesc( G50_2_RS528, sysLanes, G50_2_RS528, lineLanes ),
               MappingDesc( G50_2_FCFEC, sysLanes, G50_2_FCFEC, lineLanes ),
               MappingDesc( G50_2_NOFEC, sysLanes, G50_2_NOFEC, lineLanes ),
            ] )
            lineLanes = [ 2 * lane, 2 * lane + 1 ]
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysLanes, G100_2_RS544, lineLanes ),
               MappingDesc( G50_2_RS528, sysLanes, G50_2_RS528, lineLanes ),
               MappingDesc( G50_2_FCFEC, sysLanes, G50_2_FCFEC, lineLanes ),
               MappingDesc( G50_2_NOFEC, sysLanes, G50_2_NOFEC, lineLanes ),
            ] )
         for lane in range( 4, 8, 2 ):
            sysLanes = [ lane, lane + 1 ]
            lineLanes = [ 2 * lane, 2 * lane + 1 ]
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysLanes, G100_2_RS544, lineLanes ),
               MappingDesc( G50_2_RS528, sysLanes, G50_2_RS528, lineLanes ),
               MappingDesc( G50_2_FCFEC, sysLanes, G50_2_FCFEC, lineLanes ),
               MappingDesc( G50_2_NOFEC, sysLanes, G50_2_NOFEC, lineLanes ),
            ] )
         # single lane modes
         for ( sysLane, lineLane ) in ( zip( 2 * range( 4 ), range( 8 ) ) +
                                        zip( range( 4, 8 ), range( 8, 12 ) ) ):
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ sysLane ], G50_1_RS544, [ lineLane ] ),
               MappingDesc( G25_1_RS528, [ sysLane ], G25_1_RS528, [ lineLane ] ),
               MappingDesc( G25_1_FCFEC, [ sysLane ], G25_1_FCFEC, [ lineLane ] ),
               MappingDesc( G25_1_NOFEC, [ sysLane ], G25_1_NOFEC, [ lineLane ] ),
               MappingDesc( G10_1_NOFEC, [ sysLane ], G10_1_NOFEC, [ lineLane ] ),
            ] )
      else:
         assert False, "Invalid mode: %s" % self.mode
      return mappings

   def addOtherSideSerdes( self, phySide, serdesTopology, coreId ):
      lane = serdesTopology.physicalLane
      base = ( lane // 8 ) * 8
      offset = lane % 4
      if self.mode == "Gearbox":
         if phySide == Tac.Type( "PhyEee::PhySide" ).phySideSystem:
            serdesTopology.otherSideSerdes[ base + offset * 2 ] = True
            serdesTopology.otherSideSerdes[ base + ( offset * 2 ) + 1 ] = True
         else:
            serdesTopology.otherSideSerdes[ base + ( lane % 8 ) // 2 ] = True
      elif self.mode == "Mux":
         if phySide == Tac.Type( "PhyEee::PhySide" ).phySideSystem:
            serdesTopology.otherSideSerdes[ base + offset ] = True
            if lane < 8:
               serdesTopology.otherSideSerdes[ base + offset + 4 ] = True
         else:
            serdesTopology.otherSideSerdes[ base + offset ] = True
      else:
         assert False, "unsupported Millenio mode"

@registerHwL1Component
class T3X2( CoreBasedComponent ):
   cores = {
      GPhy : range( 0, 6 ),
      MerlinQ : range( 6, 8 ),
      Falcon : [ 8 ],
      Merlin : [ 9 ],
   }

@registerHwL1Component
class T3X3( CoreBasedComponent ):
   cores = {
      MerlinQ : range( 0, 3 ),
      Merlin : range( 3, 7 ),
      Falcon : range( 7, 10 ),
   }

@registerHwL1Component
class T3X4( CoreBasedComponent ):
   cores = {
      Falcon : range( 0, 17 ),
   }

@registerHwL1Component
class T3X5( CoreBasedComponent ):
   cores = {
      Falcon : range( 0, 20 )
   }

@registerHwL1Component
class TH4( CoreBasedComponent ):
   cores = {
      BlackhawkGen3 : range( 0, 64 )
   }

@registerHwL1Component
class Firelight( CoreBasedComponent ):
   cores = {
      MerlinQ : range( 0, 3 ),
      Falcon : range( 3, 7 ),
   }

@registerHwL1Component
class BCM54998E( ExternalPhy ):
   # Phy that splits 2 system serdes at 10G into 8 line side serdes at 2.5G

   #  system   +-------------------+    line
   #  serdes   |     BCM54998E     |   serdes
   #           |                   |    
   #    0  ____|___________________|____  0
   #           |        |__________|____  1
   #           |        |__________|____  2
   #           |        |__________|____  3
   #           |                   |
   #    5  ____|___________________|____  4
   #           |        |__________|____  5
   #           |        |__________|____  6
   #           |        |__________|____  7
   #           |                   |
   #           +-------------------+

   def getSerdes( self, serdesId ):
      physicalLaneId = serdesId % self.getNumSerdesPerCore()
      coreId = serdesId // self.getNumSerdesPerCore()
      return ( coreId, physicalLaneId )

   def getNumCores( self ):
      return 1

   def getNumSerdesPerCore( self, coreId=None ):
      return 8
   
   def addOtherSideSerdes( self, phySide, serdesTopology, coreId ):
      lane = serdesTopology.physicalLane
      if phySide == Tac.Type( "PhyEee::PhySide" ).phySideSystem:
         if lane == 0:
            serdesTopology.otherSideSerdes[ 0 ] = True
            serdesTopology.otherSideSerdes[ 1 ] = True
            serdesTopology.otherSideSerdes[ 2 ] = True
            serdesTopology.otherSideSerdes[ 3 ] = True
         elif lane == 5:
            serdesTopology.otherSideSerdes[ 4 ] = True
            serdesTopology.otherSideSerdes[ 5 ] = True
            serdesTopology.otherSideSerdes[ 6 ] = True
            serdesTopology.otherSideSerdes[ 7 ] = True
      else:
         if lane >= 0 and lane <= 3:
            serdesTopology.otherSideSerdes[ 0 ] = True
         elif lane >= 4 and lane <= 7:
            serdesTopology.otherSideSerdes[ 5 ] = True

   def getLogicalSerdesMappings( self, coreId ):
      # Mux 4x <=2.5G Base-T signals onto a shared USXGMII signal ( 10G serdes )
      # TODO: See BUG458080; Convert these mappings to specify serdes rate rather
      #       than the data rate once channels are supported.
      return [
         MappingDesc( G2P5_1_NOFEC, [ 0 ], G2P5_1_NOFEC, [ 0 ] ),
         MappingDesc( G2P5_1_NOFEC, [ 0 ], G2P5_1_NOFEC, [ 1 ] ),
         MappingDesc( G2P5_1_NOFEC, [ 0 ], G2P5_1_NOFEC, [ 2 ] ),
         MappingDesc( G2P5_1_NOFEC, [ 0 ], G2P5_1_NOFEC, [ 3 ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( M100_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 0 ] ),
         MappingDesc( M100_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 1 ] ),
         MappingDesc( M100_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 2 ] ),
         MappingDesc( M100_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 3 ] ),
         MappingDesc( G2P5_1_NOFEC, [ 1 ], G2P5_1_NOFEC, [ 4 ] ),
         MappingDesc( G2P5_1_NOFEC, [ 1 ], G2P5_1_NOFEC, [ 5 ] ),
         MappingDesc( G2P5_1_NOFEC, [ 1 ], G2P5_1_NOFEC, [ 6 ] ),
         MappingDesc( G2P5_1_NOFEC, [ 1 ], G2P5_1_NOFEC, [ 7 ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 4 ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 5 ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 6 ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 7 ] ),
         MappingDesc( M100_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 4 ] ),
         MappingDesc( M100_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 5 ] ),
         MappingDesc( M100_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 6 ] ),
         MappingDesc( M100_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 7 ] ),
      ]

@registerHwL1Component
class BCM54182( ExternalPhy ):
   #  system   +-------------------+    line
   #  serdes   |     BCM54182      |   serdes
   #           |                   |
   #    0  ____|___________________|____  0
   #           |        |__________|____  1
   #           |        |__________|____  2
   #           |        |__________|____  3
   #           |                   |
   #    1  ____|___________________|____  4
   #           |        |__________|____  5
   #           |        |__________|____  6
   #           |        |__________|____  7
   #           |                   |
   #           +-------------------+

   def getSerdes( self, serdesId ):
      physicalLaneId = serdesId % self.getNumSerdesPerCore()
      coreId = serdesId // self.getNumSerdesPerCore()
      return ( coreId, physicalLaneId )

   def getNumCores( self ):
      return 1

   def getNumSerdesPerCore( self, coreId=None ):
      return 8

   def addOtherSideSerdes( self, phySide, serdesTopology, coreId ):
      lane = serdesTopology.physicalLane
      if phySide == Tac.Type( "PhyEee::PhySide" ).phySideSystem:
         if lane == 0:
            serdesTopology.otherSideSerdes[ 0 ] = True
            serdesTopology.otherSideSerdes[ 1 ] = True
            serdesTopology.otherSideSerdes[ 2 ] = True
            serdesTopology.otherSideSerdes[ 3 ] = True
         elif lane == 1:
            serdesTopology.otherSideSerdes[ 4 ] = True
            serdesTopology.otherSideSerdes[ 5 ] = True
            serdesTopology.otherSideSerdes[ 6 ] = True
            serdesTopology.otherSideSerdes[ 7 ] = True
      else:
         if lane >= 0 and lane <= 3:
            serdesTopology.otherSideSerdes[ 0 ] = True
         elif lane >= 4 and lane <= 7:
            serdesTopology.otherSideSerdes[ 1 ] = True

   def getLogicalSerdesMappings( self, coreId ):
      # Mux 4x <=1G Base-T signals onto a single QSGMII signal ( 5G serdes )
      # TODO: See BUG458080; Convert these mappings to specify serdes rate rather
      #       than the data rate once channels are supported.
      return [
         MappingDesc( G1_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( M100_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 0 ] ),
         MappingDesc( M100_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 1 ] ),
         MappingDesc( M100_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 2 ] ),
         MappingDesc( M100_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 3 ] ),
         MappingDesc( M10_1_NOFEC, [ 0 ], M10_1_NOFEC, [ 0 ] ),
         MappingDesc( M10_1_NOFEC, [ 0 ], M10_1_NOFEC, [ 1 ] ),
         MappingDesc( M10_1_NOFEC, [ 0 ], M10_1_NOFEC, [ 2 ] ),
         MappingDesc( M10_1_NOFEC, [ 0 ], M10_1_NOFEC, [ 3 ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 4 ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 5 ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 6 ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 7 ] ),
         MappingDesc( M100_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 4 ] ),
         MappingDesc( M100_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 5 ] ),
         MappingDesc( M100_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 6 ] ),
         MappingDesc( M100_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 7 ] ),
         MappingDesc( M10_1_NOFEC, [ 1 ], M10_1_NOFEC, [ 4 ] ),
         MappingDesc( M10_1_NOFEC, [ 1 ], M10_1_NOFEC, [ 5 ] ),
         MappingDesc( M10_1_NOFEC, [ 1 ], M10_1_NOFEC, [ 6 ] ),
         MappingDesc( M10_1_NOFEC, [ 1 ], M10_1_NOFEC, [ 7 ] ),
      ]

@registerHwL1Component
class BCM84898( ExternalPhy ):
   # 10GBASE-T Phy

   def getSerdes( self, serdesId ):
      physicalLaneId = serdesId % self.getNumSerdesPerCore()
      coreId = serdesId // self.getNumSerdesPerCore()
      return ( coreId, physicalLaneId )

   def getNumCores( self ):
      return 1

   def getNumSerdesPerCore( self, coreId=None ):
      return 8
   
   def addOtherSideSerdes( self, phySide, serdesTopology, coreId ):
      lane = serdesTopology.physicalLane
      serdesTopology.otherSideSerdes[ lane ] = True

   def getLogicalSerdesMappings( self, coreId ):
      # Passthrough of XFI with idle stuffing for >1G ( 10G Serdes ), or SGMII with
      # symbol replication for <=1G ( 1G Serdes )
      # TODO: We can technically also do USXGMII on these phys, but I don't see a
      #       reason we'd wire up one of these for that vs a BCM54899E
      # TODO: See BUG458080; Convert these mappings to specify serdes rate rather
      #       than the data rate once channels are supported.
      return [
         MappingDesc( G10_1_NOFEC, [ 0 ], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], G5_1_NOFEC, [ 0 ] ),
         MappingDesc( G2P5_1_NOFEC, [ 0 ], G2P5_1_NOFEC, [ 0 ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( M100_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 0 ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], G10_1_NOFEC, [ 1 ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], G5_1_NOFEC, [ 1 ] ),
         MappingDesc( G2P5_1_NOFEC, [ 1 ], G2P5_1_NOFEC, [ 1 ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( M100_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 1 ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], G10_1_NOFEC, [ 2 ] ),
         MappingDesc( G5_1_NOFEC, [ 2 ], G5_1_NOFEC, [ 2 ] ),
         MappingDesc( G2P5_1_NOFEC, [ 2 ], G2P5_1_NOFEC, [ 2 ] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( M100_1_NOFEC, [ 2 ], M100_1_NOFEC, [ 2 ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], G10_1_NOFEC, [ 3 ] ),
         MappingDesc( G5_1_NOFEC, [ 3 ], G5_1_NOFEC, [ 3 ] ),
         MappingDesc( G2P5_1_NOFEC, [ 3 ], G2P5_1_NOFEC, [ 3 ] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( M100_1_NOFEC, [ 3 ], M100_1_NOFEC, [ 3 ] ),
         MappingDesc( G10_1_NOFEC, [ 4 ], G10_1_NOFEC, [ 4 ] ),
         MappingDesc( G5_1_NOFEC, [ 4 ], G5_1_NOFEC, [ 4 ] ),
         MappingDesc( G10_1_NOFEC, [ 4 ], G2P5_1_NOFEC, [ 4 ] ),
         MappingDesc( G1_1_NOFEC, [ 4 ], G1_1_NOFEC, [ 4 ] ),
         MappingDesc( M100_1_NOFEC, [ 4 ], M100_1_NOFEC, [ 4 ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], G10_1_NOFEC, [ 5 ] ),
         MappingDesc( G5_1_NOFEC, [ 5 ], G5_1_NOFEC, [ 5 ] ),
         MappingDesc( G2P5_1_NOFEC, [ 5 ], G2P5_1_NOFEC, [ 5 ] ),
         MappingDesc( G1_1_NOFEC, [ 5 ], G1_1_NOFEC, [ 5 ] ),
         MappingDesc( M100_1_NOFEC, [ 5 ], M100_1_NOFEC, [ 5 ] ),
         MappingDesc( G10_1_NOFEC, [ 6 ], G10_1_NOFEC, [ 6 ] ),
         MappingDesc( G5_1_NOFEC, [ 6 ], G5_1_NOFEC, [ 6 ] ),
         MappingDesc( G2P5_1_NOFEC, [ 6 ], G2P5_1_NOFEC, [ 6 ] ),
         MappingDesc( G1_1_NOFEC, [ 6 ], G1_1_NOFEC, [ 6 ] ),
         MappingDesc( M100_1_NOFEC, [ 6 ], M100_1_NOFEC, [ 6 ] ),
         MappingDesc( G10_1_NOFEC, [ 7 ], G10_1_NOFEC, [ 7 ] ),
         MappingDesc( G5_1_NOFEC, [ 7 ], G5_1_NOFEC, [ 7 ] ),
         MappingDesc( G2P5_1_NOFEC, [ 7 ], G2P5_1_NOFEC, [ 7 ] ),
         MappingDesc( G1_1_NOFEC, [ 7 ], G1_1_NOFEC, [ 7 ] ),
         MappingDesc( M100_1_NOFEC, [ 7 ], M100_1_NOFEC, [ 7 ] ),
      ]

@registerHwL1Component
class Cortina( ExternalPhy ):

   #               +-------------------+
   #               |      Cortina      |
   #     serdes    | +---------------+ |    serdes
   #       0   ----| |0             0| |----  0
   #       1   ----| |1             1| |----  1
   #       2   ----| |2   Core 0    2| |----  2
   #       3   ----| |3             3| |----  3
   #               | +---------------+ |
   #               +-------------------+

   def getSerdes( self, serdesId ):
      return ( 0, serdesId )

   def getNumCores( self ):
      return 1

   def getComponentType( self, coreId ):
      return "cs4317"

   def getNumSerdesPerCore( self, coreId=None ):
      return 4

   def addOtherSideSerdes( self, phySide, serdesTopology, coreId ):
      # Map will be:
      # A[ 0-3 ]: B[ 0-3 ]
      lane = serdesTopology.physicalLane
      serdesTopology.otherSideSerdes[ lane ] = True

   def getLogicalSerdesMappings( self, coreId ):
      return [
         MappingDesc( G10_1_NOFEC, [ 0 ], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], G10_1_NOFEC, [ 1 ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], G10_1_NOFEC, [ 2 ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], G10_1_NOFEC, [ 3 ] )
      ]

class Crosspoint( Component ):
   # TODO: See BUG500705; Remove Crosspoint
   supportedModes = { None, "Crosspoint" }

   def getSerdes( self, serdesId ):
      return ( 0, serdesId )

   def getNumCores( self ):
      return 1

   def addOtherSideSerdes( self, phySide, serdesTopology, coreId ):
      pass

   def getLogicalSerdesMappings( self, coreId ):
      # No mapping
      return []

   @abstractmethod
   def getComponentType( self, coreId ):
      pass

   @abstractmethod
   def getNumSerdesPerCore( self, coreId=None ):
      pass

@registerHwL1Component
class M21605( Crosspoint ):

   def getComponentType( self, coreId ):
      return "m21605"

   def getNumSerdesPerCore( self, coreId=None ):
      return 160

@registerHwL1Component
class M21048( Crosspoint ):

   def getComponentType( self, coreId ):
      return "m21048"

   def getNumSerdesPerCore( self, coreId=None ):
      return 48

class CpuNic( Component ):

   def getSerdes( self, serdesId ):
      return( 0, 0 )

   def getNumCores( self ):
      return 1

   def getNumSerdesPerCore( self, coreId=None ):
      return 1

   def addOtherSideSerdes( self, phySide, serdesTopology, coreId ):
      pass

   @abstractmethod
   def getLogicalSerdesMappings( self, coreId ):
      pass

   def laneRemapCapable( self ):
      return False

@registerHwL1Component
class TenGCpuNic( CpuNic ):
   # TODO: See BUG500705; Remove Default
   supportedModes = { None, "Default" }

   def getLogicalSerdesMappings( self, coreId ):
      return [
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ] ),
      ]

@registerHwL1Component
class OneGCpuNic( CpuNic ):
   # TODO: See BUG500705; Remove Default
   supportedModes = { None, "Default" }

   def getLogicalSerdesMappings( self, coreId ):
      return [
         MappingDesc( G1_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
      ]

class App( Component ):
   # TODO: See BUG500705; Remove Default
   supportedModes = { None, "Default" }

   def getSerdes( self, serdesId ):
      return ( 0, serdesId )

   def getNumCores( self ):
      return 1

   @abstractmethod
   def getNumSerdesPerCore( self, coreId=None ):
      pass

   def addOtherSideSerdes( self, phySide, serdesTopology, coreId ):
      pass

   @abstractmethod
   def getLogicalSerdesMappings( self, coreId ):
      pass

   def laneRemapCapable( self ):
      return False

@registerHwL1Component
class XCKU095_2FFVB2104E( App ):

   def getNumSerdesPerCore( self, coreId=None ):
      return 64

   def getLogicalSerdesMappings( self, coreId ):
      mapping = []
      sgm = G10_1_NOFEC
      for serdes in range( self.getNumSerdesPerCore() ):
         mapping.append( MappingDesc( sgm, [ serdes ], None, [] ) )
         mapping.append( MappingDesc( None, [], sgm, [ serdes ] ) )
      return mapping

@registerHwL1Component
class XCVU7P_2FLVB2104E( App ):

   def getNumSerdesPerCore( self, coreId=None ):
      return 76

   def getLogicalSerdesMappings( self, coreId ):
      mapping = []
      sgm = G10_1_NOFEC
      for serdes in range( self.getNumSerdesPerCore() ):
         mapping.append( MappingDesc( sgm, [ serdes ], None, [] ) )
         mapping.append( MappingDesc( None, [], sgm, [ serdes ] ) )
      return mapping

@registerHwL1Component
class XCVU9P_3FLGB2104E( App ):

   def getNumSerdesPerCore( self, coreId=None ):
      return 76

   def getLogicalSerdesMappings( self, coreId ):
      mapping = []
      sgm = G10_1_NOFEC
      for serdes in range( self.getNumSerdesPerCore() ):
         mapping.append( MappingDesc( sgm, [ serdes ], None, [] ) )
         mapping.append( MappingDesc( None, [], sgm, [ serdes ] ) )
      return mapping

def getComponent( name ):
   componentClass = registeredComponents.get( name )
   if not componentClass:
      raise HwL1ComponentError( name, 'Invalid component class name.',
                                validNames=registeredComponents.values() )
   return componentClass
