# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

""" GrubHelper is a collection of utilities for interacting with GRUB, the bootloader
on ONIE platforms. Arista platforms use a different bootloader so EOS doesn't
natively support GRUB. """

import os

class InvalidEnvironmentBlock( Exception ):
   pass

class EditEnv( object ):
   """ This class is used to read/write/modify a Grub "Environment Block"
   This is a small pre-allocated block of memory containing key/value pairs
   used to pass information to GRUB or save data between reboots."""

   BLOCK_SIZE = 1024
   COMMENT_CHAR = '#'
   GRUB_HEADER = '# GRUB Environment Block\n'

   def __init__( self, path=None ):
      self.environment = {}
      self.defaultPath = path

   def _selectPath( self, givenPath=None ):
      """ Use the default path if one isn't given."""
      if givenPath:
         return givenPath
      elif self.defaultPath:
         return self.defaultPath
      else:
         raise ValueError( "No path specified" )

   def read( self, givenPath=None ):
      path = self._selectPath( givenPath )
      if not os.path.exists( path ):
         raise IOError( "File does not exist: %s" % path )
      with open( path, 'r' ) as f:
         # extract all key/value pairs from envBlock
         for l in f:
            if not l.startswith( self.COMMENT_CHAR ):
               l = l.strip()
               splat = l.split( '=' )
               if len( splat ) != 2:
                  raise InvalidEnvironmentBlock( "Invalid format for key: %s" % l )
               self.environment[ splat[ 0 ].strip() ] = splat[ 1 ].strip()

   def write( self, givenPath=None ):
      outStr = self.GRUB_HEADER
      for kv in self.environment.iteritems():
         outStr += "%s=%s\n" % kv
      # if outStr is less than BLOCK_SIZE pad with COMMENT_CHARs
      outStrLen = len( outStr )
      padLength = self.BLOCK_SIZE - outStrLen
      if padLength > 0:
         outStr += self.COMMENT_CHAR * padLength
      if padLength < 0:
         raise InvalidEnvironmentBlock( "size %d > max %d" % ( outStrLen,
                                                               self.BLOCK_SIZE ) )
      path = self._selectPath( givenPath )
      if not os.access( path, os.W_OK ):
         raise IOError( "Can't write file: %s" % path )
      with open( path, 'w+' ) as f:
         f.write( outStr )

   def list( self ):
      for kv in self.environment.iteritems():
         print "%s=%s" % kv

   def set( self, key, value ):
      self.environment[ key ] = value

   def unset( self, key ):
      self.environment.pop( key, None )

   def get( self, key ):
      return self.environment.get( key, None )
