#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import GrpcCounterLib
import Smash
import Tracing

trace2 = Tracing.trace2

class IntfProviderBase( GrpcCounterLib.DataProvider ):

   def __init__( self, collector ):
      GrpcCounterLib.DataProvider.__init__( self, collector )
      self.lagCounterDir = None

   def doInit( self, mg, shmemEm ):
      smi = Smash.mountInfo( 'reader' )
      self.lagCounterDir = shmemEm.doMount( 'interface/counter/lag/current',
                                            'Smash::Interface::AllIntfCounterDir',
                                            smi )

   def fillIntfCounters( self, intfStat ):
      raise NotImplementedError

   def getCounters( self ):
      outDiscardsTotal = 0
      inDiscardsTotal = 0
      intfC = {}
      allEthIntfCounterDir = self.collector.allEthIntfCounterDir()
      for intf, counter in allEthIntfCounterDir.intfCounterDir.iteritems():
         currentCounter = counter.intfCounter.get( 'current', None )
         if currentCounter is None:
            continue
         intfStat = currentCounter.statistics
         trace2( 'Get interface counters for %s. outUcast: %d' %
                 ( intf, intfStat.outUcastPkts ) )
         inDiscardsTotal += intfStat.inDiscards
         outDiscardsTotal += intfStat.outDiscards
         intfC[ "%s" % intf ] = self.fillIntfCounters( intfStat )

      # Add port-channel interfaces
      for intf, intfCounter in self.lagCounterDir.counter.iteritems():
         intfStat = intfCounter.statistics
         trace2( 'Get interface counters for %s. outUcast: %d' %
                 ( intf, intfStat.outUcastPkts ) )
         intfC[ "%s" % intf ] = self.fillIntfCounters( intfStat )

      counter = { "inDiscardsTotal" : inDiscardsTotal,
                  "outDiscardsTotal" : outDiscardsTotal,
                  "interfaces" : intfC }
      return counter

class IntfProvider( IntfProviderBase ):
   COUNTER_TYPE = "INTERFACE_COUNTER"
   COMMAND_OPT = "interface"

   def fillIntfCounters( self, intfStat ):
      # This is consistent with EthIntfCli
      if intfStat.inTotalPkts:
         inTotalPkts = intfStat.inTotalPkts
      else:
         inTotalPkts = ( intfStat.inUcastPkts +
                         intfStat.inMulticastPkts +
                         intfStat.inBroadcastPkts )
      return { "inDiscards": intfStat.inDiscards,
               "outDiscards": intfStat.outDiscards,
               "inUcastPkts" : intfStat.inUcastPkts,
               "inMcastPkts" : intfStat.inMulticastPkts,
               "inBcastPkts" : intfStat.inBroadcastPkts,
               "outUcastPkts" : intfStat.outUcastPkts,
               "outMcastPkts" : intfStat.outMulticastPkts,
               "outBcastPkts" : intfStat.outBroadcastPkts,
               "inTotalPkts" : inTotalPkts,
               "outTotalPkts": ( intfStat.outUcastPkts +
                                 intfStat.outMulticastPkts +
                                 intfStat.outBroadcastPkts ),
               "inErrors": intfStat.inErrors,
               "outErrors": intfStat.outErrors,
               "inOctets": intfStat.inOctets,
               "outOctets": intfStat.outOctets }

class IntfOctetsProvider( IntfProviderBase ):
   COUNTER_TYPE = "INTERFACE_OCTETS_COUNTER"
   COMMAND_OPT = "interface-octets"

   def fillIntfCounters( self, intfStat ):
      return { "inOctets": intfStat.inOctets,
               "outOctets": intfStat.outOctets }

def Plugin( context ):
   context.register( IntfProvider )
   context.register( IntfOctetsProvider )
