#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import SharedMem
import Tac
import Tracing
from GrpcCounterLib import DataProvider

trace1 = Tracing.trace1
trace2 = Tracing.trace2

class DataCollector( object ):
   """ Mount sysdb/smash paths for subscribed data. Collect data and format
   it in json.
   """
   def __init__( self, em, sysdb, providerClasses ):
      trace1( 'Initialize DataCollector' )
      self.entityMgr_ = em
      self.sysdb_ = sysdb
      self.smashEm_ = SharedMem.entityManager( sysdbEm=em )
      self.providerClasses_ = providerClasses
      self.providers_ = []
      self.ethIntfWriterStatusDir_ = None
      self.ethIntfCounterReaderSm_ = None
      self.allEthIntfCounterDir_ = None

   def doInit( self ):
      # initialize
      mg = self.entityMgr_.mountGroup()
      shmemEm = SharedMem.entityManager( sysdbEm=self.entityMgr_ )

      # allEthIntfCounterDir is common to a lot of providers, so provide it here
      self.ethIntfWriterStatusDir_ = mg.mount(
         "interface/ethIntfCounter/writerStatus", "Tac::Dir", "ri" )

      # sort providerClasses to put the classes with requestUpdate() last
      self.providerClasses_.sort(
         key=lambda x: x.requestUpdate != DataProvider.requestUpdate )

      # initialize all providers
      for providerClass in self.providerClasses_:
         provider = providerClass( self )
         self.providers_.append( provider )
         provider.doInit( mg, shmemEm )

      mg.close( blocking=True )

      # initializr allEthIntfCounterDir
      self.ethIntfCounterReaderSm_ = Tac.newInstance(
         'Interface::EthIntfCounterReaderSm', self.smashEm_,
         self.ethIntfWriterStatusDir_ )
      self.ethIntfCounterReaderSm_.enableLegacyShmemManSupport()
      self.ethIntfCounterReaderSm_.handleInitialized()
      self.allEthIntfCounterDir_ = \
         self.ethIntfCounterReaderSm_.legacyShmemManCounterAccessor

      # call initDone on all providers
      for provider in self.providers_:
         provider.initDone()

   def entityManager( self ):
      return self.entityMgr_

   def smashEntityManager( self ):
      return self.smashEm_

   def sysdb( self ):
      return self.sysdb_

   def ethIntfWriterStatusDir( self ):
      return self.ethIntfWriterStatusDir_

   def allEthIntfCounterDir( self ):
      return self.allEthIntfCounterDir_

   def getCounters( self ):
      trace2( 'Get counters' )
      counters = {}
      for provider in self.providers_:
         provider.requestUpdate()
      Tac.runActivities( 0 )

      for provider in self.providers_:
         counters[ provider.COUNTER_TYPE ] = provider.getCounters()
      return counters
