# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import Tac
import telnetlib
from Bunch import Bunch
from collections import namedtuple
import Tracing
from RibdDumpParser import SharedAdjacencyTable
from RibdDumpParser import RoutemapChain, PrefixList, AspathList
from RibdDumpParser import CommunityEntry, CommunityList
from RibdDumpParser import CommunityListEntry, ExtCommunityEntry, LargeCommunityEntry
from RibdDumpParser import AsPathAttrEntry, AsPathEntry, AsPathInfoTable
from RibdDumpParser import RtSyncTasks

_adjEntryT = namedtuple( 'AdjEntry', 'id ref hash nGw vias' )
_viaEntryT = namedtuple( 'ViaEntry', 'nexthop intf label' )

th = Tracing.Handle( "GiiTestLib" )
t0 = th.trace0
t2 = th.trace2

def AdjEntry( adjId, ref, adjHash, nGw, vias ):
   return _adjEntryT( adjId, ref, adjHash, nGw, vias )

def ViaEntry( nh, intf, label=0 ):
   return _viaEntryT( nh, intf, label)

class GiiEcmpRtEntry(Bunch):
   # This class is used for both v4 and v6 routes.
   def __init__( self, **kwargs ):
      Bunch.__init__(self, **kwargs)

def parseGiiBgpPeerPassword( output ):
   '''
   Parse output of show bgp password peer <peerIp>
   '''
   output = output + "\n"
   lines = output.splitlines()
   regex = '.*Password: (.*)'
   for line in lines:
      m = re.match( regex, line ) 
      if not m:
         continue 
      else:
         return m.group( 1 )
   return None

def parseGiiBribRtEntry( output ):
   '''
   Parse output of show bgp brib X.X.X.X/M command
   '''
   output = output + "\n"
   pattern = re.compile( "\d+\s+(?P<rib>[um])\s+"
                         "(?P<proto>[a-zA-Z]+)\s+(?P<dest>[0-9a-f:\./]+)\s+"
                         "(?P<nextHop>(?:-+|(?:[%0-9a-zA-Z:\.]+)))\s+"
                         "(?P<med>[0-9]+)\s+"
                         "(?P<aspath>([0-9}()[\]{},<>]+\s)*)"
                         "(?P<origin>(i|e|\?))\s*")

   lines = output.splitlines() 
   skipline = 2
   rtCol = {}
   for line in lines[skipline:]:
      m = pattern.match( line ) 
      if not m:
         assert False, "Unexpected output from line: %s" % ( line )
      if m.group('nextHop') != '---':
         rtCol.setdefault( str( m.group( 'dest' ) ), [] ).append(
               GiiEcmpRtEntry( **m.groupdict() ) )
   return rtCol

def parseGiiEcmpRtEntry( output ):
   '''
   Parse output of show bgp ecmp X.X.X.X/M command
   '''
   output = output + "\n"

   pattern = re.compile( "\d+\s+(?P<rib>[um])\s+"
                         "(?P<proto>[a-zA-Z]+)\s+(?P<dest>[0-9a-f:\./]+)\s+"
                         "(?P<nextHop>(?:-+|(?:[%0-9a-zA-Z:\.]+)))\s+"
                         "(?P<med>[0-9]+)\s+"
                         "(?P<aspath>([0-9}()[\]{},<>]+\s)*)"
                         "(?P<origin>(i|e|\?))\s*")

   lines = output.splitlines() 
   skipline = 2
   rtCol = {}
   for line in lines[skipline:]:
      m = pattern.match( line ) 
      if not m:
         assert False, "Unexpected output from line: %s" % ( line )
      if m.group('nextHop') != '-':
         rtCol.setdefault( str( m.group( 'dest' ) ), [] ).append(
               GiiEcmpRtEntry( **m.groupdict() ) )
   return rtCol

class GiiRouteHead( Bunch ):
   # This class is used for both v4 and v6 routes.
   def __init__( self, **kwargs ):
      Bunch.__init__(self, **kwargs)
      self.routeEntries = {}
      # Collection indexed by source gateway. At least for now, we don't have
      # one source sending multiple paths (add-path will change that of course)
      self.bgpPaths = {}

   def addBgpEntry( self, **kwargs ):
      gw = kwargs[ 'sourceGwt' ]
      self.bgpPaths.setdefault( gw, [] ).append(GiiRouteEntry( **kwargs ))

   def addRouteEntry( self, **kwargs ):
      # kwargs should be same as the arguments for GiiRouteEntry.__init__
      if kwargs[ 'proto' ] == 'BGP':
         self.addBgpEntry( **kwargs )
      else:
         self.routeEntries.setdefault( kwargs[ 'proto' ], [] ).append(
             GiiRouteEntry( **kwargs ) )

class GiiRouteEntry( Bunch ):
   # This class is used for both v4 and v6 routes.
   def __init__( self, **kwargs ):
      Bunch.__init__(self, **kwargs)

   def isEcmp( self ):
      return 'BGPEcmp' in self.states

def parseGiiFlashList( output ):
   rthList = []
   for line in output.split('\n'): 
      m = re.match( '.*Destination: (.*) <(.*?)>', line)
      if m:
         rthList.append( m.group(1) )
   return rthList

class ForwardStateFlags( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__(self, **kwargs)

class GracefulRestartFlags( Bunch ):
   RESTART_BIT = 'restartflags'
   FWD_STATE_BIT = 'fwdstateflags'
   V4_UNI  = 'v4_uni_preserved'
   V4_MULTI  = 'v4_multi_preserved'
   V6_UNI  = 'v6_uni_preserved'
   V6_MULTI  = 'v6_multi_preserved'
   def __init__( self, **kwargs ):
      Bunch.__init__( self, **kwargs )

def parseGiiGRFlags( output ):
   output = output + "\n"
   lines = output.splitlines()
   regex = r'\s+Graceful Restart Bit: <(?P<%s>\d)>\s+' % \
         GracefulRestartFlags.RESTART_BIT + \
         'Graceful Forwarding State Bit: <(?P<%s>[\w\s]*)>' % \
         GracefulRestartFlags.FWD_STATE_BIT

   mapnames = { 'V4UniRestart' : GracefulRestartFlags.V4_UNI,
                'V4MultiRestart' : GracefulRestartFlags.V4_MULTI, 
                'V6UniRestart' : GracefulRestartFlags.V6_UNI, 
                'V6MultiRestart' : GracefulRestartFlags.V6_MULTI }

   d = { mapnames[ key ] : False for key in mapnames.keys() }
   grflags = {}
   for line in lines:
      match = re.search( regex, line )
      if not match:
         continue
      if match.group( GracefulRestartFlags.FWD_STATE_BIT ):
         flags = re.split( " ", match.group( GracefulRestartFlags.FWD_STATE_BIT ) )
         for flag in flags: 
            d[ mapnames[ flag ] ] = True
      grflags[ GracefulRestartFlags.FWD_STATE_BIT ] = ForwardStateFlags( **d )
      grflags[ GracefulRestartFlags.RESTART_BIT ] = \
            bool( int( match.group( GracefulRestartFlags.RESTART_BIT ) ) )
      break
   return GracefulRestartFlags( **grflags )

def parseGiiMaxRoutes( output ):
   output = output + "\n"
   lines = output.splitlines()
   regex = r'\s+Max Routes: (\d+)\s+'
   maxRoutes = None
   for line in lines:
      match = re.search( regex, line )
      if not match == None:
         maxRoutes = int( match.group( 1 ) )
         break
   return maxRoutes

def parseGiiMaxRoutesThreshold( output ):    
   output = output + "\n"
   lines = output.splitlines()
   regex = r'\s+Max Routes Threshold: (\d+)\s+'
   maxRoutesThreshold = None
   for line in lines:
      match = re.search( regex, line )
      if not match == None:
         maxRoutesThreshold = int( match.group( 1 ) )
         break
   return maxRoutesThreshold

class AddPathOptions( Bunch ):
   AP_RECV_V4_UNI = 'ap_recv_v4_uni'
   AP_RECV_V6_UNI = 'ap_recv_v6_uni'
   AP_SEND_V4_UNI = 'ap_send_v4_uni'
   AP_SEND_V6_UNI = 'ap_send_v6_uni'
   AP_RECV_V4_LABELED_UNI = 'ap_recv_v4_labeled_uni'
   AP_RECV_V6_LABELED_UNI = 'ap_recv_v6_labeled_uni'
   AP_SEND_V4_LABELED_UNI = 'ap_send_v4_labeled_uni'
   AP_SEND_V6_LABELED_UNI = 'ap_send_v6_labeled_uni'
   def __init__( self, **kwargs ):
      Bunch.__init__( self, **kwargs )

def parseGiiAddPathOptions( output ):
   output = output + "\n"
   lines = output.splitlines()
   regex = r'\s+Options: <([\w\s]*)>'

   mapnames = { "AddPathRecvV4Uni" : AddPathOptions.AP_RECV_V4_UNI,
                "AddPathRecvV6Uni" : AddPathOptions.AP_RECV_V6_UNI,
                "AddPathSendV4Uni" : AddPathOptions.AP_SEND_V4_UNI,
                "AddPathSendV6Uni" : AddPathOptions.AP_SEND_V6_UNI,
                "AddPathRecvV4LabeledUni" : AddPathOptions.AP_RECV_V4_LABELED_UNI,
                "AddPathRecvV6LabeledUni" : AddPathOptions.AP_RECV_V6_LABELED_UNI,
                "AddPathSendV4LabeledUni" : AddPathOptions.AP_SEND_V4_LABELED_UNI,
                "AddPathSendV6LabeledUni" : AddPathOptions.AP_SEND_V6_LABELED_UNI }

   keys = mapnames.keys()
   d = { mapnames[ key ] : False for key in keys }
   for line in lines:
      match = re.search( regex, line )
      if not match == None:
         options = match.group( 1 )
         optList = options.split( " " )
         for opt in optList:
            if opt in keys:
               d[ mapnames[ opt ] ] = True
         break
   return AddPathOptions( **d )
    
def parseGiiRouteHead( output ):
   '''
   Parse a route head output by show ip route X.X.X.X/M command.
   Sample output (\ indicates a linewrap):
   100 Route 1.0.0.0/24 entries 2 Announced 1 Depth <>
   100   Proto           Next Hop        Interface            Source Gwt      \
   Preference/2 Metric/2 \
   Tag           Age Aspath/State/Adjacency/AttrId
   100 * Direct          1.0.0.1  et_1_1  ---             0/0          1/0      \
   0        05:32:56 i (HashID 1) <Int Retain ActiveU ActiveM Unicast Multicast> \
   1000b4cee
   100   Static          1.0.0.1  et_1_1 ---             1/0          0/0      \
   0        04:28:57 ? (HashID 2) <Int Unicast> 1000b4cf2
   '''
   # Add a newline to the output; the parser expects it.
   output = output + "\n"
   routeHeadRe = '\d+\s+Route\s+(?P<dest>\S+/\d+)\s+entries\s+' \
         '(?P<entries>\d+)\s+Announced\s+(?P<announce>\d+)\s+Depth\s+' \
         '(?P<depth>(\S+-\d+ )*)<(?P<rthState>.*?)>\s*\n'
   routeEntryHeaderRe = '\d+\s+Proto\s+Next Hop\s+Interface\s+Source Gwt\s+' \
         'Preference/2\s+Metric/2\s+Label\s+Tag\s+Age\s+' \
         'Aspath/State/Adjacency/AttrId\s*\n'
   routeEntryRe = \
         '\d+\s+(?P<active>[*|+|-|/])?\s+(?P<proto>\S+)\s+(?P<nexthop>\S+)' \
         '\s+(?P<interface>\S+)' \
         '\s+(?P<sourceGwt>\S+)\s+(?P<preference>-?\d+/\d+)\s+' \
         '(?P<metric>\d+/\d+)\s+(?P<label>\d+)\s+(?P<tag>\S+)\s+'\
         '(?P<age>\S+)\s+(?P<aspath>.*?)' \
         '<(?P<states>.*?)>\s+(?P<adjacency>[0-9a-f]+)\s*' \
         '(?P<attrid>[0-9a-f]+)\s*\n'
   routeRe = '\s*%s%s(?P<routeEntry>(.*\n)+)' % ( routeHeadRe, routeEntryHeaderRe )
   match = re.search( routeRe, output )
   if not match:
      return None
   groupdict = match.groupdict()
   routeEntries = groupdict.pop( 'routeEntry' )
   # 'entries' and 'announce' are integers
   for i in [ 'entries', 'announce' ]:
      groupdict[ i ] = int( groupdict[ i ] )
   # 'depth' and 'rthState' are lists
   for i in [ 'depth', 'rthState' ]:
      groupdict[ i ] = groupdict[ i ].strip().split()
   ret = GiiRouteHead( **groupdict )
   for rem in re.finditer( routeEntryRe, routeEntries ):
      groupdict = rem.groupdict()
      for i in [ 'nexthop', 'sourceGwt' ]:
         if groupdict[ i ] == '---':
            groupdict[ i ] = None
      for i in [ 'preference', 'metric' ]:
         groupdict[ i ] = map( int, groupdict[i].split( '/' ) )
      for i in [ 'tag' ]:
         groupdict[ i ] = int( groupdict[ i ], 16 )
      for i in [ 'states' ]:
         groupdict[ i ] = groupdict[ i ].strip().split()
      groupdict[ 'label' ] = int(groupdict[ 'label' ])
      for i in ( 'adjacency', ):
         groupdict[ i ] = long( groupdict[ i ], 16 )
      for i in ( 'attrid', ):
         groupdict[ i ] = int( groupdict[ i ], 16 )
      ret.addRouteEntry( **groupdict )
   return ret

def parseGiiKernelParams( output ):
   '''
   Sample Output - 
   100 Kernel options: <> Support: <Reject Blackhole VarMask Host Multipath>
   100 Remnant Timer: <180> Routes <-1>
   100 IP forwarding: 1 UDP checksums 1
   100 IPv6 forwarding: 1
   100 The time is 08:30:16
   100 Monotonic time is 746

   NOTE: The functions just returns the Monotonic time (time_sec). If other params
   are needed, this can be enhanced.
   '''
   m = re.search( "Monotonic time is (\d+)", output )
   assert m
   return { "time_sec": int( m.groups()[ 0 ] ) }

def getEcmpAdjFromGiiRtEntry( rt ):
   '''Returns BGP ECMP adjacency associated with "head" i.e. active rt_entry
   within a route head if one exists. Otherwise, returns None'''
   ecmpAdj = None
   winnerGw = filter(lambda k: rt.bgpPaths[k][0].active == '*', rt.bgpPaths.keys())
   if winnerGw and rt.bgpPaths[ winnerGw[0] ][ 0 ].active == '*' and \
         'BGPEcmp' in rt.bgpPaths[ winnerGw[0] ][ 0 ].states:
      ecmpAdj = rt.bgpPaths[ winnerGw[0] ][ 0 ].adjacency
   return ecmpAdj

# Returns the id of the adjacency used by the prefix
def giiGetAdjUsedByPfx( dut, prefix, adjtable=None, ipv6=False, vrfName=None ):
   rt = dut.giiGetRouteHead( prefix, ipv6=ipv6, vrfName=vrfName )
   if not rt:
      t2( "giiGetRouteHead( %s ) returned None" % prefix )
      return None

   if not adjtable:
      adjtable = SharedAdjacencyTable(
                     dut.giiGetAdjacencyDetails( vrfName=vrfName ) )
   adjId = None
   ecmpAdj = getEcmpAdjFromGiiRtEntry( rt )
   for sourceGwt in rt.bgpPaths:
      if rt.bgpPaths[ sourceGwt ][ 0 ].active != '*':
         if len( rt.bgpPaths[ sourceGwt ] ) != 1 and ecmpAdj is not None and \
            rt.bgpPaths[ sourceGwt ][ 0 ].adjacency != ecmpAdj:
            # In case of BGP paths that resolve via IGP like ISIS/OSPF, it is
            # possible that they have more than one resolved nexthops, despite not
            # being ECMP head
            continue
         # Nexthop restore doesn't happen for negative preference
         # routes, BUG81516.
         #assert len( rt.bgpPaths[ sourceGwt ] ) == 1 or \
         #    rt.bgpPaths[ sourceGwt ][ 0 ].preference[ 0 ] == -1, \
         #    "Nexthop restore didn't happen for %s from gateway %s" % \
         #    ( prefix, sourceGwt )
         # This is not the active sourceGwt in use
         continue
      # Get the adjId from each routeEntry in GiiRouteHead object,
      # it's a long integer, convert it to hex, exclude the trailing 'L'
      adjId = rt.bgpPaths[ sourceGwt ][ 0 ].adjacency
      for i in range( 1, len(rt.bgpPaths[ sourceGwt ]) ):
         newAdjId = rt.bgpPaths[ sourceGwt ][ i ].adjacency
         assert newAdjId == adjId, "Prefix %s is not using a single adjacency" % \
             prefix
   return adjtable.getAdjById( adjId ) if adjId else None

#
# Given a set of prefixes, for each prefix in list; checks if any of of the
# non-activate BGP paths in ECMP group has more than one nexthops set. Returns
# False if yes, otherwise returns True to indicate that Nexthop Restore isn't
# pending for any of the BGP ECMP paths
#
def giiIsNexthopRestoreComplete( dut, routeSet, ipv6=False ):
   for prefix in routeSet:
      rt = dut.giiGetRouteHead( prefix, ipv6=ipv6 )
      if not rt:
         return False
      ecmpAdj = getEcmpAdjFromGiiRtEntry( rt )
      for sourceGwt in rt.bgpPaths:
         if rt.bgpPaths[ sourceGwt ][ 0 ].active != '*':
            if len( rt.bgpPaths[ sourceGwt ] ) != 1 and ecmpAdj is not None and \
               rt.bgpPaths[ sourceGwt ][ 0 ].adjacency != ecmpAdj:
               # In case of BGP paths that resolve via IGP like ISIS/OSPF, it
               # is possible that they have more than one resolved nexthops,
               # despite not being ECMP head
               continue
            # Nexthop restore doesn't happen for negative preference
            # routes, BUG81516.
            if len( rt.bgpPaths[ sourceGwt ] ) > 1 or \
                rt.bgpPaths[ sourceGwt ][ 0 ].preference[ 0 ] == -1:
               print "Nexthop restore pending for %s from gateway %s" % \
                     ( prefix, sourceGwt )
               return False
   return True

# Returns the common adjacency used by all the prefixes in routeSet
# Returns None if all prefixes in routeSet do not use the same adjacency
def giiGetAdjUsedByPfxs( dut, routeSet, ipv6=False, vrfName=None, agent=None ):
   adjtable = SharedAdjacencyTable(
                  dut.giiGetAdjacencyDetails( vrfName=vrfName, agent=agent ) )
   adj = giiGetAdjUsedByPfx( dut, routeSet[0], adjtable=adjtable, ipv6=ipv6,
                             vrfName=vrfName )
   if not adj:
      t2( "No adjacency for %s" % routeSet[0] )
      return None

   for i in range( 1, len(routeSet) ):
      newAdj = giiGetAdjUsedByPfx( dut, routeSet[i], adjtable=adjtable, ipv6=ipv6,
                                   vrfName=vrfName )
      if not newAdj:
         t2( "No adjacency for %s" % routeSet[i] )
         return None

      if newAdj.id() != adj.id():
         return None
   return adj

# Verifies that the ref count in nexthop_t data structure equals the number of
# adjacencies using this nexthop. If there is a mismatch or if the nexthop is used
# by an adjacency which is already freed, assert
def checkForSharedNhLeak( dut, vrfName, ipv6=False ):
   nhtable = dut.giiShowIpNexthop( vrfName=vrfName )
   sharednhs = nhtable.parse( ipv6=ipv6 )

   output = dut.giiGetAdjacencyDetails( rtsync=True, vrfName=vrfName )
   adjtable = SharedAdjacencyTable( output )
   adjnhs = adjtable.getAdjListNhs( ipv6=ipv6 )

   diff = set( sharednhs.items() ).symmetric_difference( set( adjnhs.items() ) )
   # TODO: Enable this assert (and remove the trace below) once we make sure
   # that there are no tests failing because of this leak.
   # assert not diff, "Shared NH leak. Leaked Nhs -> %s " % diff
   if diff:
      t0( "Shared NH leak. Leaked Nhs -> %s " % diff )
   return

def checkForPolicyObjectLeaks( dut, ctx ):
   policyDump = dut.giiGetPolicy( vrfName=ctx.vrfName() )
   # NOTE: This is 2 and not 0 because we create two AS paths with DIRECT and STATIC
   # origin (see aspath_create). We do not free these presently.
   assert len( policyDump.asPathEntries() ) == 2
   assert policyDump.asPathAttrEntries() == {}
   assert policyDump.communityLists() == {}
   assert policyDump.communityEntries() == {}
   assert policyDump.extCommunityEntries() == {}
   assert policyDump.largeCommunityEntries() == {}

def checkForAdjLeak( dut, ctx, sharedNhLeakCheck=True ):
   vrfName = ctx.vrfName()
   ipv6 = ctx.ipv6()
   output = dut.giiGetAdjacencyDetails( rtsync=True, vrfName=vrfName )
   adjtable = SharedAdjacencyTable( output )
   nhAdjDict = RtSyncTasks( output ).nexthopAdjRefCountDict()
   for adj in adjtable.adjacencyList().itervalues():
      if adj.flags() and ( 'BgpEcmp' in adj.flags() ):
         expected = 1
      else:
         expected = len( adj.rtEntries() ) + len( adj.isisAdjSids() ) + \
                    len( adj.isisPrefixSids() ) + nhAdjDict.get( adj.id(), 0 )
      assert int( adj.refCount() ) == expected, \
          "Adjacency 0x%x is leaked, refCount: %s, Expected: %s" % \
          ( adj.id(), adj.refCount(), expected )
   if sharedNhLeakCheck:
      if hasattr( ctx, 'bgpLu' ) == False or ctx.bgpLu == False:
         checkForSharedNhLeak( dut, vrfName, ipv6=ipv6 )

# Verifies that the adjacency used by the routes in routeSet in RIB is same
# as that used in FIB and returns that adjacency, returns None incase of mismatch
def ribFibAdjMatch( dut, routeSet, ipv6=False ):
   ribAdj = giiGetAdjUsedByPfxs( dut, routeSet, ipv6=ipv6 )
   if not ribAdj:
      t2( "No shared ribAdj found for {}".format( routeSet ) )
   if not ipv6:
      ipRouteFec = dut.ipRouteFec
   else:
      ipRouteFec = dut.ip6RouteFec
   fecInFib = ipRouteFec( routeSet[ 0 ] )

   if ( None in ( ribAdj, fecInFib ) or
        ribAdj.id() != fecInFib.fecId ):
      return None

   for i in range( 1, len( routeSet ) ):
      fecInFibNew = ipRouteFec( routeSet[ i ] )
      if fecInFibNew is None or fecInFibNew.fecId != fecInFib.fecId:
         return None

   return ribAdj

class GiiPolicy:
   '''Wraps gii 'show policy' command output parser'''
   def __init__( self, content ):
      self.dump_ = content

   @Tac.memoize
   def routemapChains( self ):
      return RoutemapChain.parse( self.dump_ )

   @Tac.memoize
   def prefixLists( self ):
      return PrefixList.parse( self.dump_ )

   @Tac.memoize
   def nUsedPolicyCommits( self ):
      '''Number of mio commits that resulted in 'used' policy being changed
      NOTE: Gated now avoids rt_new_policy calls when unused policy objects are
      changed'''
      m = re.match( r".*Useful policy commits:\s(?P<num>\d+)", self.dump_, re.S )
      if m:
         return int( m.group('num') )
      return 0

   @Tac.memoize
   def nMioReconfigCalls( self ):
      '''Number of calls to mio_reconfigure'''
      m = re.match( r".*mio_reconfigure calls:\s(?P<num>\d+)", self.dump_, re.S )
      if m:
         return int( m.group('num') )
      return 0

   @Tac.memoize
   def nRtNewpolicyCalls( self ):
      '''Number of calls to rt_new_policy API, which flashes all routes in RIB'''
      m = re.match( r".*rt_new_policy calls:\s(?P<num>\d+)", self.dump_, re.S )
      if m:
         return int( m.group('num') )
      return 0

   @Tac.memoize
   def nRtAsyncNewpolicyCalls( self ):
      '''Number of calls to rt_async_new_policy_job, which 
      flashes all routes in RIB'''
      m = re.match( r".*rt_new_policy async calls:\s(?P<num>\d+)", self.dump_, re.S )
      if m:
         return int( m.group('num') )
      return 0

   @Tac.memoize
   def nDirtyPrefixListEntries( self ):
      '''Number of dirty prefix list entries collected in prefix list modification'''
      m = re.match( r".*Dirty prefix list entries:\s(?P<num>\d+)", self.dump_, re.S )
      if m:
         return int( m.group('num') )
      return -1

   @Tac.memoize
   def nTotalPolicyCommits( self ):
      '''Number of mio commits that resulted in one or more bits in
      mio_mask_taskpolicy to be set'''
      m = re.match( r".*Total policy commits:\s(?P<num>\d+)", self.dump_, re.S )
      if m:
         return int( m.group('num') )
      return 0

   @Tac.memoize
   def nCyclePolicyHits( self ):
      '''Number of loop hits during policy evaluation'''
      m = re.match( r".*Cycle policy hits:\s(?P<num>\d+)", self.dump_, re.S )
      if m:
         return int( m.group('num') )
      return 0

   @Tac.memoize
   def srdChangesShadow( self ):
      '''Value of srd_changes prior to last reset'''
      m = re.match( r".*SRD changes shadow:\s(?P<num>\d+)", self.dump_, re.S )
      if m:
         return int( m.group('num') )
      return 0

   @Tac.memoize
   def nPolicyCaches( self ):
      '''Number of policy caches currently active in the system'''
      m = re.match( r".*Number of caches:\s(?P<num>\d+)", self.dump_, re.S )
      if m:
         return int( m.group('num') )
      return 0

   @Tac.memoize
   def nPolicyCacheEntries( self ):
      '''Number of policy cache entires considering all active caches in the
      system'''
      m = re.match( r".*Number of cache entries:\s(?P<num>\d+)", self.dump_, re.S )
      if m:
         return int( m.group('num') )
      return 0

   @Tac.memoize
   def aspathLists( self ):
      return AspathList.parse( self.dump_ )

   @Tac.memoize
   def asPathAttrEntries( self ):
      return AsPathAttrEntry.parse( self.dump_ )

   @Tac.memoize
   def asPathEntries( self ):
      return AsPathEntry.parse( self.dump_ )

   @Tac.memoize
   def asPathInfoTable( self ):
      return AsPathInfoTable.parse( self.dump_ )

   @Tac.memoize
   def communityEntries( self ):
      return CommunityEntry.parse( self.dump_ )

   @Tac.memoize
   def communityEntriesAll( self ):
      return CommunityEntry.parseAll( self.dump_ )

   @Tac.memoize
   def communityLists( self ):
      return CommunityList.parse( self.dump_ )

   @Tac.memoize
   def communityListEntries( self ):
      return CommunityListEntry.parse( self.dump_ )

   @Tac.memoize
   def extCommunityEntries( self ):
      return ExtCommunityEntry.parse( self.dump_ )

   @Tac.memoize
   def largeCommunityEntries( self ):
      return LargeCommunityEntry.parse( self.dump_ )

class GiiPiPolicyCache:
   '''
   Wraps gii 'show policy-cache <rm-name>' command output parser
   '''
   def __init__( self, output ):
      self.cache = output

   @Tac.memoize
   def nPiPolicyCacheEntries( self ):
      return len( self.cacheEntries() )

   @Tac.memoize
   def cacheEntries( self ):
      '''
      Each entry in this section is as follows (each is on one line)
          150             AsPath 3 useCount 59 flags 0x0 nexthop - -> res 1 api <0, >
      result 3 preference 200
          150             AsPath 4 useCount 59 flags 0x0 nexthop - -> res 1 api <0, >
      result 4 preference 200
          150             AsPath 5 useCount 59 flags 0x0 nexthop - -> res 1 api <0, >
      result 5 preference 200
      '''
      pat = r'AsPath (?P<aspathid>\d+) useCount (?P<useCount>\d+) ' + \
            r'flags (?P<flag>\S+) nexthop (?P<nexthop>\S+)' + \
            r'( link_bw (?P<link_bw>\S+))* ' + \
            r'-> res (?P<res>\S+) api (?P<api><\d+, \S*>) ' +\
            r'result (?P<result>\S+) ' + \
            r'preference (?P<preference>\S+)'
      lst = re.finditer( pat, self.cache, re.S | re.M )
      return [ Bunch( **m.groupdict() ) for m in lst ]

class RibdMemBlock( Bunch ):
   def __init__( self, **kwargs ):
      for k in ( 'size', 'pages', 'freepages', 'ninit', 'nalloc',
                  'nused', 'nfree', 'nbytes' ):
         if k in kwargs:
            kwargs[k] = int( kwargs[k] )
      Bunch.__init__(self, **kwargs)

class GiiMemory:
   """
   Parse the output of GII 'show memory' command

   What we are trying to parse is as follows (header appears only once at the top)

   Size   Pgs Free  Block Name        Init   Alloc  Free   InUse  (bytes)

   72     1   55    lsp_t             1      0      0      0      (0)
   72               nospf_te_t        1      0      0      0      (0)
   72               pimdm_source_t    1      0      0      0      (0)
   72               pimsm_ifstate_t   1      0      0      0      (0)
   72               evt_class_node_t  1      1      0      1      (72)
   """
   def __init__( self, content ):
      self.blocks = {}
      self.content_ = re.sub('^1\d0\s', '', content, flags=re.M).splitlines()
      self.total, self.totalUsed, self.totalFree = 0, 0, 0
      self.parse()
      self.parseTotals()

   @Tac.memoize
   def parse( self ):
      sz_pat = '(?P<size>\d+)\s+'
      header_middle_pat = '(?P<pages>\d+)\s+(?P<freepages>\d+)\s+'
      common_pat = '(?P<name>\S+)\s+(?P<ninit>\d+)\s+(?P<nalloc>\d+)\s+' + \
                   '(?P<nfree>\d+)\s+(?P<nused>\d+)\s+\((?P<nbytes>\d+)\)'
      for l in self.content_:
         m = re.match( sz_pat + header_middle_pat + common_pat, l )
         if not m:
            m = re.match( sz_pat + common_pat, l )
         if m:
            blk = RibdMemBlock( **m.groupdict() )
            self.blocks[ blk.name ] = blk

   @Tac.memoize
   def nonZeroBlocks( self ):
      return dict(filter( lambda i: i[1].nused != 0, self.blocks.items() ) )

   @Tac.memoize
   def parseTotals( self ):
      # Parse total memory on last-line
      pat = 'Total Memory:\s+(?P<total>\d+)\s+' + \
            'Total Free:\s+(?P<totalFree>\d+)\s+' + \
            'Total Allocated:\s+(?P<totalUsed>\d+)\s+'
      m = re.match( pat, self.content_[-1] )
      if m:
         self.total = int( m.group( 'total' ) )
         self.totalFree = int( m.group( 'totalFree' ) )
         self.totalUsed = int( m.group( 'totalUsed' ) )

class RibdRtAttr( Bunch ):
   def __init__( self, **kwargs ):
      for k in ( 'id', 'refcount', 'preference', 'preference2', 'metric', 'metric2',
                  'tag', 'aspathid', 'rcvaspathid' ):
         if k in kwargs:
            kwargs[k] = int( kwargs[k] )
      Bunch.__init__(self, **kwargs)

class GiiRtAttr:
   """
   Parse the output of GII 'show ip rtattr' command
   Shared-Attributes Table
      Id 74   Refs 10 Hash 19 Pref 200 Pref2 0 Metric 0 Metric2 100 Tag 2 \
                     AspathID 92 RcvAspathID 10 Gwp 0xdeadbeef Rd 0:0
      Id 75   Refs 10 Hash 123 Pref 200 Pref2 0 Metric 0 Metric2 100 Tag 2 \
                     AspathID 94 RcvAspathID 100 Gwp 0xdeadbeef Rd 0:0
   """
   def __init__( self, content ):
      self.attrs = {}
      self.content_ = re.sub('^1\d0\s+', '', content, flags=re.M).splitlines()
      self.parse()

   @Tac.memoize
   def parse( self ):
      common_pat = 'Id (?P<id>\d+)\s+Refs (?P<refcount>\d+)\s+Hash (?P<hash>\d+)' + \
                   '\s+Pref (?P<preference>\d+)\s+Pref2 (?P<preference2>\d+)' + \
                   '\s+Metric (?P<metric>\d+)\s+Metric2 (?P<metric2>\d+)' + \
                   '\s+Tag (?P<tag>\d+)\s+AspathID (?P<aspathid>\d+)' + \
                   '\s+RcvAspathID (?P<rcvaspathid>\d+)' + \
                   '\s+Gwp (?P<gwp>\S+)'
      for l in self.content_:
         m = re.search( common_pat, l )
         if m:
            a = RibdRtAttr( **m.groupdict() )
            self.attrs[ a.id ] = a

class AttrInfoIdSet( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__(self, **kwargs)
      self.id = int( self.id )
      self.refcount = int( self.refcount )
      self.hash = int( self.hash )
      self.count = int( self.count )
      self.aiiset = set()

class GiiAttrInfoIdSet:
   """
   Parse the output of GII 'show ip aiiset' command
   Attr Info Id Set Table
      Id %u   Refs %u   Hash %u   N_gw %u
         AttrInfoId: %u
   Attr Info Id Set Table End
   """
   def __init__( self, content ):
      self.aiisets = {}
      self.content_ = re.sub( '^1\d0\s+', '', content, flags=re.M )
      self.parse()

   @Tac.memoize
   def parse( self ):
      m = re.match( r'.*Attr Info Id Set Table(?P<aiiset>.*?)' + \
                     '^Attr Info Id Set Table End',
                     self.content_, re.S | re.M )
      if m:
         content = m.group( 'aiiset' )
         aiipat = '^AttrInfoId:\s+(?P<aii>\d+)\n'
         aiisetRe = '^Id\s+(?P<id>\d+)\s+Refs\s+(?P<refcount>\d+)\s+Hash' + \
                     '\s+(?P<hash>\d+)\s+N_gw\s+(?P<count>\d+)\n' + \
                     '(?P<aiis>(' + aiipat + ')*)'
         lst = re.finditer( aiisetRe, content, re.S | re.M )
         self.aiisets = {}
         for m in lst:
            aiisetobj = AttrInfoIdSet( **m.groupdict() )
            self.aiisets[ int( m.group( 'id' ) ) ] = aiisetobj
            seqs = re.finditer( aiipat, aiisetobj.aiis, re.S | re.M )
            for m2 in seqs:
               aiisetobj.aiiset.add( int( m2.group( 'aii' ) ) )

def giiRthProperty( dut, prefix, rtEntryProps=None ):
   '''This method calls giiRthProperty( prefix ) and
   looks for at least one rt_entry matching the properties
   requested.
   
   Here is an example:

      rtEntryProps = { "Static": { "active": "+", nexthop="8.0.0.10" } }

   which returns true if a Static route entry is found with active flag
   set to "+" and nexthop="8.0.0.10".

   This uses gii to fetchstate.
   '''
   rth = dut.giiGetRouteHead( prefix )

   if not rth:
      t2( "Prefix %s not found" % prefix )
      return False

   if rtEntryProps:
      # we want to validate some route entry properties
      entriesFound = rth.routeEntries
      if not entriesFound:
         t2( "No route entries found" )
         return False

      # for each protocol we want to verify
      for proto, attrs in rtEntryProps.iteritems():
         if proto not in entriesFound:
            t2( "Protocol %s rtEntry not found" % proto )
            return False
         rtEntry = entriesFound[proto]

         matchFound = False
         # a single protocol can publish multiple rt_entries
         # (especially in the context of a change where the
         # previous rt_entry is marked for deletion and a new
         # rt_entry might become active).
         # Walk through each of the given protocol's route
         # looknig for an rt_entry matching our criterion
         for rt in rtEntry:
            numMatchedProperties = 0
            # Validate each requested property
            for attr, expValue in attrs.iteritems():
               foundValue = rt.__getattribute__( attr )
               if foundValue != expValue:
                  t2( "Route {}, attr {}, expected {}, found {}".format(
                        rt, attr, expValue, foundValue ) )
                  break
               numMatchedProperties += 1

            if numMatchedProperties == len( attrs ):
               matchFound = True
               break

         if not matchFound:
            return False

   return True
RIBD_GII_DUMP_PORT = 616
HOST = "127.0.0.1"
PROMPT = '>'

def giiModeCmdIs( vrfName=None, *cmd ):
   """
   Get show output from gii interface, return string value of show result
   Gii is not vrf aware yet, vrfName is not used in this function.
   """
   giiOutput = ""
   cmd = ' '.join( cmd ) + '\n'
   if not cmd:
      return giiOutput
   try:
      telnetSession = telnetlib.Telnet( HOST, RIBD_GII_DUMP_PORT, 10 )
      telnetSession.read_until( PROMPT )
      telnetSession.write( cmd )
      giiOutput =  telnetSession.read_until( PROMPT )
      telnetSession.write( "quit\n" )
      telnetSession.close()
   except:
      raise Exception( "gii connect error" )
   else:
      return giiOutput

def memUsage( data ):
   """
   return RibdDumpParser friendly dict, input data is dict of RibdMemBlock
   """
   usage = {}
   if data:
      dataParsed =  GiiMemory( data )
      for key in dataParsed.blocks:
         size = dataParsed.blocks[ key ].size
         name = dataParsed.blocks[ key ].name
         nused = dataParsed.blocks[ key ].nused
         if not size in usage:
            usage[ size ] = {}
         usage[ size ][ name ] = nused
   return usage

class SingleMioCommit(object):
   """
   Helper class to commit MIO changes as a single mio commit

   This can be used as:

      with SingleMioCommit( dut, ctx.vrfName() ):
         # do config change 1
         # do config change 2
         # ...
   """
   def __init__( self, dut, vrfName, forceCommitOnExit=True ):
      """
      @dut - dut
      @vrfName - vrfName (ctx.vrfName() should work in most cases)
      @forceCommitOnExit - if an mio-commit should be forced on exit
      """
      self._dut = dut
      self._vrfName = vrfName
      self._forceCommitOnExit = forceCommitOnExit

   def __enter__( self ):
      self._dut.giiSetMioCommitFault( 1, vrfName=self._vrfName )

   def __exit__( self, _type, _value, _traceback ):
      self._dut.giiSetMioCommitFault( 0, vrfName=self._vrfName )
      if self._forceCommitOnExit:
         self._dut.giiSetMioCommitForce( vrfName=self._vrfName )

def giiRtHeadToRibdDump( rtDump ):
   '''
   Take the output of giiBgpRouteDump as input and formats it so that
   RibdDumpParser.RouteHead() can parse it. Returns the formatted rtDump
   '''
   rtDump = rtDump.replace( '150 ', '' ) # each line is prefixed with 150 '
   # remove the first line which has the GII command executed
   rtDump = rtDump[ rtDump.index( '\n' ) : ]
   rtDump = rtDump.rstrip() + '\n\n\n' #RouteHead expects 3 \n at the end
   return rtDump

def waitGiiListen( dut, port ):
   '''
   Waits for the GII listener on the given dut/port. There can be a startup
   synchronization problem for certain duts (ribd) created outside of a topology
   and/or gated instances created for agents (Isis, Ospf, Ospf3) launched after the
   dut is created.  Wait for warmup does not work in these cases. When the startup
   problem occurs, changing to giiMode causes an annoying 60 second timeout since
   the telnet session gets rejected.
   '''
   def verifyGiiListen():
      if dut.hostType == 'ribd' :
         # On ribd duts, need to use consoleCli for bash access.
         output = dut.consoleCli().runCmd( 'netstat -ltn' )
      else:
         output = '\n'.join( dut.showCmdIs( 'bash netstat -ltn' ) )
      if "127.0.0.1:%d" % port in output:
         return True
      return False
   Tac.waitFor( verifyGiiListen, description="GII ready on DUT" )

def verifyScheduler( dut, vrfName=None, agent=None ):
   '''
   Verifies that the scheduler stats are actively being collected for the gated
   agent on the given dut.
   '''
   def assertInOutput( output ):
      assert 'Priority RunnableCount     IdleCount DoneCount   TotalRunCount' \
         in output
      assert 'GII_SESSION.127.0.0.1' in output
      assert 'Last 10000 scheduler events' in output

   def verifyStatsAccumulation():
      taskCount = 3
      # Create multiple instances of GII_SESSION and mio tasks
      for x in range( taskCount ):
         if dut.hostType == 'ribd' :
            dut.cliribdShowCmdIs( "show version", agent=agent )
         else:
            dut.cliribdShowCmdIs( "show version", vrf=vrfName, agent=agent )
         output = dut.giiSchedulerShow( vrfName=vrfName, agent=agent )
      # Partition the output into the active and completed task sections
      part1 = output.partition( "150 TaskName (completed)" )
      part2 = part1[ 2 ].partition( "150 Last 10000 scheduler events" )
      activeTasks = part1[ 0 ]
      completedTasks = part2[ 0 ]

      def verifyTaskStats( task ):
         # Only 1 instance of the task in the completed tasks
         assert completedTasks.count( task ) == 1
         # Verify the RunCount and Completed count of the task
         columns = completedTasks.split()
         # Verify we are looking at the right fields
         assert columns[ 1 ] == 'RunCount'
         assert columns[ 7 ] == 'Completed'
         columns = completedTasks.partition( task )[ 2 ].split()
         # The RunCount may be > then taskCount since it may be suspended/resumed
         assert int( columns[ 1 ] ) >= taskCount
         # The completed count is field 7 in the header but field 8 in the data since
         # Last/Max-Delay field takes 1 field in header and 2 in the data.
         assert int( columns[ 8 ] ) == taskCount

      # The active instance will have a '+port' in the name.
      assert 'GII_SESSION.127.0.0.1+' in activeTasks
      # The completed instance will not have a '+port' in the name.
      assert 'GII_SESSION.127.0.0.1+' not in completedTasks
      # No mio:%d instances in the completed tasks
      for x in range( taskCount ):
         assert 'mio:%d' % x not in completedTasks
      verifyTaskStats( 'GII_SESSION.127.0.0.1::recv_method' )
      verifyTaskStats( 'mio::recv_method' )
   assertInOutput( dut.giiSchedulerShow( vrfName=vrfName, agent=agent ) )
   assertInOutput( dut.ribdDump( agent=agent ).dump_ )
   verifyStatsAccumulation()
