#!/var/virtualenv/CloudVirtualEnv/bin/python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import json
import os
import random
import simplejson
import threading
from googleapiclient import discovery
from googleapiclient.errors import HttpError
from oauth2client.client import GoogleCredentials
import ArPyUtils.Decorators
from CloudException import BackendException
import CloudHaBackend
import Tracing
import Tac

t0Saved = Tracing.t0
t1Saved = Tracing.t1
t2Saved = Tracing.t2

def debugMsg( *s ):
   msg = '** %s **: ' % ( threading.current_thread().name )
   for i in s:
      msg += str( i )
   return msg

def t0( *msg ):
   t0Saved( debugMsg( msg ) )

def t1( *msg ):
   t1Saved( debugMsg( msg ) )

def t2( *msg ):
   t2Saved( debugMsg( msg ) )

def _httpErrorMessage( err ):
   if err.resp.get( 'content-type', '' ).startswith( 'application/json' ):
      error = simplejson.loads( err.content ).get( 'error' )
      if error:
         return error.get( 'message', '' )
   return ''

class GcpAccess( object ):
   def __init__( self, project ):
      self.project = project

   def accessKwargs( self ):
      raise NotImplementedError()

class GcpServiceAccountAccess( GcpAccess ):
   def __init__( self, project, serviceFile ):
      super( GcpServiceAccountAccess, self ).__init__( project )
      self._serviceFile = serviceFile

   def accessKwargs( self ):
      os.environ[ 'GOOGLE_APPLICATION_CREDENTIALS' ] = self._serviceFile
      return { 'credentials': GoogleCredentials.get_application_default() }

class GcpDefaultCredentialsAccess( GcpAccess ):
   def accessKwargs( self ):
      return { 'credentials': GoogleCredentials.get_application_default() }

class GcpParsedConfig( object ):
   def __init__( self, access, localRoutes, peerRoutes, httpProxy, httpsProxy ):
      assert access and isinstance( access, GcpAccess )
      self.access = access
      self.localRoutes = localRoutes
      self.peerRoutes = peerRoutes
      self.httpProxy = httpProxy
      self.httpsProxy = httpsProxy

class GcpBackend( CloudHaBackend.BackendBase ):
   resourceBaseUrl = 'https://www.googleapis.com/compute/v1/projects/{project}/'
   vpcResourceUrl = 'global/networks/{vpc}'
   instanceResourceUrl = 'zones/{zone}/instances/{instance}'

   intfsPath = 'instance/network-interfaces/?recursive=true'
   instancePath = 'instance/name'
   zonePath = 'instance/zone'

   filterTemplate = '( destRange="{destRange}" ) AND ( network="{network}" )'

   badProjectMsg = 'Failed to find project {project}'
   badVpcMsg = ( 'The resource \'projects/{project}/global/'
                 'networks/{vpc}\' was not found' )
   badRouteMsg = ( 'The resource \'projects/{project}/global/'
                   'routes/{route}\' was not found' )

   defaultRoutePriority = 1000

   def __init__( self, parsedConfig ):
      super( GcpBackend, self ).__init__()
      assert parsedConfig and isinstance( parsedConfig, GcpParsedConfig )
      self._parsedConfig = parsedConfig
      self._service = None
      self._defaultVpc = ''
      self._macAddrVpcMap = {}
      self._configureProxy()
      self._initializeComputeService()
      self._project = self._parsedConfig.access.project
      self._setMacAddrVpcMap()
      instance = self._getGCPMetadata( self.instancePath )
      zone = self._getGCPMetadata( self.zonePath ).split( '/' )[ -1 ]
      self._nextHop = self._getInstanceResourceUrl( instance, zone )

   def _initializeComputeService( self ):
      accessKwargs = self._parsedConfig.access.accessKwargs()
      self._service = discovery.build( 'compute', 'v1', **accessKwargs )

   def _setMacAddrVpcMap( self ):
      intfs = json.loads( self._getGCPMetadata( self.intfsPath ) )
      for intf in intfs:
         mac = intf[ 'mac' ]
         # intf[ 'network' ] returns the resource id of the vpc (which includes the
         # project id) in the following format:
         # "projects/<project-id>/networks/<vpc-name>"
         # We need just the vpc name, i.e., the part after the last '/'
         vpc = intf[ 'network' ].split( '/' )[ -1 ]
         # The gcp instance metadata is used to construct a map of mac address to vpc
         # for each network interface of the instance.
         # For HA routes where the interface name is specified in the config,
         # self._macAddrVpcMap is used to derive the vpc containing the interface,
         # using the mac address of the interface.
         self._macAddrVpcMap[ mac ] = vpc
         # Use vpc of the first interface as default vpc.
         if not self._defaultVpc:
            self._defaultVpc = vpc

   def _configureProxy( self ):
      if self._parsedConfig.httpProxy:
         os.environ[ 'http_proxy' ] = self._parsedConfig.httpProxy
      if self._parsedConfig.httpsProxy:
         os.environ[ 'https_proxy' ] = self._parsedConfig.httpsProxy

   def _getRoute( self, vpc, tags, destination ):
      filterStr = self.filterTemplate.format( destRange=destination,
            network=self._getVpcResourceUrl( vpc ) )
      request = self._service.routes().list( project=self._project,
            filter=filterStr )
      while request is not None:
         response = request.execute()
         for route in response.get( 'items', [] ):
            if ( route.get( 'name', '' ).startswith( 'cloudha-' ) and
                  route.get( 'tags', [] ) == tags ):
               return route
         request = self._service.routes().list_next(
               previous_request=request, previous_response=response )
      return None

   def _replaceRoute( self, oldRoute, vpc, tags, destination ):
      # Delete 'oldRoute' and create a new route in GCP.
      # 'tags' is the list of network tags that will be applied to the new route.
      # 'destination' specifies the destRange of the new route.
      def routeExists( routeName ):
         request = self._service.routes().get( project=self._project,
               route=routeName )
         try:
            _response = request.execute()
         except HttpError as e:
            message = _httpErrorMessage( e )
            if message == self.badRouteMsg.format(
                  project=self._project, route=routeName ):
               return False
            raise e
         return True

      def routeName( maxTries=10 ):
         while maxTries:
            name = 'cloudha-%0.12d' % random.randrange( 10**12 )
            if not routeExists( name ):
               return name
            else:
               maxTries = maxTries - 1
         msg = "Failed to find available route name"
         t0( msg )
         raise BackendException( msg )

      newRouteExists = False
      if oldRoute is not None:
         if ( oldRoute[ 'network' ] == self._getVpcResourceUrl( vpc ) and
               oldRoute[ 'destRange' ] == destination and
               oldRoute[ 'nextHopInstance' ] == self._nextHop and
               oldRoute[ 'tags' ] == tags ):
            newRouteExists = True
         else:
            request = self._service.routes().delete(
                  project=self._project, route=oldRoute[ 'name' ] )
            _response = request.execute()
      if not newRouteExists:
         route_body = {
               'destRange': destination,
               'name': routeName(),
               'network': self._getVpcResourceUrl( vpc ),
               'priority': self.defaultRoutePriority,
               'nextHopInstance': self._nextHop,
               'tags': tags,
         }
         request = self._service.routes().insert( project=self._project,
               body=route_body )
         _response = request.execute()

   def _updateRouteTables( self, peer ):
      try:
         routes = ( self._parsedConfig.peerRoutes if peer else
                    self._parsedConfig.localRoutes )
         for route in routes:
            macAddr, vpc, destination, tag = route
            tags = [ tag ] if tag else []
            if not vpc:
               if macAddr:
                  vpc = self._macAddrVpcMap[ macAddr ]
               else:
                  vpc = self._defaultVpc
            oldRoute = self._getRoute( vpc, tags, destination )
            self._replaceRoute( oldRoute, vpc, tags, destination )
      # pylint: disable-msg=broad-except
      except Exception as e:
         t0( 'Failed to update %s routes: %s' % ( 'peer' if peer else 'local', e ) )
         self.updateResultMessage( str( e ) )
         return False
      return True

   def updatePeerRouteTables( self ):
      t0( 'UpdatePeerRouteTables called' )
      return self._updateRouteTables( peer=True )

   def updateLocalRouteTables( self ):
      t0( 'UpdateLocalRouteTables called' )
      return self._updateRouteTables( peer=False )

   def _getGCPMetadata( self, path ):
      return Tac.run( [
            'curl', '-s', '--noproxy', '*', '-H', 'Metadata-Flavor:Google',
            'http://metadata.google.internal/computeMetadata/v1/' + path ],
         stdout=Tac.CAPTURE )

   def _getInstanceResourceUrl( self, instance, zone ):
      return ( self.resourceBaseUrl.format( project=self._project ) +
               self.instanceResourceUrl.format( instance=instance, zone=zone ) )

   def _getVpcResourceUrl( self, vpc ):
      return ( self.resourceBaseUrl.format( project=self._project ) +
               self.vpcResourceUrl.format( vpc=vpc ) )

   def _validateProject( self ):
      # Verify that the project exists
      project = self._parsedConfig.access.project
      request = self._service.projects().get( project=project )
      try:
         _response = request.execute()
      except HttpError as e:
         message = _httpErrorMessage( e )
         if message == self.badProjectMsg.format( project=project ):
            msg = 'Project %s does not exist' % project
            t0( msg )
            raise BackendException( msg )
      except Exception as e:
         raise BackendException( str( e ) )
   
   def _validateRoutes( self, peer=False ):
      routes = ( self._parsedConfig.peerRoutes if peer else
                 self._parsedConfig.localRoutes )
      for route in routes:
         macAddr, vpc, _, _ = route
         if macAddr and macAddr not in self._macAddrVpcMap:
            msg = 'Interface with mac address %s does not exist' % macAddr
            t0( msg )
            raise BackendException( msg )
         if vpc:
            self._validateVpc( vpc )
   
   def _validateVpc( self, vpc ):
      # Verify that the vpc exists
      project = self._parsedConfig.access.project
      request = self._service.networks().get( project=project, network=vpc )
      try:
         _response = request.execute()
      except HttpError as e:
         message = _httpErrorMessage( e )
         if message == self.badVpcMsg.format( project=project, vpc=vpc ):
            msg = 'VPC %s does not exist' % vpc
            t0( msg )
            raise BackendException( msg )
      except Exception as e:
         raise BackendException( str( e ) )

   @ArPyUtils.Decorators.retry( retryCheckEmbeddedMethodName='transientFailure',
                                attempts=4, retryInterval=30 )
   def configValidator( self ):
      t1( 'Invoked AWS Backend configValidator' )
      try:
         self._validateProject()
         self._validateRoutes( peer=True )
         self._validateRoutes( peer=False )
      except BackendException as e:
         t0( str( e ) )
         self.updateResultMessage( str( e ) )
         return False
      return True
