#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This plugin provides a FruFactory that is used when EOS in running
# inside a Container. We synthesize a FDL based on what we can read out of
# the Linux kernel. We assume that we're on a fixed system. 
# Interfaces that begin with 'docker' are assumed to be Management Ethernet
# All other interfaces are numbered.

import Ark
import Cell
import FirmwareRev
import Fru
from Fru.Driver import FruDriver
import Tac
import Tracing

import os
import re
from eunuchs.if_arp_h import ARPHRD_ETHER
from FruPlugin.GenericPc import generateSystemMacAddr
from EbraTestBridgeConstants import bridgingEtbaConfigPath
from EbraTestBridgeConstants import bridgingEtbaConfigType
from Fru.FruBaseVeosDriver import parseVeosConfig, handleEtbaOverride

__defaultTraceHandle__ = Tracing.Handle( "Fru.cEOSLab" )
t0 = Tracing.trace0
t2 = Tracing.trace2
t9 = Tracing.trace9

class CEosConfig( object ):
   def __init__( self, ceosConfigPath=None, systemMacAddrPath=None ):
      t0( 'CEosConfig: Building a list of interfaces for cEOS Lab' )
      if ceosConfigPath is None:
         ceosConfigPath = '/mnt/flash/ceos-config'
      kwargs = { 'veosConfigPath': ceosConfigPath }
      if systemMacAddrPath is not None:
         kwargs[ 'systemMacAddrPath' ] = systemMacAddrPath

      self.ceosConfig = parseVeosConfig( **kwargs )

   def getInterfaces( self, intfType ):
      t0( "cEOSLabDriver: Getting Interfaces" )
      if intfType is None:
         intfType = "eth"
      devNames = {}
      # Get all ethernet interface device names
      sysClassNet = "/sys/class/net"
      for devName in os.listdir( sysClassNet ):
         if not os.path.isdir( os.path.join( sysClassNet, devName ) ):
            # Possibly a file, like "bonding_masters".
            continue

         devType = int(file( os.path.join( sysClassNet, devName, "type" )).read())
         t9( devName, devType )
         # This type apparently corresponds to include/linux/if_arp.h
         # ARPHRD_* values.
         if devType != ARPHRD_ETHER:
            continue
         if not os.path.exists( os.path.join( sysClassNet, devName, "device" ) ):
            if not devName.startswith( "Ethernet" ) and \
               not devName.startswith( "eth" ) and \
               not devName.startswith( "et" ) and \
               not devName.startswith( "ma" ) and \
               not devName.startswith( intfType ):
               continue
         mac = file( os.path.join( sysClassNet, devName, "address" )).read().strip()
         devNames[ devName ] = mac
      return devNames

   def matchWithRegex( self, devName ):
      intfNamesRe = [
               r"(\S+?)(\d+)$",
               r"(\S+?)(\d+)_(\d+)$",
               r"(\S+?)(\d+)_(\d+)_(\d+)$",
               ]
      intftype = ""
      label = 0
      subLabel = 0
      subSubLabel = 0
      m1 = re.match( intfNamesRe[0], devName )
      m2 = re.match( intfNamesRe[1], devName )
      m3 = re.match( intfNamesRe[2], devName )
      if m3:
         intftype, label, subLabel, subSubLabel = m3.groups()
      elif m2:
         intftype, label, subLabel = m2.groups()
      elif m1:
         intftype, label = m1.groups()
      else:
         t0( 'Unsupported device %s' % devName )
      return intftype, int( label ), int( subLabel), int( subSubLabel )

   def getSerialNumber( self ):
      return self.ceosConfig[ "SERIALNUMBER" ]

   def getSystemMacAddr( self ):
      return self.ceosConfig[ 'SYSTEMMACADDR' ]

   def handleEtbaOverride( self ):
      handleEtbaOverride( self.ceosConfig )

class cEOSLabDriver( FruDriver ):
   managedTypeName = "Eos::CEOSLabFru"
   managedApiRe = ".*"
   driverPriority = 1

   def __init__( self, cEOSLab, parentMibEntity, parentDriver, driverCtx ):
      t0( "cEOSLabDriver: Populating the Entity Mib" )
      mapping = {}
      self.portId = 0
      assert parentMibEntity is None
      sysdbRoot = driverCtx.sysdbRoot
      self.entityMibStatus = sysdbRoot[ "hardware" ][ "entmib" ]
      self.entityMibStatus.fixedSystem = ( 1, 0, "FixedSystem" )
      self.systemMib_ = self.entityMibStatus.fixedSystem
      self.systemMib_.modelName = "cEOSLab"
      self.systemMib_.description = "cEOSLab"
      self.ceosConfig = CEosConfig()
      self.systemMib_.serialNum = self.ceosConfig.getSerialNumber()
      self.systemMib_.swappability = "notSwappable"
      # cEOSLab can be found at /ar/Sysdb/hardware/inventory/cEOSLab
      self.cEOSLab = cEOSLab
      if self.entityMibStatus.nextPhysicalIndex < 2:
         self.entityMibStatus.nextPhysicalIndex = 2

      self.ceosConfig.handleEtbaOverride()

      # -------------
      # Emulate the FixedConfig FruPlugin, i.e.
      #
      #    o create per-cell state ("sys/config/cell/<cellId>",
      #      "hardware/cell/<cellId>")
      #
      #    o map the appropriate roles to our cell
      #
      #    o Set the default port roles we support

      myCellId = Cell.cellId()
      roles = [ "AllCells", "AllSupervisors", "ActiveSupervisor" ]
      Fru.createAndConfigureCell( driverCtx.sysdbRoot,
                                  cEOSLab,
                                  myCellId,
                                  roles )

      for portRole in [ "Management", "Switched" ]:
         cEOSLab.portRole[ portRole ] = portRole

      # -------------
      # Populate the rest of the inventory tree based on the
      # interfaces we find in the kernel
      myIntfType = os.environ.get( "INTFTYPE" )
      devNames = self.ceosConfig.getInterfaces( myIntfType )
      t0( "cEOSLabDriver: Creating Interfaces", devNames )
      self.lowestMacAddr = None
      etbaConfig = sysdbRoot[ 'bridging' ][ 'etba' ][ 'config' ]
      linuxConfig = sysdbRoot[ 'bridging' ][ 'linux' ][ 'config' ]
      ethPortDir = cEOSLab.component.newEntity(
            "Inventory::VirtualEthPortDir", "ethPortDir" )
      pciPortDir = cEOSLab.component.newEntity(
            "Inventory::PciPortDir", "pciPortDir" )
      phyEthtoolPhyDir = cEOSLab.component.newEntity(
            "Inventory::Phy::EthtoolDir", "phy" )
      intfConfigDir = sysdbRoot[ 'interface' ][ 'config' ]\
            [ 'eth' ][ 'phy' ][ 'default' ]
      defaultSwitchedEthIntfConfig = intfConfigDir.defaultIntfConfig[
            "DefaultEthSwitchedPort" ]
      defaultSwitchedEthIntfConfig.linkModeLocal = "linkModeForced1GbpsFull"

      mapEth0Intf = os.environ.get( "MAPETH0" )
      mgmtIntf = os.environ.get( "MGMT_INTF" )
      maIntfAllowed = os.environ.get( "MA_INTF_ALLOWED" )
      for devName in devNames:
         # Support following types of interfaces:
         #  - EthernetX, EthernetX_X, EthernetX_X_X
         #  - ethX, ethX_X, ethX_X_X 
         #  - etX, etX_X, etX_X_X
         #  - maX
         intftype, label, subLabel, subSubLabel = \
               self.ceosConfig.matchWithRegex( devName )
         mac = devNames[ devName ]

         if ( maIntfAllowed and devName.startswith( "ma" ) ) or \
            ( mgmtIntf and devName == mgmtIntf ):
            if label in mapping and mgmtIntf:
               # prefer MGMT_INTF over interface 'maX', i.e, when MGMT_INTF env
               # variable is set to 'eth0' and 'ma0' interface is also present
               # then use 'eth0'.
               if mapping[ label ] != mgmtIntf:
                  # Delete conflicting interface that we already created
                  existingPhyName = "Phy%s" % ( mapping[ label ] )
                  existingPhyEnt = phyEthtoolPhyDir.phy.get( existingPhyName )
                  existingPort = existingPhyEnt.port
                  del phyEthtoolPhyDir.phy[ existingPhyName ]
                  del pciPortDir.port[ existingPort.id ]
                  del existingPort
                  del existingPhyEnt 
               else:
                  # We already created the one we desired, skip the conflicting one
                  continue

            port = pciPortDir.newPort( self.nextPortId() )
            port.description = "Management Ethernet Port %d" % label
            port.role = "Management"
            # Don't set a pci address, the EthPciPort fru driver will assign
            # things correctly based on the device name

            # Create an Inventory::PhyEthtoolPhy
            phy = phyEthtoolPhyDir.newPhy( "Phy%s" % ( devName ) )
            phy.port = port
            port.label = label
            port.macAddr = mac
            mapping[ label ] = devName
            continue

         # If Ethernet510 is being used, report conflict and terminate the test
         if label == 510 and mapEth0Intf:
            assert False, "eth0 is conflict with %s" % devName

         # eth0 interface on the host is not avaliable in EOS for configurations
         # Rename eth0 to unused Ethernet510 in order to manipulate the interface
         if not label and mapEth0Intf:
            label = 510
         
         # The upper limit of ethernet interface portID is 511
         # If portID exceed 511, it is a unvalid port
         # The lower limit is 1, label can not be zero
         if label > 511 or label == 0:
            continue
         
         # Track the lowest mac address in the system. This is
         # used below to generate the system mac address
         if not self.lowestMacAddr or ( mac < self.lowestMacAddr ):
            self.lowestMacAddr = mac

         # Create an Inventory::VirtualEthPort
         if intftype != myIntfType:
            continue
         port = ethPortDir.newPort( self.nextPortId() )
         port.description = "Front-Panel Port %d" % label
         port.role = "Switched"
         tapDevice = Tac.Value( "Bridging::Etba::Config::TapDevice" )
         tapDevice.name = devName
         tapDevice.fileno = 0
         if subSubLabel:
            etbaConfig.extTapDevice[ "Ethernet%d/%d/%d" %
                  ( label, subLabel, subSubLabel ) ] = tapDevice
         elif subLabel:
            etbaConfig.extTapDevice[ "Ethernet%d/%d" %
                  ( label, subLabel ) ] = tapDevice
         else:
            etbaConfig.extTapDevice[ "Ethernet%d" % label ] = tapDevice

         port.label = label
         port.macAddr = mac
         if subLabel:
            port.subLabel = subLabel
         if subSubLabel:
            port.subSubLabel = subSubLabel

         t0( "device %s, mac %s" % ( devName, mac ) )

      FruDriver.__init__(
            self, cEOSLab, parentMibEntity, parentDriver, driverCtx )
      self.instantiateChildDrivers( cEOSLab, self.systemMib_ )

      self.genSystemMacAddr()
      self.genEosVersion()
      self.setDeviceName( mapping )
      etbaEnabled = "ETBA" in os.environ
      etbaConfig.complete = etbaEnabled
      linuxConfig.enabled = not etbaEnabled

      # Declare FruReady
      hwCellDir = driverCtx.sysdbRoot[ 'hardware' ][ 'cell' ][ '%d' % myCellId ]
      assert hwCellDir
      hwCellDir.newEntity( 'Tac::Dir', 'FruReady' )

      # Declare success at the end
      self.systemMib_.initStatus = "ok"

   def nextPortId( self ):
      self.portId += 1
      return self.portId

   def genSystemMacAddr( self ):
      t0( 'Generating system MAC address' )
      # Set the system mac address. We have no choice but to choose
      # this randomly - see the comment in generateSystemMac for details.
      #
      # We allow the mac address to be overwritten by creating the
      # file /mnt/flash/system_mac_address. This gives us a hook in
      # case the automatically generated mac address is not
      # sufficient. Care must be taken when cloning systems with
      # a /mnt/flash/system_mac_address file.
      systemMacAddr = self.ceosConfig.getSystemMacAddr()
      if self.lowestMacAddr and not systemMacAddr:
         systemMacAddr = generateSystemMacAddr( self.lowestMacAddr )
      if systemMacAddr:
         self.entityMibStatus.systemMacAddr = systemMacAddr

      self.systemMib_.firmwareRev = FirmwareRev.abootFirmwareRev()

   def genEosVersion( self ):
      t0( 'Generating EOS version' )
      import EosVersion
      vi = EosVersion.VersionInfo( sysdbRoot=None )
      if vi.version() is None:
         self.systemMib_.softwareRev = ""
      else:
         self.systemMib_.softwareRev = vi.version()

   def setDeviceName( self, mapping ):
      t0( 'Setting deviceName in PhyEthtool' )
      # For all front-panel interfaces managed by PhyEthtool,
      # we have to set the deviceName on the EthIntfStatus
      # ourselves (just like we do in Fru for the management
      # interfaces).
      phyEthtoolPhyDir = self.cEOSLab.component.get( "phy" )
      if phyEthtoolPhyDir:
         for invPhy in phyEthtoolPhyDir.phy.values():
            intfStatus = invPhy.port.intfStatus
            if not intfStatus.deviceName:
               intfStatus.deviceName = mapping[ invPhy.port.label ]

def containerFactory( parent, fdl, idInParent ):
   assert idInParent is None
   cEOSLab = parent.newEntity( "Eos::CEOSLabFru", "cEOSLab" )
   cEOSLab.component = ( "component", )
   cEOSLab.api = Ark.getPlatform() or ""
   cEOSLab.managingCellId = Cell.cellId()
   cEOSLab.valid = True
   cEOSLab.generationId = 1
   return cEOSLab

def Plugin( context ):
   context.registerDriver( cEOSLabDriver )
   context.registerFruFactory( containerFactory, Fru.containerFactoryId )
   mg = context.entityManager.mountGroup()
   mg.mount( bridgingEtbaConfigPath, bridgingEtbaConfigType, 'w' )
   mg.mount( 'bridging/linux/config', 'Bridging::Linux::Config', 'w' )
   mg.close( None )
