#!/usr/bin/env python
# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Fru, Tracing, XcvrLib

__defaultTraceHandle__ = Tracing.Handle( "XcvrFru" )

t0 = Tracing.trace0
t1 = Tracing.trace1

xcvrInit = "xcvr"

RS528 = 'ReedSolomon'
RS544 = 'ReedSolomon544'
FCFEC = 'FireCode'
NOFEC = 'Disabled'

class XcvrSlotDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::Xcvr::SlotDir"
   # managedApiRe = "$"
   provides = [ xcvrInit ]
   requires = [ Fru.FruDriver.interfaceInit, Fru.FruDriver.systemInit ]

   def __init__( self, slotDir, fruEntMib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, slotDir, fruEntMib,
                              parentDriver, driverCtx )
      t0( "Handling Inventory::Xcvr::SlotDir for %s" % Fru.fruBaseName( slotDir ) )

      xcvrDir = driverCtx.entity( "hardware/xcvr/config" ).parent
      envDir = driverCtx.sysdbRoot[ "environment" ]
      envTempConfig = envDir[ "temperature" ][ "config" ]
      topoDir = driverCtx.entity( "hardware/l1/topology" )

      sliceId = Fru.fruBaseName( slotDir )

      # Now that we're know that we have xcvrs in the system, load the
      # attentuation map into sysdb
      attnTable = xcvrDir[ "attenuation" ]
      XcvrLib.populateAttenuationMap( attnTable )
      
      fruHelper = Tac.newInstance("Xcvr::FruPluginHelper", "")
      fruHelper.slotDir = slotDir
      fruHelper.xcvrDir = xcvrDir
      fruHelper.envTempConfig = envTempConfig
      fruHelper.attnTable = attnTable
      fruHelper.xgc = driverCtx.entity( "hardware/xcvr/xgc" )
      fruHelper.fruEntMib = fruEntMib
      fruHelper.fruUtil = Fru.fruUtil()
      fruHelper.generationId = Fru.powerGenerationId( slotDir )
      fruHelper.sliceName = sliceId
      fruHelper.statusRootDir = driverCtx.entity( "hardware/archer/xcvr/status" )
      fruHelper.coherentSliceDir = driverCtx.entity(
                             "hardware/archer/phy/config/data/coherent/slice" )
      fruHelper.xcvrL1TopoFruHelper = Tac.newInstance(
                             "Hardware::L1Topology::XcvrFruPluginHelper", topoDir )
      self.maybeUpdateCoherentConfig( fruHelper )
      fruHelper.doPopulateHardware()
      self.fruHelper = fruHelper
      self.maybeUpdateTuningParameters( fruHelper )
      self.maybePopulateL1Topology( topoDir, sliceId, slotDir )

      topoHelper = Tac.newInstance(
         "Hardware::L1Topology::XcvrFruPluginHelper", topoDir )
      xcvrTopoDir = topoHelper.xcvrTopologyDir( sliceId )
      if xcvrTopoDir:
         for i, xcvrId in slotDir.xcvrIdOrder.items():
            xcvrTopoDir.xcvrIdOrder[ i ] = xcvrId

         t0( "xcvrIdOrder:", xcvrTopoDir.xcvrIdOrder.values() )

   def maybeUpdateTuningParameters( self, fruHelper ):
      t0( "maybeUpdateTuningParameters" )
      for slotKey in fruHelper.slotDir.xcvrSlot:
         slot = fruHelper.slotDir.xcvrSlot[ slotKey ]
         if slot.slotType == 'qsfp':
            intfName = slot.derivedName
            conf = fruHelper.allConfigDir.xcvrConfig[ intfName ]
            self.updateQsfpTuningParameters( conf, slot )
         elif slot.slotType in [ 'qsfpDd', 'osfp', 'qsfpCmis' ]:
            intfName = slot.derivedName
            conf = fruHelper.allConfigDir.xcvrConfig[ intfName ]
            self.updateCmisTuningParameters( conf, slot )

   def updateQsfpTuningParameters( self, xcvrConfig, slot ):
      t0( "updateQsfpTuningParameters for %s" % xcvrConfig.name )
      for lane in range( 0, 4 ):
         if lane in slot.scaledXcvrRxTraceLength:
            pre, amp = self.translateRxTrace( slot.scaledXcvrRxTraceLength[ lane ] )
         else:
            pre = Tac.Value( "Xcvr::TuningValue", False, 0, False )
            amp = Tac.Value( "Xcvr::TuningValue", False, 0, False )
         xcvrConfig.rxOutputPreEmphasis[ lane ] = pre
         xcvrConfig.rxOutputAmplitude[ lane ] = amp

      for lane in range( 0, 4 ):
         if lane in slot.scaledXcvrTxTraceLength:
            eq = self.translateTxTrace( slot.scaledXcvrTxTraceLength[ lane ] )
         else:
            eq = Tac.Value( "Xcvr::TuningValue", False, 0, False )
         xcvrConfig.txInputEqualization[ lane ] = eq

   def updateCmisTuningParameters( self, xcvrConfig, slot ):
      t0( "updateCmisTuningParameters for %s" % xcvrConfig.name )
      for lane in range( 0, 8 ):
         # TODO: Currently all fdl osfp/qsfpDd tuning config are not valid.
         #       We don't know how to translate them yet. Update the logic
         #       of translating trace length to tuning value. Bug 345549
         pre = Tac.Value( "Xcvr::TuningValue", False, 0, False )
         post = Tac.Value( "Xcvr::TuningValue", False, 0, False )
         amp = Tac.Value( "Xcvr::TuningValue", False, 0, False )
         adaEq = Tac.Value( "Xcvr::TuningValue", False, 0, False )
         fixEq = Tac.Value( "Xcvr::TuningValue", False, 0, False )

         laneTuningParam = Tac.Value( "Xcvr::LaneTuningParameter" )
         laneTuningParam.rxOutputPreEmphasis = pre
         laneTuningParam.rxOutputPostEmphasis = post
         laneTuningParam.rxOutputAmplitude = amp
         laneTuningParam.txInputAdaptiveEqEnable = adaEq
         laneTuningParam.txInputEqualization = fixEq
         xcvrConfig.fdlTuningParam[ lane ] = laneTuningParam

   def maybeUpdateCoherentConfig( self, fruHelper ):
      for _, slot in fruHelper.slotDir.xcvrSlot.iteritems():
         if slot.slotType == 'cfp2' and slot.xcvrSlotCapabilities.cfp2DcoSupported:
            inv = Fru.fruBase( fruHelper.slotDir )
            cellId = Fru.fruBase( fruHelper.slotDir ).managingCellId
            sliceId = Fru.fruBase( fruHelper.slotDir ).sliceId
            if cellId:
               coherentConfig = Fru.Dep( fruHelper.coherentSliceDir, inv ).newEntity(
                 "Phy::Coherent::PhyCoherentConfig", "FixedSystem" )
            else:
               coherentConfig = Fru.Dep( fruHelper.coherentSliceDir, inv ).newEntity(
                  "Phy::Coherent::PhyCoherentConfig", sliceId )

            coherentConfig.generation = Tac.Value( "Ark::Generation",
                                                Fru.powerGenerationId( inv ) + 1,
                                                True )

   def maybePopulateL1Topology( self, topoDir, sliceId, slotDir ):
      topoHelper = None
      for slot in slotDir.xcvrSlot.values():
         if not slot.txTraceId or not slot.rxTraceId:
            continue
         if not topoHelper:
            topoHelper = Tac.newInstance(
               "Hardware::L1Topology::XcvrFruPluginHelper", topoDir )
         for lane, txTraceId in slot.txTraceId.iteritems():
            topoHelper.addXcvrLaneTopology( sliceId, slot.slotId, lane, True,
                                            txTraceId )
            topoHelper.addXcvrLaneTopology( sliceId, slot.slotId, lane, False,
                                            slot.rxTraceId[ lane ] )

   # Returns tuple ( rxOutputPreEmphasis, rxOutputAmplitude )
   @staticmethod
   def translateRxTrace( traceLength ):
      # Configure tuning, and don't rely on any module defaults
      return ( Tac.Value( "Xcvr::TuningValue", bool( traceLength ), 0, False ),
               Tac.Value( "Xcvr::TuningValue", True, 2, False ) )

   # Returns txInputEqualization
   @staticmethod
   def translateTxTrace( traceLength ):
      if traceLength:
         # Configure tuning, and don't rely on any module defaults
         return Tac.Value( "Xcvr::TuningValue", True, 0, False )
      else:
         return Tac.Value( "Xcvr::TuningValue", False, 0, False )

#----------------------------------------------------------------------
#
# Mount the relevant state from Sysdb.
#
#----------------------------------------------------------------------
def Plugin( context ):
   context.registerDriver( XcvrSlotDriver )

   mg = context.entityManager.mountGroup()
   mg.mount( 'hardware/xcvr/attenuation', 'Xcvr::ModelToAttenuation', 'w' )
   mg.mount( 'hardware/xcvr/config', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/xcvr/xgc', 'Xcvr::Xgc', 'r' )
   # NOTE - we don't need to mount any other subdirectories of
   # hardware/xcvr/{config,status} (like the all subdirectories), as
   # we're doing an immediate mount above. This has to be
   # mount-immediate to ensure that Fru mounts everything that it has
   # previously created in the agent restart case.
   mg.mount( 'environment/temperature/config',
             'Environment::Temperature::Config', 'w' )
   mg.mount( 'hardware/l1/topology', 'Tac::Dir', 'wi' )
   mg.close( None )

   # BUG48760 workaround. Objects in status keep pointers to those in
   # config. Tacc attrlog has problem dealing with it. See comments
   # attached to the bug for detailed analysis.
   context.deferredMount( 'hardware/archer/xcvr/status', 'Tac::Dir', 'ri' )
