#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This plugin provides a FruFactory that is used when EOS in running
# in a VM (vEOS). We synthesize a FDL based on what we can read out of
# the Linux kernel. We assume that we're on a fixed system. Interfaces
# that begin with 'ma' are assumed to be Management Ethernet
# Interfaces, while interfaces that begin with 'vmnicet' are assumed
# to be virtual nic interfaces for front-panel switchports. All other
# interfaces are ignored.

import Cell
import FirmwareRev
import Fru
import Tac
import Tracing

import os
import re
from eunuchs.if_arp_h import ARPHRD_ETHER
from FruPlugin.GenericPc import renameInterfaces, generateSystemMacAddr
from EbraTestBridgeConstants import bridgingEtbaConfigPath, bridgingEtbaConfigType
from Fru.FruBaseVeosDriver import FruBaseVeosDriver, parseVeosConfig, \
                                  handleEtbaOverride

t0 = Tracing.trace0
t9 = Tracing.trace9

class VEosDriver( FruBaseVeosDriver ):
   managedTypeName = "Eos::GenericPcFru"
   managedApiRe = "veos$"
   # Higher priority than the default GenericPcDriver
   driverPriority = 3

   def __init__( self, genericPc, parentMibEntity, parentDriver, driverCtx ):
      t0( "VEosDriver: Populating the Entity Mib" )
      assert ( parentMibEntity is None )
      sysdbRoot = driverCtx.sysdbRoot
      entityMibStatus = driverCtx.entity( "hardware/entmib" )
      entityMibStatus.fixedSystem = ( 1, 0, "FixedSystem" )
      self.systemMib_ = entityMibStatus.fixedSystem
      veosConfig = parseVeosConfig()
      self.systemMib_.modelName = "vEOS"
      self.systemMib_.description = "vEOS"
      self.systemMib_.serialNum = veosConfig[ "SERIALNUMBER" ]
      self.systemMib_.swappability = "notSwappable"
      if entityMibStatus.nextPhysicalIndex < 2:
         entityMibStatus.nextPhysicalIndex = 2

      handleEtbaOverride( veosConfig )

      # -------------
      # Emulate the FixedConfig FruPlugin, i.e.
      #
      #    o create per-cell state ("sys/config/cell/<cellId>",
      #      "hardware/cell/<cellId>")
      #
      #    o map the appropriate roles to our cell
      #
      #    o Set the default port roles we support

      myCellId = Cell.cellId()
      roles = [ "AllCells", "AllSupervisors", "ActiveSupervisor" ]
      Fru.createAndConfigureCell( driverCtx.sysdbRoot,
                                  genericPc,
                                  myCellId,
                                  roles )

      for portRole in [ "Management", "Switched" ]:
         genericPc.portRole[ portRole ] = portRole

      # -------------
      # Populate the rest of the inventory tree based on the
      # interfaces we find in the kernel

      t0( "VEosDriver: Getting Interfaces" )
      # Get all ethernet interface device names
      sysClassNet = "/sys/class/net"
      devNames = []
      for devName in os.listdir( sysClassNet ):
         if not os.path.isdir( os.path.join( sysClassNet, devName ) ):
            # Possibly a file, like "bonding_masters".
            continue

         devType = int(file( os.path.join( sysClassNet, devName, "type" )).read())
         t9( devName, devType )
         # This type apparently corresponds to include/linux/if_arp.h
         # ARPHRD_* values.
         if devType != ARPHRD_ETHER:
            continue
         
         devNames.append( devName )

      devNames = renameInterfaces( devNames, veosConfig[ "MODE" ] )
         
      t0( "VEosDriver: Creating Interfaces", devNames )
      lowestMacAddr = None
      etbaConfig = sysdbRoot[ 'bridging' ][ 'etba' ][ 'config' ]
      linuxConfig = sysdbRoot[ 'bridging' ][ 'linux' ][ 'config' ]      
      for devName in devNames:
         # We support 3 types of interfaces:
         #    - maN, which we assume are management interfaces named
         #      by the boot loader
         #    - vmnicetN, which we assume are test front-panel interfaces
         #      created as part of vEOS which will be managed by Etba.
         #    - etN, which we assume are test front-panel interfaces
         #      created as part of vEOS which will be managed by
         #      PhyEthtool
         # All other interfaces are ignored
         m = re.match( "(ma|vmnicet|et)(\d+)", devName )
         if not m:
            continue

         ethPortDir = genericPc.component.newEntity(
            "Inventory::EthPortDir", "ethPortDir" )
         pciPortDir = genericPc.component.newEntity(
            "Inventory::PciPortDir", "pciPortDir" )
         phyEthtoolPhyDir = genericPc.component.newEntity(
            "Inventory::Phy::EthtoolDir", "phy" )

         ( intftype, index ) = m.groups()
         label = int( index )
         if intftype == "ma":
            portId = label + 100
         else:
            portId = label

         mac = file( os.path.join( sysClassNet, devName, "address" )
                     ).read().strip()

         # Track the lowest mac address in the system. This is
         # used below to generate the system mac address
         if not lowestMacAddr or ( mac < lowestMacAddr ):
            lowestMacAddr = mac
         
         if intftype == "et":
            # There's a front-panel interface that we're going to manage
            # with PhyEthtool.
            # Hack - change the default switched interface config
            # to be linkModeForced1GbpsFull instead of
            # linkModeForced10GbpsFull, as otherwise the PhyEthtool
            # agent will fail trying to manage front-panel ports.
            intfConfigDir = driverCtx.entity( 'interface/config/eth/phy/default' )
            defaultSwitchedEthIntfConfig = intfConfigDir.defaultIntfConfig[
               "DefaultEthSwitchedPort" ]
            defaultSwitchedEthIntfConfig.linkModeLocal = "linkModeForced1GbpsFull"

         # Create an Inventory::EthPort
         if intftype == "ma":
            port = pciPortDir.newPort( portId )
            port.description = "Management Ethernet Port %d" % label
            port.role = "Management"
            # Don't set a pci address, the EthPciPort fru driver will assign
            # things correctly based on the device name

            # Create an Inventory::PhyEthtoolPhy
            phy = phyEthtoolPhyDir.newPhy( "Phy%s%d" %( intftype, label ) )
            phy.port = port

         elif intftype == "et":
            linuxConfig.enabled = True
            
            port = ethPortDir.newPort( portId )
            port.description = "Front-Panel Port %d" % label
            port.role = "Switched"
            port.macAddr = mac
            
            # Create an Inventory::PhyEthtoolPhy
            phy = phyEthtoolPhyDir.newPhy( "Phy%s%d" %( intftype, label ) )
            phy.port = port
            
         elif intftype == "vmnicet":
            tapDevice = Tac.Value( "Bridging::Etba::Config::TapDevice" )
            tapDevice.name = devName
            tapDevice.fileno = 0
            etbaConfig.extTapDevice[ "Ethernet%d" % label ] = tapDevice

            port = ethPortDir.newPort( portId )
            port.description = "Front-Panel Port %d" % label
            port.role = "Switched"
            port.macAddr = mac

            # Force the MTU of these ports to 10000. This is how
            # real hardware ports behave.
            try:
               Tac.run( [ "ifconfig", devName, "mtu", "10000" ] )
            except Tac.SystemCommandError:
               # Shoot, we're seem to be using a virtual network
               # adapter that does not support jumbo frames
               # (ie hyper-v).
               # Fall back to 1500 bytes
               Tac.run( [ "ifconfig", devName, "mtu", "1500" ] )

            # Make sure our vmnicet interfaces don't respond to arp
            # requests. This is done by the etN interface created on
            # top of it
            f = file( "/proc/sys/net/ipv4/conf/%s/arp_ignore" % devName, "w" )
            f.write( "1" )
            f.close()
            # Make sure the device is actually up. This is required
            # before the Etba agent attempts to bind to it
            Tac.run( [ "ifconfig", devName, "promisc", "up" ] )
         else:
            assert False, "Unknown port %s" % intftype
         port.label = label
         
         t0( "device %s, mac %s" %( devName, mac ) )

      FruBaseVeosDriver.__init__( self, genericPc, parentMibEntity,
                              parentDriver, driverCtx )
      # Now that we've built up the full inventory tree, run
      # drivers across it
      self.instantiateChildDrivers( genericPc, self.systemMib_ )

      # Set the system mac address. We have no choice but to choose
      # this randomly - see the comment in generateSystemMac for details.
      #
      # We allow the mac address to be overwritten by creating the
      # file /mnt/flash/system_mac_address. This gives us a hook in
      # case the automatically generated mac address is not
      # sufficient. Care must be taken when cloning systems with
      # a /mnt/flash/system_mac_address file.
      systemMacAddr = veosConfig[ 'SYSTEMMACADDR' ]
      if lowestMacAddr and not systemMacAddr:
         systemMacAddr = generateSystemMacAddr( lowestMacAddr )
      if systemMacAddr:
         entityMibStatus.systemMacAddr = systemMacAddr
      
      self.systemMib_.firmwareRev = FirmwareRev.abootFirmwareRev()

      import EosVersion
      vi = EosVersion.VersionInfo( sysdbRoot=None )
      if vi.version() == None:
         self.systemMib_.softwareRev = ""
      else:
         self.systemMib_.softwareRev = vi.version()

      # For all front-panel interfaces managed by PhyEthtool,
      # we have to set the deviceName on the EthIntfStatus
      # ourselves (just like we do in Fru for the management
      # interfaces).
      phyEthtoolPhyDir = genericPc.component.get( "phy" )
      if phyEthtoolPhyDir:
         for invPhy in phyEthtoolPhyDir.phy.values():
            intfStatus = invPhy.port.intfStatus
            if not intfStatus.deviceName:
               intfStatus.deviceName = invPhy.port.intfId.replace( "Ethernet", "et" )

      etbaConfig.complete = True

      # Declare FruReady
      hwCellDir = driverCtx.sysdbRoot[ 'hardware' ][ 'cell' ][ '%d' % myCellId ]
      assert hwCellDir
      hwCellDir.newEntity( 'Tac::Dir', 'FruReady' )
         
      # Declare success at the end
      self.systemMib_.initStatus = "ok"


def Plugin( context ):
   context.registerDriver( VEosDriver )
   mg = context.entityManager.mountGroup()
   mg.mount( bridgingEtbaConfigPath, bridgingEtbaConfigType, 'w' )
   mg.mount( 'bridging/linux/config', 'Bridging::Linux::Config', 'w' )
   mg.close( None )
