# Copyright (c) 2006-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Tracing
import Fru, EntityMib
import FruPlugin.Smbus
from EntityMib import IndexAllocator
t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2

__defaultTraceHandle__ = Tracing.Handle( "Fru" )

def createSensorEnvConfig( envTemperature, sensorName, invSensor ):
   sensorEnvConfig = Fru.Dep( envTemperature[ 'config' ].tempSensor,
                              invSensor ).newMember( sensorName, invSensor.position )
   sensorEnvConfig.description = invSensor.description
   sensorEnvConfig.overheatThreshold = invSensor.overheatThreshold
   sensorEnvConfig.criticalThreshold = invSensor.criticalThreshold
   sensorEnvConfig.targetTemperature = invSensor.targetTemperature
   sensorEnvConfig.ultimateGainFrontToBack = invSensor.ultimateGainFrontToBack
   sensorEnvConfig.ultimateGainBackToFront = invSensor.ultimateGainBackToFront
   sensorEnvConfig.ultimatePeriodFrontToBack = invSensor.ultimatePeriodFrontToBack
   sensorEnvConfig.ultimatePeriodBackToFront = invSensor.ultimatePeriodBackToFront
   sensorEnvConfig.generationId = Fru.powerGenerationId( invSensor )
   sensorEnvConfig.sliceName = Fru.fruBaseName( invSensor )

   if hasattr( invSensor, 'hystTemp' ):
      sensorEnvConfig.hystTemp = invSensor.hystTemp
   # BUG65208
   if hasattr( invSensor, 'ignoreForBoardInitialization' ):
      sensorEnvConfig.ignoreForBoardInitialization = \
          invSensor.ignoreForBoardInitialization
   if hasattr( invSensor, 'pidIgnored' ):
      sensorEnvConfig.pidIgnored = invSensor.pidIgnored
   if hasattr( invSensor, 'coolingDomainName' ):
      sensorEnvConfig.coolingDomainName = invSensor.coolingDomainName
   return sensorEnvConfig

def createSensorEnvStatus( envTemperature, sensorName, invSensor ):
   return Fru.Dep( envTemperature[ 'status' ].tempSensor,
                   invSensor ).newMember( sensorName )

def createSensorEnv( envTemperature, sensorName, invSensor ):
   sensorEnvConfig = createSensorEnvConfig( envTemperature, sensorName, invSensor )
   sensorEnvStatus = createSensorEnvStatus( envTemperature, sensorName, invSensor )
   return ( sensorEnvConfig, sensorEnvStatus )

class SmbusTempSensorDriverBase( Fru.FruDriver ):

   provides = [ Fru.FruDriver.environmentInit ]

   tempAllSuiteCreated = False
   supportsDisablingAlerts = False

   def __init__( self, sensorFru, parentMibEntity, parentDriver, ctx ):
      Fru.FruDriver.__init__( self, sensorFru, parentMibEntity,
                              parentDriver, ctx )
      self.sysdbRoot_ = ctx.sysdbRoot
      self.parentMibEntity_ = parentMibEntity

      self.fruBase_ = Fru.fruBase( sensorFru )
      tempSensorId = sensorFru.sensorId # BUG1731
      sensorName = EntityMib.componentNameFromParent(
         parentMibEntity, "TempSensor", tempSensorId )

      # ------------------------------------------
      # Create the associated Environment::TempSensorConfig and
      # Environment::TempSensorStatus objects
      # ------------------------------------------

      self.envTempDir_ = self.sysdbRoot_[ 'environment' ][ 'temperature' ]
      sensorEnvConfig = createSensorEnvConfig(
         self.envTempDir_, sensorName, sensorFru )

      # ------------------------------------------
      # Create the associated hardware config
      # ------------------------------------------

      topology = ctx.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]

      # First, figure out if the temp sensor is managed by the system
      # or its cell.
      if FruPlugin.Smbus.managedByLocalCell( topology, sensorFru ) and (
         # On linecards, we need to differentiate between temp sensors on the standby
         # power domain from the ones on the main power domain, because the ones on
         # the standby are accessed via the supervisor scd's smbus accelerators as
         # opposed to the linecards. Here, the ignoreForBoardInitialization
         # attribute, if true, indicates that the temp sensor is on the main power
         # domain.
         not hasattr( sensorFru, 'ignoreForBoardInitialization' ) or
         not sensorFru.ignoreForBoardInitialization ):
         cell = self.fruBase_.managingCellId
         hardwareDir = ctx.sysdbRoot[ 'hardware' ][ 'cell' ][ str(cell) ]
      else:
         cell = None
         hardwareDir = ctx.sysdbRoot[ 'hardware' ]
      hwConfigDir = self._sensorHardwareConfigDir( hardwareDir )

      ahamArgs = [ sensorFru.fullName ]
      ahamArgs.extend( FruPlugin.Smbus.ahamDesc( topology, sensorFru ) )
      ahamDesc = Fru.Dep( hwConfigDir.ahamDesc, sensorFru ).newMember( *ahamArgs )

      Fru.Dep( hwConfigDir.tempSensor, sensorFru ).newMember( 
         sensorName, ahamDesc, self.fruBase_.generationId,
         sensorEnvConfig )
      self.sensorHwConfig_ = hwConfigDir.tempSensor[ sensorName ]
      self.sensorHwConfig_.offset = sensorFru.offset

      # Currently only the max6697 agent does anything with the
      # alertDisabled attr
      assert ( ( not sensorFru.alertDisabled ) or
               ( self.supportsDisablingAlerts ) ), \
               'alertDisabled unsupported on this sensor'
      self.sensorHwConfig_.alertDisabled = sensorFru.alertDisabled

      # ------------------------------------------
      # Create the associated EntityMib object
      # ------------------------------------------

      sensorMib = self._createEntityMib( sensorFru )
      EntityMib.populateMib( sensorMib, sensorFru )
      sensorMib.groupName = sensorFru.groupName

      for rd in sensorFru.remoteDiode.values():
         self._setupRemoteDiode( rd, parentMibEntity )

      for offset, value in sensorFru.registerInitializer.iteritems():
         self.sensorHwConfig_.registerInitializer[ offset ] = value

      # Declare the sensor as initialized
      self.sensorHwConfig_.initialized = True

      # Declare success.
      # Note that this MUST be the last thing that we do, in order to ensure
      # that at the point that the initStatus is "ok", we know that the
      # TempSensorEnvConfig objects have been created.
      sensorMib.initStatus = "ok"

   def _sensorHardwareConfigDir( self, hardwareDir ):
      """ Create and return the Hardware::TempSensor::Config entity
      under which this sensor's config should go """
      raise NotImplementedError

   def _createEntityMib( self, sensorFru ):
      sensorMib = self.parentMibEntity_.sensor.get( sensorFru.sensorId )
      if sensorMib is None:
         physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                         ( self.parentMibEntity_.sensor, sensorFru.sensorId )
         sensorMib = Fru.Dep( self.parentMibEntity_.sensor, sensorFru ).\
                     newMember( physicalIndex, sensorFru.sensorId, "TempSensor" )
      sensorMib.label = str( sensorFru.sensorId )
      # Copy description from Config
      sensorMib.description = sensorFru.description
      return sensorMib

   def _setupRemoteDiode( self, invRemote, parentMibEntity ):
      # HW config object.
      sensorName = EntityMib.componentNameFromParent(
         parentMibEntity, "TempSensor", invRemote.sensorId )
      envConfig = createSensorEnvConfig( self.envTempDir_, sensorName, invRemote )
      remote = self.sensorHwConfig_.newRemoteDiode(
         invRemote.localId, invRemote.sensorId, envConfig )
      remote.offset = invRemote.offset
      remote.truThermEnable = invRemote.truThermEnable
      # Currently only the max6697 agent does anything with the
      # alertDisabled attr
      assert ( ( not invRemote.alertDisabled ) or
               ( self.supportsDisablingAlerts ) ), \
               'alertDisabled unsupported on this sensor'
      remote.alertDisabled = invRemote.alertDisabled

      # Entity MIB
      mib = self._createEntityMib( invRemote )
      mib.description = invRemote.description
      mib.groupName = invRemote.groupName
      mib.initStatus = "ok"

def Plugin( context ):
   mg = context.entityManager.mountGroup()
   mg.mount( 'environment/temperature/config',
             'Environment::Temperature::Config', 'w' )
   mg.close( None )
