# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import EntityMib
import Fru
import StorageInvLib
import Tracing

__defaultTraceHandle__ = Tracing.Handle( "Fru.Storage" )
t0 = Tracing.trace0 

def getStorageEntMib( deviceName, fixedSystemMib ):
   for entityMibDevice in fixedSystemMib.storageDevices.values():
      if entityMibDevice.tag == deviceName:
         return entityMibDevice

   # If there are existing device ids, add one to the highest value
   devId = 0
   devIdList = fixedSystemMib.storageDevices.keys()
   devId = ( sorted( devIdList )[ -1 ] if devIdList else 0 ) + 1

   physicalIndex = EntityMib.IndexAllocator.collectionItemPhysicalIndex(
      fixedSystemMib.storageDevices, devId )
   storageDevicesMibEnt = fixedSystemMib.newStorageDevices( physicalIndex,
      devId, deviceName )

   return storageDevicesMibEnt

class StorageDeviceDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::StorageDeviceDir"
   managedApiRe = "$"
   def __init__( self, invStorageDeviceDir, fixedSystemMib, parentDriver, ctx ):
      Fru.FruDriver.__init__( self, invStorageDeviceDir, fixedSystemMib,
                              parentDriver, ctx )
      assert invStorageDeviceDir, 'storage device dir does not exist'

      expectedMounts = [ 'flash' ]
      if invStorageDeviceDir.storageMount:
         expectedMounts += invStorageDeviceDir.storageMount.keys()
      self.inventory( expectedMounts, fixedSystemMib )

   def inventory( self, expectedMounts, fixedSystemMib ):
      mounts = StorageInvLib.Mounts()
      devices = StorageInvLib.DeviceFiles( mounts )
      possibleMounts = [ 'flash', 'drive' ]
      for mount in possibleMounts:
         mountExpected = mount in expectedMounts
         device = devices.deviceFactory( mount, mountExpected=mountExpected )
         if not device and not mountExpected:
            continue
         mibEnt = getStorageEntMib( mount, fixedSystemMib )
         if device:
            device.populateInventory( mibEnt )

            # For devices that need Write-Cache disabled, disable writeCache
            # here. This is the best place to do it as we don't have any agent
            # responsible for bringing up these devices.
            # Only Disable write cache if the drive requires it to be
            # disabled AND is currently enabled.
            device.maybeDisableWriteCache()
         else:
            mibEnt.modelName = "Unknown"
            mibEnt.serialNum = "Unknown"
            mibEnt.firmwareRev = "0.0"
            mibEnt.sizeGB = 0
         mibEnt.initStatus = 'ok'

      # Check for an eMMC device
      mount = 'flash'
      mounts = StorageInvLib.Mounts()
      devices = StorageInvLib.DeviceFiles( mounts,
                                           deviceFilter=r'e?mmcblk[0-9]+$' )
      device = devices.getDeviceFile( mount )
      if device:
         eMmcDevice = StorageInvLib.EmmcFlash( mounts, mount, device )
         if eMmcDevice:
            eMmcDeviceMibEnt = getStorageEntMib( mount, fixedSystemMib )
            eMmcDevice.populateInventory( eMmcDeviceMibEnt )
            eMmcDeviceMibEnt.initStatus = 'ok'

def Plugin( context ):
   context.registerDriver( StorageDeviceDriver )

