# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=F0401

import Tac, Fru, FruPlugin.Smbus, FruPlugin.Gpio
import sys
import Tracing
import FruPlugin.Led as LedHelper
import re
import EntityMib
from EntityMib import IndexAllocator
import FpgaUtil

__defaultTraceHandle__ = Tracing.Handle( "Fru.StandbyCpld" )
t0 = Tracing.trace0

class CpldDriver( Fru.FruDriver ):
   managedApiRe = ".*$"

   provides = [ "StandbyCpld", FruPlugin.Gpio.gpioInit ]

   def aham( self, cpld, configDir, ctx ):
      raise NotImplementedError

   def gpoAham( self, cpld, gpo, ctx ):
      raise NotImplementedError

   def __init__( self, cpld, parentMibEntity, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, cpld, parentMibEntity,
                              parentDriver, driverCtx )
      fruBase = Fru.fruBase( cpld )
      cellId = fruBase.managingCellId
      hardwareDir = driverCtx.sysdbRoot.entity[ "hardware/cell/%d" % cellId ]
      cpldDir = hardwareDir.mkdir( "cpld" )
      configDir = cpldDir.newEntity( "Hardware::StandbyCpld::SystemConfig",
                                     "config" )

      #-----------------------------------
      # Create the cpld hw config
      ahamDesc = self.aham( cpld, configDir, driverCtx )

      hwCpldConfig = Fru.Dep( configDir.cpldConfig, cpld ).newMember( cpld.name,
                                                                      ahamDesc )
      hwCpldConfig.customerName = Fru.fruBaseName( cpld )

      hwCpldConfig.firmwareRevReg = cpld.firmwareRevReg
      hwCpldConfig.hardwareMajorRevReg = cpld.hardwareMajorRevReg

      if cpld.suicideRegisterOffset:
         hwCpldConfig.suicideRegisterOffset = cpld.suicideRegisterOffset

      for offset, value in cpld.registerInitializer.iteritems():
         hwCpldConfig.registerInitializer[ offset ] = value

      gpoDir = driverCtx.sysdbRoot.entity[ 'hardware/archer/gpo' ]
      for block in cpld.registerGpoBlock.itervalues():
         for bit in block.bit.itervalues():
            systemName = '%s-%s-%d' % ( cpld.name, block.name, bit.bitpos )
            fruGpo = Fru.Dep( gpoDir, cpld ).newEntity(
               'Hardware::CpldRegisterGpo',
               systemName )
            fruGpo.ahamDesc = self.gpoAham( cpld, fruGpo, driverCtx )
            fruGpo.offset = block.offset
            fruGpo.bit = bit.bitpos
            fruGpo.activeType = bit.activeType
            bit.fruGpo = fruGpo
            bit.systemName = systemName

      self.hwSystemCpldConfig = hwCpldConfig

class I2cCpldDriver( CpldDriver ):
   managedTypeName = "Inventory::I2cCpld"

   def __init__( self, cpld, parentMibEntity, parentDriver, driverCtx ):
      CpldDriver.__init__( self, cpld, parentMibEntity, parentDriver, driverCtx )
      self.hwSystemCpldConfig.ready = True

   def aham( self, cpld, configDir, ctx ):
      ahamDesc = Fru.Dep( configDir.i2cAhamDesc, cpld ).newMember(
         cpld.name, cpld.master.id, cpld.deviceId )
      return ahamDesc

   def gpoAham( self, cpld, gpo, ctx ):
      gpo.i2cAhamDesc = ( cpld.name, cpld.master.id, cpld.deviceId )
      return gpo.i2cAhamDesc

class SmbusCpldDriver( CpldDriver ):
   managedTypeName = "Inventory::SmbusCpld"

   def __init__( self, cpld, parentMibEntity, parentDriver, driverCtx ):
      if cpld.enablePowerCycleOnRailFault:
         cpld.registerInitializer[ cpld.powerCycleRegister ] = \
            cpld.powerCycleRegisterValue
      CpldDriver.__init__( self, cpld, parentMibEntity, parentDriver, driverCtx )

      # Create ArcherPinInit
      t0( "creating archerpinInit for smbuscpld " )
      gpioInitDir = driverCtx.sysdbRoot.entity[ 'hardware/archer/gpio/init' ]

      for( pinBlockSubAddr, invPinBlock ) in cpld.pinBlock.items():
         hwPinBlockConfig = self.hwSystemCpldConfig.newPinBlock( pinBlockSubAddr,
                                                                 invPinBlock.count )
         for ( pinId, invPin ) in invPinBlock.gpioPin.items():
            assert ( 1 <= pinId + invPin.count <= invPinBlock.count ), (
               "Bit location of pin (pin id: %d) plus its count (%d) are outside "
               "the range of [ 1, %d ]."
               % ( pinId, invPin.count, invPinBlock.count ) )
            systemName = "%s-%d-%d" % ( cpld.name, pinBlockSubAddr, pinId )
            t0( "Creating system cpld gpioPins for %s and id %d with systemName %s"
               % ( invPin.pinName, pinId, systemName ) )
            pinInit = Fru.Dep( gpioInitDir, cpld ).newEntity(
               "Hardware::Gpio::ArcherPinInit", systemName )
            pinInit.pinName = invPin.pinName
            pinInit.pinId = pinId
            pinInit.count = invPin.count
            pinInit.activeType = invPin.activeType
            pinInit.setToInputIfInactive = invPin.setToInputIfInactive
            pinInit.setToInputIfActive = invPin.setToInputIfActive
            pinInit.direction = invPin.direction
            pinInit.defaultVal = invPin.defaultVal
            invPin.systemName = systemName
            hwPinBlockConfig.gpioPinName[ systemName ] = True

      self.hwSystemCpldConfig.ready = True

   def aham( self, cpld, configDir, ctx ):
      topology = ctx.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]
      ahamArgs = FruPlugin.Smbus.ahamDesc( topology, cpld )
      ahamDesc = Fru.Dep( configDir.smbusAhamDesc, cpld ).newMember(
         cpld.name, *ahamArgs )
      return ahamDesc

   def gpoAham( self, cpld, gpo, ctx ):
      topology = ctx.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]
      ahamArgs = FruPlugin.Smbus.ahamDesc( topology, cpld )
      gpo.smbusAhamDesc = ( cpld.name, ) + ( ahamArgs )
      return gpo.smbusAhamDesc

class StandbyCpldDriver( Fru.FruDriver ):

   managedTypeName = "Inventory::StandbyCpld"
   managedApiRe = ".*"

   def __init__( self, invStandbyCpld, parentMibEnt, parentDriver, ctx ):
      Fru.FruDriver.__init__( self, invStandbyCpld, parentMibEnt, parentDriver, ctx )

      standbyCpldConfig = ctx.sysdbRoot[ 'hardware' ][ 'standbyCpld' ][ 'config' ]
      standbyCpldStatus = ctx.entity( 'hardware/standbyCpld/status' )
      topology = ctx.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]
      gpioInitDir = ctx.sysdbRoot.entity[ 'hardware/archer/gpio/init' ]

      name = "%d-%s" % ( parentMibEnt.relPos, invStandbyCpld.name )
      hwCpldConfig = Fru.Dep( standbyCpldConfig.cpldConfig,
                              invStandbyCpld ).newMember( name )
      hwCpldConfig.generationId = Fru.fruBase( invStandbyCpld ).generationId
      hwCpldConfig.ahamDesc = ( "cpldAhamDesc", ) + \
                              FruPlugin.Smbus.ahamDesc( topology, invStandbyCpld )

      hwCpldConfig.revisionRegister = invStandbyCpld.revisionRegister

      # Pin blocks to define gpio registers
      for ( pinBlockSubAddr, invPinBlock ) in invStandbyCpld.pinBlock.items():
         hwPinBlockConfig = hwCpldConfig.newPinBlock( pinBlockSubAddr,
                                                      invPinBlock.count )
         # Pins within the pin blocks (i.e. bits within gpio registers)
         for ( pinId, invPin ) in invPinBlock.gpioPin.items():
            # When writing pin values, we don't want them to wrap around
            assert ( 1 <= pinId + invPin.count <= invPinBlock.count ), \
                   "Bit location of pin (pin id: %d) plus its count (%d) are " \
                   "outside the range of [ 1, %d ]." \
                   % ( pinId, invPin.count, invPinBlock.count )
            t0( "  Creating gpioPins for %s and id %d " % (invPin.pinName, pinId ) )
            # Archer GPIO initialization
            systemName = "%s-%d-%d" % ( name, pinBlockSubAddr, pinId )
            pinInit = Fru.Dep( gpioInitDir, invStandbyCpld ).newEntity(
               "Hardware::Gpio::ArcherPinInit",
               systemName )
            pinInit.pinName = invPin.pinName
            pinInit.pinId = pinId
            pinInit.count = invPin.count
            pinInit.activeType = invPin.activeType
            pinInit.setToInputIfInactive = invPin.setToInputIfInactive
            pinInit.setToInputIfActive = invPin.setToInputIfActive
            pinInit.direction = invPin.direction
            pinInit.defaultVal = invPin.defaultVal
            invPin.systemName = systemName
            hwPinBlockConfig.gpioPinName[ systemName ] = True
      if invStandbyCpld.controllerInterface:
         hwCpldConfig.fanController = ( 'fanController', )
         invStandbyCpld.controllerInterface.hwFanController = \
               hwCpldConfig.fanController
      hwCpldConfig.pwmRegisterBase = invStandbyCpld.pwmRegisterBase
      hwCpldConfig.tachRegisterBase = invStandbyCpld.tachRegisterBase
      hwCpldConfig.pwmRegisterStride = invStandbyCpld.pwmRegisterStride
      hwCpldConfig.tachRegisterStride = invStandbyCpld.tachRegisterStride
      hwCpldConfig.fanFaultRegister = invStandbyCpld.fanFaultRegister

      # Leds
      fruBaseName = Fru.fruBaseName( invStandbyCpld )
      for ( ledName, cpldLedBlock ) in invStandbyCpld.cpldLedBlock.items():
         t0( "Create Led %s" % ledName )
         # For each Led, create a ledConfig that will be updated by LedPolicyAgent.
         ledConfig = LedHelper.addLed( ledName, fruBaseName,
                                       invStandbyCpld, ctx )
         # Right now we use LedAttribute defined in InvScd to store blueLedOffset,
         # which is defined as U32 (since Scd's register is 32 bits), to keep the 
         # implementation consistent with Leds controlled by Scd.
         # But the offset for Leds controlled by StandbyCpld is 8 bit address (U8).
         # So we count on fdl implemetator to set the right address (0x00 to 0xff),
         # but it's worth to have an assert here.
         assert 0x00 <= cpldLedBlock.capability.blueLedOffset <= 0xff
         ledBlock = Tac.Value( "Hardware::StandbyCpld::CpldLedBlock", 
                               cpldLedBlock.offset,
                               cpldLedBlock.greenBit,
                               cpldLedBlock.redBit,
                               cpldLedBlock.capability.blueLedOffset, 
                               cpldLedBlock.blueBit )
         hwCpldConfig.cpldLedBlock[ ledName ] = ledBlock
         ledAttribute = Tac.Value( "Led::LedAttribute",
                                    blue=cpldLedBlock.capability.blue )
         ledConfig.ledAttribute = ledAttribute
      hwCpldConfig.maxSequentialTachFailures = \
          invStandbyCpld.maxSequentialTachFailures

      hwCpldConfig.ready = True

      # Mendocino doesn't have a separate standby cpld to control its fans,
      # so it doesn't make sense to track its revision
      if invStandbyCpld.revisionRegisterValid:
         sliceId = Fru.fruBase( invStandbyCpld ).sliceId
         sliceRegex = re.compile \
               ( '^(FixedSystem|Fabric|Switchcard|Linecard|Supervisor|Cell)(\d*)' )
         sliceMatch = sliceRegex.match( sliceId )
         sliceType, index1 = sliceMatch.groups()
         # The 100 is a magic offset because chipId
         # needs to be globally unique in the mib
         # See the Ucd9012 Fru plugin for something similar
         chipId = 100 + invStandbyCpld.deviceId + ( 0 if sliceType == 'FixedSystem'
                                                    else int( index1 ) )

         chipMib = self.parentMibEntity_.chip.get( chipId )
         if chipMib is None:
            physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                            ( self.parentMibEntity_.chip, chipId )
            chipMib = self.parentMibEntity_.newChip( physicalIndex, chipId,
                                                     "stdbycpld" )
            chipMib.label = str( chipId )
            chipMib.description = "Standby Cpld Chip %d" % chipId
            EntityMib.populateMib( chipMib, invStandbyCpld )

         self.statusReactor = StatusReactor( standbyCpldStatus,
                                             name,
                                             chipMib )

class StandbyCpldDirDriver( Fru.FruDriver ):

   # Technically, StandbyCpldDriver provides gpioInit. However, because the
   # provides-requires mechanism is only effective within one stage/generation
   # of FruPlugin's, we need to put it here. In other words, standbyDomain
   # won't look for "provides" in *Cpld* since it is a grandchild. But it will
   # know to look for "provides" in *CpldDir* since it is a direct child.
   provides = [ FruPlugin.Gpio.gpioInit ]

   managedTypeName = "Inventory::StandbyCpldDir"
   managedApiRe = ".*"

   # TODO: rework this part

   def __init__( self, invStandbyCpldDir, parentMibEnt, parentDriver, ctx ):
      Fru.FruDriver.__init__( self, invStandbyCpldDir,
                              parentMibEnt, parentDriver, ctx )
      self.standbyCpldDriver_ = {}
      for invStandbyCpld in invStandbyCpldDir.standbyCpld.values():
         t0( "StandbyCpldDriver looking for driver for cpld", invStandbyCpld.name )
         try:
            self.standbyCpldDriver_[ invStandbyCpld.name ] = ctx.driverRegistry.\
                  newDriver( invStandbyCpld, parentMibEnt, parentDriver, ctx )
         except NotImplementedError:
            sys.excepthook( *sys.exc_info() )
            #XXX What should be done here?
            raise

class StandbyCpldStatusReactor( Tac.Notifiee ):
   """propagates StandbyCpld version from hardware to entmib"""
   notifierTypeName = "Hardware::StandbyCpld::CpldStatus"

   def __init__( self, cpldStatus, chipMib ):
      Tac.Notifiee.__init__( self, cpldStatus )
      self.chipMib = chipMib
      self.handleFirmwareRevision()

   @Tac.handler( 'revision' )
   def handleFirmwareRevision( self ):
      major = self.notifier_.revision
      minor = FpgaUtil.parseVersion( self.chipMib.firmwareRev ).minor
      self.chipMib.firmwareRev = "0x%02x.%02x" % ( major, minor )

class StatusReactor( Tac.Notifiee ):
   """propagates StandbyCpld version from hardware to entmib"""
   notifierTypeName = "Hardware::StandbyCpld::Status"

   def __init__( self, status, cpldName, chipMib ):
      Tac.Notifiee.__init__( self, status )
      self.status = status
      self.cpldName = cpldName
      self.chipMib = chipMib
      self.cpldStatusReactor = None
      self.handleCpldStatus( self.cpldName )

   @Tac.handler( 'cpldStatus' )
   def handleCpldStatus( self, name ):
      if name == self.cpldName:
         if name in self.status.cpldStatus:
            self.cpldStatusReactor = StandbyCpldStatusReactor(
               self.status.cpldStatus[ name ],
               self.chipMib )
         else:
            del self.cpldStatusReactor

def Plugin( ctx ):
   ctx.registerDriver( StandbyCpldDriver )
   ctx.registerDriver( StandbyCpldDirDriver )
   ctx.registerDriver( I2cCpldDriver )
   ctx.registerDriver( SmbusCpldDriver )

   mg = ctx.entityManager.mountGroup()
   mg.mount( 'hardware/standbyCpld/config', 'Hardware::StandbyCpld::Config', 'w' )
   mg.mount( 'hardware/standbyCpld/status', 'Hardware::StandbyCpld::Status', 'r' )
   mg.mount( 'hardware/archer/gpo', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/archer/gpio/init', 'Tac::Dir', 'wi' )
   mg.close( None )
