#!/usr/bin/env python
# Copyright (c) 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.


import Tac, Tracing
import Fru
import FruPlugin.Smbus, FruPlugin.Gpio
import Cell

th = Tracing.Handle( "SolFru" )
t9 = th.trace9
t0 = th.trace0

class SolChipFruDriver( Fru.FruDriver ):

   # provided by Pca9555 on modular, not used on fixed
   requires = [ FruPlugin.Gpio.gpioInit ]
   
   managedTypeName = "Inventory::SolChip"
   managedApiRe = ".*"

   def getNode( self, topology, fruBase ):
      if fruBase.managingCellId:
         key = str( fruBase.managingCellId )
      elif fruBase.sliceId:
         key = str( fruBase.sliceId )
      else:
         assert False, "cell or slice id must always be set"
      return topology.node.get( key )

   def checkDedicatedSmbus( self, topology, smbusDeviceInv ):
      """ Return true if the Sol chip is on the only device on the Smbus,
      otherwise return false. """
      fruBase = Fru.fruBase( smbusDeviceInv )
      n = self.getNode( topology, fruBase )
      assert n
      bus = n.smbus[ smbusDeviceInv.upstreamBus.name ]
      assert bus
      if len( bus.device ) == 1:
         return True
      else:
         return False

   def __init__ ( self, solFru, parentMibEntity, parentDriver, ctx ):
      Fru.FruDriver.__init__( self, solFru, parentMibEntity, parentDriver, ctx )
      t0( "Handling Inventory::SolChip for %s" % Fru.fruBaseName( solFru ) )

      fruBase = Fru.fruBase( solFru )
      solName = fruBase.sliceId
      genId = fruBase.generationId
      cellId = fruBase.managingCellId or Cell.cellId()

      solConfigDir = ctx.sysdbRoot[ "hardware" ][ "sol" ][ "config" ] \
                                  [ "cell" ].newEntity( "Hardware::Sol::CellConfig",
                                                        str( cellId ) )

      topology = ctx.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]
      ahamArgs = [ solName ]
      ahamArgs.extend( FruPlugin.Smbus.ahamDesc( topology, solFru ) )
      ahamDesc = Fru.Dep( solConfigDir.ahamDesc, solFru ).newMember( *ahamArgs )

      solConfig = Fru.Dep(solConfigDir.solConfig, solFru).newMember(
         solName, genId, cellId, ahamDesc )

      # BUG6174
      if solFru.reset is not None:
         solConfig.resetGpoName = solFru.reset.systemName
      if solFru.resetGpio is not None:
         solConfig.resetPinName = \
                      solFru.resetGpio.pin[ solFru.resetGpioIdx ].systemName

      # Set whether to use the MAC value that is read from the board
      solConfig.hasBoardMac = solFru.hasBoardMac
      # Determine if this device is on a dedicated Smbus
      solConfig.dedicatedSmbus = self.checkDedicatedSmbus( topology, solFru )

def Plugin( context ):
   context.registerDriver( SolChipFruDriver )

   mg = context.entityManager.mountGroup()
   mg.mount( 'hardware/sol/config/cell', 'Tac::Dir', 'wi' )
   mg.close( None )
