#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import Fru

th = Tracing.Handle( "Fru.SflowAccelFpga" )
t0 = th.trace0

class SflowAccelFpgaSliceDriver( Fru.FruDriver ):
   requires = [ Fru.FruDriver.systemInit ]
   provides = [ "SflowAccelFpgaSliceInit" ]

   managedTypeName = "Inventory::SflowAccelFpga::SflowAccelFpgaDir"

   def __init__( self, inv, parentMib, parentDriver, ctx ):
      super( SflowAccelFpgaSliceDriver, self ).__init__( inv, parentMib,
                                                    parentDriver, ctx )
      t0( "Creating a SflowAccelFpga driver", inv.name )

      fruBase = Fru.fruBase( inv )
      sliceId = fruBase.sliceId
      self.instantiateChildDrivers( self.invEntity_, self.parentMibEntity_ )
      configBaseDir = ctx.sysdbRoot.entity[ 'hardware/sflowAccelFpga/config/slice' ]

      t0( "sliceId: ", sliceId )
      t0( "ConfigBase: ", configBaseDir )
      t0( "fruBase: ", fruBase )

      hwConfigCollection = Fru.Dep(
         configBaseDir, inv ).newEntity(
         'Hardware::SflowAccelFpga::ConfigDir',
         sliceId )

      hwConfigCollection.sliceReady = True

class SflowAccelFpgaDriver( Fru.FruDriver ):
   requires = [ Fru.FruDriver.systemInit, "SflowAccelFpgaSliceInit" ]
   provides = [ "SflowAccelFpgaInit" ]

   managedTypeName = "Inventory::SflowAccelFpga::SflowAccelFpga"

   def __init__( self, inv, parentMib, parentDriver, ctx ):
      super( SflowAccelFpgaDriver, self ).__init__( inv, parentMib,
                                                    parentDriver, ctx )
      t0( "Creating a SflowAccelFpga driver", inv.name )

      fruBase = Fru.fruBase( inv )
      sliceId = fruBase.sliceId
      configBaseDir = ctx.sysdbRoot.entity[ 'hardware/sflowAccelFpga/config/slice' ]

      t0( "sliceId: ", sliceId )
      t0( "ConfigBase: ", configBaseDir )
      t0( "fruBase: ", fruBase )

      hwConfigCollection = Fru.Dep(
         configBaseDir, inv ).newEntity(
         'Hardware::SflowAccelFpga::ConfigDir', sliceId )

      hwConfig = Fru.Dep( hwConfigCollection.config, inv ).newMember( inv.name )

      t0( "hwConfig", hwConfig )

      # Create the Ham for the SflowAccel Fpga
      pciComponentInv = inv.sflowAccelFpga if inv.sflowAccelFpga else inv
      if inv.useBar1:
         hwConfig.ham = Fru.pciHotplugHam( pciComponentInv, ctx,
                                           filename="resource1" )
      else:
         hwConfig.ham = Fru.pciHotplugHam( pciComponentInv, ctx,
                                           filename="resource0" )

      # Instantiate the resetHam. For platforms using Bongo, the Sflow ham and sflow
      # reset ham are same since the reset happens through bongo. For platforms using
      # Scd for Sflow, the Sflow ham is in bar1 and Sflow logic is reset using Scd
      # registers in bar0.
      resetDevice = inv.resetDevice if inv.resetDevice else pciComponentInv
      hwConfig.resetHam = Fru.pciHotplugHam( resetDevice, ctx )

      # set the accelId. This is a globally unique id per fpga.
      hwConfig.accelId = inv.accelId

      hwConfig.resetBit = inv.resetBit

      # set the fpga type
      hwConfig.sflowAccelFpgaType = inv.sflowAccelFpgaType

      hwConfig.configReady = True


def Plugin( context ):
   context.registerDriver( SflowAccelFpgaSliceDriver )
   context.registerDriver( SflowAccelFpgaDriver )
   mg = context.entityManager.mountGroup()
   mg.mount( 'hardware/sflowAccelFpga/config/slice', 'Tac::Dir', 'wi' )

   # NOTE - we don't need to mount any other subdirectories of
   # hardware/sflowAccelFpga, as we're doing an immediate mount above.
   # This has to be mount-immediate to ensure that Fru mounts everything
   # that it has previously created in the agent restart case.
   mg.close( None )
