#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re
import weakref

import Cell
import FirmwareRev
import Fru
from Fru.FruBaseVeosDriver import FruBaseVeosDriver, parseVeosConfig
from SfFruHelper import DEVICE, FIRST_MAC
import Tac
import Tracing
import VeosHypervisor

__defaultTraceHandle__ = Tracing.Handle( "Fru.Sfa" )
Tracing.traceSettingIs( ",".join( [ os.environ.get( "TRACE", "" ) ] +
   [ 'Fru.Sfa/*' ] ) )

t0 = Tracing.trace0
t9 = Tracing.trace9
veosConstants = Tac.Type( "VeosCommon::VeosConstants" )

# pylint: disable-msg=broad-except

# This class provides hot plug capability to veos. We hookup with the
# udev net rules handler script which passes us the dev name and action
# ( currently only add ), we reinitialize the Fru drivers which handles
# the inventory objects. We don;t deal with "removal" for now.
class HotPlugHandler( Tac.File ):
   def __init__( self, driver ):
      self._driver = weakref.proxy( driver )
      eventFifoName = self._driver.fruHelper.getEventFifoPath()
      self.fragmentedEvent = ""

      # Create pipe File Descriptor
      if not os.path.exists( eventFifoName ):
         os.mkfifo( eventFifoName )

      # It is critical to open for RDWR as otherwise we keep getting
      # spurious readable events after we read something  and writer is
      # gone...
      self._eventFd = os.open( eventFifoName , os.O_RDWR | os.O_NONBLOCK )
      Tac.File.__init__( self, self._eventFd, self.readHandler )

   def readHandler( self, intfName ):
      t0( 'Invoked IntfEventHandler %s ' % intfName )
      if not intfName:
         return

      intfName = self.fragmentedEvent + intfName
      self.fragmentedEvent = ""
      intfs = intfName.splitlines( True )

      # There can be too many incoming events streamed continuously on a
      # Kubernetes setup depending on the POD creation/deletion.
      # The buffer used for reading the input is of limited size, so
      # when the incoming data is more than the buffer size then the last event in
      # the buffer can be fragmented.
      # We need to handle this fragmented data in the next PASS.
      # As each event is delimited by a newline character, the fragmented event can
      # be easily identified if there is no newline character present in the event.
      # Currently we handle this case only for cEOSR and this will not affect other
      # platforms.
      if VeosHypervisor.platformCeosR() and not intfs[ -1 ].endswith( "\n" ):
         # Last event looks fragmented
         self.fragmentedEvent = intfs.pop()
         t0( 'fragmented event: %s' % self.fragmentedEvent )

      for i in intfs:
         i = i.rstrip()
         t0( 'readHandler : Calling handleInterfaceEvent for %s' % i )
         self._driver.fruHelper.handleInterfaceEvent( i )

class SfaPhyDriver( Fru.FruDriver ):
   """This Fru plugin manages the runnability of the Sfa agent"""

   requires = [ Fru.FruDriver.interfaceInit ]
   managedTypeName = "Inventory::Phy::SfaPhyDir"
   managedApiRe = "$"

   def __init__( self, phyDir, parentMib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, phyDir, parentMib, parentDriver, driverCtx )

      # Create the qualPath for the Sfa agent, only if one or more
      # interfaces are found. This will avoid Sfa crashing when no
      # interfaces are present BUG148456. Sfa agent will not be
      # started if Sfe mode is enabled.
      if phyDir.phy:
         sysdbRoot = driverCtx.sysdbRoot
         launchConfig = sysdbRoot[ 'hardware' ][ 'sfa' ][ 'launcherConfig' ]
         launchConfig.newEntity( 'Tac::Dir', 'Sfa' )
      # To support sfe_failsafe mode which is really SFA mode with all interfaces
      # disappearing except for et1 to use as management port. we will unbind all
      # other interfaces from kernel.
      platform = VeosHypervisor.getPlatform()
      if parentDriver and hasattr( parentDriver, 'veosConfig' ) and \
         parentDriver.veosConfig[ 'MODE' ] == 'sfe_failsafe' and \
         platform != 'BareMetal':
         self.handleFailSafeMode( phyDir )

      # On baremetal platforms, setDeviceNames below doesn't get to run so we
      # need to do the same here.
      if platform == 'BareMetal':
         for invPhy in phyDir.phy.values():
            port = invPhy.port
            intfStatus = port.intfStatus
            if port.role != "Management" and intfStatus is not None:
               intfStatus.deviceName = port.intfId.replace( "Ethernet", "et" )

   def handleFailSafeMode( self, phyDir ):
      t0( "handleFailSafeMode" )
      for fruPhy in phyDir.phy.values():
         try:
            # In AWS we don't use management port but eth1.
            if re.search( 'Phyma', fruPhy.name ) \
                  or re.search( 'Phyet1', fruPhy.name ):
               t0( "Skipping interface in failsafe mode", fruPhy.name )
               continue
            devName = fruPhy.name.replace( "Phyet", "et", 1 )
            op = Tac.run( [ 'ethtool' , '-i', devName ], stdout=Tac.CAPTURE ).\
                    splitlines()
            busInfo = [ i for i in op if i.startswith( 'bus-info:' ) ][ 0 ].\
                   split()[ 1 ]
            driver = Tac.run( [ 'readlink' ,
                        '/sys/bus/pci/devices/%s/driver' % busInfo ],
                        stdout=Tac.CAPTURE ).split( '/') [ -1 ][ : -1 ]
            cmd = [ "bash" , "-c" , "echo %s > " \
                     "/sys/bus/pci/drivers/%s/unbind" % ( busInfo, driver ) ]
            Tac.run( cmd, asRoot=True )
            t0( 'Unbound driver %s from dev: %s ' % (driver , devName ) )
         except Exception as e:
            t0( 'Ignoring Caught exception in handleFailSafeMode:' , e )
            continue

class SfaDriver( FruBaseVeosDriver ):
   managedTypeName = "Eos::GenericPcFru"
   managedApiRe = "veos$"
   driverPriority = 2

   def __init__( self, genericPc, parentMibEntity, parentDriver, driverCtx,
                 mgmtConfigEnabled=True ):
      t0( "SfaDriver: Enable Sfa to run" )
      assert parentMibEntity is None

      self.veosConfig = parseVeosConfig()
      if self.veosConfig[ "MODE" ] != "sfe" and \
         self.veosConfig[ "MODE" ] != "sfe_failsafe" :
         self.veosConfig[ "MODE" ] = "linux"

      if VeosHypervisor.platformCeosR():
         from CEOSRFruHelper import CEOSRFruHelper
         self.fruHelper = CEOSRFruHelper( self )
      elif VeosHypervisor.platformAWS():
         from AWSFruHelper import AWSFruHelper
         self.fruHelper = AWSFruHelper( self )
      elif VeosHypervisor.platformAzure():
         from AzureFruHelper import AzureFruHelper
         self.fruHelper = AzureFruHelper( self )
      elif VeosHypervisor.platformGCP():
         from GCPFruHelper import GCPFruHelper
         self.fruHelper = GCPFruHelper( self )
      elif VeosHypervisor.platformKVM():
         from KVMFruHelper import KVMFruHelper
         self.fruHelper = KVMFruHelper( self )
      elif VeosHypervisor.platformESXi():
         from ESXiFruHelper import ESXiFruHelper
         self.fruHelper = ESXiFruHelper( self )
      else:
         t0( "SfaDriver: unknown platform" )
         return

      self.entityMibStatus_ = driverCtx.entity( "hardware/entmib" )
      self.entityMibStatus_.fixedSystem = ( 1, 0, "FixedSystem" )
      self.systemMib_ = self.entityMibStatus_.fixedSystem

      if VeosHypervisor.platformCeosR():
         t0( "SfaDriver - running inside a container" )
         self.systemMib_.modelName = veosConstants.ceosrModelName
         self.systemMib_.description = "EOS in container"
      else:
         self.systemMib_.modelName = "vEOS"
         self.systemMib_.description = "EOS in a virtual machine"

      self.systemMib_.mfgName = "Arista"
      self.mgmtConfigEnabled_ = mgmtConfigEnabled

      # entmib/FixedSystem/vendorType will be used to detect
      # the environment vEOS is deployed in and initiate
      # the correct license detection sequence.
      self.systemMib_.vendorType = VeosHypervisor.getPlatform() or ''

      serialNumberGenerator = Tac.newInstance( "License::SerialNumberGenerator" )
      self.systemMib_.serialNum = serialNumberGenerator.getSerialNumber(
                                    self.systemMib_.vendorType )

      self.systemMib_.swappability = "notSwappable"
      if self.entityMibStatus_.nextPhysicalIndex < 2:
         self.entityMibStatus_.nextPhysicalIndex = 2
      self.genericPc = genericPc
      myCellId = Cell.cellId()
      roles = [ "AllCells", "AllSupervisors", "ActiveSupervisor" ]
      Fru.createAndConfigureCell( driverCtx.sysdbRoot,
                                  genericPc,
                                  myCellId,
                                  roles )
      for portRole in [ "Management", "Switched" ]:
         genericPc.portRole[ portRole ] = portRole
      # -------------
      # Populate the rest of the inventory tree based on the
      # interfaces we find in the kernel

      sysClassDmi = "/sys/class/dmi/id/"
      sysVendor = None
      procKernelPath = "/proc/sys/kernel/"
      procWatchdog = None
      t0( "SfaDriver: Enabling the watchdog" )
      if os.path.exists( os.path.join( sysClassDmi, "sys_vendor" ) ):
         sysVendor = str(file( os.path.join( sysClassDmi, "sys_vendor" )). \
                         read()).rstrip()
      if os.path.exists( os.path.join( procKernelPath, "watchdog" ) ):
         procWatchdog = str(file( os.path.join( procKernelPath, "watchdog" )). \
                            read()).rstrip()
      if sysVendor == "Microsoft Corporation" and procWatchdog == "0":
         if os.path.exists( os.path.join( procKernelPath, "watchdog_thresh" ) ):
            file( os.path.join( procKernelPath, "watchdog_thresh" ), "w" ). \
            write("30\n")
         file( os.path.join( procKernelPath, "watchdog"), "w" ).write("1\n")

      t0( "SfaDriver: Getting Interfaces" )
      # If we are going to sfe_failsafe mode, all the interfaces are owned by dpdk.
      # We will not see any devices as searched below due to being vport interfaces
      # and not the pci devices. We need to unbind from dpdk to kernel to be able to
      # inventory these. We don't care for KVM/ESXI etc as console will still be
      # there.
      if self.veosConfig[ "MODE" ] == "sfe_failsafe":
         self.fruHelper.bindInterfacesToKernel()
      self.fruHelper.loadDeviceCache()

      t9( "SfaDriver: Creating Interfaces %s" % \
          self.fruHelper.deviceCache[ DEVICE ] )
      self.ethPortDir = genericPc.component.newEntity(
            "Inventory::EthPortDir", "ethPortDir" )
      self.pciPortDir = genericPc.component.newEntity(
            "Inventory::PciPortDir", "pciPortDir" )
      self.phyEthtoolPhyDir = genericPc.component.newEntity(
            "Inventory::Phy::EthtoolDir", "phy" )
      self.sfePhyDir = genericPc.component.newEntity(
            "Inventory::Phy::SfePhyDir", "sfePhy" )
      self.sfaPhyDir = genericPc.component.newEntity(
            "Inventory::Phy::SfaPhyDir", "sfaPhy" )
      self.containerIntfStatusDir = \
         driverCtx.sysdbRoot[ 'interface' ][ 'status' ][ 'containerintf' ]

      # Cleanup stale entries that exists in Sysdb, but not in deviceCache
      # When Fru is restarted, we lose DependentSet and cannot do complete
      # cleanup of dependents like inftStatus and phyEthTool config.
      # Need to find a better way of doing the cleanup.
      # Filed BUG424310 to track this issue.
      # For now comment out below cleanup code.
      #for invPhy in self.phyEthtoolPhyDir.phy.values():
      #   dName = invPhy.name.replace( "Phy", "", 1 )
      #   portId = invPhy.port.id if invPhy.port else None
      #   if dName not in self.fruHelper.deviceCache[ DEVICE ]:
      #      self.fruHelper.cleanupInventory( dName, portId )

      for devName in self.fruHelper.deviceCache[ DEVICE ]:
         self.fruHelper.addInterface( devName )
      self.fruHelper.updateIntfConfigAll()

      FruBaseVeosDriver.__init__( self, genericPc, parentMibEntity,
         parentDriver, driverCtx )
      # Now that we've built up the full inventory tree, run
      # drivers across it
      self.instantiateChildDrivers( genericPc, self.systemMib_ )

      # Set the system mac address. We have no choice but to choose
      # this randomly - if we choose the mac address of any of our
      # existing interfaces, then we end up with weird behavior as
      # traffic sent on that interface will then be treated as if it
      # were sent to the system mac address. This can cause weird
      # issues (double responses to ping packets, for
      # example). Unfortunately we can't just choose a locally
      # administered address, as MLAG assumes that the system mac
      # address is not locally administered.
      #
      # To randomly generate an address, we start with the first 3
      # bytes of the mac address, which differs based on the
      # hypervisor being used. We then seed random with the mac
      # address of the first interface of the system(basically management interface).
      # This ensures the same behavior every
      # time we run (assuming management interface is not removed / changed, at
      # which point we consider this to be a 'new' machine), and makes
      # it much more unlikely that two systems will come up with the
      # same 3 random bytes (which might happen if two VMs booted with
      # the exact same system time)
      #
      # We allow the mac address to be overwritten by creating the
      # file /mnt/flash/system_mac_address. This gives us a hook in
      # case the automatically generated mac address is not
      # sufficient. Care must be taken when cloning systems with
      # a /mnt/flash/system_mac_address file.
      systemMacAddr = self.veosConfig[ 'SYSTEMMACADDR' ]
      firstMacAddr = self.fruHelper.deviceCache[ FIRST_MAC ]
      if firstMacAddr and not systemMacAddr:
         systemMacAddr = firstMacAddr[ : 8 ]
         import random

         # consider only lower 32bit of hash value as we need same
         # seed value on both 32bit and 64bit python binary.
         seedValue = hash( firstMacAddr ) & 0xFFFFFFFFL
         random.seed( seedValue )
         for _ in range( 3 ):
            systemMacAddr += ":%x" % ( random.randint( 0, 255 ) )
         # In case others are using the random package, reseed.
         random.seed()
      if systemMacAddr:
         systemMacAddr = systemMacAddr.encode( 'utf-8' )
         self.entityMibStatus_.systemMacAddr = \
            Tac.Value( 'Arnet::EthAddr', stringValue=systemMacAddr )

      self.systemMib_.firmwareRev = FirmwareRev.abootFirmwareRev()

      import EosVersion
      vi = EosVersion.VersionInfo( sysdbRoot=None )
      if vi.version() is None:
         self.systemMib_.softwareRev = ""
      else:
         self.systemMib_.softwareRev = vi.version()

      # set the kernel device names into IntfStatus
      self.setDeviceNames()

      # Create/hookup hotplug mechanism. Ignore if we fail to instantiate it
      # as it is not critical for system bringup and failure may result in
      # blackholing the  instance if running in AWS.
      t0( "initialize hot plug handler. : " )
      try :
         self._hotPlugHandler = HotPlugHandler( self )
      except Exception as e :
         t0( "Failed to initialize hot plug handler. Bailing out : %s " % e )

      # Declare success at the end
      self.systemMib_.initStatus = "ok"

      # Declare FruReady
      hwCellDir = driverCtx.sysdbRoot[ 'hardware' ][ 'cell' ][ '%d' % myCellId ]
      assert hwCellDir
      hwCellDir.newEntity( 'Tac::Dir', 'FruReady' )

   def setDeviceNames( self ):
      # For all front-panel interfaces managed by PhyEthtool,
      # we have to set the deviceName on the EthIntfStatus
      # ourselves (just like we do in Fru for the management
      # interfaces).
      phyEthtoolPhyDir = self.genericPc.component.get( "phy" )
      if phyEthtoolPhyDir:
         for invPhy in phyEthtoolPhyDir.phy.values():
            port = invPhy.port
            intfStatus = port.intfStatus
            if port.role != "Management" and intfStatus is not None:
               if self.fruHelper.useKernelDevNames():
                  if port.id in self.fruHelper.kernelDeviceNames:
                     intfStatus.deviceName = \
                        self.fruHelper.kernelDeviceNames[ port.id ]
                  else:
                     # Stale phyEthtool entry, ignore.
                     # BUG424310 tracks this issue
                     intfStatus.deviceName = ""
               else:
                  intfStatus.deviceName = port.intfId.replace( "Ethernet", "et" )
               t9( "Set device name for %s to %s" % ( port.intfId,
                                                      intfStatus.deviceName ) )

   def reinstantiateDrivers( self ):
      t0( " recreating drivers..")
      self.instantiateChildDrivers( self.genericPc, self.systemMib_ )
      self.setDeviceNames()

def veosPluginFn( driver ):
   def fn( context ):
      context.registerDriver( driver )
      mg = context.entityManager.mountGroup()
      mg.mount( 'hardware/sfa/launcherConfig', 'Tac::Dir', 'wi' )
      mg.mount( 'bridging/hwcapabilities', 'Bridging::HwCapabilities', 'w')
      mg.mount( 'interface/status/containerintf',
                'Interface::ContainerIntfStatusDir', 'w' )
      mg.close( None )
   return fn

def Plugin( context ):
   context.registerDriver( SfaPhyDriver )
   veosPluginFn( SfaDriver )( context )
