# Copyright (c) 2007-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import EntityMib
from EntityMib import IndexAllocator
import Tac, Fru, XcvrLib
import FruPlugin.Led as LedHelper
import FruPlugin.XcvrCtrl as XcvrHelper
import FruPlugin.TempSensorFru as TempSensorFru
import FruPlugin.Gpio
import Tracing
import Cell
import FpgaUtil

__defaultTraceHandle__ = Tracing.Handle( "Fru.Scd" )
t1 =  Tracing.trace1
t6 =  Tracing.trace6
sliceStatusDir = None
sliceStatusReactor = None
cellStatusDir = None
cellStatusReactor = None
entMib = None
redundancyReactor = None
redundancyStatus = None 

class PciScdDriver( Fru.FruDriver ):

   requires = [ Fru.FruDriver.pciInit ]
   provides = [ Fru.FruDriver.systemInit, FruPlugin.Gpio.gpioInit ]

   managedTypeName = "Inventory::PciScd"
   managedApiRe = ".*$"

   @staticmethod
   def populateGpoHwState( scd, hwScdConfig, gpoDir ):
      fruHelper = Tac.newInstance( "Hardware::Scd::FruPluginHelper", 
                                   "ScdFruPluginHelper" )
      fruHelper.scdInv = scd
      fruHelper.scdConfig = hwScdConfig
      fruHelper.gpoDir = gpoDir
      fruHelper.fruBaseName = Fru.fruBaseName( scd )
      fruHelper.dependentSet = Fru.Dep( fruHelper, scd ).dependentSet()
      fruHelper.populateGpoHwState()

   @staticmethod
   def populateUserFpgaHwState( scd, hwScdConfig ):
      ''' Create the hardware customer-programmable fpgas if they exists in the
      inventory.'''
      for userFpga in scd.userFpga.itervalues():
         hwUserFpga = hwScdConfig.newUserFpgaConfig( userFpga.name,
                                                     userFpga.revisionBlock.offset,
                                                     userFpga.scratchBlock.offset,
                                                     userFpga.pollInterval )
         invProgrammingBlock = userFpga.programmingBlock
         invErrorBlock = userFpga.errorBlock
         invStatusBlock = userFpga.statusBlock

         hwUserFpga.errorBlock = ( invErrorBlock.name, )
         hwUserFpga.errorBlock.errorReset = invErrorBlock.errorReset.systemName
         hwUserFpga.programmingBlock = \
             ( invProgrammingBlock.name,
               invProgrammingBlock.offset + invProgrammingBlock.ctrlOffset )
         hwUserFpga.programmingBlock.programmingDone = \
             invProgrammingBlock.programmingDone.systemName
         hwUserFpga.statusBlock = ( userFpga.name, invStatusBlock.offset, )
         for name, statusGpo in invStatusBlock.statusGpo.iteritems():
            hwUserFpga.statusBlock.statusGpo[ name ] = statusGpo.systemName
         for name, resetGpo in userFpga.resetGpo.iteritems():
            hwUserFpga.resetGpo[ name ] = resetGpo.systemName
         hwUserFpga.pcieAccessGpo = userFpga.pcieAccessGpo.systemName

   @staticmethod
   def populateSatelliteAccessState( scd, hwScdConfig ):
      for satelliteData in scd.satelliteAccess.itervalues():
         hwScdConfig.newSatelliteHam( satelliteData.name, 0, 0,
                                      satelliteData.pageReg,
                                      satelliteData.pageOffsetReg,
                                      satelliteData.mask, hwScdConfig.ham )

   @staticmethod
   def populateTempSensorHwState( scd, hwScdConfig, parentMib,
                                  envTempDir ):
      for sensor in scd.tempSensor.itervalues():
         tempSensorId = sensor.sensorId
         name = EntityMib.componentNameFromParent( parentMib,
                                                   "TempSensor",
                                                   tempSensorId )
         TempSensorFru.createSensorEnvConfig( envTempDir, name, sensor )
         # make config
         tsConfig = hwScdConfig.newTempSensor( name, sensor.regOffset )
         tsConfig.offset = sensor.offset
         # make status
         # Create the sensor EntityMib object
         sensorMib = parentMib.sensor.get( tempSensorId )
         if sensorMib is None:
            physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                ( parentMib.sensor, tempSensorId )
            sensorMib = parentMib.newSensor( physicalIndex,
                                             tempSensorId,
                                             "TempSensor" )
         sensorMib.label = str( tempSensorId )
         sensorMib.description = sensor.description
         # Declare success for the sensor. This MUST be done after the
         # TempSensorEnvConfig has been created
         sensorMib.initStatus = "ok"

   @staticmethod
   def populateModeControlState( scd, hwScdConfig ):
      if scd.modeControlBlock:
         hwScdConfig.modeControlBlock = ( scd.modeControlBlock.name, )
         hwScdConfig.modeControlBlock.driveResetsGpo = \
             scd.modeControlBlock.driveResetsGpo.systemName

         if scd.modeControlBlock.portLogicQsfpGpo:
            hwScdConfig.modeControlBlock.portLogicQsfpGpo = \
                scd.modeControlBlock.portLogicQsfpGpo.systemName
            
   @staticmethod
   def populateVoltageRailBlockHwState( scd, hwScdConfig ):
      for block in scd.voltageRailBlock.itervalues():
         voltageRailConfig = hwScdConfig.newVoltageRailBlock( block.name )
         voltageRailConfig.description = block.description
         voltageRailConfig.regOffset = block.offset
         voltageRailConfig.controlBitOffset = block.controlBitOffset
         voltageRailConfig.voltageMarginBitOffset = block.voltageMarginBitOffset
         voltageRailConfig.voltageMarginValue = block.voltageMarginValue

   @staticmethod
   def populateFaultPowerCycleEnableBlockState( scd, hwScdConfig ):
      if scd.faultPowerCycleEnableBlock:
         hwScdConfig.faultPowerCycleEnableBlock = \
            ( scd.faultPowerCycleEnableBlock.name, )
         if scd.faultPowerCycleEnableBlock.scdCrcRebootActionGpo:
            hwScdConfig.faultPowerCycleEnableBlock.scdCrcRebootActionGpo = \
               scd.faultPowerCycleEnableBlock.scdCrcRebootActionGpo.systemName
         if scd.faultPowerCycleEnableBlock.userRebootActionGpo:
            hwScdConfig.faultPowerCycleEnableBlock.userRebootActionGpo = \
               scd.faultPowerCycleEnableBlock.userRebootActionGpo.systemName
         if scd.faultPowerCycleEnableBlock.watchdogRebootActionGpo:
            hwScdConfig.faultPowerCycleEnableBlock.watchdogRebootActionGpo = \
               scd.faultPowerCycleEnableBlock.watchdogRebootActionGpo.systemName

   @staticmethod
   def populateRegisterOffsets( scd, hwScdConfig ):
      if scd.ptpTimeSyncBlock:
         Fru.Dep( scd.ptpTimeSyncBlock, scd ).ham = hwScdConfig.ham

      if scd.quartzyBlock:
         Fru.Dep( scd.quartzyBlock, scd ).ham = hwScdConfig.ham

      if scd.max24305ConfigBlock:
         Fru.Dep( scd.max24305ConfigBlock, scd ).ham = hwScdConfig.ham

      if scd.max24305StatusBlock:
         Fru.Dep( scd.max24305StatusBlock, scd ).ham = hwScdConfig.ham

      if scd.crosspointBlock:
         Fru.Dep( scd.crosspointBlock, scd ).ham = hwScdConfig.ham

      if scd.extResetCauseBlock:
         hwScdConfig.extResetCauseConfig = ('extResetCauseConfig', 
                                            scd.extResetCauseBlock.offset )
         for value in scd.extResetCauseBlock.reasons.itervalues():
            hwScdConfig.extResetCauseConfig.reasons.addMember( value )

      if scd.scratchBlock:
         hwScdConfig.scratchRegisterOffset = scd.scratchBlock.offset

      if scd.testFeaturesBlock:
         hwScdConfig.testFeaturesRegisterOffset = scd.testFeaturesBlock.offset

      if scd.baliErrorBlock:
         hwScdConfig.baliErrorRegisterOffset = scd.baliErrorBlock.offset

      def _copySoftBlocks( hwSeuBlock, invSeuBlock ):
         hwSeuBlock.offset = invSeuBlock.offset
         hwSeuBlock.errorBit = invSeuBlock.errorBit
         hwSeuBlock.errorBitProvided = invSeuBlock.errorBitProvided
         hwSeuBlock.errorBitValidVal = invSeuBlock.errorBitValidVal
         hwSeuBlock.errorBitValidMask = invSeuBlock.errorBitValidMask
         hwSeuBlock.enableBitProvided = invSeuBlock.enableBitProvided
         hwSeuBlock.enableBit = invSeuBlock.enableBit
         hwSeuBlock.uncorrErrorMaskProvided = invSeuBlock.uncorrErrorMaskProvided
         hwSeuBlock.uncorrErrorMask = invSeuBlock.uncorrErrorMask

      for invSeuBlock in scd.softErrorBlock.itervalues():
         hwSeuBlock = hwScdConfig.newSoftErrorBlock( invSeuBlock.name )
         _copySoftBlocks( hwSeuBlock, invSeuBlock )
         
      if scd.externalAlarmBlock:
         hwScdConfig.externalAlarmBlockOffset = scd.externalAlarmBlock.offset
         hwScdConfig.externalAlarmCount = scd.externalAlarmCount

      for invSatScd in scd.satelliteScd.itervalues():
         satScd = hwScdConfig.newSatelliteScd( invSatScd.name )
         for invSeuBlock in invSatScd.softErrorBlock.itervalues():
            hwSeuBlock = satScd.newSoftErrorBlock( invSeuBlock.name )
            _copySoftBlocks( hwSeuBlock, invSeuBlock )
         for invDB in invSatScd.debugBlock.itervalues():
            satScd.newDebugBlock( invDB.name, invDB.offset, invDB.clearOnPowerOn,
                                  invDB.clearVal, invDB.logNormVal,
                                  invDB.logMask )
         satScd.latestLogTime = invSatScd.latestLogTime

      for block in scd.phyTxEnableBlock.itervalues():
         # setup the phy tx enable
         # this is a bit of a hack, ideally someone other than the scd should do this
         # BUG6161
         if block.name == 'phyTxEnable':
            hwScdConfig.phyTxEnableOffset = block.offset

      hwScdConfig.revisionRegisterOffset = scd.revisionRegister
      hwScdConfig.secondRevRegisterOffset = scd.secondRevRegister

   @staticmethod
   def createChipMib( scdName, scd, parentMib, driverCtx ):
      #
      # Create the associated EntityMib object
      #
      t6( "create chipMib for: %s-%d" % ( scdName, scd.chipId ) )
      chipMib = parentMib.chip.get( scd.chipId )
      t6( "chipMib: %s" % chipMib )
      if chipMib is None:
         physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                         ( parentMib.chip, scd.chipId )
         chipMib = parentMib.newChip( physicalIndex,
                                      scd.chipId, "scd-%s" % scd.chipId )
      chipMib.label = str( scd.chipId )
      chipMib.description = "Scd Chip %s" % scd.chipId
      EntityMib.populateMib( chipMib, scd )
      return chipMib

   def __init__( self, scd, parentMib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, scd, parentMib, parentDriver, driverCtx )

      fruBase = Fru.fruBase(scd)
      cellId = fruBase.managingCellId
      sliceId = fruBase.sliceId
      slotId = getattr( fruBase, 'slot', 1 )
      if Cell.cellType() == "supervisor":
         modular = True
      else:
         modular = False if cellId else True
      gpioInitDir = driverCtx.sysdbRoot.entity[ 'hardware/archer/gpio/init' ]

      # create xcvrCtrl config path befor Scd Agent starts up.
      XcvrHelper.createXcvrCtrlPaths( fruBase, driverCtx )
      if cellId:
         assert slotId == cellId
         cellDir =  driverCtx.sysdbRoot[ 'hardware' ][ 'cell' ][ str( cellId ) ]
         scdDir = cellDir.mkdir( 'scd' )

         hwScdConfigCollection = scdDir.newEntity(
            'Hardware::Scd::Config',
            'config' )

         # Here, 'FixedSystem'.
         # scdName needs to be just sliceId to for Arsys btest to
         # pass.  Make build happy for now.  BUG54649
         if scd.name != 'scd':
            scdName = sliceId + '-' + scd.name
         else:
            scdName = sliceId

      else:
         # The Scd must reside on a cell or a slice.
         assert sliceId

         configBaseDir = \
             driverCtx.sysdbRoot[ 'hardware' ][ 'scd' ][ 'config' ][ 'slice' ]

         hwScdConfigCollection = Fru.Dep(
            configBaseDir, scd ).newEntity(
            'Hardware::Scd::Config',
            sliceId )

         scdName = "%u" % scd.chipId

      # Environment dir is unaffeceted by above
      envTempDir = driverCtx.sysdbRoot['environment']['temperature']

      # Must create status before config. The Scd agent is created
      # in response to the config object. If config is created first, the
      # agent might run between the two stmts and try to mount the status
      # object before the Fru plugin has created it.
      genId = Fru.powerGenerationId( scd )
      hwScdConfig = Fru.Dep(hwScdConfigCollection.scdConfig, scd).\
                                             newMember(scdName, genId, slotId,
                                                       scd.chipId)
      fruBaseName = Fru.fruBaseName( scd )

      fruHelper = Tac.newInstance( "Hardware::Scd::FruPluginHelper", 
                                   "ScdFruPluginHelper" )
      fruHelper.cellId = cellId
      fruHelper.scdInv = scd
      fruHelper.fruBaseName = fruBaseName
      fruHelper.scdConfig = hwScdConfig
      fruHelper.dependentSet = Fru.Dep( fruHelper, scd ).dependentSet()
      fruHelper.ledHelper = LedHelper.createLedFruPluginHelper( driverCtx )
      fruHelper.gpoDir = driverCtx.sysdbRoot.entity[ 'hardware/archer/gpo' ]

      if scd.fpgaName:
         hwScdConfig.fpgaName = scd.fpgaName

      if scd.powerManagementBlock:
         hwScdConfig.powerManagementBlock = ( scd.powerManagementBlock.name, )
         hwScdConfig.powerManagementBlock.powerCapacityOffset = \
            scd.powerManagementBlock.powerCapacityOffset
         hwScdConfig.powerManagementBlock.powerSystemOffset = \
            scd.powerManagementBlock.powerSystemOffset
         hwScdConfig.powerManagementBlock.powerAvailableOffset = \
            scd.powerManagementBlock.powerAvailableOffset
         hwScdConfig.powerManagementBlock.powerReserveOffset = \
            scd.powerManagementBlock.powerReserveOffset
         hwScdConfig.powerManagementBlock.linkedListOffset = \
            scd.powerManagementBlock.linkedListOffset
         hwScdConfig.powerManagementBlock.headOfListIndex = \
            scd.powerManagementBlock.headOfListIndex

      hwScdConfig.isSatelliteScd = scd.isSatelliteScd
      if scd.isSatelliteScd:
         mainScd = scd.satelliteEndpoint.otherEndConnectedTo
         t1( 'satelliteScd', mainScd, 'offset=', mainScd.offset, 'size=',
             mainScd.size, "masterComponent=", mainScd.masterComponent )
         hwScdConfig.ham = Fru.pciHotplugHam( mainScd.masterComponent, driverCtx,
            offset=mainScd.offset, size=mainScd.size )
      elif scd.pciFpga:
         hwScdConfig.ham = Fru.pciHotplugHam( scd.pciFpga, driverCtx )
      else:
         hwScdConfig.ham = Fru.pciHotplugHam( scd, driverCtx )

      # BUG4710 We're stashing pointers to the hardware tree in the
      # inventory tree. However, when a fru is powered off, these
      # pointers need to get cleaned up. I'm not sure how widespread
      # this technique is, but we may need to rethink what we're
      # doing here (like deleting and re-exec'ing parts of the
      # inventory tree.
      Fru.Dep(scd, scd).ham = hwScdConfig.ham

      for name in scd.localBus:
         # scd includes a PCI-to-LocalBus bridge
         assert len( scd.localBus ) == 1
         if scd.pciFpga:
            hamCtors = Fru.pciHotplugHam( scd.pciFpga, driverCtx,
                                          filename="resource1" )[1:]
         else:
            hamCtors = Fru.pciHotplugHam( scd, driverCtx, filename="resource1" )[1:]
         hwScdConfig.localBusHam.newMember( name, *hamCtors )
         Fru.Dep(scd.localBus[name], scd).ham = hwScdConfig.localBusHam[ name ]

      #
      # Driver config
      #
      interrupts = False
      for block in scd.interruptBlock.itervalues():
         interrupts = True
         break
      watchdogBlock = scd.watchdogBlock
      if interrupts or watchdogBlock:
         hwScdConfig.driverConfig = ( "driverConfig",
                                      scd.interruptIrqConfigured,
                                      scd.interruptIrq,
                                      scd.crcErrorInterruptConfigured,
                                      scd.crcErrorInterruptIrq,
                                      scd.ptpOffsetsConfigured,
                                      scd.ptpHighOffset,
                                      scd.ptpLowOffset,
                                      scd.ptpSampleSecOffset,
                                      scd.ptpSampleNsecOffset,
                                      scd.ptpGpioSampleStatusOffset,
                                      scd.msiConfigured,
                                      scd.msiRearmOffset,
                                      scd.setIntrPoll,
                                      bool(watchdogBlock),
                                      scd.nmiHandlerConfigured,
                                      scd.nmiHandlerType,
                                      scd.nmiGpioBit )

      # Microblaze block
      # For now only supported in Rosa type switches
      if scd.microblazeBlock:
         hwScdConfig.microblazeBlock = ( "microblazeBlock", 
                                         scd.microblazeBlock.controlBitOffset,
                                         scd.microblazeBlock.imemOffset, 
                                         scd.microblazeBlock.dmemOffset )

      fruHelper.populateScdHwModel()

      # WatchdogPowerCycle block
      # Only used in Denali modular so far
      if scd.watchdogPowerCycleBlock:
         hwScdConfig.watchdogPowerCycleBlock = ( scd.watchdogPowerCycleBlock.name, )
         hwScdConfig.watchdogPowerCycleBlock.offset = \
            scd.watchdogPowerCycleBlock.offset

         hwScdConfig.watchdogPowerCycleBlock.enableBit = \
            scd.watchdogPowerCycleBlock.enableBit

      # ardma Driver config
      for block in scd.ardmaBlock.itervalues():
         assert len( scd.ardmaBlock ) == 1
         hwScdConfig.driverConfig.ardmaConfigured = True
         hwScdConfig.driverConfig.ardmaBlockOffset = block.offset
         hwScdConfig.driverConfig.ardmaTxInterruptBit = block.txInterruptBit
         hwScdConfig.driverConfig.ardmaRxInterruptBit = block.rxInterruptBit

      if scd.systemResetBlock:
         hwScdConfig.systemSoftReset = ( "systemSoftReset", hwScdConfig.ham,
                                         scd.systemResetBlock.offset,
                                         "soft" )
         hwScdConfig.systemHardReset = ( "systemHardReset", hwScdConfig.ham,
                                         scd.systemResetBlock.offset,
                                         "hard" )

      for block in scd.fpgaProgrammingBlock.itervalues():
         fp = hwScdConfig.newFpgaProgrammer( block.name )
         fp.ham = ( "ham", "hamTypeRecursive", hwScdConfig.ham, block.offset, 0, "" )
         fpgaSel = 0
         for fpga in block.fpgaToBeProgrammed.values():
            # make sure that the same select is not used for different fpgas
            # pylint: disable-msg=superfluous-parens
            assert not ( fpgaSel & fpga.fpgaSel )
            fpgaSel |= fpga.fpgaSel
            # add the fpga
            hwFpga = fp.newFpgaToBeProgrammed( fpga.name, fpga.image, fpga.fpgaSel )
            hwFpga.fpgaBitFlipByteSwap = fpga.fpgaBitFlipByteSwap
            hwFpga.dynamicImageSuffix = fpga.dynamicImageSuffix

            if fpga.fpgaTermPcie:
               hwFpga.fpgaTermPcie = fpga.fpgaTermPcie.systemName

            # the resetPins
            for resetPinGroupName in [ 'firstReset', 'secondReset' ]:
               resetPinGroup = getattr( fpga, resetPinGroupName )
               if resetPinGroup:
                  setattr( hwFpga, resetPinGroupName,
                           ( resetPinGroup.name, resetPinGroup.resetHoldTime ) )
                  hwRPG = getattr( hwFpga, resetPinGroupName )
                  for pin in resetPinGroup.resetPins.values():
                     hwRPG.resetPins[ pin.systemName ] = True

      PciScdDriver.populateRegisterOffsets( scd, hwScdConfig )
      PciScdDriver.populateTempSensorHwState( scd, hwScdConfig,
                                              parentMib, envTempDir )
      PciScdDriver.populateUserFpgaHwState( scd, hwScdConfig )
      PciScdDriver.populateSatelliteAccessState( scd, hwScdConfig )
      PciScdDriver.populateModeControlState( scd, hwScdConfig )
      PciScdDriver.populateVoltageRailBlockHwState( scd, hwScdConfig )
      PciScdDriver.populateFaultPowerCycleEnableBlockState( scd, hwScdConfig )

      # Create Fan Controller
      if scd.controllerInterface:
         hwScdConfig.fanController = ( 'fanController', )
         hwScdConfig.fanController.ham = hwScdConfig.ham
         scd.controllerInterface.hwFanController = hwScdConfig.fanController

      # Create fan LED PinInits
      for invPinBlock in scd.pinBlock.itervalues():
         t6( "create pinBlock: %s" % invPinBlock.name )
         systemNamePrefix = "%d-%s" % ( self.parentMibEntity_.relPos, scd.name )
         hwPinBlock = hwScdConfig.newPinBlock( invPinBlock.name, invPinBlock.offset,
                                               invPinBlock.count )
         for ( pinId, invPin ) in invPinBlock.gpioPin.items():
            systemName = "%s-%d-%d" % ( systemNamePrefix, invPinBlock.offset, pinId )
            t6( "create ArcherPinInit: %s" % systemName )
            pinInit = Fru.Dep( gpioInitDir, scd ).newEntity(
               "Hardware::Gpio::ArcherPinInit",
               systemName )
            pinInit.pinName = invPin.pinName
            pinInit.pinId = pinId
            pinInit.count = invPin.count
            pinInit.activeType = invPin.activeType
            pinInit.setToInputIfInactive = invPin.setToInputIfInactive
            pinInit.setToInputIfActive = invPin.setToInputIfActive
            pinInit.direction = invPin.direction
            pinInit.defaultVal = invPin.defaultVal
            invPin.systemName = systemName
            hwPinBlock.gpioPinName.add( systemName )

      if not scd.scdXcvrController:
         # create enough xcvrController directories to work with passive mounts
         cellId = fruBase.managingCellId
         xcvrController = driverCtx.sysdbRoot[ 'hardware' ][ 'xcvrController' ]
         xcvrControllerConfig = xcvrController[ 'config' ]
         if cellId:
            xcvrControllerConfig.mkdir( 'cell' ).mkdir( str( cellId ) ).newEntity(
               "Hardware::XcvrController::CellConfig", "config" )
            xcvrControllerConfig.newEntity( "Hardware::XcvrController::AllConfigDir",
                                            "all" )
         else:
            sliceId = fruBase.sliceId
            assert sliceId
            controllerConfigs = Fru.newStyleHardwareDir( xcvrControllerConfig,
                                                         'Tac::Dir', scd )
            controllerConfigs.newEntity( "Hardware::XcvrController::CellConfig",
                                         "config" )
            controllerConfigs.newEntity( "Hardware::XcvrController::AllConfigDir",
                                         "xcvrConfig" )

      # create the xcvr controllers
      topologySmbus = None
      for xcvrBlock in scd.scdXcvrController.itervalues():
         xcvrType = xcvrBlock.xcvrController.xcvrType
         ( xcvrCtrlName, xcvrNames ) = XcvrLib.xcvrInfo( 
               [ conn for conn in xcvrBlock.xcvrController.xcvrConnection ],
               xcvrType, modular )
         xcvrConfig = XcvrHelper.addXcvr( xcvrCtrlName, fruBase, scd, 
                                          xcvrBlock.xcvrController,
                                          driverCtx, xcvrType, 
                                          xcvrNames )
         communicationBus = xcvrBlock.xcvrController.mgmtType
         if xcvrType == 'sfpPlus':
            scdXcvrCfg = hwScdConfig.newSfpPlusConfig( xcvrCtrlName, 
                                                       xcvrBlock.offset )
         elif xcvrType == 'qsfpPlus':
            scdXcvrCfg = hwScdConfig.newQsfpPlusConfig( xcvrCtrlName, 
                                                        xcvrBlock.offset )
         elif xcvrType == 'mbo':
            # Control/status offset not available for MBO, so is set to 0
            scdXcvrCfg = hwScdConfig.newMboConfig( xcvrCtrlName, 0,
                                                   xcvrBlock.reset.systemName )
         elif xcvrType == 'cfp2':
            scdXcvrCfg = hwScdConfig.newCfp2Config( xcvrCtrlName, xcvrBlock.offset )
         elif xcvrType == 'rcx':
            assert communicationBus == 'none'
            assert len( xcvrNames ) == 1
            # RCx has multiple scdXcvrCfg for different offset of memory space, but
            # point to the same RcxGroupControllerConfig.
            scdXcvrCfg = hwScdConfig.newRcxConfig( xcvrNames[ 0 ], xcvrBlock.offset, 
                                                   xcvrCtrlName )
            hwScdConfig.rcxControllerConfig.addMember( xcvrConfig )
         elif xcvrType in ( 'osfp', 'qsfpDd', 'qsfpCmis', ):
            scdXcvrCfg = hwScdConfig.newOsfpConfig( xcvrCtrlName,
                                                    xcvrBlock.offset )
         else:
            assert 0

         for name, gpoBit in xcvrBlock.scdProgrammableGpoPins.iteritems():
            xcvrConfig.programmableGpoPins[ name ] = gpoBit.systemName
         if xcvrBlock.interruptBit:
            scdXcvrCfg.interruptBlockBit = xcvrBlock.interruptBit.fruIntCtrl
         if xcvrBlock.portLogicEnable:
            scdXcvrCfg.portLogicEnable = xcvrBlock.portLogicEnable.systemName
         if communicationBus == 'smbusMgmt':
            if not topologySmbus:
               topologySmbus = \
                  driverCtx.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]
            import FruPlugin.Smbus # pylint: disable-msg=W0621
            ahamDesc = xcvrConfig.parent.newAhamDesc( xcvrCtrlName, 
                                 *FruPlugin.Smbus.ahamDesc( 
                                 topologySmbus,
                                 xcvrBlock.xcvrController.smbusDeviceBase ) )
         elif communicationBus == 'mdioMgmt':
            mdioDevice = xcvrBlock.xcvrController.mdioDeviceBase
            portAddr = mdioDevice.portAddr

            assert len( mdioDevice.upstreamBus.mdioMasters ) == 1
            accelBlock = mdioDevice.upstreamBus.mdioMasters.values()[ 0 ]
            busId = mdioDevice.upstreamBus.mdioBusId[ accelBlock ]
   
            assert accelBlock.tacType.fullTypeName == "Inventory::Mdio::AccelBlock"
            scdXcvrCfg.mdioAhamDesc = ( xcvrCtrlName, accelBlock.id, 
                                        busId, portAddr )
            ahamDesc = scdXcvrCfg.mdioAhamDesc
         elif communicationBus == 'none':
            # we expect RCx only
            assert xcvrType == 'rcx'
            ahamDesc = None
         else:
            assert 0
         if ahamDesc:
            xcvrConfig.ahamDesc = ahamDesc

      # signal that the scd is ready for use by the ScdAgent reactor
      # this makes sure that all the InvScd attrs are set before the ScdAgent
      # starts to configure the HwScd entities
      hwScdConfig.configReady = True
      parentMib = self.parentMibEntity_
      PciScdDriver.createChipMib( hwScdConfig.name, scd,
                                  parentMib, driverCtx )

def setupReactors():
   # setupReactors() is a callback that is called everytime Fru agent starts.
   # This runs independently from the PciScdDriver. On, Fru restart, runs 
   # setupReactors() without driver causing the reactors to run immediately.
   global redundancyReactor
   redundancyReactor = RedundancyStatusReactor( redundancyStatus )

class RedundancyStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Redundancy::RedundancyStatus"

   def __init__( self, redStatus ):
      Tac.Notifiee.__init__( self, redStatus )
      self.protocol = redStatus.protocol
      self.updateReactors()
   
   @Tac.handler( 'mode' )
   def updateReactors( self ):
      global sliceStatusReactor, cellStatusReactor
      sysMode = self.notifier_.mode
      t6( 'mode: %s, protocol: %s' % ( sysMode, self.protocol ) )
      if ( sysMode == 'active' or
           ( self.protocol != 'sso' and sysMode == 'standby' ) ):
         cellStatusReactor = StatusDirReactor( cellStatusDir )
         sliceStatusReactor = StatusDirReactor( sliceStatusDir )


class StatusDirReactor( Tac.Notifiee ):
   notifierTypeName = "Tac::Dir"

   def __init__( self, hwStatus ):
      Tac.Notifiee.__init__( self, hwStatus )
      t6( "StatusDirReactor.__init__" )
      self.hwScdStatusReactor_ = {}
      for key in self.notifier_:
         self.handleHwScdStatus( key )

   @Tac.handler( 'entityPtr' )
   def handleHwScdStatus( self, key ):
      keyDir = self.notifier_.get( key )
      keyDirEntry = self.notifier_.entryState.get( key )
      if not keyDir and not keyDirEntry:
         t1( "Deleting reactor for:", key )
         del self.hwScdStatusReactor_[ key ]
      elif keyDir and keyDirEntry:
         self.hwScdStatusReactor_[ key ] = HardwareStatusReactor( keyDir )

class HardwareStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Hardware::Scd::Status"

   def __init__( self, hwStatus ):
      Tac.Notifiee.__init__( self, hwStatus )
      t6( "HardwareStatusReactor.__init__" )
      self.scdStatusReactor_ = {}
      self.handleScdStatus( None )

   @Tac.handler( 'scdStatus' )
   def handleScdStatus( self, key ):
      t6( "Changes in Scd::ScdStatus %s" % key )
      Tac.handleCollectionChange( ScdStatusReactor,
                                  key,
                                  self.scdStatusReactor_,
                                  self.notifier_.scdStatus )
      
class ScdStatusReactor( Tac.Notifiee ):
   """propagates SCD version from hardware to entmib"""
   notifierTypeName = "Hardware::Scd::ScdStatus"

   def __init__( self, hwScdStatus ):
      Tac.Notifiee.__init__( self, hwScdStatus )
      t6( "hwScdStatus: Scd%d/%d" % ( hwScdStatus.slotId, hwScdStatus.chipId ) )
      self.handleFirmwareRevision()
      self.handleHardwareMajorRev()

   def getScdChipMib( self ):
      # retrive the chipMib created by the PciScdDriver
      slotId = self.notifier_.slotId
      chipId = self.notifier_.chipId
      t6( "get chipMib for Scd%d/%d" % ( slotId, chipId ) )
      cardMib = None
      entityMibRoot = entMib.root
      if( entityMibRoot.tacType.fullTypeName == "EntityMib::Chassis" and
          slotId in entityMibRoot.cardSlot and 
          entityMibRoot.cardSlot[ slotId ].card ):
         cardMib = entityMibRoot.cardSlot[ slotId ].card
      elif entityMibRoot.tacType.fullTypeName == "EntityMib::FixedSystem":
         cardMib = entityMibRoot
      chipMib = cardMib.chip.get( chipId, None ) if cardMib else None
      return chipMib

   @Tac.handler( 'firmwareRev' )
   def handleFirmwareRevision( self ):
      chipMib = self.getScdChipMib()
      if chipMib:
         major = self.notifier_.firmwareRev
         minor = FpgaUtil.parseVersion( chipMib.firmwareRev ).minor
         chipMib.firmwareRev = "0x%02x.%02x" % ( major, minor )

   @Tac.handler( 'hardwareMajorRev' )
   def handleHardwareMajorRev( self ):
      chipMib = self.getScdChipMib()
      if chipMib:
         major = FpgaUtil.parseVersion( chipMib.firmwareRev ).major
         minor = self.notifier_.hardwareMajorRev
         chipMib.firmwareRev = "0x%02x.%02x" % ( major, minor )

def Plugin( context ):
   global sliceStatusDir, cellStatusDir, entMib, redundancyStatus  
   context.registerDriver( PciScdDriver )

   mg = context.entityManager.mountGroup()
   mg.mount( 'hardware/scd/config/slice', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/pciDeviceMap/config', 'Hardware::PciDeviceConfigDir', 'w' )
   mg.mount( 'hardware/xcvrController/config', 'Tac::Dir', 'wi' )
   mg.mount( 'environment/temperature/config', 
             'Environment::Temperature::Config', 'w' )
   mg.mount( 'hardware/archer/gpo', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/archer/gpio/init', 'Tac::Dir', 'wi' )
   cellStatusDir = mg.mount( 'hardware/archer/scd/status/cell', 'Tac::Dir', 'ri' )
   sliceStatusDir = mg.mount( 'hardware/archer/scd/status/slice', 'Tac::Dir', 'ri' )
   entMib = mg.mount( 'hardware/entmib', 'EntityMib::Status', 'wi' )
   # redundancy/status is already mounted through CAgent layer
   redundancyStatus = context.entityManager.lookup( Cell.path('redundancy/status' ) )
   mg.close( setupReactors )


