# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable-msg=F0401
import Fdl, Fru, Tac
import FruPlugin.Smbus
import sys
import Cell
from EntityMib import IndexAllocator
from FruPlugin.Health import registerHealthSource
from PowerSupplyIncompatibilities import incompatibilityList

import Logging
import Tracing
t0 = Tracing.trace0
__defaultTraceHandle__ = Tracing.Handle( "Fru.PowerSupply" )

Logging.logD( id="FRU_POWERSUPPLY_INSERTED",
              severity=Logging.logInfo,
              format="Power Supply in slot %d has been inserted.%s",
              explanation="Power supply insertion detected",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

Logging.logD( id="FRU_POWERSUPPLY_REMOVED",
              severity=Logging.logInfo,
              format="Power Supply in slot %d has been removed.%s",
              explanation="Power supply removal detected",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

Logging.logD( id="FRU_POWERSUPPLY_UNSUPPORTED",
              severity=Logging.logError,
              format="Power Supply in slot %d is not supported",
              explanation=( "The power supply inserted in the above slot is "
                            "unknown and not supported by the software" ),
              recommendedAction=( "Replace the power supply with a "
                                  "power supply supported by the version of your "
                                  "software." ) )

Logging.logD( id="FRU_POWERSUPPLY_FAILURE",
              severity=Logging.logError,
              format="Power Supply in slot %d failed ",
              explanation=( "The power supply inserted in the above slot could "
                            "not be identified due to hardware errors" ),
              recommendedAction=( "Check that the power supply is properly "
                                  "inserted. It may be necessary to replace "
                                  "the power supply." ) )

Logging.logD( id="FRU_POWERSUPPLY_DRIVER",
              severity=Logging.logError,
              format="Fatal software failure for the power supply in slot %s",
              explanation="A fatal error occurred when attempting to instantiate"
              " management software for the power supply.",
              recommendedAction="Upgrade software to the current release. Contact "
              " support if the problem persists following an upgrade. " )

Logging.logD( id="FRU_POWERSUPPLY_UNSUPPORTED_IN_SYSTEM",
              severity=Logging.logError,
              format="Power supply in slot %s is unsupported in system.",
              explanation="Software detected unsupported power supply in system.",
              recommendedAction=( "Replace unsupported power supply to improve"
              " system performance." ) )
Logging.logD( id="FRU_POWERSUPPLY_OUT_OF_DATE",
              severity=Logging.logWarning,
              format="Power supply in slot %s is out-of-date.",
              explanation="Software detected out-of-date power supply in system.",
              recommendedAction=( "Out-of-date power supply should be replaced"
              " with newer model." ) )
Logging.logD( id="FRU_POWERSUPPLY_INCOMPATIBILITY",
              severity=Logging.logWarning,
              format=( "Power supply %s in slot %s is not fully compatible with "
                       "the %s power supply in slot %s" ),
              explanation=( "Software detected incompatible power supplies in "
                            "system. Some features may not work as intended" ),
              recommendedAction=( "Incompatible power supply should be replaced"
                                  " with a compatible model." ) )

class UnidentifiedPowerSupplyDriver( Fru.FruDriver ):
   """ Driver for power supplies that could not be identified by the
   PowerSupplyDetector. """

   driverPriority = 0

   managedTypeName = "Inventory::PowerSupply::PowerSupplyFru"
   managedApiRe = ".*$"

   def __init__( self, powerSupplyFru, parentEntityMib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, powerSupplyFru, parentEntityMib,
                              parentDriver, driverCtx )

      self.powerSupplyFru_ = powerSupplyFru
      self.driverCtx_ = driverCtx
      self.incommunicable_ = False

      # Create the Environment::Power::SupplyStatus. Set hwstatus to totalFailure
      powerEnvStatusDir = driverCtx.sysdbRoot.entity['environment/archer/power/'
                                                     'status/powerSupply' ]
      psName = "PowerSupply%d" % powerSupplyFru.slotId
      registerHealthSource( powerSupplyFru, psName )
      status = powerEnvStatusDir.newEntity( 'Environment::Power::SupplyStatus',
                                            psName )
      status.slot = powerSupplyFru.slotId
      # Create the entity mib object as a dependent of powerSupplyFru
      # Set the entity mib information to "unknown"
      powerSupplySlotEntmib = parentEntityMib.powerSupplySlot[
         powerSupplyFru.slotId ]
      if not powerSupplySlotEntmib.powerSupply:
         physicalIndex = IndexAllocator.physicalIndex\
                         ( powerSupplySlotEntmib,
                           'PowerSupply',
                           powerSupplySlotEntmib.relPos )
         Fru.Dep( powerSupplySlotEntmib, powerSupplyFru ).powerSupply = ( 
            physicalIndex, powerSupplySlotEntmib.relPos, "PowerSupply" )
         powerSupplySlotEntmib.powerSupply.description = psName
         powerSupplySlotEntmib.powerSupply.swappability = "hotSwappable"
      self.powerSupplyEntmib_ = powerSupplySlotEntmib.powerSupply
      self.powerSupplyEntmib_.label = powerSupplySlotEntmib.label
      self.powerSupplyEntmib_.mfgName = "Unknown"

   @property
   def incommunicable( self ):
      return self.incommunicable_

   @incommunicable.setter
   def incommunicable( self, value ):
      self.incommunicable_ = value
      if not self.incommunicable_:
         return
      self.powerSupplyEntmib_.initStatus = "ok"

class HwPowerSupplySlotStatusReactor( Tac.Notifiee ):
   """ Waits for the Hardware::PowerSupplySlot::SlotStatus object
   associated with the given SlotConfig to be created, and then
   creates a SlotStatusReactor. """

   notifierTypeName = "Hardware::PowerSupplySlot::Status"

   def __init__( self, status, slotDir, hwSlotConfig, slotInv,
                 parentEntmib, parentDriver, driverCtx ):
      self.slotDir_ = slotDir
      self.hwSlotConfig_ = hwSlotConfig
      self.slotInv_ = slotInv
      self.parentEntmib_ = parentEntmib
      self.parentDriver_ = parentDriver
      self.driverCtx_ = driverCtx
      self.slotStatusReactor_ = None
      Tac.Notifiee.__init__( self, status )
      self.handleSlotStatus( key=hwSlotConfig.name )

   @Tac.handler( "slotStatus" )
   def handleSlotStatus( self, key ):
      if key != self.hwSlotConfig_.name:
         return
      hwSlotStatus = self.notifier_.slotStatus.get( key )
      if ( hwSlotStatus and
           ( self.hwSlotConfig_.generationId == hwSlotStatus.generationId ) ):
         assert not self.slotStatusReactor_
         self.slotStatusReactor_ = SlotStatusReactor(
            self.slotDir_, hwSlotStatus, self.slotInv_, self.parentEntmib_,
            self.parentDriver_, self.driverCtx_ )
      else:
         self.slotStatusReactor_ = None

class SlotStatusReactor( Tac.Notifiee ):
   # pylint: disable-msg=E0602
   """ Reacts to the Hardware::SlotStatus object, waiting for a power supply
   to be inserted and identified. Logs appropriate messages. Once a power supply
   is inserted and identified, exec the fdl, and find and runs the fru driver for
   the resulting object. When the power supply is removed, cleanup after it. """
   notifierTypeName = "Hardware::PowerSupplySlot::SlotStatus"

   def __init__( self, slotDir, slotStatus, slotInv, parentEntmib, parentDriver,
                 driverCtx ):
      self.slotDir_ = slotDir
      self.slotInv_ = slotInv
      self.parentEntmib_ = parentEntmib
      self.parentDriver_ = parentDriver
      self.driverCtx_ = driverCtx
      self.powerSupplyDriver_ = None
      self.powerSupplyFru_ = None
      self.prevState_ = "unknown"
      self.slotStatus_ = slotStatus
      Tac.Notifiee.__init__( self, slotStatus )
      self.handleState() # BUG1869 Required since we're doing
                         # immediate notifications

   def _logPsuChange( self, logMsg, psu ):
      info = ""

      if psu:
         if psu.modelName:
            info += " Model: %s" % psu.modelName
         if psu.hardwareRev:
            info += " Rev: %s" % psu.hardwareRev
         if psu.serialNum:
            info += " Serial number: %s" % psu.serialNum

      Logging.log( logMsg, self.slotInv_.slotId, info )

   @Tac.handler( 'state' )
   def handleState( self ):
      prevState = self.prevState_
      curState = self.notifier_.state
      name = self.notifier_.name
      incommunicable = self.slotStatus_.incommunicable
      t0( 'Slot %s: CurState=%s, PrevState=%s' % ( name, curState, prevState ) )
      
      # On agent start, prevState is also unknown
      if curState != "unknown":
         assert prevState != curState, "Unexpected call to handleState"

      # Allowed state transitions:
      #   unknown -> present
      #   notPresent -> present
      #   unknown -> notPresent
      #   present -> notPresent
      #   present -> identified
      #   present -> unknownPowerSupply
      #   present -> identificationFailed
      #   identified -> notPresent
      #   identified -> present
      #   unknownPowerSupply -> notPresent
      #   unknownPowerSupply -> present
      #   identificationFailed -> notPresent

      slotId = self.slotInv_.slotId

      psSlotMib = self.parentEntmib_.powerSupplySlot[ slotId ] 
      assert psSlotMib is not None

      if curState == "unknown":
         # Should only be in this state at the beginning of time
         assert prevState in ["unknown", "notPresent"], \
             "should only be true at the beginning of time"
      elif curState == "notPresent":
         if self.slotInv_.fru is not None:
            # Power supply has been removed
            self._logPsuChange( FRU_POWERSUPPLY_REMOVED, self.powerSupplyFru_ )
            self.powerSupplyDriver_ = None
            self.powerSupplyFru_ = None
            Fru.deleteDependents( self.slotInv_.fru )
            self.slotInv_.fru = None
            powerEnvStatusDir = self.driverCtx_.sysdbRoot.entity[
               'environment/archer/power/status/powerSupply' ]
            psName = "PowerSupply%d" % slotId
            if prevState == "unknownPowerSupply" and psName in powerEnvStatusDir:
               powerEnvStatusDir.deleteEntity( psName )
         t0( "last change time getting set" )
         psSlotMib.lastChangeTime = Tac.now()
      elif curState == "present":
         # Power supply has been inserted
         assert prevState in [ "unknown", "notPresent", "unknownPowerSupply",
                               "identified" ]
      else:
         if self.slotInv_.fru is not None:
            # This should only occur when an unknown power supply was inserted
            # due to the power supply becoming incommunicable
            Fru.deleteDependents( self.slotInv_.fru )
         if curState == "unknownPowerSupply":
            self._logPsuChange( FRU_POWERSUPPLY_INSERTED, None )
            if not incommunicable:
               Logging.log( FRU_POWERSUPPLY_UNSUPPORTED, slotId )
            self.slotInv_.fru = ( slotId, )
            powerSupplyFru = self.slotInv_.fru
         elif curState == "identificationFailed":
            self._logPsuChange( FRU_POWERSUPPLY_INSERTED, None )
            Logging.log( FRU_POWERSUPPLY_FAILURE, slotId )
            self.slotInv_.fru = ( slotId, )
            powerSupplyFru = self.slotInv_.fru
         elif curState == "identified":
            assert prevState in [ "unknown", "present" ]

            # Power supply has been identified
            fdl = Fdl.Rfc822Fdl( self.notifier_.fdl )
            try:
               powerSupplyFru = self.driverCtx_.fruFactoryRegistry.newFru(
                  self.slotInv_, fdl )
            except Fru.FruFactoryError:
               # XXX_APECH Can this even ever happen? The fdl is not baked into
               # hardware. We'd end up with an identificationFailed
               raise Tac.InternalException

            self._logPsuChange( FRU_POWERSUPPLY_INSERTED, None )
            self.powerSupplyFru_ = powerSupplyFru

            if self.slotInv_.fru is not None:
               # Power supply was previously identified so we need to keep the 
               # generation id the same so the status does not get deleted
               powerEnvStatusDir = self.driverCtx_.sysdbRoot.entity[ \
                     'environment/archer/power/status/powerSupply' ]
               psName = "PowerSupply%d" % slotId
               if psName in powerEnvStatusDir:
                  status = powerEnvStatusDir[ psName ]
                  powerSupplyFru.generationId = status.generationId

            psuModel = powerSupplyFru.modelName

            # Gather the other PSU model names
            parentDir = self.parentEntmib_.powerSupplySlot
            psus = [ slot.powerSupply for slot in parentDir.values() if
                     slot.powerSupply ]
            systemModelNames = [ ( psu.modelName, psu.label ) for psu in psus if
                                 ( psu.label != name and psu.modelName ) ]

            # Check for any PSUs incompatible with this newly inserted PSU
            if systemModelNames and incompatibilityList( psuModel ):
               for model, slot in systemModelNames:
                  if model in incompatibilityList( psuModel ):
                     Logging.log( FRU_POWERSUPPLY_INCOMPATIBILITY, psuModel,
                                  name, model, slot )

            if self.slotDir_.powerSupplyVerificationNeeded:
               psuFanName = powerSupplyFru.fan.values()[ 0 ].physicalProperties.name
               registerHealthSource( powerSupplyFru.fan.values()[ 0 ], psuFanName )
               if psuFanName not in self.slotDir_.supportedPowerSupplies.values():
                  Logging.log( FRU_POWERSUPPLY_UNSUPPORTED_IN_SYSTEM, slotId )
               elif psuFanName in self.slotDir_.outOfDatePowerSupplies.values():
                  Logging.log( FRU_POWERSUPPLY_OUT_OF_DATE, slotId )

            if self.slotStatus_.alternateBus:
               powerSupplyFru.deviceBase = self.slotInv_.alternateDeviceBase
            else:
               powerSupplyFru.deviceBase = self.slotInv_.deviceBase

         # In the case that we didn't use the power supply factory (like for an 
         # unsupported power supply), need to create a new dependent set 
         Fru.newDependentSet( powerSupplyFru ) 

         # instantiate the associated driver class
         try:
            self.powerSupplyDriver_ = self.driverCtx_.driverRegistry.newDriver(
               powerSupplyFru, self.parentEntmib_,
               self.parentDriver_, self.driverCtx_ )
            self.powerSupplyDriver_.incommunicable = incommunicable
         except NotImplementedError:
            sys.excepthook( *sys.exc_info() )
            Logging.log( FRU_POWERSUPPLY_DRIVER, slotId )
         # NOTE - do not catch general exceptions thrown by trying to
         # create a power supply fru driver. Since the Power Supply
         # FDL is software-generated, we know that the power supply is
         # supported, so any exception thrown here is simply a
         # software bug that we shouldn't attempt to explicitly
         # handle.

         t0( "last change time getting set" )
         psSlotMib.lastChangeTime = Tac.now()
         
      self.prevState_ = curState

def powerSupplyFactory( powerSupplySlot, fdl, idInParent ):
   """ Create the power supply fru within the given slot. This factory is
   valid for all system types. """
   assert idInParent is None

   powerSupplySlot.fru = ( powerSupplySlot.slotId, )
   powerSupplySlot.fru.component = ( "component", )
   powerSupplySlot.fru.managingCellId = 0
   powerSupplySlot.fru.sliceId = "PowerSupply%d" % powerSupplySlot.slotId
   if not powerSupplySlot.fru.generationId:
      # New power supply insertion (as opposed to a Fru restart).
      slotDir = powerSupplySlot.parent
      slotDir.generationIdMarker += 1
      powerSupplySlot.fru.generationId = slotDir.generationIdMarker

   # Create a new DependentSet for the power supply
   Fru.newDependentSet( powerSupplySlot.fru )

   fdl.execFdl( "fru", powerSupplySlot.fru )

   return powerSupplySlot.fru
   

class SlotDirDriver( Fru.FruDriver ):
   """ Manages the PowerSupplySlotDir object, create corresponding slot config and
   slot status objects, then create a reactor to the slot status """

   managedTypeName = "Inventory::PowerSupply::SlotDir"
   managedApiRe = "$"

   def __init__( self, slotdir, parentEntmib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, slotdir, parentEntmib, parentDriver, driverCtx )

      self.statusReactor_ = {}

      config = driverCtx.sysdbRoot.entity[ 'hardware/powerSupply/slot/config' ]
      cellConfig = driverCtx.sysdbRoot.entity[ 'hardware/powerSupply/config/cell/%d'
                                               % Cell.cellId() ]
      status = driverCtx.entity( 'hardware/powerSupply/slot/status' )

      config.basePower = slotdir.basePower
      config.deratingFactor = slotdir.deratingFactor
      for ( slotId, slotInv ) in slotdir.powerSupplySlot.items():
         midplaneSlotId = slotInv.midplaneSlotId
         topology = driverCtx.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]
         powerSupplyBaseAhamDesc = config.newAhamDesc(
            "slot%d" % slotId,
            *FruPlugin.Smbus.ahamDesc( topology, slotInv.deviceBase ) )
         if slotInv.alternateDeviceBase:
            powerSupplyAlternateBaseAhamDesc = config.newAhamDesc(
               "slot%dAlternate" % slotId,
               *FruPlugin.Smbus.ahamDesc( topology, slotInv.alternateDeviceBase ) )
         else:
            powerSupplyAlternateBaseAhamDesc = None
         # create the Hardware::PowerSupply::SlotConfig
         # The presentGpo and presentInterrupt bits may be set either directly in
         # the powerSupplySlot (currently in fixed systems) or in the InvScd
         # (necessary for modular systems).
         invModularSystem = driverCtx.sysdbRoot[ 'hardware' ][ 'inventory' ].get(
            'modularSystem' )
         if invModularSystem:
            assert not slotInv.present
            assert not slotInv.presentInterrupt
            invScd = invModularSystem.card[ Cell.cellId() ].component[ 'scd' ]
            if ( invScd.psuPresentGpo and invScd.psuPresentInterrupt ):
               Fru.Dep( cellConfig.psuPresentInterrupt,
                        invModularSystem.card[ Cell.cellId() ] )[ slotId ] = \
                        invScd.psuPresentInterrupt[ midplaneSlotId ].fruIntCtrl
               Fru.Dep( cellConfig.psuPresentGpo,
                        invModularSystem.card[ Cell.cellId() ] )[ slotId ] = \
                        invScd.psuPresentGpo[ midplaneSlotId ].systemName
            if invScd.psuPresentChangedClearGpo:
               Fru.Dep( cellConfig.psuPresentChangedClearGpo,
                        invModularSystem.card[ Cell.cellId() ] )[ slotId ] = \
                        invScd.psuPresentChangedClearGpo[ midplaneSlotId ].systemName

         else:
            cellConfig.psuPresentInterrupt[ slotId ] = \
                slotInv.presentInterrupt and slotInv.presentInterrupt.fruIntCtrl
            cellConfig.psuPresentGpo[ slotId ] = \
                slotInv.present.systemName if slotInv.present else ''

         powerOkGpo = slotInv.powerOk.systemName if slotInv.powerOk else ''
         slotConfig = config.newSlotConfig(
            str( slotId ),
            Fru.fruBase( slotInv ).generationId,
            slotId,
            powerOkGpo,
            powerSupplyBaseAhamDesc,
            powerSupplyAlternateBaseAhamDesc,
            slotInv.slotCoolingIndependent,
            slotInv.pollInterval,
            slotInv.pollIntervalWhenInterruptsEnabled,
            slotInv.addrOffset )
         # Setup a reactor to the Hardware::PowerSupply::SlotStatus object
         # being created (by the PowerSupplyDetector in response to the config
         # being created).
         self.statusReactor_[ str( slotId ) ] = HwPowerSupplySlotStatusReactor(
            status,
            slotdir,
            slotConfig,
            slotInv,
            parentEntmib,
            parentDriver,
            driverCtx )

def Plugin( ctx ):
   ctx.registerDriver( SlotDirDriver )
   ctx.registerDriver( UnidentifiedPowerSupplyDriver )
   ctx.registerFruFactory( powerSupplyFactory, "PowerSupply" )
   
   mg = ctx.entityManager.mountGroup()
   mg.mount( 'environment/archer/power/status/powerSupply', 'Tac::Dir', 'wi' )
   mg.mount( 'environment/cooling/config', 'Environment::Cooling::Config', 'w' )
   mg.mount( 'environment/temperature/config',
             'Environment::Temperature::Config', 'w' ) 
   mg.mount( 'hardware/powerSupply/slot/config',
             'Hardware::PowerSupplySlot::Config', 'w' )
   mg.mount( 'hardware/powerSupply/config/cell', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/powerSupply/slot/status',
             'Hardware::PowerSupplySlot::Status', 'r' )

   mg.close( None )
