#!/usr/bin/env python
# Copyright (c) 2008, 2009, 2010 Arastra, Inc.  All rights reserved.
# Arastra, Inc. Confidential and Proprietary.

# pylint: disable-msg=F0401
import Fru, EntityMib, SmbusClient, Tac
import FruPlugin.Smbus as Smbus
from EntityMib import IndexAllocator
import os
from FruPlugin.Health import registerHealthSource

class PmbusPowerSupplyDriverBase( Fru.FruDriver ):
   """ Base class for fru power supply drivers. Create component simulations,
   entity mib objects, anything else a card / fixed system driver would do. """

   # NOTE - driverPriority has to be > 0 so that the driver is used before
   # the UnidentifiedPowerSupplyDriver.
   driverPriority = 1

   provides = [ Fru.FruDriver.environmentInit ]
   checkRegisterValidity = False
   supportedRegisterValues = {}

   directM = 0
   directB = 0
   directR = 0
   tempSensorEncoding = "pmbusEncodingLinear11"
   fanCommandEncoding = "pmbusEncodingLinear11"
   readFanSpeedEncoding = "pmbusEncodingLinear11"
   inputVoltageEncoding = "pmbusEncodingLinear11"
   inputCurrentEncoding = "pmbusEncodingLinear11"
   inputPowerEncoding = "pmbusEncodingLinear11"
   outputCurrentEncoding = "pmbusEncodingLinear11"
   outputPowerEncoding = "pmbusEncodingLinear11"
   overTempEncoding = "pmbusEncodingLinear11"
   extraDelays = False
   outputCurrentCanSignalFailure = False
   fanFailureAndPowerLossDifferentiator = False
   sendByteSupported = False
   cmlFaultsLenient = True
   voltageWarningsLenient = True
   fanWarningsLenient = False
   powerLossMask = 0x0840
   powerLossValues = [ 0x0840, 0x0800 ]
   typeStr = "Hardware::PowerSupply::Pmbus::BlackBoxHandlerType"
   BlackBoxHandlerType = Tac.Type( typeStr )
   blackBoxHandlerType = BlackBoxHandlerType.Unsupported
   fanOperation = False
   noOperationCmd = False
   overTempLenientDelay = 15
   negativeVinAllowed = False
   dualFan = False
   mfrData = [
      ( "MFR_ID", 0x99, None ),
      ( "MFR_MODEL", 0x9a, None ),
      ( "MFR_REVISION", 0x9b, None ),
      ( "MFR_SERIAL", 0x9e, None ),
      ( "PRI_FIRMWARE", 0xe0, None ),
      ( "SEC_FIRMWARE", 0xe1, None ),
      ( "FW_REVISION", 0xd5, 8 ),
      ( "FW_REVISION_TW", 0xef, 8 ), 
      ]
   inputCurrentExponentFix = False
   pec = False

   def __init__( self, powerSupplyFru, parentEntityMib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, powerSupplyFru, parentEntityMib,
                              parentDriver, driverCtx )
      self.powerSupplyFru_ = powerSupplyFru
      slot = powerSupplyFru.slotId

      # ---------------------------------------------------------
      # Create the Hardware::PowerSupply::Pmbus::Device config
      topology = driverCtx.sysdbRoot.entity[ 'hardware/smbus/topology' ]
      slotConfig = driverCtx.sysdbRoot.entity[ 'hardware/powerSupply/slot/config' ]
      hwPsDir = self.hardwareDir( driverCtx.sysdbRoot )
      idPromOffset = SmbusClient.ahamAddress(
         accelId=0, busId=0, addrSize='addrSize1',
         deviceId=powerSupplyFru.idPromAddrOffset, offset=0 )
      pmbusDeviceOffset = SmbusClient.ahamAddress(
         accelId=0, busId=0, addrSize='addrSize1',
         deviceId=powerSupplyFru.deviceAddrOffset, offset=0 )
      baseAhamDesc = hwPsDir.newAhamDesc(
         str( slot ),
         *Smbus.ahamDesc( topology, powerSupplyFru.deviceBase ) )
      muxOffset = 0
      if powerSupplyFru.muxAddrOffset:
         muxOffset = SmbusClient.ahamAddress(
            accelId=0, busId=0, addrSize='addrSize1',
            deviceId=powerSupplyFru.muxAddrOffset, offset=1,
            pec=self.pec )

      pmbusPowerSupply = Fru.Dep( hwPsDir.powerSupply, powerSupplyFru ).newMember(
         slot,
         idPromOffset,
         pmbusDeviceOffset,
         baseAhamDesc,
         powerSupplyFru.generationId )
      pmbusPowerSupply.muxOffset = muxOffset
      pmbusPowerSupply.directM = self.directM
      pmbusPowerSupply.directB = self.directB
      pmbusPowerSupply.directR = self.directR
      pmbusPowerSupply.tempSensorEncoding = self.tempSensorEncoding
      pmbusPowerSupply.fanCommandEncoding = self.fanCommandEncoding
      pmbusPowerSupply.readFanSpeedEncoding = self.readFanSpeedEncoding
      pmbusPowerSupply.inputVoltageEncoding = self.inputVoltageEncoding
      pmbusPowerSupply.inputCurrentEncoding = self.inputCurrentEncoding
      pmbusPowerSupply.inputPowerEncoding = self.inputPowerEncoding
      pmbusPowerSupply.outputCurrentEncoding = self.outputCurrentEncoding
      pmbusPowerSupply.outputPowerEncoding = self.outputPowerEncoding
      pmbusPowerSupply.overTempEncoding = self.overTempEncoding
      pmbusPowerSupply.extraDelays = self.extraDelays
      pmbusPowerSupply.outputCurrentCanSignalFailure = \
          self.outputCurrentCanSignalFailure
      pmbusPowerSupply.fanFailureAndPowerLossDifferentiator = \
          self.fanFailureAndPowerLossDifferentiator
      pmbusPowerSupply.sendByteSupported = self.sendByteSupported
      pmbusPowerSupply.cmlFaultsLenient = self.cmlFaultsLenient
      pmbusPowerSupply.voltageWarningsLenient = self.voltageWarningsLenient
      pmbusPowerSupply.fanWarningsLenient = self.fanWarningsLenient
      pmbusPowerSupply.fanOperation = self.fanOperation
      pmbusPowerSupply.noOperationCmd = self.noOperationCmd
      pmbusPowerSupply.overTempLenientDelay = self.overTempLenientDelay
      pmbusPowerSupply.negativeVinAllowed = self.negativeVinAllowed
      pmbusPowerSupply.inputCurrentExponentFix = self.inputCurrentExponentFix
      pmbusPowerSupply.blackBoxHandlerType = self.blackBoxHandlerType
      pmbusPowerSupply.dualFan = self.dualFan
      pmbusPowerSupply.pecOffset = SmbusClient.ahamAddress(
         accelId=0, busId=0, addrSize='addrSize1', deviceId=0, offset=0,
         pec=self.pec )

      # Populate the power supply with valid register values.
      for regName in self.supportedRegisterValues:
         validRegVals = pmbusPowerSupply.validRegisterValues.newMember( regName )
         for regVal in self.pmbusGetValidRegisterValues( regName ):
            validRegVals.validValues[ regVal ] = True
      pmbusPowerSupply.checkRegisterValidity = \
            self.checkRegisterValidity

      # Populate powerLoss mask and values
      pmbusPowerSupply.powerLossMask = self.powerLossMask
      for val in self.powerLossValues:
         pmbusPowerSupply.powerLossValues[ val ] = True

      psName = "PowerSupply%d" % slot
      registerHealthSource( powerSupplyFru, psName )

      if "PMBUS_SIMULATION" not in os.environ:
         for mfrLabel, mfrAddr, mfrLen in self.mfrData:
            m = pmbusPowerSupply.newMfrData( mfrLabel, mfrAddr )
            if mfrLen is not None:
               m.len = mfrLen
               m.blockRead = False

      # ---------------------------------------------------------
      # Create the Environment::Power::SupplyConfig
      powerEnvConfig = driverCtx.sysdbRoot.entity[ 'environment/power/config' ]
      pmbusPowerSupply.capacity = powerSupplyFru.capacity
      pmbusPowerSupply.dualInput = powerSupplyFru.dualInput
      # --------------------------------------------------------
      # Create the power supply entity mib object
      powerSupplySlotEntmib = parentEntityMib.powerSupplySlot[ slot ]
      if not powerSupplySlotEntmib.powerSupply:
         physicalIndex = IndexAllocator.physicalIndex \
                         ( powerSupplySlotEntmib,
                           'PowerSupply',
                           powerSupplySlotEntmib.relPos )
         Fru.Dep( powerSupplySlotEntmib, powerSupplyFru ).powerSupply = (
            physicalIndex,
            powerSupplySlotEntmib.relPos,
            "PowerSupply" )
         powerSupplySlotEntmib.powerSupply.description = psName
         powerSupplySlotEntmib.powerSupply.swappability = "hotSwappable"
      powerSupplyEntmib = powerSupplySlotEntmib.powerSupply
      powerSupplyEntmib.label = powerSupplySlotEntmib.label
      EntityMib.populateMib( powerSupplyEntmib, powerSupplyFru )
      pmbusPowerSupply.parentLabel = powerSupplySlotEntmib.label

      # ---------------------------------------------------------
      envTemperature = driverCtx.sysdbRoot.entity[ 'environment/temperature' ]
      import FruPlugin.TempSensorFru
      for ( tsId, tsInv ) in powerSupplyFru.tempSensor.items():
         sensorName = EntityMib.componentNameFromParent( powerSupplyEntmib,
                                                         "TempSensor",
                                                         tsId )
         tsEnvConfig = FruPlugin.TempSensorFru.createSensorEnvConfig(
            envTemperature, sensorName, tsInv )
         tsEnvConfig.offOnPowerLoss = tsInv.offOnPowerLoss
         pmbusPowerSupply.newTempSensor( tsId, tsEnvConfig )
         sensorMib = powerSupplyEntmib.sensor.get( tsId )
         if sensorMib is None:
            physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                            ( powerSupplyEntmib.sensor, tsId )
            sensorMib = Fru.Dep( powerSupplyEntmib.sensor, tsInv ).newMember(
               physicalIndex, tsId, "TempSensor" )
            # Copy description from config.
            sensorMib.description = tsEnvConfig.description
         sensorMib.initStatus = "ok"

      # ---------------------------------------------------------
      coolingEnvConfig = driverCtx.sysdbRoot.entity[ 'environment/cooling/config' ]
      for ( fanId, fanInv ) in powerSupplyFru.fan.items():
         fanName = EntityMib.componentNameFromParent( powerSupplyEntmib,
                                                      "Fan",
                                                      fanId )         
         registerHealthSource( fanInv, fanName )
         if fanInv.physicalProperties:
            import FruPlugin.FanFru
            properties = FruPlugin.FanFru.copyFanProperties(
               coolingEnvConfig, fanInv.physicalProperties,
               powerSupplyFru.parent.invertFanDirection )
         else:
            properties = None
         fanEnvConfig = Fru.Dep( coolingEnvConfig.fan, powerSupplyFru ).newMember(
            fanName )
         fanEnvConfig.managed = fanInv.managed
         fanEnvConfig.offOnPowerLoss = (
            fanInv.offOnPowerLoss or
            ( pmbusPowerSupply.fanOperation and
              slotConfig.slotConfig[ str( slot ) ].slotCoolingIndependent ) )
         fanEnvConfig.properties = properties
         fanEnvConfig.fanTrayName = fanName
         fanEnvConfig.generationId = Fru.powerGenerationId( powerSupplyFru )
         if fanInv.controllerVars:
            fanEnvConfig.prop = fanInv.controllerVars.p
            fanEnvConfig.integ = fanInv.controllerVars.i
            fanEnvConfig.deriv = fanInv.controllerVars.d
            fanEnvConfig.pband = fanInv.controllerVars.pband
            fanEnvConfig.pidOverride = True

         Fru.Dep( coolingEnvConfig.fanPtr, powerSupplyFru ).addMember( fanEnvConfig )

         pmbusFan = pmbusPowerSupply.newFan( fanId, fanEnvConfig )
         pmbusFan.maxRpm = properties and properties.maxRpm or 0

         # Create the fan EntityMib object
         fanMib = powerSupplyEntmib.fan.get( fanId )
         if fanMib is None:
            physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                            ( powerSupplyEntmib.fan, fanId )
            fanMib = powerSupplyEntmib.newFan( physicalIndex,
                                               fanId,
                                               "Fan" )
         fanMib.description = "%s Fan %s" % (
            powerSupplyEntmib.description, fanId )

         # All PMBus power supply fans expose their speed
         fanSensorMib = fanMib.sensor.get( fanId )
         if fanSensorMib is None:
            physicalIndex = IndexAllocator.collectionItemPhysicalIndex(
               fanMib.sensor, fanId )
            fanSensorMib = fanMib.sensor.newMember(
               physicalIndex, fanId, "FanSensor" )
         fanSensorMib.description = "%s Sensor %d" % (
            fanMib.description, fanId )
         fanSensorMib.initStatus = "ok"
         fanMib.initStatus = "ok"

      if powerSupplyEntmib.sensor.keys():
         # Keep the number of sensors idempotent by making sensorIdBase
         #   be a constant offset from number of temp sensors.
         sensorIdBase = len( powerSupplyFru.tempSensor.items() )
      else:
         sensorIdBase = 0

      # Make sure we added exactly 1 output current sensor with name '1' for now
      # More code needs to be written to have > 1 current sensor
      assert len( powerSupplyFru.outputCurrentSensor ) == 1
      assert powerSupplyFru.outputCurrentSensor[ '1' ]
      assert powerSupplyFru.inputCurrentSensor[ '1' ]
      currentInfo = [ ( sensorIdBase + 1, 1,
                        powerSupplyFru.inputCurrentSensor[ '1' ].description ),
                      ( sensorIdBase + 2, 2, 
                        powerSupplyFru.outputCurrentSensor[ '1' ].description ) ]
      if '2' in powerSupplyFru.inputCurrentSensor:
         currentInfo.append( ( sensorIdBase + 5, 3,
                        powerSupplyFru.inputCurrentSensor[ '2' ].description ) )
      for sensorId, sensorLabel, description in currentInfo:
         sensorName = EntityMib.componentNameFromParent( powerSupplyEntmib,
                                                         "CurrentSensor",
                                                         sensorLabel )
         sensorMib = powerSupplyEntmib.sensor.get( sensorId )
         if sensorMib is None:
            physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                            ( powerSupplyEntmib.sensor, sensorId )
            sensorMib = powerSupplyEntmib.newSensor( physicalIndex,
                                                     sensorId,
                                                     "CurrentSensor" )
         sensorMib.label = str( sensorLabel )
         sensorMib.description = description
         sensorMib.initStatus = "ok"

      # Make sure we added exactly 1 output voltage sensor with name '1' for now
      # More code needs to be written to have > 1 current sensor
      assert len( powerSupplyFru.outputVoltageSensor ) == 1
      assert powerSupplyFru.outputVoltageSensor[ '1' ]
      assert powerSupplyFru.inputVoltageSensor[ '1' ]
      voltageInfo = [ 
         ( sensorIdBase + 3, powerSupplyFru.inputVoltageSensor[ '1' ], 1,
            powerSupplyFru.inputVoltageSensor[ '1' ].description,
            powerSupplyFru.inputVoltageSensor[ '1' ].expectedVoltage ),
         ( sensorIdBase + 4, powerSupplyFru.outputVoltageSensor[ '1' ], 2,
           powerSupplyFru.outputVoltageSensor[ '1' ].description,
           powerSupplyFru.outputVoltageSensor[ '1' ].expectedVoltage )
         ]
      if '2' in powerSupplyFru.inputVoltageSensor:
         voltageInfo.append( ( sensorIdBase + 6, 
            powerSupplyFru.inputVoltageSensor[ '2' ], 3,
            powerSupplyFru.inputVoltageSensor[ '2' ].description,
            powerSupplyFru.inputVoltageSensor[ '2' ].expectedVoltage ) )
      for sensorId, sensor, sensorLabel, description, expectedVoltage in voltageInfo:
         sensorName = EntityMib.componentNameFromParent( powerSupplyEntmib,
                                                         "VoltageSensor",
                                                         sensorLabel )
         # Create config for power supply voltage sensors
         marginLow = 0
         marginHigh = 0
         vsEnvConfig = Fru.Dep( powerEnvConfig.voltageSensor,
                                powerSupplyFru ).newMember( sensorName,
                                                            marginLow, marginHigh )
         vsEnvConfig.description = description
         vsEnvConfig.expectedVoltage = expectedVoltage
         vsEnvConfig.sliceName = Fru.fruBaseName( sensor )
         sensorMib = powerSupplyEntmib.sensor.get( sensorId )
         if sensorMib is None:
            physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                            ( powerSupplyEntmib.sensor, sensorId )
            sensorMib = powerSupplyEntmib.newSensor( physicalIndex,
                                                     sensorId,
                                                     "VoltageSensor" )
         sensorMib.label = str( sensorLabel )
         sensorMib.description = description
         sensorMib.initStatus = "ok"

      # Power supply was successfully initialized
      powerSupplyEntmib.initStatus = "ok"
      
      # Set the ready attribute
      pmbusPowerSupply.ready = True

   # pylint: disable-msg=R0201

   def pmbusGetValidRegisterValues( self, register ):
      return None

   @staticmethod
   def hardwareDir( sysdb ):
      """ Return the mountpoint where the power supply hardware config should
      be written. Deriving classes can override this if the default pmbus driver
      is not sufficient to manage their power supply """
      return sysdb.entity[ 'hardware/smbus/pmbus' ]

class PmbusV11Driver( PmbusPowerSupplyDriverBase ):
   managedTypeName = "Inventory::PowerSupply::PowerSupplyFru"
   managedApiRe = "PMBus-1.1$"

   @staticmethod
   def hardwareDir( sysdb ):
      return sysdb.entity[ 'hardware/powerSupply/pmbus11' ]

class PowerFuseRelayDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::Pmbus::PowerFuseRelay"
   managedApiRe = "$"

   def __init__( self, relayFru, parentEntmib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, relayFru, parentEntmib, parentDriver,
                              driverCtx )

      sysdbRoot = driverCtx.sysdbRoot

      smbusTopo = sysdbRoot.entity[ "hardware/smbus/topology" ]
      pmbusConfig = sysdbRoot.entity[ "hardware/powerSupply/pmbus11" ]
      ahamDesc = pmbusConfig.newAhamDesc(
         relayFru.name, *Smbus.ahamDesc( smbusTopo, relayFru.receiver ) )

      Fru.Dep( pmbusConfig.powerFuseRelay, relayFru ).newMember(
         relayFru.name, relayFru.shutdownKey, ahamDesc )


def Plugin( ctx ):
   ctx.registerDriver( PmbusV11Driver )
   ctx.registerDriver( PowerFuseRelayDriver )

   mg = ctx.entityManager.mountGroup()
   mg.mount( 'hardware/powerSupply/pmbus11',
             'Hardware::PowerSupply::Pmbus::Config', 'w' )

   mg.mount( 'hardware/smbus/topology', 'Hardware::SmbusTopology', 'r' )

   mg.mount( "hardware/powerSupply/slot/config",
             "Hardware::PowerSupplySlot::Config", "r" )
   mg.close( None )

