#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from itertools import chain

import Tac
import Fru
import Tracing
import Cell
from TypeFuture import TacLazyType

from HwL1TopologyComponentLib import Components
from HwL1TopologyComponentLib.Serdes import UNKNOWN_SERDES_GROUP_ID
from HwL1TopologyComponentLib.Errors import SerdesLogicalMappingConsistencyError
from HwL1TopologyComponentLib.Tuning import NoTuning, MEDIUM_ANY

__defaultTraceHandle__ = Tracing.Handle( 'PhyTopoFru' )

t0 = Tracing.trace0
t1 = Tracing.trace1
t5 = Tracing.trace5

PhySide = TacLazyType( 'PhyEee::PhySide' )
ManagementScope = TacLazyType(
   'Hardware::L1Topology::ManagementScope::ManagementScope' )
PhySideDesc = TacLazyType( 'PhyEee::PhySideDescriptor' )
MidplanePinDesc = TacLazyType( "PhyEee::MidplanePinDescriptor" )
TuningInfo = TacLazyType( "Inventory::L1TuningInfo" )

class PhyComponentTopoDriver( Fru.FruDriver ):

   managedTypeName = 'Inventory::L1ComponentTopoDir'
   managedApiRe = '$'

   provides =  []
   requires = [ "PhyTopoConfig", "PhyTraceTopo", "MidplanePositions",
                "MidplaneConnections" ]

   def __init__( self, l1ComponentTopo, fruEntMib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, l1ComponentTopo, fruEntMib,
                              parentDriver, driverCtx )

      sliceId = Fru.fruBase( l1ComponentTopo ).sliceId
      if not sliceId:
         assert Cell.cellType() == "fixed"
         sliceId = "FixedSystem"

      t0( 'Handling Inventory::L1ComponentTopoDir for %s' % sliceId )

      topoDir = driverCtx.entity( 'hardware/l1/topology' )
      tuningDir = driverCtx.entity( 'hardware/l1/tuning' )
      mappingDir = driverCtx.entity( 'hardware/l1/mapping' )
      phyTopoDir = driverCtx.entity( 'hardware/phy/topology/allPhys' )
      hwSliceDir = driverCtx.entity( 'hardware/slice' )
      phyFruHelper = Tac.newInstance(
            'Hardware::L1Topology::PhyFruPluginHelper', topoDir )
      self.midplaneFruHelper = Tac.newInstance(
            'Hardware::L1Topology::MidplaneFruPluginHelper', topoDir )
      xcvrFruHelper = Tac.newInstance(
            'Hardware::L1Topology::XcvrFruPluginHelper', topoDir )
      tuningFruHelper = Tac.newInstance(
            'Hardware::L1Topology::TuningFruPluginHelper', tuningDir )
      self.serdesMappingHelper = Tac.newInstance(
            'Hardware::L1Topology::SerdesMappingFruHelper', mappingDir )

      # add a lane for xcvr slots
      def addLane( xcvrFruHelper, sliceId, slotId, laneId, tx, traceId ):
         xcvrFruHelper.addXcvrLaneTopology( sliceId, slotId, laneId, tx, traceId )
         serdesStr = 'tx' if tx else 'rx'
         t1( 'added xcvr lane topology for physical %s lane: %d,'
             'xcvr slot: %d' % ( serdesStr, laneId, slotId ) )

      # add a serdes for normal l1 components ( not xcvr slots )
      def addSerdes( phyFruHelper, sliceId, serdesId, componentDef,
                     phySide, tx, chipId, traceId ):
         # TODO: See BUG489342; We assume that all of the components are of the same
         #       component type to calculate the offset.
         ( coreId, physicalLaneId ) = componentDef.getSerdes( serdesId )
         phyCoreId = chipId * componentDef.getNumCores() + coreId
         phyType = componentDef.getComponentType( coreId )
         serdesPtr = phyFruHelper.addSerdesTopology( sliceId, phyType, phyCoreId,
                                                     phySide,
                                                     physicalLaneId, tx, traceId )
         phyTypeTopoPtr = phyFruHelper.phyTypeTopology( sliceId, phyType )
         phyTypeTopoPtr.laneRemapCapable = componentDef.laneRemapCapable()
         
         # TODO BUG486097 adding componentLocalCoreId directly in PhyTopology.py only
         # instead of as a c'tor parameter carried through addSerdesTopology
         # in order to reduce impact of this new attribute which is only needed
         # by a limited set of L1Topo users (namely Tundra). Longterm
         # we would like to work on a cleaner design for platforms that benefit
         # from having the notion of cores scoped to an L1 component.
         pct = phyFruHelper.phyCoreTopology( sliceId, phyType, phyCoreId )
         pct.componentLocalCoreId = coreId

         # Add other side serdes for phys unable to do lane remapping
         # See /src/PhyEee/HwL1Topology.tac for more information
         # If the phy does not have an "other side" this is a no-op
         componentDef.addOtherSideSerdes( phySide, serdesPtr, coreId )

         serdesStr = 'tx' if tx else 'rx'
         t1( 'added serdes topology for physical %s serdes lane id: %d,'
             'phy core id: %d phy side: %s phy type: %s' %
             ( serdesStr, physicalLaneId, phyCoreId, phySide, phyType ) )

      def addExpectedNoTuning( l1Component, componentDef ):
         serdesIdBase = 0
         for coreId in range( componentDef.getNumCores() ):
            numSerdesPerCore = componentDef.getNumSerdesPerCore( coreId )
            for speed, tapGroup in componentDef.getTapGroups( coreId ).iteritems():
               if not issubclass( tapGroup, NoTuning ):
                  continue
               tuningInfo = TuningInfo( speed, MEDIUM_ANY, "defaultTuning" )
               for serdesId in range( serdesIdBase,
                                      serdesIdBase + numSerdesPerCore ):
                  lineTuning = l1Component.newLineSerdesTuning( serdesId )
                  systemTuning = l1Component.newSystemSerdesTuning( serdesId )
                  # All we need to do is instantiate an empty L1TuningData, as
                  # we expect that for the NoTuning case ( initial/no data )
                  lineTuning.newTuningData( tuningInfo )
                  systemTuning.newTuningData( tuningInfo )
            serdesIdBase += numSerdesPerCore

      # add tuning information for normal l1 component serdes ( not xcvr slots )
      def addTuning( tuningFruHelper, sliceId, serdesId, componentDef,
                     phySide, chipId, serdesTuning ):
         # TODO: See BUG489342; We assume that all of the components are of the same
         #       component type to calculate the offset.
         ( coreId, physicalLaneId ) = componentDef.getSerdes( serdesId )
         phyCoreId = chipId * componentDef.getNumCores() + coreId
         phyType = componentDef.getComponentType( coreId )
         serdesDesc = Tac.Value( "PhyEee::SerdesDescriptor", sliceId, phyType,
                                 phyCoreId, phySide, physicalLaneId, True ) # tx=True
         supportedTapGroups = componentDef.getTapGroups( coreId )
         assert supportedTapGroups, "Cannot tune %s phy." % phyType

         # TODO: See BUG489341; Make sure to add a pre-init for the slice path for
         #       the PhyType in question! See the PhyEee preinit file.
         Fru.Dep( tuningDir[ phyType ][ "slice" ], l1ComponentTopo ).newEntity(
            "Hardware::Phy::PhyTypeTuning", sliceId )

         # TODO: See BUG489469; Consider if we want to assert that we exactly match
         #       the speeds required by the component
         for tuningInfo, tuningData in serdesTuning.tuningData.iteritems():
            # tuningInfo describes when we should apply tuning, while the tuningData
            # describes what actual tuning values to apply.
            tapGroup = supportedTapGroups.get( tuningInfo.speed )
            assert tapGroup, \
                   "%s does not support tuning at %s" % ( phyType, tuningInfo.speed )
            tuningDesc = Tac.Value( "Hardware::Phy::SerdesTuningInfo",
                                    tuningInfo.tuningType, tuningInfo.speed,
                                    tuningInfo.medium )

            tuningParams = tuningData.tuningParams
            if tuningParams:
               # Convert the collection of "index to value" into a tuple in order
               _, tuningParams = zip( *sorted( tuningParams.iteritems() ) )
               assert len( tuningParams ) == len( tapGroup.parameters._fields ), \
                      "The tuningParams don't match the width of the tapGroup"
            else:
               # If we have no tuningParams, then we must be in the NoTuning case, so
               # override the tapGroup to handle it
               tapGroup = NoTuning

            # Create the actual model and add the actual tuning values to it.
            serdesTuningData = \
               tuningFruHelper.addSerdesTuningData( serdesDesc,
                                                    tuningDesc )

            tapGroup.addTuning( serdesTuningData,
                                tapGroup.parameters( *tuningParams ) )

         t1( 'added serdes tuning for serdes lane id: %d, phy core id: %d, '
             'phy side: %s, phy type: %s' %
             ( physicalLaneId, phyCoreId, phySide, phyType ) )

      # TODO: See BUG512914; We should investigate converting xcvrs into a
      #       first-class component similar to midplanes to allow for subclass
      #       checking
      if l1ComponentTopo.chipType == "XcvrSlot":
         # for the xcvr slot we do not need to get component class
         for slotId, l1Component in l1ComponentTopo.l1ComponentTopo.iteritems():
            t5( "processing %s slot %s" % ( l1ComponentTopo.name, slotId ) )
            for laneId, traceId in l1Component.systemRxSerdesId.iteritems():
               t5( "  processing rx trace id %s lane %s" % ( traceId, laneId ) )
               addLane( xcvrFruHelper, sliceId, slotId, laneId, False, traceId )
            for laneId, traceId in l1Component.systemTxSerdesId.iteritems():
               t5( "  processing tx trace id %s lane %s" % ( traceId, laneId ) )
               addLane( xcvrFruHelper, sliceId, slotId, laneId, True, traceId )
            # no need to check line side since this is the xcvr slot and
            # nothing is on the line side
            assert ( not l1Component.systemSerdesTuning and
                     not l1Component.lineSerdesTuning ), \
                   "Shouldn't have tuning information for a xcvr in L1Topo"
         return

      linePhySide = PhySide.phySideLine
      systemPhySide = PhySide.phySideSystem
      componentClass = Components.getComponent( l1ComponentTopo.chipType )
      if issubclass( componentClass, Components.Midplane ):
         # Have no mode for midplanes components yet, and they are modeled as having
         # a single core, so just use None and 0 here
         connType = componentClass( None ).getComponentType( 0 )
         for connId, l1Component in l1ComponentTopo.l1ComponentTopo.iteritems():
            rxPins = l1Component.lineRxSerdesId
            txPins = l1Component.lineTxSerdesId
            side = linePhySide
            if rxPins or txPins:
               assert not l1Component.systemRxSerdesId, \
                      "Both system and line RX serdes specified."
               assert not l1Component.systemTxSerdesId, \
                      "Both system and line TX serdes specified."
            else:
               rxPins = l1Component.systemRxSerdesId
               txPins = l1Component.systemTxSerdesId
               side = systemPhySide

            # Midplane pin ids are not scoped by rx vs tx; we assume this in the
            # later layers so assert it here
            assert not ( set( rxPins ) & set( txPins ) ), \
                   "Conflicting pin ids across RX and TX for midplane."
            assert ( not l1Component.systemSerdesTuning and
                     not l1Component.lineSerdesTuning ), \
                   "Shouldn't have tuning information for a midplane in L1Topo"

            for pinId, traceId in chain( rxPins.iteritems(), txPins.iteritems() ):
               self.addMidplane( sliceId, connType, connId, pinId, side, traceId )
      else:
         # for most components we fetch the component class and use it to get extra
         # information about the serdes
         coreModes = dict( l1ComponentTopo.coreMode.iteritems() )
         componentDef = componentClass( l1ComponentTopo.mode or None,
                                        coreModes=coreModes or None )
         for chipId, l1Component in l1ComponentTopo.l1ComponentTopo.iteritems():
            t5( "processing %s chip %s" % ( l1ComponentTopo.name, chipId ) )
            # Create the line/system serdes for the component
            for serdesId, traceId in l1Component.lineRxSerdesId.iteritems():
               addSerdes( phyFruHelper, sliceId, serdesId, componentDef,
                          linePhySide, False, chipId, traceId )
            for serdesId, traceId in l1Component.lineTxSerdesId.iteritems():
               addSerdes( phyFruHelper, sliceId, serdesId, componentDef,
                          linePhySide, True, chipId, traceId )
            for serdesId, traceId in l1Component.systemRxSerdesId.iteritems():
               addSerdes( phyFruHelper, sliceId, serdesId, componentDef,
                          systemPhySide, False, chipId, traceId )
            for serdesId, traceId in l1Component.systemTxSerdesId.iteritems():
               addSerdes( phyFruHelper, sliceId, serdesId, componentDef,
                          systemPhySide, True, chipId, traceId )

            # Add the tuning for the tx serdes on the component
            addExpectedNoTuning( l1Component, componentDef )
            for serdesId, serdesTuning in l1Component.systemSerdesTuning.iteritems():
               addTuning( tuningFruHelper, sliceId, serdesId, componentDef,
                          systemPhySide, chipId, serdesTuning )
            for serdesId, serdesTuning in l1Component.lineSerdesTuning.iteritems():
               addTuning( tuningFruHelper, sliceId, serdesId, componentDef,
                          linePhySide, chipId, serdesTuning )

         self.addSerdesMapping( sliceId, componentDef )

         # instantiate l1TopologySm to resolve lane and polarity swaps
         self.startTopologySm( sliceId, hwSliceDir, topoDir, phyTopoDir, mappingDir )


   def startTopologySm( self, sliceId, hwSliceDir, topoDir, phyTopoDir, mappingDir ):
      l1TopologyConfig = topoDir[ 'config' ]
      l1TopologySmCol = Tac.singleton( "Hardware::L1Topology::HwL1TopologySmCol" )

      if l1TopologyConfig.managementScope == ManagementScope.slot:
         self.l1TopoSm = l1TopologySmCol.newHwL1TopologySm(
            sliceId,
            ManagementScope.slot,
            hwSliceDir,
            topoDir,
            phyTopoDir,
            mappingDir )
      elif l1TopologyConfig.switchSliceId not in l1TopologySmCol.hwL1TopologySm:
         self.l1TopoSm = l1TopologySmCol.newHwL1TopologySm(
            l1TopologyConfig.switchSliceId,
            ManagementScope.chassis,
            hwSliceDir,
            topoDir,
            phyTopoDir,
            mappingDir )

   def addSerdesMapping( self, sliceId, component ):
      '''Given a HW L1 Topology Component, this function parses the statically
      defined logical SerDes mappings from the class, validates them, and creates the
      HW models in SysDb.

      Args:
         component ( object ): The HW L1 Component ( defined in the HW L1 Component
                                                     Lib )
      '''
      # TODO: BUG463069, investigate using tx/rx
      for coreId in range( component.getNumCores() ):
         for desc in component.getLogicalSerdesMappings( coreId ):

            systemGroupId = UNKNOWN_SERDES_GROUP_ID
            lineGroupId = UNKNOWN_SERDES_GROUP_ID
            systemPhySide = PhySide.phySideUnknown
            linePhySide = PhySide.phySideUnknown

            # Because there are two decoupled fields: Mode and Serdes, they need to
            # be kept in sync. e.g if Mode is None we expect Serdes to be an empty
            # list or also to be None.
            if bool( desc.sysMode ) is not bool( desc.sysSerdes ):
               raise SerdesLogicalMappingConsistencyError( component, coreId,
                                                           'system',
                                                           serdesMode=desc.sysMode,
                                                           serdes=desc.sysSerdes )

            if bool( desc.lineMode ) is not bool( desc.lineSerdes ):
               raise SerdesLogicalMappingConsistencyError( component, coreId, 'line',
                                                           serdesMode=desc.lineMode,
                                                           serdes=desc.lineSerdes )

            if desc.sysMode:
               systemPhySide = PhySide.phySideSystem
               phySideDesc = PhySideDesc(
                  sliceId, component.getComponentType( coreId ), coreId,
                  systemPhySide )
               systemGroup = (
                  self.serdesMappingHelper.serdesMappingHelper.thisSideSerdesGroup(
                     phySideDesc, desc.sysMode, desc.sysSerdes[ 0 ] ) )
               if not systemGroup:
                  systemGroup = self.serdesMappingHelper.addNextSerdesGroup(
                     component.getComponentType( coreId ), PhySide.phySideSystem  )
                  systemGroup.mode = desc.sysMode
                  for serdes in desc.sysSerdes:
                     systemGroup.serdes.add( serdes )

               systemGroupId = systemGroup.id

            if desc.lineMode:
               linePhySide = PhySide.phySideLine
               phySideDesc = PhySideDesc(
                  sliceId, component.getComponentType( coreId ), coreId,
                  linePhySide )
               lineGroup = (
                  self.serdesMappingHelper.serdesMappingHelper.thisSideSerdesGroup(
                     phySideDesc, desc.lineMode, desc.lineSerdes[ 0 ] ) )
               if not lineGroup:
                  lineGroup = self.serdesMappingHelper.addNextSerdesGroup(
                     component.getComponentType( coreId ), PhySide.phySideLine )
                  lineGroup.mode = desc.lineMode
                  for serdes in desc.lineSerdes:
                     lineGroup.serdes.add( serdes )

               lineGroupId = lineGroup.id

            if desc.lineMode or desc.sysMode:
               self.serdesMappingHelper.addNextBidirSerdesMapping(
                  component.getComponentType( coreId ),
                  Tac.newInstance( "Hardware::L1Topology::SerdesGroupDescriptor",
                     systemGroupId, systemPhySide ),
                  Tac.newInstance( "Hardware::L1Topology::SerdesGroupDescriptor",
                     lineGroupId, linePhySide ) )

   def addMidplane( self, sliceId, connType, connId, pinId, phySide, traceId ):
      pinDesc = MidplanePinDesc( sliceId, connType, connId, pinId )
      self.midplaneFruHelper.addMidplanePin( pinDesc, phySide, traceId )
      
      t1( 'added midplane pin for physical pin id: %d '
          'connector id: %d connector type: %s phy side: %s with trace id %s' %
          ( pinId, connId, connType, phySide, traceId ) )


def Plugin( context ):
   '''Register the plugins.'''
   context.registerDriver( PhyComponentTopoDriver )

   mg = context.entityManager.mountGroup()
   mg.mount( 'hardware/l1/mapping', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/l1/topology', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/l1/topology/config',
             'Hardware::L1Topology::L1TopologyConfig', 'wc' )
   mg.mount( 'hardware/l1/tuning', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/phy/topology/allPhys',
             'Hardware::PhyTopology::AllPhyIntfStatuses',
             'wi' )
   mg.close( None )

