# Copyright (c) 2008-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Fru
import Tracing
import FruPlugin.ModularSystem as ModularSystem
import FruPlugin.Smbus

__defaultTraceHandle__ = Tracing.Handle( "Fru.NorCalCard" )
t0 = Tracing.trace1
t1 = Tracing.trace1
t2 = Tracing.trace2

class HwCardStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Hardware::Card::Status"

   def __init__( self, hwCardStatus, name, generationId,
                 invNorCalCard, cardMibEnt, driverCtx ):
      self.name_ = name
      self.generationId_ = generationId
      self.invNorCalCard_ = invNorCalCard
      self.cardMibEnt_ = cardMibEnt
      self.driverCtx_ = driverCtx
      self.norCalCardStatusReactor_ = None
      Tac.Notifiee.__init__( self, hwCardStatus )
      self.handleCardStatus( None )

   @Tac.handler( 'cardStatus' )
   def handleCardStatus( self, name=None ):
      cardStatus = self.notifier_.cardStatus.get( self.name_ )
      if cardStatus:
         if cardStatus.generationId != self.generationId_:
            # Previous CardStatus object has not yet been deleted
            assert not self.norCalCardStatusReactor_
            return
         elif not self.norCalCardStatusReactor_:
            self.norCalCardStatusReactor_ = NorCalCardStatusReactor(
               cardStatus,
               self.invNorCalCard_,
               self.cardMibEnt_,
               self.driverCtx_ )
      else:
         if self.norCalCardStatusReactor_:
            assert False, "CardStatus %s should only be deleted when " \
                "the card is removed" % self.name_

class NorCalCardStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Hardware::Card::CardStatus"

   def __init__( self, hwCardStatus, invNorCalCard, cardMibEnt, driverCtx ):
      self.invNorCalCard_ = invNorCalCard
      self.cardMibEnt_ = cardMibEnt
      self.driverCtx_ = driverCtx
      self.preMainPowerDomainDriver_ = None
      self.mainPowerDomainDriver_ = None
      self.generationId_ = 0
      Tac.Notifiee.__init__( self, hwCardStatus )
      self.handleState()

   def powerStatus( self ):
      if Fru.fruBase( self.invNorCalCard_ ).parentAttrName == 'supeUplinkCard':
         return self.cardMibEnt_.uplinkPowerStatus
      else:
         return self.cardMibEnt_.powerStatus

   def powerStatusIs( self, status ):
      if Fru.fruBase( self.invNorCalCard_ ).parentAttrName == 'supeUplinkCard':
         self.cardMibEnt_.uplinkPowerStatus = status
      else:
         self.cardMibEnt_.powerStatus = status

   def _createPreMainPowerDomain( self ):
      invPreMainPowerDomain = self.invNorCalCard_.preMainPowerDomain
      assert invPreMainPowerDomain
      if ( ( invPreMainPowerDomain in Fru.Driver.dependentDir().dependentSetMap )
            and ( self.powerStatus() != 'poweredOn' )
            and self.generationId_ == 0 ):
         # Fru must have died while executing main power domain drivers
         t0( "Cleaning up after partially run preMainPowerDomain drivers" )
         self._deletePreMainPowerDomain()
         assert not Fru.dependentSet( invPreMainPowerDomain )

      self.generationId_ += 1
      card = Fru.fruBase( self.invNorCalCard_ )
      t0( "Instantiating pre main power domain drivers" )
      # Create a new dependent set for the pre main power domain, as we need
      # it to get cleaned up separately when the power to a card is enabled
      # disabled
      #
      # NOTE: If we were restarted, we will get back the same dependentSet
      nds = Fru.newDependentSet( invPreMainPowerDomain )
      if self.invNorCalCard_.preMainPowerDomainDependentSetKey != nds.key:
         # Change the generation ID only when the dependent set we use is different.
         # This way we will also cover config deletions when Fru restarted during
         # main or pre-main power domain creation or after creation but before
         # successful power on.
         self.invNorCalCard_.preMainPowerDomainDependentSetKey = nds.key
         card.powerOnAttempts += 1

      # Run the main domain fru plugins
      self.preMainPowerDomainDriver_ = ModularSystem.PreMainPowerDomainDriver(
                                             invPreMainPowerDomain,
                                             self.cardMibEnt_,
                                             self.driverCtx_ )

      # For the PCIe switch config
      pcieConfigDir = self.driverCtx_.sysdbRoot.entity[
            'hardware/pcieSwitch/config/system' ]
      if len( pcieConfigDir.pcieSwitchConfig ):
         switchName = 'Slot%dcardSwitch' % card.slot
         switchConfig = pcieConfigDir.pcieSwitchConfig.get( switchName )
      else:
         # This can happen if the first card is a fan spinner with no pcie chip
         switchConfig = None

      cardConfig = self.driverCtx_.sysdbRoot[ 'hardware' ][ 'card' ]\
                                [ 'config' ].pcieCardConfig[ card.sliceId ]
      Fru.Dep( cardConfig, invPreMainPowerDomain ).pcieSwitchConfig = switchConfig

      for invPciFpga in self.invNorCalCard_.pciFpga.values():
         hwPciFpgaConfig = invPciFpga.hwPciFpgaConfig
         assert hwPciFpgaConfig, (
            "Fru plugins probably ran in wrong order." )
         Fru.Dep( cardConfig.pciFpgaConfig,
                  invPreMainPowerDomain )[ hwPciFpgaConfig.name ] = True

      for invPartialReconfig in self.invNorCalCard_.partialReconfig.values():
         partialReconfigConfig = invPartialReconfig.partialReconfigConfig
         assert partialReconfigConfig, "Fru plugins probably ran in wrong order"
         Fru.Dep( cardConfig.partialReconfigConfig,
                  invPreMainPowerDomain ).addMember( partialReconfigConfig )

      cardConfig.scdPciHotpluggable = self.invNorCalCard_.scdPciHotpluggable

      # Must be set last
      cardConfig.pcieReadyGenerationId += 1
      cardConfig.pcieReady = True

   def _createMainPowerDomain( self ):
      invMainPowerDomain = self.invNorCalCard_.mainPowerDomain
      if ( ( invMainPowerDomain in Fru.Driver.dependentDir().dependentSetMap ) and
           ( self.powerStatus() != 'poweredOn' ) ):
         # Fru must have died while executing main power domain drivers
         t0( "Cleaning up after partially run mainPowerDomain drivers" )
         self._deleteMainPowerDomain()
         assert not Fru.dependentSet( invMainPowerDomain )

      t0( "Instantiating main power domain drivers" )
      # Create a new dependent set for the main power domain, as we need
      # it to get cleaned up separately when the power to a card is enabled
      # disabled
      Fru.newDependentSet( invMainPowerDomain )

      # Run the main domain fru plugins
      self.mainPowerDomainDriver_ = ModularSystem.MainPowerDomainDriver(
                                          invMainPowerDomain,
                                          self.cardMibEnt_,
                                          self.driverCtx_,
                                          self.notifier_.name )

   def _deletePreMainPowerDomain( self ):
      card = Fru.fruBase( self.invNorCalCard_ )
      cardConfig = self.driverCtx_.sysdbRoot[ 'hardware' ][ 'card' ] \
                                [ 'config' ].pcieCardConfig[ card.sliceId ]
      t0( "Deleting pre main power domain drivers" )
      Fru.deleteDependents( self.invNorCalCard_.preMainPowerDomain )
      cardConfig.pcieReady = False
      self.preMainPowerDomainDriver_ = None

   def _deleteMainPowerDomain( self ):
      t0( "Deleting main power domain drivers" )
      Fru.deleteDependents( self.invNorCalCard_.mainPowerDomain )
      self.mainPowerDomainDriver_ = None

   @Tac.handler( 'state' )
   def handleState( self ):
      t0( self.notifier_.name, "state changed to", self.notifier_.state )
      if self.notifier_.state == 'booting':
         self._deleteMainPowerDomain()
         # Create pre main power domain drivers required for power on.
         self._createPreMainPowerDomain()
      elif self.notifier_.state == 'running':
         if not self.preMainPowerDomainDriver_:
            # After switchover, all drivers need to be recreated with the same state;
            # Pre-Main Power domain drivers too and in the most common case, card
            # state will initially be running.
            self._createPreMainPowerDomain()
         # Card is powered on.
         self._createMainPowerDomain()
         self.powerStatusIs( 'poweredOn' )
      else:
         self.powerStatusIs( 'poweredOff' )
         self._deleteMainPowerDomain()
         if self.notifier_.state in [ 'unknown', 'poweredOff' ]:
            # Delete pre main power domain drivers only in these states.
            self._deletePreMainPowerDomain()

class NorCalCardBaseDriver( Fru.FruDriver ):
   """ Responsible for initializing the components of Inventory::NorCalCardBase """
   provides = [ Fru.FruDriver.systemInit ]
   requires = []

   def __init__( self, invNorCalCard, cardMibEnt, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, invNorCalCard, cardMibEnt,
                              parentDriver, driverCtx )
      invCard = Fru.fruBase( invNorCalCard )

      # --------------------------------------------------------------
      # Build up the smbus topology tree
      topology = driverCtx.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]
      node = FruPlugin.Smbus.createNode( topology, invCard )
      for smbusInv in invNorCalCard.smbus.values():
         FruPlugin.Smbus.addSmbusToTopology( topology, node, smbusInv )
      node.initialized = True

class NorCalCardDriver( NorCalCardBaseDriver ):
   """ Responsible for initializing the shared components of
   Inventory::NorCalLinecard, Inventory::NorCalFabriccard, and
   Inventory::NorCalSwitchcard """

   def __init__( self, invNorCalCard, cardMibEnt, parentDriver, driverCtx ):
      NorCalCardBaseDriver.__init__( self, invNorCalCard, cardMibEnt,
                                      parentDriver, driverCtx )
      self.invCard = Fru.fruBase( invNorCalCard )
      if not self.invCard.valid:
         # Bail if the card is not valid
         return

      # Run standby domain drivers
      self.standbyDomainDriver_ = ModularSystem.StandbyDomainDriver(
         invNorCalCard.standbyDomain, cardMibEnt, self, driverCtx )

      self.hwCardConfig = driverCtx.sysdbRoot[ 'hardware' ][ 'card' ][ 'config' ]
      self.hwCardStatus = driverCtx.entity( 'hardware/card/status' )

      self.cardName = self.getCardName( cardMibEnt )

   def getCardName( self, cardMibEnt ):
      return "%s%s" % ( cardMibEnt.tag, cardMibEnt.label )

   def logicalCardName( self, cardMibEnt ):
      return self.cardName

# Common things for PCIe cards on the Merced chassis
class NorCalPcieCardDriver( NorCalCardDriver ):
   requires = [ Fru.FruDriver.pciInit ]

   def __init__( self, invNorCalCard, cardMibEnt, parentDriver, driverCtx ):
      NorCalCardDriver.__init__( self, invNorCalCard, cardMibEnt,
                                 parentDriver, driverCtx )
      if not self.invCard.valid:
         # If we didn't succeed at fully executing the fdl of the
         # card, then bail here as we don't want to try to power on
         # the card. Plus the code below will fail given that we bail
         # on invalid cards in the NorCalCardDriver base class too.
         return

      invCard = Fru.fruBase( invNorCalCard )
      assert invCard.generationId

      cardConfig = Fru.Dep(
         self.hwCardConfig.pcieCardConfig, self.invCard ).newMember(
         self.cardName,
         invCard.generationId,
         invCard.serialNum,
         invCard.modelName,
         invCard.hardwareRev,
         invCard.maxPowerDraw,
         invCard.maxPoePowerDraw,
         invCard.typicalPowerDraw,
         invNorCalCard.pinPowerGood.systemName
         if invNorCalCard.pinPowerGood else '',  # main or primary
         invNorCalCard.pinTempAlert.systemName
                        if invNorCalCard.pinTempAlert else '',
         invNorCalCard.pinClkReset.systemName
                        if invNorCalCard.pinClkReset else '',
         invNorCalCard.pinFpgaReset.systemName
                        if invNorCalCard.pinFpgaReset else '',
         invNorCalCard.pinEcbOn1.systemName
                        if invNorCalCard.pinEcbOn1 else '',
         invNorCalCard.pinEcbOn2.systemName
                        if invNorCalCard.pinEcbOn2 else '',
         invNorCalCard.pinPort1Upstream.systemName
                        if invNorCalCard.pinPort1Upstream else '',
         invNorCalCard.pinPcieReset.systemName
                        if invNorCalCard.pinPcieReset else '',
         invNorCalCard.pinPcieFatalErr.systemName
                        if invNorCalCard.pinPcieFatalErr else '',
         invNorCalCard.pinFpgaProgram.systemName
                        if invNorCalCard.pinFpgaProgram else '',
         invNorCalCard.pinEjectorClosed.systemName
                        if invNorCalCard.pinEjectorClosed else '',
         invNorCalCard.pinDpmAlert.systemName
                        if invNorCalCard.pinDpmAlert else '',
         invNorCalCard.pinSmbusEnable.systemName
                        if invNorCalCard.pinSmbusEnable else '',
         invNorCalCard.pinPowerCycle.systemName
         if invNorCalCard.pinPowerCycle else '',
         invNorCalCard.secondaryCardPinPowerGood.systemName
         if invNorCalCard.secondaryCardPinPowerGood else '',
         invNorCalCard.secondaryCardPinTempAlert.systemName
         if invNorCalCard.secondaryCardPinTempAlert else '',
         invNorCalCard.secondaryCardPinEjectorClosed.systemName
         if invNorCalCard.secondaryCardPinEjectorClosed else '',
         invNorCalCard.secondaryCardPinPcieReset.systemName
         if invNorCalCard.secondaryCardPinPcieReset else '',
         invNorCalCard.secondaryCardPinPowerCycle.systemName
         if invNorCalCard.secondaryCardPinPowerCycle else '' )

      cardConfig.logicalSlotName = self.logicalCardName( cardMibEnt )
      for invPowerController in invNorCalCard.powerController.values():
         name = invPowerController.controllerName
         assert name, "Fru plugins probably ran in wrong order"
         cardConfig.powerControllerName[ name ] = True

      for invPointOfLoad in invNorCalCard.pointOfLoad.values():
         name = invPointOfLoad.controllerName
         assert name, "ControllerName not provided"
         cardConfig.pointOfLoadName[ name ] = True

      cardConfig.hasTempAlertPin = invNorCalCard.hasTempAlertPin
      cardConfig.hasPowerGoodPin = invNorCalCard.hasPowerGoodPin
      cardConfig.hasPcieSwitch = invNorCalCard.hasPcieSwitch

      cardConfig.powerOffDelay = invNorCalCard.powerOffDelay

      cardConfig.ready = True

      # Create a reactor to the associated Hardware::CardStatus object,
      # so that we can run the main power fru plugins once the card
      # is actually powered on. In order to ensure that we're reacting
      # to the correct HwCardStatus object, we need to also pass in
      # the generation id from the HwCardConfig
      self.hwCardStatusReactor_ = HwCardStatusReactor( self.hwCardStatus,
                                                       self.cardName,
                                                       cardConfig.generationId,
                                                       invNorCalCard,
                                                       cardMibEnt,
                                                       driverCtx )

class NorCalPcieLinecardDriver( NorCalPcieCardDriver ):
   managedTypeName = "Inventory::NorCalPcieLinecard"
   managedApiRe = ".*$"

   def __init__( self, invNorCalLinecard, cardMibEnt, parentDriver, driverCtx ):
      NorCalPcieCardDriver.__init__( self, invNorCalLinecard, cardMibEnt,
                                     parentDriver, driverCtx )

      # Populate the card entity mib object
      cardMibEnt.minAddr = invNorCalLinecard.minAddr
      cardMibEnt.maxAddr = invNorCalLinecard.maxAddr

   def logicalCardName( self, cardMibEnt ):
      return "%s%d" % ( cardMibEnt.tag, self.invCard.slotDesc.logicalSlotId )

class NorCalPcieFabriccardDriver( NorCalPcieCardDriver ):
   managedTypeName = "Inventory::NorCalPcieFabriccard"
   managedApiRe = ".*$"

   def __init__( self, invNorCalFabriccard, cardMibEnt, parentDriver, driverCtx ):
      NorCalPcieCardDriver.__init__( self, invNorCalFabriccard, cardMibEnt,
                                     parentDriver, driverCtx )
      cardMibEnt.maxFanTrayMissingAllowed = \
          invNorCalFabriccard.maxFanTrayMissingAllowed
      cardMibEnt.maxFanFailuresAllowed = invNorCalFabriccard.maxFanFailuresAllowed

class NorCalPcieSwitchcardDriver( NorCalPcieCardDriver ):
   managedTypeName = "Inventory::NorCalPcieSwitchcard"
   managedApiRe = ".*$"

   def __init__( self, invNorCalSwitchcard, cardMibEnt, parentDriver, driverCtx ):
      NorCalPcieCardDriver.__init__( self, invNorCalSwitchcard, cardMibEnt,
                                     parentDriver, driverCtx )
      cardMibEnt.maxFanTrayMissingAllowed = \
          invNorCalSwitchcard.maxFanTrayMissingAllowed
      cardMibEnt.maxFanFailuresAllowed = invNorCalSwitchcard.maxFanFailuresAllowed
      cardMibEnt.shutdownSupervisorOnThermalShutdown = \
          invNorCalSwitchcard.shutdownSupervisorOnThermalShutdown

class SupervisorUplinkDriver( NorCalPcieCardDriver ):
   managedTypeName = "Inventory::NorCalSupervisorUplink"
   managedApiRe = ".*$"

   def __init__( self, invNorCalSupeUplink, cardMibEnt, parentDriver, driverCtx ):
      NorCalPcieCardDriver.__init__( self, invNorCalSupeUplink, cardMibEnt,
                                     parentDriver, driverCtx )

      # Populate the card entity mib object
      cardMibEnt.minAddr = invNorCalSupeUplink.minAddr
      cardMibEnt.maxAddr = invNorCalSupeUplink.maxAddr

   def getCardName( self, cardMibEnt ):
      return "Linecard%s" % ( cardMibEnt.label )

class SupervisorDriver( NorCalCardBaseDriver ):
   managedTypeName = "Inventory::NorCalSupervisor"
   managedApiRe = ".*$"

   def __init__( self, invNorCalSupe, cardMibEnt, parentDriver, driverCtx ):
      NorCalCardBaseDriver.__init__( self, invNorCalSupe, cardMibEnt,
                                      parentDriver, driverCtx )
      # Populate the card entity mib object
      cardMibEnt.minAddr = invNorCalSupe.minAddr
      cardMibEnt.maxAddr = invNorCalSupe.maxAddr

def Plugin( context ):
   mg = context.entityManager.mountGroup()
   t1( "running NorCalCard plugins for system ", context.sysname )

   context.registerDriver( SupervisorDriver )
   context.registerDriver( SupervisorUplinkDriver )
   context.registerDriver( NorCalPcieSwitchcardDriver )
   context.registerDriver( NorCalPcieFabriccardDriver )
   context.registerDriver( NorCalPcieLinecardDriver )

   # Mount the Modular System Agent config and status objects
   mg.mount( 'hardware/card/config', 'Hardware::Card::Config', 'w' )
   mg.mount( 'hardware/card/status', 'Hardware::Card::Status', 'r' )

   def _finish():
      t1( "Mounts complete" )
   mg.close( _finish )
