#!/usr/bin/env python
# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Fru, EntityMib
import FruPlugin.Smbus as Smbus
import FruPlugin.Gpio
import FruPlugin.TempSensorFru as TempSensorFru
from EntityMib import IndexAllocator

class FanControllerDriverBase( Fru.FruDriver ):
   """ Base class for fan controller fru drivers """

   def __init__( self, fanControllerFru, parentMibEntity, parentDriver, ctx ):
      super( FanControllerDriverBase, self ).__init__(
         fanControllerFru, parentMibEntity, parentDriver, ctx )

      fruBase = Fru.fruBase( fanControllerFru )
      self.fanControllerName = self._fanControllerName()

      # Create the driver config and status dir.
      hwConfigDir = self._hardwareConfigDir()
      ahamDesc = self._ahamDesc()
      controllerHwConfig = Fru.Dep(
         hwConfigDir.fanController, fanControllerFru ).newMember(
         self.fanControllerName, fanControllerFru.modelName,
         ahamDesc, fruBase.generationId )
      fanControllerFru.hwFanController = controllerHwConfig

      self._doDriverSpecificStuff()

      # Signal that configuration is complete.
      controllerHwConfig.ready = True

   def _fanControllerName( self ):
      fanControllerFru = self.invEntity_
      parentMibEntity = self.parentMibEntity_
      return EntityMib.componentNameFromParent(
         parentMibEntity, "FanController", fanControllerFru.id )

   def _hardwareConfigDir( self ):
      raise NotImplementedError

   def _ahamDesc( self ):
      smbusFanControllerFru = self.invEntity_
      ctx = self.driverCtx_
      hwConfigDir = self._hardwareConfigDir()
      topology = ctx.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]
      ahamArgs = [ smbusFanControllerFru.fullName ]
      ahamArgs.extend( Smbus.ahamDesc( topology, smbusFanControllerFru ) )
      ahamDesc = Fru.Dep( hwConfigDir.ahamDesc,
                          smbusFanControllerFru ).newMember( *ahamArgs )
      return ahamDesc

   def _doDriverSpecificStuff( self ):
      pass

class SmbusFanControllerDriverBase( FanControllerDriverBase ):
   """ Base class for smbus fan controller fru drivers """

   provides = [ Fru.FruDriver.environmentInit, FruPlugin.Gpio.gpioInit ]

   def _doDriverSpecificStuff( self ):
      smbusFanControllerFru = self.invEntity_
      parentMibEntity = self.parentMibEntity_
      ctx = self.driverCtx_

      environmentDir = ctx.sysdbRoot[ 'environment' ]

      # Create the driver config and status dir
      hwConfigDir = self._hardwareConfigDir()
      controllerHwConfig = hwConfigDir.fanController[ self.fanControllerName ]

      entmib = ctx.entity( 'hardware/entmib' )
      gpioInitDir = ctx.sysdbRoot.entity[ 'hardware/archer/gpio/init' ]

      # For each fan controller temp sensor, create the associated
      # Environment::TempSensorConfig, Environment::TempSensorStatus, and
      # Hardware::FanController::TempSensorConfig, then add it to the
      # Hardware::FanController::SmbusControllerConfig
      temperatureConfig = environmentDir[ 'temperature' ][ 'config' ]
      for tempSensorFru in smbusFanControllerFru.fanControllerTempSensor.values():
         tempSensorId = tempSensorFru.sensorId # BUG1731
         tempSensorName = EntityMib.componentNameFromParent( parentMibEntity,
                                                             "TempSensor",
                                                             tempSensorId )

         TempSensorFru.createSensorEnvConfig( environmentDir[ 'temperature' ],
                                              tempSensorName, tempSensorFru )

         fanTempSensor = controllerHwConfig.tempSensorConfig.newMember(
            tempSensorFru.idFromFanController)
         fanTempSensor.tempSensorName = tempSensorName
         assert ( ( not tempSensorFru.alertDisabled ) or
               ( self.managedApiRe.startswith( "ADT-1" ) ) ), \
               'alertDisabled currently only supported on Adt7462'
         fanTempSensor.alertDisabled = tempSensorFru.alertDisabled
         # Create the sensor EntityMib object
         sensorMib = parentMibEntity.sensor.get( tempSensorFru.sensorId )
         if sensorMib is None:
            physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                            ( parentMibEntity.sensor, tempSensorFru.sensorId )
            sensorMib = Fru.Dep( parentMibEntity.sensor, tempSensorFru ).\
                        newMember( physicalIndex, tempSensorFru.sensorId,
                                   "TempSensor" )
         sensorMib.label = str( tempSensorId )
         sensorMib.description = tempSensorFru.description
         sensorMib.groupName = tempSensorFru.groupName

         # Declare success for the sensor. This MUST be done after the
         # TempSensorEnvConfig has been created
         sensorMib.initStatus = "ok"

      # For each fan controller voltage sensor, create the associated
      # Environment::Power::VoltageRail{Config,Status}, 
      # Hardware::FanController::VoltageRailConfig, then add it to the
      # Hardware::FanController::SmbusControllerConfig
      powerEnvConfig = environmentDir[ 'power' ][ 'config' ]
      marginLow = 0
      marginHigh = 0
      for voltageSensorFru in smbusFanControllerFru.voltageSensor.values():
         voltageSensorId = voltageSensorFru.sensorId # BUG1731
         voltageSensorName = EntityMib.componentNameFromParent( parentMibEntity,
                                                                "VoltageSensor",
                                                                voltageSensorId )
         envSensorConfig = Fru.Dep( powerEnvConfig.voltageSensor,
                                    voltageSensorFru ).newMember\
                                    ( voltageSensorName, \
                                         marginLow, marginHigh )
         envSensorConfig.expectedVoltage = voltageSensorFru.expectedVoltage
         envSensorConfig.sliceName = Fru.fruBaseName( voltageSensorFru )
         envSensorConfig.description = voltageSensorFru.description
         voltageSensorFru.envSensorConfig = envSensorConfig

         voltageSensor = controllerHwConfig.voltageSensorConfig.newMember(
            voltageSensorFru.idFromFanController )
         voltageSensor.voltageSensorName = voltageSensorName

         # Create the sensor EntityMib object
         sensorMib = parentMibEntity.sensor.get( voltageSensorFru.sensorId )
         if sensorMib is None:
            physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                            ( parentMibEntity.sensor, voltageSensorFru.sensorId )
            sensorMib = parentMibEntity.newSensor( physicalIndex,
                                                   voltageSensorFru.sensorId,
                                                   "VoltageSensor" )
         sensorMib.label = str( voltageSensorId )
         sensorMib.description = voltageSensorFru.description
         
         # Declare success for the sensor
         sensorMib.initStatus = "ok"

      #For each gpio pin used in the fan controller, populate the controller's
      #config hardware tree.
      for pin in smbusFanControllerFru.gpio.values():
         systemName = "%s-%d" % ( self.fanControllerName, pin.id )
         pinInit = Fru.Dep( gpioInitDir, smbusFanControllerFru ).newEntity(
            "Hardware::Gpio::ArcherPinInit",
            systemName )
         pinInit.pinName = pin.pinName
         pinInit.pinId = pin.id
         pinInit.count = pin.count
         pinInit.activeType = pin.activeType
         pinInit.setToInputIfInactive = pin.setToInputIfInactive
         pinInit.setToInputIfActive = pin.setToInputIfActive
         pinInit.direction = pin.direction
         pinInit.defaultVal = pin.defaultVal
         pin.systemName = systemName
         controllerHwConfig.gpioName[ systemName ] = True
         controllerHwConfig.addGpioInit( pinInit )

   def _hardwareConfigDir( self ):
      raise NotImplementedError

def Plugin( context ):
   mountGroup = context.entityManager.mountGroup()
   mountGroup.mount( 'environment/power/config',
                     'Environment::Power::Config', 'w' )
   mountGroup.mount( 'hardware/archer/gpio/init', 'Tac::Dir', 'wi' )
   mountGroup.close( None )
