#!/usr/bin/python
# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import os
import Tac, Fru, Fru.Port, Tracing
from EthIntf import MAX_SUPPORTED_MTU
import cjson
from AsuPStore import pStoreFilePathDefault, pStoreFileNameJson

__defaultTraceHandle__ = Tracing.Handle( "Fru.EthPort" )
t0 = Tracing.trace0
t1 = Tracing.trace1

ethIntfConfigSliceDir = None
ethIntfStatusSliceDir = None
ethIntfDefaultConfigDir = None
ethIntfSliceDefaultConfigDir = None
ethIntfCapabilitiesConfigSliceDir = None
ethPhyAutonegConfigSliceDir = None
ethPhyIntfDescSliceDir = None
generationConfigSliceDir = None
sysName = None

# Early reading of the PSTORE - this happens well before the PStore restorer
# code runs. Seconds before. Need to do it now so we get right info for MTU
# before early agents try using it. Not a problem in cold boot but when we do
# ASU boot connected ports are already up (thanks to # AsuFastPktRestore)
# before the correct information is written into Sysdb.
def getEthIntfPStoreMtus():
   mtuConfigs = {}
   cmdLinePath = os.environ.get( "ASU_CMDLINE_PATH", "/proc/cmdline" )
   cmdFile = file( cmdLinePath )
   m = re.search( r"arista\.asu_(reboot|hitless)", cmdFile.read() )
   if m is not None:
      t0( "ASU Reboot" )
      pStoreFilePath = os.environ.get( "ASU_PSTORE_PATH", pStoreFilePathDefault )
      cfgPath = pStoreFilePath + "/" + pStoreFileNameJson
      try:
         with open( cfgPath, "r" ) as f:
            config = f.read()
            jsonConfig = cjson.decode( config )
            ethIntfCfg = jsonConfig.get( "EthIntf", None )
            if ethIntfCfg is not None:
               mtuConfigs = ethIntfCfg.get( "intfMtus", {} )
               if not mtuConfigs:
                  t0( "intfMtus not in PSTORE or is empty. Using defaults." )
            else:
               t0( "EthIntf not in PSTORE. Using defaults." )
      except cjson.DecodeError:
         t0( "File %s not valid JSON format. Using defaults." % ( cfgPath ) )
      except IOError as e:
         t0( "Could not read %s: %s. Using defaults." % ( cfgPath, e ) )
   else:
      t0( "Not an ASU reboot. Not looking at PSTORE. Using defaults." )
   return mtuConfigs

class EthIntfFactory( Fru.Port.PortIntfFactory ):
   """ The EthIntfFactory provides manages, intfStatus, and cleanup methods
   for creating the appropriate EthIntfStatus object for a Fru::Port.
   This class also is responsible for creating the default EthIntfConfig objects,
   and thus is a reactor to the Fru::Status in order to be notified when the
   supported port roles are set """

   managedTypeName = "Inventory::EthPort"
   managedPortRoles = [ "Switched", "Internal" ]

   # The default switched intf id part string to use for a default link mode.
   # linkModeUnknown & 10GbpsFull map to the same id since they result in a
   # defaultIntfConfig to have linkModeLocal = 'linkModeForced10GbpsFull'.
   # This id mapping must be consistent with defaultSwitchedIntfConfigName's
   # return in EthIntfLib.py and this file.
   linkModeDefaultIntfId = { "linkModeUnknown" : "",
                             "linkModeForced10GbpsFull" : "",
                             "linkModeForced25GbpsFull" : "2",
                             "linkModeForced40GbpsFull" : "3",
                             "linkModeForced50GbpsFull" : "4",
                             "linkModeForced100GbpsFull" : "5",
                             "linkModeForced1GbpsFull" : "6",
                             "linkModeForced50GbpsFull1Lane" : "7",
                             "linkModeForced400GbpsFull8Lane" : "8",
                             "linkModeForced200GbpsFull4Lane" : "9", }

   mtuConfigs = getEthIntfPStoreMtus()

   @staticmethod
   def defaultSwitchedIntfConfigName( linkMode ):
      assert linkMode in EthIntfFactory.linkModeDefaultIntfId
      return 'DefaultEthSwitchedPort%s' % \
          EthIntfFactory.linkModeDefaultIntfId[ linkMode ]

   @staticmethod
   def getIntfCfgMtu( intfName ):
      mtuCfg = MAX_SUPPORTED_MTU
      intfCfg = EthIntfFactory.mtuConfigs.get( intfName, None )
      if intfCfg is not None:
         mtuCfg = intfCfg.get( "mtu", MAX_SUPPORTED_MTU )
      return mtuCfg

   def createIntfStatus( self, fruBase, port, intfName ):
      t0( "Create EthIntfStatus for", intfName )
      if fruBase.managingCellId != 0:
         sliceName = str( fruBase.managingCellId )
         capsSliceName = 'FixedSystem'
         phyAnConfigSliceName = 'FixedSystem'
         generationConfigSliceName = 'FixedSystem'
      else:
         sliceName = fruBase.sliceId
         capsSliceName = sliceName
         phyAnConfigSliceName = sliceName
         generationConfigSliceName = sliceName

      # Create the intfStatus object under the corresponding slice dir
      # Create the slice dir first if it does not already exist
      # XXX_BUG96584
      # For breadth tests, if the celltype is not explicitly set to 
      # 'supervisor', we assume its a fixed system and set the slice
      # name to the cell id
      # This is a temporary change till we can figure out how to set 
      # celltype to 'supervisor' in breadth tests
      # Currently if we attempt to set the CELLTYPE before calling
      # Artest.beginTest(), we hit an exception
      import Cell
      if os.environ.get( "SIMULATION_VMID" ) and\
         os.environ.get( "CELLTYPE" ) != 'supervisor':
         sliceName = str( Cell.cellId() )
         t0( "Forcing sliceName to", sliceName, "for breadth tests" )

      # Create the slice for the EthPhyIntfDesc entity
      ethPhyIntfDescDir = ethPhyIntfDescSliceDir.get( sliceName )
      if not ethPhyIntfDescDir:
         Fru.Dep( ethPhyIntfDescSliceDir, port ).newEntity(
               'Interface::EthPhyIntfDescDir', sliceName )
         t0( "Created status slice dir for", sliceName )
         ethPhyIntfDescDir = ethPhyIntfDescSliceDir[ sliceName ]
      ethPhyIntfDescDir.generation = Tac.Value(
            "Ark::Generation", Fru.powerGenerationId( fruBase ), True )

      intfStatusDir = ethIntfStatusSliceDir.get( sliceName )
      if not intfStatusDir:
         Fru.Dep( ethIntfStatusSliceDir, port ).newEntity(
            'Interface::EthPhyIntfStatusDir', sliceName )
         t0( "Created status slice dir for ", sliceName )
         intfStatusDir = ethIntfStatusSliceDir[ sliceName ]

      ethIntfCapConfig = ethIntfCapabilitiesConfigSliceDir.get( capsSliceName )
      if not ethIntfCapConfig:
         # Create CapabilitiesConfig
         Fru.Dep( ethIntfCapabilitiesConfigSliceDir, fruBase ).newEntity(
            'Interface::Capabilities::CapabilitiesConfig',
            capsSliceName )
         ethIntfCapConfig = ethIntfCapabilitiesConfigSliceDir[ capsSliceName ]
      ethIntfCapConfig.generation = Tac.Value( "Ark::Generation",
                                               Fru.powerGenerationId( fruBase ),
                                               True )
      ethPhyAnConfig = ethPhyAutonegConfigSliceDir.get( phyAnConfigSliceName )
      if not ethPhyAnConfig:
         # Create PhyAutonegConfig
         Fru.Dep( ethPhyAutonegConfigSliceDir, fruBase ).newEntity(
                  'Interface::Autoneg::PhyAutonegConfig', phyAnConfigSliceName )
         ethPhyAnConfig = ethPhyAutonegConfigSliceDir[ phyAnConfigSliceName ]
      ethPhyAnConfig.generation = Tac.Value( 'Ark::Generation',
                                             Fru.powerGenerationId( fruBase ),
                                             True )

      generationConfig = generationConfigSliceDir.get( generationConfigSliceName )
      if not generationConfig:
         # Create generationConfig
         generationConfig = Fru.Dep( generationConfigSliceDir, fruBase ).newEntity(
                  'Interface::GenerationConfig', generationConfigSliceName )
      generationConfig.generation = Tac.Value( 'Ark::Generation',
                                               Fru.powerGenerationId( fruBase ),
                                               True )

      t0( "intfStatusDir is ", intfStatusDir,
          "; ethIntfCapConfig is ", ethIntfCapConfig,
          "; ethIntfCapConfig.gen ", ethIntfCapConfig.generation,
          "; ethPhyAnConfig is ", ethPhyAnConfig,
          "; ethPhyAnConfig.gen ", ethPhyAnConfig.generation,
          "; ethPhyIntfDescDir is", ethPhyIntfDescDir,
          "; descStatusDir.gen is ", ethPhyIntfDescDir.generation )

      # Check if the interface already exists in the instantiating
      # collection. For why we should check the instantiating collection
      # only and not one of the *all* collections, see BUG94023
      if not intfStatusDir.intfStatus.has_key( intfName ):
         dsicn = EthIntfFactory.defaultSwitchedIntfConfigName(
            port.defaultLinkMode ) if port.role == "Switched" else \
                                          ( 'DefaultEth%sPort' % port.role )
         d = ethIntfDefaultConfigDir.defaultIntfConfig.get( dsicn )
         t0( 'Creating new intfStatus for', intfName )
         intf = Fru.Dep( intfStatusDir.intfStatus, port ).newMember(
            intfName, d, Fru.powerGenerationId( port ), port.macAddr,
            port.relativeIfindex )

         intf.mtu = EthIntfFactory.getIntfCfgMtu( intfName )
         t0( "Setting MTU for %s to %d" % ( intfName, intf.mtu ) )
         intf.maxMtu = MAX_SUPPORTED_MTU
      else:
         t0( 'intfStatus for', intfName, 'already exists' )
         intf = intfStatusDir[ intfName ]

      # Create the EthPhyIntfDesc entity in the corresponding slice
      # mount point
      if not ethPhyIntfDescDir.ethPhyIntfDesc.has_key( intfName ):
         dscin = EthIntfFactory.defaultSwitchedIntfConfigName(
            port.defaultLinkMode ) if port.role == "Switched" else \
                                          ( 'DefaultEth%sPort' % port.role )
         desc = Fru.Dep( ethPhyIntfDescDir.ethPhyIntfDesc, port ).newMember(
                                    intfName,
                                    ethPhyIntfDescDir.generation,
                                    port.macAddr, port.relativeIfindex,
                                    Tac.Value( "Arnet::IntfId", dscin ),
                                    "dataPlanePort" )
      else:
         t0( 'EthPhyIntfDesc for %s already exists' % intfName )
         desc = ethPhyIntfDescDir.ethPhyIntfDesc[ intfName ]

      desc.mtu = EthIntfFactory.getIntfCfgMtu( intfName )
      desc.maxMtu = MAX_SUPPORTED_MTU

      # If we stop creating an EthPhyIntfStatus in interface/status/eth/phy,
      # we should return desc here (uncomment the next line)

      # return desc

      return intf

   
   def handlePortRole( self, portRole ):
      # For each port role that we know about, create a default eth intf config.
      # XXX This assumes, perhaps incorrectly, that roles cannot be removed from
      # the list.

      assert portRole is not None 
      
      defaultEthIntfConfigs = ethIntfDefaultConfigDir.defaultIntfConfig

      def maybeAddNewDefaultIntfConfig( defaultIntfConfigName,
                                        linkMode="linkModeUnknown" ):
         if defaultIntfConfigName not in defaultEthIntfConfigs:
            deic = defaultEthIntfConfigs.newMember( defaultIntfConfigName )
            deic.adminEnabledStateLocal = "enabled"
            # BUG18018: We really ought to get rid of
            #           this whole DefaultIntfConfig business...
            # For backwards compatibility, linkModeUnknown is 10G
            deic.linkModeLocal = "linkModeForced10GbpsFull" \
                if linkMode == "linkModeUnknown" else linkMode
            deic.rxFlowcontrolLocal = "flowControlConfigOff"
            deic.txFlowcontrolLocal = "flowControlConfigOff"

      if portRole in EthIntfFactory.managedPortRoles:
         if portRole == "Switched":
            for linkMode in EthIntfFactory.linkModeDefaultIntfId.iterkeys():
               dsicn = EthIntfFactory.defaultSwitchedIntfConfigName( linkMode )
               maybeAddNewDefaultIntfConfig( dsicn, linkMode )
         else:
            maybeAddNewDefaultIntfConfig( 'DefaultEth%sPort' % portRole )
      else:
         # We don't manage this port role. Ignore.
         pass

# what should be the speed for 'linkModeUnknown' and 'linkModeAutoneg'?
ethLinkModes = Tac.Type( "Interface::EthLinkMode" )
linkModeToSpeed = {
      ethLinkModes.linkMode10MbpsFull : 10,
      ethLinkModes.linkModeForced10MbpsHalf : 10,
      ethLinkModes.linkModeForced10MbpsFull : 10,
      ethLinkModes.linkMode100MbpsFull : 100,
      ethLinkModes.linkModeForced100MbpsHalf : 100,
      ethLinkModes.linkModeForced100MbpsFull : 100,
      ethLinkModes.linkModeForced1GbpsHalf : 1000,
      ethLinkModes.linkModeForced1GbpsFull : 1000,
      ethLinkModes.linkModeForced10GbpsFull : 10000,
      ethLinkModes.linkModeForced25GbpsFull : 25000,
      ethLinkModes.linkModeAuto40GbpsFull : 40000,
      ethLinkModes.linkModeForced40GbpsFull : 40000,
      ethLinkModes.linkModeForced50GbpsFull : 50000,
      ethLinkModes.linkModeForced50GbpsFull1Lane : 50000,
      ethLinkModes.linkModeForced100GbpsFull : 100000,
      ethLinkModes.linkModeForced100GbpsFull2Lane : 100000,
      ethLinkModes.linkModeForced200GbpsFull4Lane : 200000,
      ethLinkModes.linkModeForced200GbpsFull8Lane : 200000,
      ethLinkModes.linkModeForced400GbpsFull8Lane : 400000,
}

def getHighestSpeed( ports ):
   speed = 10
   for p in ports:
      if p.defaultLinkMode not in [ ethLinkModes.linkModeUnknown,
                                    ethLinkModes.linkModeAutoneg ] and \
            linkModeToSpeed[ p.defaultLinkMode ] > speed:
         speed = linkModeToSpeed[ p.defaultLinkMode ]

   return speed

def getSubPortCount( ports ):
   count = 1
   for p in ports:
      if p.subLabel != 0xffffffff and p.subLabel > count:
         count = p.subLabel

   return count

# Port label is composed as label/subLabel, in context of bootstrap port marking
# we only care about /1 port, make subLabel primary key, label secondary key, so
# that all /1 ports appears before /2~4 when sorted. Treat ports without subLabel
# as having subLabel 1
def getKeyFromLabel( p ):
   if p.subLabel != 0xffffffff:
      return p.subLabel * 1000 + p.label
   else:
      return 1000 + p.label

# Assume port with subLabel at same speed, always put /1 in front of the others
def getSpeed( p ):
   return linkModeToSpeed[ p.defaultLinkMode ]

def showPorts( ports, desc="Ports" ):
   t1( desc )
   for p in ports:
      if p.subLabel != 0xffffffff:
         t1( "%d/%d" % ( p.label, p.subLabel ), p.defaultLinkMode )
      else:
         t1( "%d" % p.label, p.defaultLinkMode )

class EthPortDirDriver( Fru.Port.PortDirDriver ):
   managedTypeName = "Inventory::EthPortDir"
   managedApiRe = "$"

   def __init__( self, invPortDir, fruEntMib, parentDriver, driverCtx ):
      Fru.Port.PortDirDriver.__init__( self, invPortDir, fruEntMib,
                                       parentDriver, driverCtx )
      allFPPs = [ p for p in invPortDir.port.values()
                  if p.role == "Switched" and not p.isUnconnected ]
      allFPPsKnownSpeed = [ p for p in allFPPs
                            if p.defaultLinkMode not in
                            [ ethLinkModes.linkModeUnknown,
                              ethLinkModes.linkModeAutoneg ] ]

      # find the first 2 of all front panel ports
      firstTwoFPPs = sorted( allFPPs, key=getKeyFromLabel )[ 0:2 ]
      showPorts( firstTwoFPPs, desc="first 2 FPPs:" )
      firstTwoFastestFPPs = []
      if len( allFPPsKnownSpeed ) >= 2:
         # find the first 2 of highest speed front panel ports
         allFPPsBySpeed = sorted( allFPPsKnownSpeed, key=getSpeed, reverse=True )
         showPorts( allFPPsBySpeed, desc="FPPs sorted by speed:" )
         fastestFPPs = [ p for p in allFPPsBySpeed
                         if getSpeed( p ) == getSpeed( allFPPsBySpeed[ 0 ] ) ]
         firstTwoFastestFPPs = sorted( fastestFPPs, key=getKeyFromLabel )[ 0:2 ]
         showPorts( firstTwoFastestFPPs, desc="first 2 fastest FPPs:" )
      bsps = list( set( firstTwoFPPs ) | set( firstTwoFastestFPPs ) )
      for invPort in invPortDir.port.values():
         self.initPort( invPort, fruEntMib, parentDriver, driverCtx,
                        isBSP=( invPort in bsps ) )

   def portNamePrefix( self, port ):
      if port.role in EthIntfFactory.managedPortRoles:
         if port.role == "Internal":
            return "Internal"
         elif port.isUnconnected:
            return "UnconnectedEthernet"
         else:
            return "Ethernet"
      else:
         raise Exception( "Unknown port role %s" % port.role )

class EthPortMacUpdateDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::EthPortMacUpdate"

   def __init__( self, inv, parentMib, parentDriver, ctx ):
      t0( "Creating an EthPortMacUpdateDriver Fru driver for dir", inv.name )
      Fru.FruDriver.__init__( self, inv, parentMib, parentDriver, ctx )

      intfStatusDir = ethIntfStatusSliceDir.get( inv.sliceId )
      ethPhyIntfDescDir = ethPhyIntfDescSliceDir.get( inv.sliceId )
      for port, addr in inv.macAddr.iteritems():
         intfStatusDir[ port ].burnedInAddr = addr
         ethPhyIntfDescDir[ port ].burnedInAddr = addr

   def __del__( self ):
      inv = self.invEntity_
      t0( "Deleting an EthPortMacUpdateDriver Fru driver for dir", inv.name )

      intfStatusDir = ethIntfStatusSliceDir.get( inv.sliceId )
      ethPhyIntfDescDir = ethPhyIntfDescSliceDir.get( inv.sliceId )
      for port in inv.macAddr.keys():
         emptyAddr = '00:00:00:00:00:00'
         intfStatusDir[ port ].burnedInAddr = emptyAddr
         ethPhyIntfDescDir[ port ].burnedInAddr = emptyAddr

      Fru.FruDriver.__del__( self )

class EthIntfSliceDefaultConfigDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::EthIntfSliceDefaultConfig"

   def __init__( self, inv, parentMib, parentDriver, ctx ):
      t0( "Creating an EthIntfSliceDefaultConfig Driver Fru driver", inv.name )
      Fru.FruDriver.__init__( self, inv, parentMib, parentDriver, ctx )

      defaultConfig = ethIntfSliceDefaultConfigDir.newMember( inv.name )
      if inv.qsfpDefaultMode == Tac.Type( "Inventory::XcvrMode" ).xcvrMode4x10G:
         defaultConfig.qsfpDefaultMode = \
               Tac.Type( "Interface::XcvrMode" ).xcvrMode4x10G
      else:
         defaultConfig.qsfpDefaultMode = Tac.Type( "Interface::XcvrMode" ).noXcvrMode
      t0( "Creating an EthIntfSliceDefaultConfig Driver Fru driver", inv.sliceId )

   def __del__( self ):
      inv = self.invEntity_
      t0( "Deleting an EthIntfSliceDefaultConfig Driver Fru driver", inv.name )

      del ethIntfSliceDefaultConfigDir[ inv.name ]
      Fru.FruDriver.__del__( self )

_ethIntfFactory = {}
def Plugin( context ):
   context.registerDriver( EthPortDirDriver )
   context.registerDriver( EthPortMacUpdateDriver )
   context.registerDriver( EthIntfSliceDefaultConfigDriver )

   em = context.entityManager
   mountGroup = em.mountGroup()
   
   # Have Fru mount EthIntf state if it hasn't done so already
   hardwareInventory = mountGroup.mount( 'hardware/inventory', 'Tac::Dir', 'wi' )

   mountGroup.mount( 'interface/config/eth/phy', 'Tac::Dir', 'wi' )
   mountGroup.mount( 'interface/status/eth/phy', 'Tac::Dir', 'wi' )

   global ethIntfConfigSliceDir, ethIntfStatusSliceDir
   global ethPhyIntfDescSliceDir
   global ethIntfDefaultConfigDir, sysName
   global ethIntfSliceDefaultConfigDir
   global ethIntfCapabilitiesConfigSliceDir
   global ethPhyAutonegConfigSliceDir
   global generationConfigSliceDir
   ethIntfConfigSliceDir = mountGroup.mount( 'interface/config/eth/phy/slice',
                                             'Tac::Dir', 'w' )
   ethIntfStatusSliceDir = mountGroup.mount( 'interface/status/eth/phy/slice',
                                             'Tac::Dir', 'w' )
   ethPhyIntfDescSliceDir = mountGroup.mount( 'interface/archer/status/init/eth/'
                                              'phy/slice',
                                              'Tac::Dir', 'wi' )
   ethIntfDefaultConfigDir = mountGroup.mount(
                                  'interface/config/eth/phy/default',
                                  'Interface::EthPhyIntfDefaultConfigDir', 'w' )
   ethIntfSliceDefaultConfigDir = mountGroup.mount(
                                  'interface/archer/config/eth/phy/sliceDefault',
                                  'Interface::EthIntfSliceDefaultConfigDir', 'w' )

   # The above will go away once we finish single-writer work.
   # Create the "config" which has one purpose - to hold the generationId
   # used to qualify the capabilities input provided by other agents.
   ethIntfCapabilitiesConfigSliceDir = mountGroup.mount(
         'interface/archer/config/eth/capabilities/slice',
         'Tac::Dir', 'wi' )
   ethPhyAutonegConfigSliceDir = mountGroup.mount(
         'interface/archer/config/eth/phy/autoneg/slice',
         'Tac::Dir', 'wi' )
   generationConfigSliceDir = mountGroup.mount(
         Tac.Type( 'Interface::EthIntfModeDirMounter' ).generationConfigPath( '' ),
         'Tac::Dir', 'wi' )
   sysName = context.sysname

   def _finished():
      if not _ethIntfFactory.has_key( context.sysname ):
         _ethIntfFactory[ context.sysname ] = EthIntfFactory(
               hardwareInventory, None, None )

         # Register PortIntfFactory for ethernet switch ports
         context.registerPortIntfFactory( _ethIntfFactory[ context.sysname ] )


   mountGroup.close( _finished )
