#!/usr/bin/env python
# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Fru, Fru.Port, Tracing
from FruPlugin.Health import registerHealthSource

__defaultTraceHandle__ = Tracing.Handle( "Fru.EthPciPort" )
t0 = Tracing.trace0

ethIntfConfigSliceDir = None
ethIntfStatusSliceDir = None
ethIntfDefaultConfigDir = None
ethPhyIntfDescSliceDir = None
ethIntfCapabilitiesConfigSliceDir = None
ethPhyAutonegConfigSliceDir = None
sysName = None

class EthPciIntfFactory( Fru.Port.PortIntfFactory ):

   """ The EthPciIntfFactory provides manages, intfStatus, and cleanup methods
   for creating the appropriate EthIntfStatus object for a Fru::PciPort objects.
   This class also is responsible for creating the default EthIntfConfig objects,
   and thus is a reactor to the Fru::Status in order to be notified when the
   supported port roles are set.
   NOTE: For now we assume that all Inventory::PciPort objects are in fact
   Ethernet interfaces. """

   managedTypeName = "Inventory::PciPort"
   managedPortRoles = [ "Management", "Internal" ]

   def createIntfStatus( self, fruBase, port, intfName ):
      registerHealthSource( port, intfName )
      import Cell
      if fruBase.managingCellId == Cell.cellId():
         # This port is on our cell. Use its pci address to figure out
         # its kernel device name.
         import os
         sysclassnet = '/sys/class/net'
         dirs = os.listdir( sysclassnet )
         pciaddr = port.pciAddr
         pciname = "%04x:%02x:%02x.%1x" % ( pciaddr.domain, pciaddr.bus,
                                            pciaddr.slot, pciaddr.function )
         
         kernelDevName = ""
         for name in dirs:
            try:
               l = os.readlink( os.path.join( sysclassnet, name, "device" ) )
               if pciname in l:
                  kernelDevName = name
                  break
            except OSError:
               pass

         if not kernelDevName:
            # HACK HACK HACK
            # BUG1368 qemu doesn't assign the same pci slots as real hardware does!
            # BUG40924 Oak's ma interface's pci address is dynamic
            if Cell.cellType() == "supervisor":
               suffix = "%d_%d" % ( fruBase.managingCellId, port.label )
            else:
               suffix = "%d" % ( port.label )
            if port.role == "Management":
               kernelDevName = "ma" + suffix
            elif port.role == "Internal":
               kernelDevName = "internal" + suffix
            # END HACK

         mgmtIntf = os.environ.get( "MGMT_INTF" )
         if mgmtIntf and mgmtIntf[ -1 ] == intfName[ -1 ]:
            kernelDevName = mgmtIntf 

         if not kernelDevName:
            raise Exception( "Could not find the kernel interface associated with "
                             "pci port %s at address %s" %( intfName, pciname ) )
      else:
         # XXX_APECH We can't determine the mac address and kernel device name
         # from the pci address, although we do know how Aboot/Uboot will assign
         # them for internal interfaces.
         t0( "PciPort", intfName, "is managed by cell", fruBase.managingCellId )
         # We should revisit the following code once we find a proper way to
         # add deviceName to standby Interfaces. See BUG24955.
         if port.role == "Management":
            kernelDevName = "ma%d_%d" % ( fruBase.managingCellId, port.label )
         elif port.role == "Internal":
            kernelDevName = "internal%d_%d" % ( fruBase.managingCellId, port.label )
         else:
            kernelDevName = ""
      macAddr = port.macAddr
      if macAddr == "00:00:00:00:00:00" and fruBase.managingCellId == Cell.cellId():
         # When we expect to see the 00:00:00:00:00:00 mac address:
         # * Fixed system
         # When we shouldn't see the 00:00:00:00:00:00 mac address:
         # * Standby Supervisor in RPR mode
         # * Activer Supervisor
         # * Standby Supervisor in SSO mode

         # Read it from the kernel.
         kernelDevPath = os.path.join( sysclassnet, kernelDevName, "address" )
         if os.path.exists(kernelDevPath):
            output = file( kernelDevPath ).read()
            macAddr = output.strip()
         else:
            print "Kernel device file %s does not exist; "\
                  "mac address unknown for %s" % (kernelDevPath, kernelDevName)

      # Create the new EthIntfStatus object
      defaultConfigIntfId = 'DefaultEth%sPciPort' % port.role
      dic = ethIntfDefaultConfigDir.defaultIntfConfig.get( defaultConfigIntfId )
      # Create the EthPhyIntfStatus object under the corresponding slice dir
      # Create the slice dir first if it does not already exist
      sliceName = str( fruBase.managingCellId )

      # Create the slice for the EthPhyIntfDesc entity
      ethPhyIntfDescDir = ethPhyIntfDescSliceDir.get( sliceName )
      if not ethPhyIntfDescDir:
         Fru.Dep( ethPhyIntfDescSliceDir, port ).newEntity(
               'Interface::EthPhyIntfDescDir', sliceName )
         t0( 'Created status slice dir for', sliceName )
         ethPhyIntfDescDir = ethPhyIntfDescSliceDir[ sliceName ]
         ethPhyIntfDescDir.generation = Tac.Value(
               "Ark::Generation", Fru.powerGenerationId( fruBase ), True )

      if not ethIntfStatusSliceDir.has_key( sliceName ):
         Fru.Dep( ethIntfStatusSliceDir, port ).newEntity(
            'Interface::EthPhyIntfStatusDir', sliceName )
         t0( "Created status slice dir for ", sliceName )
         intfStatusDir = ethIntfStatusSliceDir[ sliceName ]

      if not ethPhyIntfDescDir.ethPhyIntfDesc.has_key( intfName ):
         # pylint: disable=W0612
         desc = Fru.Dep( ethPhyIntfDescDir.ethPhyIntfDesc, port ).newMember(
               intfName,
               ethPhyIntfDescDir.generation,
               macAddr,
               Tac.Value( "Interface::EthPhyRelativeIfindex" ),
               Tac.Value( "Arnet::IntfId", defaultConfigIntfId ), "ethPciPort" )
         desc.deviceName = kernelDevName
      else:
         # pylint: disable=W0612
         desc = ethPhyIntfDescDir.ethPhyIntfDesc[ intfName ]


      intfStatusDir = ethIntfStatusSliceDir[ sliceName ]
      t0( "intfStatusDir is ", intfStatusDir,
          "intfStatusDescDir is", ethPhyIntfDescDir,
          "generation is", ethPhyIntfDescDir.generation )
      # Check if the interface already exists in the instantiating 
      # collection. For why we should check the instantiating collection
      # only and not one of the *all* collections, see BUG94023
      if intfName not in intfStatusDir:
         t0( 'Creating new intfStatus object for PciPort', intfName,
               'kernel device name', kernelDevName, 'and mac addr', macAddr )
         ethIntfStatus = Fru.Dep( intfStatusDir.intfStatus,
                                  port ).newMember(
                                     intfName, dic,
                                     Fru.powerGenerationId( fruBase ),
                                     macAddr,
                                     Tac.Value( "Interface::EthPhyRelativeIfindex") )
         ethIntfStatus.deviceName = kernelDevName
      else:
         t0( 'intfStatus object for PciPort', intfName, 'already exists.' )
         ethIntfStatus = intfStatusDir[ intfName ]

      # Create the interface capabilites config.
      capSliceName = 'FixedSystem'
      ethIntfCapConfig = ethIntfCapabilitiesConfigSliceDir.get( capSliceName )
      if not ethIntfCapConfig:
         # Create CapabilitiesConfig
         Fru.Dep( ethIntfCapabilitiesConfigSliceDir, fruBase ).newEntity(
            'Interface::Capabilities::CapabilitiesConfig', capSliceName )
         ethIntfCapConfig = ethIntfCapabilitiesConfigSliceDir[ capSliceName ]
      ethIntfCapConfig.generation = Tac.Value( "Ark::Generation",
                                               Fru.powerGenerationId( fruBase ),
                                               True )

      # Create the interface autoneg config.
      ethPhyAnConfig = ethPhyAutonegConfigSliceDir.get( capSliceName )
      if not ethPhyAnConfig:
         # Create PhyAutonegConfig
         Fru.Dep( ethPhyAutonegConfigSliceDir, fruBase ).newEntity(
                  'Interface::Autoneg::PhyAutonegConfig', capSliceName )
         ethPhyAnConfig = ethPhyAutonegConfigSliceDir[ capSliceName ]
      ethPhyAnConfig.generation = Tac.Value( 'Ark::Generation',
                                             Fru.powerGenerationId( fruBase ),
                                             True )
      return ethIntfStatus

   def handlePortRole( self, portRole ):
      # For each port role that we know about, create a default eth intf config.
      # XXX This assumes, perhaps incorrectly, that roles cannot be removed from
      # the list.

      assert portRole is not None 
      
      defaultEthIntfConfigs = ethIntfDefaultConfigDir.defaultIntfConfig

      dicn = 'DefaultEth%sPciPort' % portRole
      t0( 'handlePortRole', dicn )
      if portRole in self.managedPortRoles:
         if not defaultEthIntfConfigs.has_key( dicn ):
            deic = defaultEthIntfConfigs.newMember( dicn )
            deic.adminEnabledStateLocal = "enabled"
            deic.linkModeLocal = "linkModeAutoneg"
            deic.rxFlowcontrolLocal = "flowControlConfigDesired"
            deic.txFlowcontrolLocal = "flowControlConfigDesired"

class EthPciPortDirDriver( Fru.Port.PortDirDriver ):
   managedTypeName = "Inventory::PciPortDir"
   managedApiRe = "$"

   def __init__( self, invPciPortDir, fruEntMib, parentDriver, driverCtx ):
      Fru.Port.PortDirDriver.__init__( self, invPciPortDir, fruEntMib,
                                       parentDriver, driverCtx )
      for invPciPort in invPciPortDir.port.values():
         self.initPort( invPciPort, fruEntMib, parentDriver, driverCtx )

   def portNamePrefix( self, port ):
      if port.role == "Management":
         return "Management"
      elif port.role == "Internal":
         return "Internal"
      else:
         raise Exception( "Unknown port role %s" % port.role )

_ethPciIntfFactory = {}
def Plugin( context ):
   context.registerDriver( EthPciPortDirDriver )
   
   em = context.entityManager
   mountGroup = em.mountGroup()
   
   # Have Fru mount Ebra if it hasn't done so already
   hardwareInventory = mountGroup.mount( 'hardware/inventory', 'Tac::Dir', 'wi' )
   
   mountGroup.mount( 'interface/config/eth/phy', 'Tac::Dir', 'wi' )
   mountGroup.mount( 'interface/status/eth/phy', 'Tac::Dir', 'wi' )

   global ethPhyIntfDescSliceDir
   global ethIntfConfigSliceDir, ethIntfStatusSliceDir
   global ethIntfDefaultConfigDir, sysName
   global ethIntfCapabilitiesConfigSliceDir
   global ethPhyAutonegConfigSliceDir
   ethIntfConfigSliceDir = mountGroup.mount( 'interface/config/eth/phy/slice',
                                             'Tac::Dir', 'w' )
   ethIntfStatusSliceDir = mountGroup.mount( 'interface/status/eth/phy/slice',
                                             'Tac::Dir', 'w' )
   ethPhyIntfDescSliceDir = mountGroup.mount( 'interface/archer/status/init/eth/'
                                              'phy/slice',
                                              'Tac::Dir', 'wi' )
   ethIntfDefaultConfigDir = mountGroup.mount(
                                  'interface/config/eth/phy/default',
                                  'Interface::EthPhyIntfDefaultConfigDir', 'w' )
   ethIntfCapabilitiesConfigSliceDir = mountGroup.mount(
         'interface/archer/config/eth/capabilities/slice',
         'Tac::Dir', 'wi' )
   ethPhyAutonegConfigSliceDir = mountGroup.mount(
         'interface/archer/config/eth/phy/autoneg/slice',
         'Tac::Dir', 'wi' )
   sysName = context.sysname

   def _finished():
      if not _ethPciIntfFactory.has_key( context.sysname ):
         _ethPciIntfFactory[ context.sysname ] = EthPciIntfFactory(
                        hardwareInventory, None, None )

      # Register PortIntfFactory for Ethernet pci ports
      context.registerPortIntfFactory( _ethPciIntfFactory[ context.sysname ] )

   mountGroup.close( _finished )

