# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from datetime import datetime
import FlowTrackerConst
from FlowTrackerConst import (
   activeInterval,
   constants,
   inactiveTimeout,
   ipfixMtu,
   ipfixPort,
   sampleRate,
   templateInterval,
   reservedGroupNames,
   accessListDefault,
   mirrorConfigDefault,
   initialCopy,
   mirrorIntervalFixed,
   mirrorIntervalRandom,
)
from TypeFuture import TacLazyType
import Tac
from Toggles.FlowTrackerToggleLib import toggleHwOffloadIpv4Enabled

FtrTypeEnum = TacLazyType( 'FlowTracking::FlowTrackingType' )
ftrTypeSampled = FtrTypeEnum.sampled
ftrTypeHardware = FtrTypeEnum.hardware
ftrTypeInbandTelemetry = FtrTypeEnum.inbandTelemetry
ftrTypeMirrorOnDrop = FtrTypeEnum.mirrorOnDrop
AddressFamily = TacLazyType( "Arnet::AddressFamily" )
exporterInactiveReasonEnum = Tac.Type( "FlowTracking::ExporterInactiveReasonEnum" )

ftrTypes = [
   ftrTypeSampled,
   ftrTypeHardware,
   ftrTypeInbandTelemetry,
   ftrTypeMirrorOnDrop
]

ftrTypeKwStr = {
   ftrTypeSampled : 'sampled',
   ftrTypeHardware : 'hardware',
   ftrTypeInbandTelemetry : 'telemetry inband',
   ftrTypeMirrorOnDrop : 'mirror-on-drop',
}

ftrTypeByName = { value : key for key, value in ftrTypeKwStr.iteritems() }

ftrTypeShowStr = {
   ftrTypeSampled : 'Sampled',
   ftrTypeHardware : 'Hardware',
   ftrTypeInbandTelemetry : 'Inband telemetry',
   ftrTypeMirrorOnDrop : 'Mirror on drop',
}

encapTypeEnum = TacLazyType( 'FlowTracking::EncapType' )
encapIpv4 = encapTypeEnum.encapIpv4
encapIpv6 = encapTypeEnum.encapIpv6
encapVxlan = encapTypeEnum.encapVxlan
encapTypeKwStr = {
   encapIpv4 : 'ipv4',
   encapIpv6 : 'ipv6',
   encapVxlan : 'vxlan',
}
encapTypeShowStr = {
   encapIpv4 : 'IPv4',
   encapIpv6 : 'IPv6',
   encapVxlan : 'VXLAN',
}

seqnoOffsetType = 'FlowTracking::StaticGroupSeqnoOffset'
GroupSeqnoOffsetEnum = TacLazyType( seqnoOffsetType )
reservedFgSeqnoBase = constants.reservedFgSeqnoBase
reservedGroupToSeqnoMap = {
   reservedGroupNames.groupHwIpv4 : \
         constants.reservedGroupSeqno( GroupSeqnoOffsetEnum.offsetIpv4 ),
   reservedGroupNames.groupHwIpv6 : \
         constants.reservedGroupSeqno( GroupSeqnoOffsetEnum.offsetIpv6 ),
   reservedGroupNames.groupHwVxlanIpv4 : \
         constants.reservedGroupSeqno( GroupSeqnoOffsetEnum.offsetVxlanIpv4 ),
   reservedGroupNames.groupHwVxlanIpv6 : \
         constants.reservedGroupSeqno( GroupSeqnoOffsetEnum.offsetVxlanIpv6 ),
}

# hwFgSeqno is computed based on the equation
#       U32 hwFgSeqno = configSeqno * hwFgSeqnoMultiple + offset;
# So for hwFgSeqno to be less than reservedFgSeqnoBase, we need
#       configSeqno < ( reservedFgSeqnoBase - offsetMax ) / hwFgSeqnoMultiple
groupSeqnoOffset = Tac.Value( "FlowTracking::GroupSeqnoOffset" )
configSeqnoMax = ( ( reservedFgSeqnoBase - groupSeqnoOffset.maxOffset ) / \
                   constants.hwFgSeqnoMultiple ) - 1

collectorInactiveReason = TacLazyType( 'FlowTracking::CollectorInactiveReason' )
exporterInactiveReason = TacLazyType( 'FlowTracking::ExporterInactiveReason' )
trackerInactiveReason =  TacLazyType( 'FlowTracking::TrackerInactiveReason' )

ftrCapabilitiesPathPrefix = 'hardware/flowtracking/capabilities/'
ftrCapabilitiesType = 'HwFlowTracking::Capabilities'

ftrConfigPathPrefix = 'flowtracking/config/'
ftrConfigType = 'FlowTracking::Config'

ftrStatusPathPrefix = 'flowtracking/status/'
ftrStatusType = 'FlowTracking::Status'

ftrConfigReqPathPrefix = 'flowtracking/configReq/'
ftrConfigReqType = 'FlowTracking::ConfigReq'

hwFtrConfigPathPrefix = 'hardware/flowtracking/config/'
hwFtrConfigType = 'HwFlowTracking::Config'

hwFtrStatusPathPrefix = 'hardware/flowtracking/status/'
hwFtrStatusType = 'HwFlowTracking::Status'

swExportCmd = 'record format ipfix standard timestamps counters'
hwOffloadIpv4Cmd = 'hardware offload ipv4'

def showFlowTracking( config, ftrType, cliSave=True,
                     saveAll=False, saveAllDetail=False, space='' ):
   detail = saveAll or saveAllDetail

   lines = []
   if not cliSave:
      space += FlowTrackerConst.spaceConst

   if ftrType == ftrTypeSampled:
      if ( config.sampleRate != sampleRate.rateDefault or detail ):
         lines.append( '%ssample %d' % ( space, config.sampleRate ) )

      if toggleHwOffloadIpv4Enabled():
         if config.hwOffloadIpv4:
            lines.append( '%s%s' % ( space, hwOffloadIpv4Cmd ) )
         elif detail:
            lines.append( '%sno %s' % ( space, hwOffloadIpv4Cmd ) )

   if not cliSave:
      if lines:
         lines = [ 'flow tracking ' + ftrTypeKwStr[ ftrType ] ] + lines

   return lines

def showFlowTrackingTrailer( config, ftrType, cliSave=True,
                           saveAll=False, saveAllDetail=False, space='' ):
   detail = saveAll or saveAllDetail

   lines = []
   if not cliSave:
      space += FlowTrackerConst.spaceConst

   if ftrType == ftrTypeHardware:
      if config.swExport:
         lines.append( "%s%s" % ( space, swExportCmd ) )
      elif detail:
         lines.append( "%sno %s" % ( space, swExportCmd ) )
   if config.enabled:
      lines.append( "%sno shutdown" % space )
   elif detail:
      lines.append( "%sshutdown" % space )
   return lines

def showFtr( output, trName, ftr, cliSave=True,
             saveAll=False, saveAllDetail=False, space='' ):
   detail = saveAll or saveAllDetail
   if not cliSave:
      output.write( '%stracker %s\n' % ( space, trName ) )
      space += FlowTrackerConst.spaceConst
   if ftr.inactiveTimeout != inactiveTimeout.timeoutDefault or detail:
      output.write( '%srecord export on inactive timeout %d\n' %
            ( space, ftr.inactiveTimeout ) )

   if ftr.activeInterval != activeInterval.intervalDefault or detail:
      output.write( '%srecord export on interval %d\n'
               % ( space, ftr.activeInterval ) )

   cmd = 'record export on tcp state change'
   if ftr.tcpStateChangeExport != constants.tcpStateChangeExportDefault:
      output.write( '%s%s\n' % ( space, cmd ) )
   elif detail:
      output.write( '%sno %s' % ( space, cmd ) )

def showExporter( output, expName, exp, cliSave=True,
                  saveAll=False, saveAllDetail=False, space='' ):
   detail = saveAll or saveAllDetail
   if not cliSave:
      output.write( '%s   exporter %s\n' % ( space, expName ) )
      space += '      '

   for host in sorted( exp.collectorHostAndPort.keys() ):
      hnp = exp.collectorHostAndPort[ host ]
      fmtStr = 'collector %s' % hnp.hostname
      if detail or hnp.port != ipfixPort.ipfixPortDefault:
         fmtStr += ' port %d' % hnp.port
      output.write( '%s%s\n' % ( space, fmtStr ) )

   if detail or exp.dscpValue != constants.dscpValueDefault:
      output.write( '%sdscp %d\n' % ( space, exp.dscpValue ) )

   if detail or exp.ipfixFormatConfigured:
      fmtStr = 'format ipfix version %d' % exp.ipfixVersion
      if detail or exp.ipfixMtu != ipfixMtu.mtuDefault:
         fmtStr += ' max-packet-size %d' % exp.ipfixMtu
      if exp.ipfixFormatConfigured is False:
         fmtStr = 'default ' + fmtStr
      output.write( '%s%s\n' % ( space, fmtStr ) )

   if exp.localAddress != constants.ipAddrDefault:
      output.write( '%slocal address %s\n' % ( space, exp.localAddress ) )
   elif detail:
      output.write( '%sno local address\n' % ( space ) )

   if exp.localIntfName and exp.localIntfName != constants.intfNameDefault:
      output.write( '%slocal interface %s\n' % ( space, exp.localIntfName ) )
   elif detail:
      output.write( '%sno local interface\n' % ( space ) )

   if detail or exp.templateInterval != templateInterval.intervalDefault:
      output.write( '%stemplate interval %d\n' % ( space, exp.templateInterval ) )

def showGroup( output, groupName, group, cliSave=True,
               saveAll=False, saveAllDetail=False, space='' ):
   if not cliSave:
      output.write( '%s   group %s\n' % ( space, groupName ) )
      space += '      '

   if group.encapType:
      encapStr='encapsulation'
      for encapType in sorted( group.encapType ):
         encapStr += ' %s' % encapTypeKwStr[ encapType ]
      output.write( '%s%s\n' % ( space, encapStr ) )

   if group.ipAccessList != accessListDefault:
      output.write( '%sip access-list %s\n'
                    % ( space, group.ipAccessList.aclName ) )

   if group.ip6AccessList != accessListDefault:
      output.write( '%sipv6 access-list %s\n'
                    % ( space, group.ip6AccessList.aclName ) )

   if group.expName:
      for exp in sorted( group.expName ):
         output.write( '%sexporter %s\n' % ( space, exp ) )

   if group.mirrorConfig != mirrorConfigDefault:
      mirrorStr = 'mirror %s' % group.mirrorConfig.sessionName
      if group.mirrorConfig.initialCopy != initialCopy.pktDefault:
         mirrorStr += ' initial packets %d' % group.mirrorConfig.initialCopy
      if group.mirrorConfig.mirrorIntervalFixed != \
         mirrorIntervalFixed.intervalDefault:
         mirrorStr += ' sample interval fixed %d' \
                      % group.mirrorConfig.mirrorIntervalFixed
      elif group.mirrorConfig.mirrorIntervalRandom != \
           mirrorIntervalRandom.intervalDefault:
         mirrorStr += ' sample interval random %d' \
                      % group.mirrorConfig.mirrorIntervalRandom
      output.write( '%s%s\n' % ( space, mirrorStr ) )

def getFtrTypeFromArgs( args ):
   ftrType = None
   if ftrTypeKwStr[ ftrTypeSampled ] in args:
      ftrType = ftrTypeSampled
   elif ftrTypeKwStr[ ftrTypeHardware ] in args:
      ftrType = ftrTypeHardware
   elif ftrTypeKwStr[ ftrTypeInbandTelemetry ].split()[ 1 ] in args:
      ftrType = ftrTypeInbandTelemetry
   elif ftrTypeKwStr[ ftrTypeMirrorOnDrop ] in args:
      ftrType = ftrTypeMirrorOnDrop
   else:
      assert 0

   return ftrType

def protocolStr( protocol, protocolNumber ):
   if protocol:
      if protocol.startswith( 'ipProto' ):
         return protocol[ len( 'ipProto' ) : ].upper()
      else:
         return protocol
   else:
      return str( protocolNumber )

def tcpFlagStr( tcpFlags ):
   tcpFlagMap = { 'syn' : 'S',
                  'fin' : 'F',
                  'rst' : 'R',
                  'ack' : 'A',
                  'psh' : 'P',
                  'urg' : 'U',
                  'ns' : 'N',
                  'cwr' : 'C',
                  'ece' :'E',
   }
   flags = []
   for flag, symbol in tcpFlagMap.items():
      if getattr( tcpFlags, flag ):
         flags.append( symbol )
   return '|'.join( flags ) or 'none'

def addressStr( ip, port ):
   ipString = ipStr( ip )
   template = "[%s]:%s" if ":" in ipString else "%s:%s"
   return template % ( ipString, port )

def ipStr( ip ):
   return getattr( ip, 'stringValue', ip )

def timeStr( time ):
   return datetime.fromtimestamp( time ).strftime( "%Y-%m-%d %H:%M:%S.%f" )

def collectorInactiveReasonStr( reason ):
   if reason == collectorInactiveReason.maxCollectorsLimit:
      return " (inactive, reason - maximum collectors limit reached)"
   if reason == collectorInactiveReason.collectorNotProcessed:
      return " (inactive, reason - waiting configuration processing)"
   return " (inactive)"

def getExpReasonFlagsFromExpReasonModel( expReasonModel ):
   expInactiveReason = Tac.newInstance(
         "FlowTracking::ExporterInactiveReason", 0 )
   for expReason in expReasonModel.expReasons:
      if expReason.expReason == exporterInactiveReasonEnum.exporterNotProcessed:
         expInactiveReason.exporterNotProcessed = True
      if expReason.expReason == exporterInactiveReasonEnum.maxExportersLimit:
         expInactiveReason.maxExportersLimit = True
      if expReason.expReason == exporterInactiveReasonEnum.invalidCollector:
         expInactiveReason.invalidCollector = True
      if expReason.expReason == exporterInactiveReasonEnum.invalidVrf:
         expInactiveReason.invalidVrf = True
      if expReason.expReason == exporterInactiveReasonEnum.invalidLocalIntf:
         expInactiveReason.invalidLocalIntf = True
      if expReason.expReason == exporterInactiveReasonEnum.waitingForUDPPort:
         expInactiveReason.waitingForUDPPort = True
   return expInactiveReason

def getExpReasonEnumList( reasonInt ):
   expReasonEnumList = []
   if not reasonInt:
      expReasonEnumList.append(
            exporterInactiveReasonEnum.exporterInactiveReasonUnknown )
      return expReasonEnumList

   expReason = Tac.newInstance(
         "FlowTracking::ExporterInactiveReason", reasonInt )
   if expReason.exporterNotProcessed:
      expReasonEnumList.append( exporterInactiveReasonEnum.exporterNotProcessed )
   if expReason.maxExportersLimit:
      expReasonEnumList.append( exporterInactiveReasonEnum.maxExportersLimit )
   if expReason.invalidCollector:
      expReasonEnumList.append( exporterInactiveReasonEnum.invalidCollector )
   if expReason.invalidVrf:
      expReasonEnumList.append( exporterInactiveReasonEnum.invalidVrf )
   if expReason.invalidLocalIntf:
      expReasonEnumList.append( exporterInactiveReasonEnum.invalidLocalIntf )
   if expReason.waitingForUDPPort:
      expReasonEnumList.append( exporterInactiveReasonEnum.waitingForUDPPort )
   return expReasonEnumList

def exporterInactiveReasonStr( reason, ipVersion='' ):
   if ipVersion == AddressFamily.ipunknown:
      ipVersion = ''
   if reason.maxExportersLimit:
      return " (inactive, reason - maximum exporters limit reached)"
   if reason.exporterNotProcessed:
      return " (inactive, reason - waiting configuration processing)"
   if reason.waitingForUDPPort:
      return " (inactive, reason - waiting for UDP port for local interface)"
   if reason.value == 0:
      return " (inactive)"

   reasonStr = " (inactive, reason - invalid/missing %s" % ipVersion
   addComma = False
   if reason.invalidCollector:
      reasonStr += " collector config"
      addComma = True

   addLocalIntf = True
   if reason.invalidLocalIntf:
      if addComma:
         reasonStr += ','
      reasonStr += ' local interface IP address'
      addLocalIntf = False
      addComma = False

   if reason.invalidVrf:
      if addComma:
         reasonStr += ','
      if addLocalIntf:
         reasonStr += ' local interface VRF config'
      else:
         reasonStr += ' and VRF config'
   reasonStr += ')'
   return reasonStr

def trackerInactiveReasonStr( reason ):
   if reason == trackerInactiveReason.maxTrackersLimit:
      return " (inactive, reason - maximum trackers limit reached)"
   if reason == trackerInactiveReason.trackerNotProcessed:
      return " (inactive, reason - waiting configuration processing)"
   return " (inactive)"
