# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Logging, re, Tac, errno
import os
import CEosHelper

Logging.logD( id="FRU_NOLBTDUMP",
              severity=Logging.logError,
              format="Unable to determine Aboot firmware version (%s)",
              explanation="nvramtool/lbtdump command failed",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

def abootFirmwareRev( ):
   """Obtain the Aboot version"""
   # First check if we are running in a workspace
   etcDir = os.environ.get( "VERSION_ETC_DIR", "/etc" )
   if ( not os.path.exists( os.path.join( etcDir, "arch" ) ) ) or\
      ( not os.path.exists( os.path.join( etcDir, "swi-version" ) ) ):
      return "unknown (in workspace)"
   if CEosHelper.isCeos():
      return "not applicable (cEOS doesn't use Aboot)"
   # Read from Aboot command line
   cmdLineFile = open( '/proc/cmdline' )
   m = re.search( "Aboot=(\S+)", cmdLineFile.read() )
   if m:
      return m.group( 1 )

   # It is an old version Aboot. We need find the info from the coreboot table. 
   try:
      x = Tac.run( [ "nvramtool", "-l", "extra_version" ],
                   stdout=Tac.CAPTURE, stderr=Tac.DISCARD, asRoot=True ).strip()
      return x.lstrip( "-" )
   except OSError, e:
      if e.errno != errno.ENOENT:
         Logging.log( FRU_NOLBTDUMP, e )
         return "unknown (nvramtool error)"
   except Tac.SystemCommandError, e:
      Logging.log( FRU_NOLBTDUMP, e )
      return "unknown (nvramtool error)"
   try:
      x = Tac.run( [ "lbtdump" ], stdout=Tac.CAPTURE, stderr=Tac.DISCARD, 
                   asRoot=True )
      # (It's the line after 'Extra Version', sans any leading hyphen. e.g:
      #    lb_record #4 type 5 @ 0x00000574 Extra Version
      #    -Aboot-sthelena_sonoma-eng-eng
      m = re.search( 'Extra Version[\n\r]([^\n\r]*)', x )
      if m:
         return m.group( 1 ).lstrip( "-" )
      Logging.log( FRU_NOLBTDUMP, x )
      return "unknown (version not found)"
   except (OSError, Tac.SystemCommandError), e:
      Logging.log( FRU_NOLBTDUMP, e )
      return "unknown (lbtdump error)"


def abootFirmwareRevNumbers():
   rev = abootFirmwareRev()
   if 'unknown' in rev:
      return ""
   # The rev is of the form 'Aboot-norcal1-1.9.8'. However, I am being more
   # tolerant and will accept the first instance of num.num[ .num... ] that I see.
   match = re.search( "(\d+\.\d+[\d|\.]*)", rev )
   return match.group() if match else ""
