#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import FileUrlDefs
from Url import Context, parseUrl
import SimpleConfigFile
import EosVersion

bootConfigFile = os.environ.get( 'BOOT_CONFIG_LOCATION', "flash:/boot-config" )

def localStartupConfig( entityManager, disableAaa, cliSession=None,
                        secureMonitor=False ):
   context = Context( entityManager, disableAaa, cliSession )
   if secureMonitor:
      filename = FileUrlDefs.SECURE_MONITOR_STARTUP_CONFIG_FILE_NAME
   else:
      filename = FileUrlDefs.STARTUP_CONFIG_FILE_NAME
   return parseUrl( "flash:/" + filename, context )

def localBootConfig( entityManager, disableAaa, cliSession=None ):
   context = Context( entityManager, disableAaa, cliSession )
   return parseUrl( bootConfigFile, context )

def localRunningConfig( entityManager, disableAaa, cliSession=None,
                        showSanitized=False, showJson=False,
                        showNoSeqNum=False,
                        secureMonitor=False,
                        showProfileExpanded=False,
                        intfFilter=None ):
   context = Context( entityManager, disableAaa, cliSession )
   urlString = "system:/running-config"
   if intfFilter:
      # assuming we don't have :, @, and , in interface names
      urlString += ":interface@" + ','.join( intfFilter )
   if showSanitized:
      urlString += ":sanitized"
   if showJson:
      urlString += ":json"
   if showNoSeqNum:
      urlString += ":noseqnum"
   if secureMonitor:
      urlString += ":secure-monitor"
   if showProfileExpanded:
      urlString += ":profile-expanded"
   return parseUrl( urlString, context )

def bootConfig( mode, createIfMissing=True ):
   url = localBootConfig( mode.entityManager, mode.session_.disableAaa_, 
                          mode.session )
   bootConfigFilename = url.realFilename_ # pylint: disable-msg=E1103
   return SimpleConfigFile.SimpleConfigFileDict( bootConfigFilename,
                                                 createIfMissing=createIfMissing,
                                                 autoSync=True )

def localRunningConfigAll( entityManager, disableAaa, cliSession=None,
                           showDetail=False, showSanitized=False,
                           showJson=False, showProfileExpanded=False,
                           intfFilter=None ):
   # These urls are for internal use and are not displayed externally to user.
   # Currently, there is no other better way to pass arguments to Url.get() method. 
   # These urls are used to pass appropriate flags to saver functions.
   context = Context( entityManager, disableAaa, cliSession )
   if showDetail:
      urlString = "system:/running-config:all:detail"
   else:
      urlString = "system:/running-config:all"
   if intfFilter:
      # assuming we don't have -, @, and , in interface names
      urlString += ":interface@" + ','.join( intfFilter )
   if showSanitized:
      urlString += ":sanitized"
   if showJson:
      urlString += ":json"
   if showProfileExpanded:
      urlString += ":profile-expanded"
   return parseUrl( urlString, context )

def urlIsSwiFile( url ):
   from zipfile import BadZipfile
   ver = None
   try:
      ver = EosVersion.swiVersion( url.realFilename_ )
   except BadZipfile:
      return False
   if ver.internalVersion() is None:
      return False
   return True
