#!/usr/bin/python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import Tac, Tracing, QuickTrace

traceDebug = Tracing.trace0
qv = QuickTrace.Var
qtDebug = QuickTrace.trace0

internalAddress = { 1:'127.1.0.1', 2:'127.1.0.2' }
peerAddress = { 1:'127.1.0.2', 2:'127.1.0.1' }
internalIntfIndex = 1

ISSHD_PORT = 3601

# initialize a salt for SSHD temporary paths,
# and export it to subprocesses of the breadth test
import random
TMPSEED = os.environ.get( 'TMPSEED', "%04x" % random.randrange( 0, 65536 ) )
os.environ[ 'TMPSEED' ] = TMPSEED

def GetLogPath( supeId, simulation=False ):
   """Compute the log path for the private SSHD.

   For simulated RedSup instances, use a temporary directory.
   """
   
   if simulation or os.environ.has_key( 'SIMULATION_VMID' ):
      return "/tmp/RedSup-supe%d-sshd-%s.out" % ( supeId, TMPSEED, )
   else:
      return "/var/log/RedSup-supe%d-sshd.out" % supeId

def GetWorkingDirectory( supeId, simulation=False ):
   """Compute the working directory for the private SSHD."""
   
   if simulation or os.environ.has_key( 'SIMULATION_VMID' ):
      return "/tmp/RedSup-supe%d-sshd-%s" % ( supeId, TMPSEED, )
   else:
      return "/usr/share/RedSup"

def GetDataDirectory( simulation=False ):
   """Compute the data directory for the private SSHD."""
   
   if simulation or os.environ.has_key( 'SIMULATION_VMID' ):
      # Not using "import Artest" because it would cause a dependency on
      # Artest, which we don't want because this file is a part of EOS.swi
      import importlib
      Artest = importlib.import_module( 'Artest' )
      return Artest.findSrcDir()
   else:
      return "/usr/share/RedSup"

def _sshOptions( supeId, asList, useKey=False ):
   # issh/id_rsa can also be used as key but we are just sticking to the
   # original key to be backward compatible. The authorized key issh/id_rsa.pu
   # contains public keys of both issh (user login) and FileReplicator (root user)
   rsakey = os.path.join( GetWorkingDirectory( supeId ), 'root.id_rsa' )

   options = [ '-oPort=%d' % ISSHD_PORT,
               '-oUser=root',
               '-oStrictHostKeyChecking=no',
               '-oUserKnownHostsFile=/dev/null',
               '-oCheckHostIP=no', ]
   if useKey:
      options += [ '-oIdentityFile=%s' % rsakey, ]
         
   if not asList:
      return " ".join( options )
   return options

def _peerIp( supeId ):
   return peerAddress[ supeId ]

def shredFile( supeId, target, useKey=False, loopback=False, recursive=True ):
   # Shred but not delete target
   # Please note on flash drives this will mangle the data visible on readback
   # but the original content may still be on the flash drive if the page has not
   # been reused.
   command = []
   shredCommand = []
   if recursive:
      shredCommand += [ "ShredRecursive", target ]
   else:
      # FIXME: does this work in loopback case? We should also skip symlinks to
      # avoid shredding the wrong file.
      shredCommand += [ "touch", target + ";" ]
      shredCommand += [ "scrub", "--no-signature", target ]
   if loopback:
      command = shredCommand
   else:
      shredCommand = " ".join( shredCommand )
      command = [ 'ssh' ] + _sshOptions( supeId, True, useKey ) + \
                  [ _peerIp( supeId ), shredCommand ]

   traceDebug( " ".join( command ) )
   qtDebug( qv( " ".join( command ) ) )
   return command

def deleteFile( supeId, target, useKey=False, loopback=False, recursive=True ):
   command = []
   if loopback:
      if recursive:
         command = [ 'rm', '-rf', target ]
      else:
         command = [ 'rm', '-f', target ]
   else:
      if recursive:
         options = '-rf'
      else:
         options = '-f'
      command = [ 'ssh' ] + _sshOptions( supeId, True, useKey ) + \
                  [ _peerIp( supeId ), 'rm %s %s' % ( options, target ) ]
   traceDebug( " ".join( command ) )
   qtDebug( qv( " ".join( command ) ) )
   return command

def createDir( supeId, target, useKey=False, loopback=False ):
   command = []
   if loopback:
      command = [ 'mkdir', '-p', target ]
   else:
      command = [ 'ssh' ] + _sshOptions( supeId, True, useKey ) +\
                  [ _peerIp( supeId ), 'mkdir -p %s' % target ]
   traceDebug( " ".join( command ) )
   qtDebug( qv( " ".join( command ) ) )
   return command

def stat( supeId, target, useKey=False, loopback=False ):
   command = []
   if loopback:
      command = [ 'stat', '-f', target ]
   else:
      command = [ 'ssh' ] + _sshOptions( supeId, True, useKey ) + \
                  [ _peerIp( supeId ), 'stat -f %s' % target ]
   traceDebug( " ".join( command ) )
   qtDebug( qv( " ".join( command ) ) )
   return command

def listDir( supeId, target, useKey=False, loopback=False ):
   command = []
   if loopback:
      command = [ 'ls', '-la', '--time-style=long-iso', target ]
   else:
      command = [ 'ssh' ] + _sshOptions( supeId, True, useKey ) + \
                  [ _peerIp( supeId ), 'ls -la --time-style=long-iso %s' % target ]
   traceDebug( " ".join( command ) )
   qtDebug( qv( " ".join( command ) ) )
   return command

def symlink( supeId, target, filename, useKey=False, loopback=False ):
   # Get link name. To keep it simple, we assume both source and target 
   # share the same link name without doing any relative path calculation.
   command = []
   if loopback:
      command = [ 'ln', '-sfT', filename, target ]
   else:
      command = [ 'ssh' ] + _sshOptions( supeId, True, useKey ) + \
                  [ _peerIp( supeId ), 'ln -sfT %s %s' % ( filename, target ) ]
   traceDebug( " ".join( command ) )
   qtDebug( qv( " ".join( command ) ) )
   return command

def rename( supeId, target, source, useKey=False, loopback=False ):
   command = []
   assert source is not None
   if loopback:
      command = [ 'mv', source, target ]
   else:
      command = [ 'ssh' ] + _sshOptions( supeId, True, useKey ) + \
                  [ _peerIp( supeId ), 'mv %s %s' % ( source, target ) ]
   traceDebug( " ".join( command ) )
   qtDebug( qv( " ".join( command ) ) )
   return command

def verifyHash( supeId, target, hashName, useKey=False, loopback=False ):
   # Perform md5 or sha512 checksum on a file
   checksumType = "md5sum" if hashName == "md5" else "sha512sum"
   command = []
   if loopback:
      command = [ checksumType, target ]
   else:
      command = [ 'ssh' ] + _sshOptions( supeId, True, useKey ) + \
                  [ _peerIp( supeId ), '%s %s' % ( checksumType, target ) ]
   traceDebug( " ".join( command ) )
   qtDebug( qv( " ".join( command ) ) )
   return command

def copyFile( supeId, target, source=None, useKey=False,
              loopback=False, peerSource=False, preservePerm=False ):
   command = []
   assert source is not None

   command = [ 'rsync', '-av', '--delete' ]
   if not preservePerm:
      # The --no-p option is necessary because the vfat filesystem on
      # /mnt/flash doesn't support permissions and rsync will fail
      # attempting to set permissions.
      command.append( "--no-p" )

   if not loopback:
      command.append( '--rsh=/usr/bin/ssh ' +
                      _sshOptions( supeId, False, useKey ) )
      if peerSource:
         source = '%s:%s' % ( _peerIp( supeId ), source )
      else:
         target = '%s:%s' % ( _peerIp( supeId ), target )

   command += [ source, target ]

   traceDebug( " ".join( command ) )
   qtDebug( qv( " ".join( command ) ) )
   return command

def syncFile( supeId, target, useKey=False, loopback=False ):
   # open a file and call fsync
   syncCmd = 'SyncFile'
   command = []
   if loopback:
      command = [ syncCmd, target ]
   else:
      command = [ 'ssh', '-oLogLevel=ERROR' ] + \
                  _sshOptions( supeId, True, useKey ) + \
                  [ _peerIp( supeId ), '%s %s' % ( syncCmd, target ) ]
   traceDebug( " ".join( command ) )
   qtDebug( qv( " ".join( command ) ) )
   return command

def runCmd( supeId, cmd, useKey=False, loopback=False ):
   command = []
   if loopback:
      command = cmd.split()
   else:
      command = [ 'ssh' ] + _sshOptions( supeId, True, useKey ) + \
                [ _peerIp( supeId ) ] + cmd.split()

   traceDebug( " ".join( command ) )
   qtDebug( qv( " ".join( command ) ) )
   return command
