# Copyright (c) 2009-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

# This module provides a mechanism whereby CliPlugins can register additional
# urls that can be used with certain commands provided by FileCli, such as copy
# and diff.  FileCli itself uses this module to register urls for startup-config
# and running-config.  The module also contains some functions useful for
# implementing file-related Cli commands.

import errno
import sys
import re

import CliExtensions
import CliCommand
import CliParser
import Logging
import PyClient
import Tac
import TacSigint
import Tracing
import Url

t0 = Tracing.trace0

DIFF_MAX_FILE_SIZE = 1024 * 1024 * 8

Logging.logD( id="SYS_FILE_COPY_ERROR",
              severity=Logging.logError,
              format="An error occurred when copying from %s to %s (%s).",
              explanation="An error occurred when copying a file in the "
               "file system.",
              recommendedAction="This may indicate a serious hardware or "
               "software problem. Please check free space and integrity of the "
               "file system. If the problem persists, please contact "
               "technical support." )

def logLocalFileCopyError( surl, durl, error ):
   # We only want to log serious errors
   allowedErrno = [ errno.EFAULT, errno.EFBIG, errno.ELOOP,
                       errno.EMFILE, errno.ENOMEM, errno.ENOSPC,
                       errno.EOVERFLOW, errno.EROFS, errno.EBADF, errno.EIO ]

   # We cannot compare the types of filesystem of urls because doing so
   # would cause cyclic depencies between Cli and FileCli;
   # localFilename() should be a good way to go, as it should return None if
   # there is no local file that correspond to the url.
   fsIsLocal = surl.localFilename() or durl.localFilename()
   if fsIsLocal and ( error.errno in allowedErrno ):
      # pylint: disable-msg=undefined-variable
      Logging.log( SYS_FILE_COPY_ERROR, surl.url, durl.url, error.strerror )

def checkUrl( url ):
   """Checks a Url to make sure it isn't the special object indicating that a
   relative Url was entered but the current filesystem no longer exists.  If it is,
   prints an error message and aborts the current command."""
   if url == Url.currentFilesystemNoLongerExists:
      print( "% Error: current filesystem no longer exists" )
      raise CliParser.AlreadyHandledError

def showFile( mode, url ):
   """A Cli value function that prints the contents of a file to stdout."""
   checkUrl( url )
   bufferSize = 1024

   try:
      f = url.open()
      if hasattr( f, 'name' ):
         if f.name.endswith( ".gz" ):
            f.close()
            import gzip
            f = gzip.GzipFile( f.name, f.mode )
         elif f.name.endswith( ".bz2" ):
            f.close()
            import bz2
            f = bz2.BZ2File( f.name, f.mode )
      try:
         data = f.read( bufferSize )
         while data:
            sys.stdout.write( data )
            data = f.read( bufferSize )
            TacSigint.check()
      finally:
         f.close()
   except EOFError:
      mode.addError( "Error displaying %s (End of file)" % ( url.url ) )
   except EnvironmentError, e:
      # don't commit a double fault (sometimes that kills the cli session)
      if e.errno != errno.EPIPE:
         mode.addError( "Error displaying %s (%s)" % ( url.url, e.strerror ) )
   except PyClient.RpcError:
      mode.addError( "Error displaying %s" % ( url.url ) )
      raise
   except ValueError:
      mode.addError( "Error displaying %s" % ( url.url ) )

checkCopyFileAllowed = CliExtensions.CliHook()
copyFileNotifiers = CliExtensions.CliHook()

def copyFile( cliStatus, mode, surl, durl, commandSource="commandLine",
              ignoreENOENT=False ):
   checkUrl( surl )
   checkUrl( durl )
   try:
      if surl == durl:
         raise EnvironmentError( 0, "Source and destination are the same file" )

      if durl.isdir():
         d = durl.child( surl.basename() )
      else:
         d = durl

      for checkFun in checkCopyFileAllowed.extensions():
         if not checkFun( mode, surl, durl ):
            return "not allowed by hook"

      d.copyfrom( surl )

      for n in copyFileNotifiers.extensions():
         n( mode, surl, durl, commandSource )

      if commandSource == "commandLine":
         mode.addMessage( "Copy completed successfully." )
   except EnvironmentError, e:
      if not ( ignoreENOENT and e.errno == errno.ENOENT ):
         mode.addError( "Error copying %s to %s (%s)" % (
            surl.url, durl.url, e.strerror ) )
         logLocalFileCopyError( surl, durl, e )

         return e.strerror

   # success
   return ""

urlExpressions_ = []

class UrlExpressionFactory( CliCommand.OrExpressionFactory ):
   def __init__( self, description, fsFunc, notAllowed ):
      CliCommand.OrExpressionFactory.__init__( self )
      self.urlMatcher_ = Url.UrlMatcher( fsFunc, description,
                                         notAllowed=notAllowed )
      self |= ( "BASE", self.urlMatcher_ )

   def registerExtension( self, name, expr, token ):
      self |= ( name, expr )
      if token:
         self.urlMatcher_.notAllowed_.append( token )

def registerUrlExpression( description, fsFunc, notAllowed ):
   t0( "registerUrlExpression", description )
   expr = UrlExpressionFactory( description, fsFunc, notAllowed )
   urlExpressions_.append( expr )
   return expr

def registerUrlExprExtension( exprName, expr ):
   # Url rule exteions can only be registered after registerUrlRule()
   # therefore, check Url rules have been registerd first
   assert urlExpressions_
   for r in urlExpressions_:
      r |= ( exprName, expr )

copySourceUrlExpr_ = registerUrlExpression( 'Source file name',
                                            lambda fs: fs.supportsRead(),
                                            notAllowed=[] )
copyDestUrlExpr_ = registerUrlExpression( 'Destination file name',
                                          lambda fs: fs.supportsWrite(),
                                          notAllowed=[] )

def registerCopySource( name, expr, token=None ):
   """Register a CliParser rule that will be included in the rule that specifies
   source URLs for the copy command.  If token is not None it will be included
   in the set of tokens that UrlMatcher will not match."""
   copySourceUrlExpr_.registerExtension( name, expr, token )

def registerCopyDestination( name, expr, token=None, callback=None ):
   """Add a CliParser rule that matches destination URLs for the copy command.
   The given token will be included in the set of tokens that UrlMatcher
   will not match.
   If a callback is passed, it will be invoked as follows whenever a copy is
   initiated to the given file:
     callback( mode, srcUrl )
   and if the callback returns False, the copy will not proceed.
   """
   # pylint: disable=protected-access
   copyDestUrlExpr_.registerExtension( name, expr, token )
   # pylint: enable=protected-access
   if callback:
      assert callable( callback ), "Invalid callback %r" % ( callback, )

      def callbackWrapper( mode, srcUrl, dstUrl ):
         # registerCopyDestination only works with "flash:" URLs, so check
         # the destination URL against our filename prefixed by "flash:/"
         # or "system:/"
         if dstUrl.url not in ( "flash:/%s" % token, "system:/%s" % token ):
            return True
         return callback( mode, srcUrl )
      checkCopyFileAllowed.addExtension( callbackWrapper )

def copySourceUrlExpr():
   return copySourceUrlExpr_

def copyDestinationUrlExpr():
   return copyDestUrlExpr_

def _getValueForDiff( aUrl ):
   try:
      aFile = aUrl.open()
   except IOError, ex:
      # let's absorb any errors from a missing file and treat it like it is empty
      if ex.errno == errno.ENOENT:
         return ""
      else:
         raise
   try:
      # If the url has the expected prefix, such as startup-config or checkpoint,
      # we need to skip the first line which contains last modified info,
      # so that it is not reported in diff command output.
      # If not, don't exclude that line.
      aValue = aFile.readline()
      if not aUrl.isHeader( aValue ):
         aFile.seek( 0 )

      # to prevent out of memory errors, limit how much is read
      aValue = aFile.read( DIFF_MAX_FILE_SIZE )
   finally:
      aFile.close()
   return aValue

def diffFile( mode, aUrl, bUrl ):
   import difflib
   checkUrl( aUrl )
   checkUrl( bUrl )

   # Check size of the URL before doing the diff, since the file can be huge.
   # This means remote URLs cannot be diff'ed.
   for url in [ aUrl, bUrl ]:
      errorMessage = None
      try:
         curFileSize = url.size()
      except IOError, ex:
         errorMessage = ex.strerror
      except OSError, ex:
         # let's absorb any errors from a missing file and treat it like it is empty
         if ex.errno != errno.ENOENT:
            errorMessage = ex.strerror
      else:
         if curFileSize is None:
            mode.addError( "%s is not a local file" % url )
            return
         if curFileSize > DIFF_MAX_FILE_SIZE:
            mode.addError( "The size of %s is %s which "
                           "exceeds the maximum size of %s" % (
                  url, curFileSize, DIFF_MAX_FILE_SIZE ) )
            return

      if errorMessage is not None:
         # same message as below to have consistent output in the two error paths
         mode.addError( "Error comparing %s to %s (%s)" % (
               aUrl.url, bUrl.url, errorMessage ) )
         return

   try:
      aValue = _getValueForDiff( aUrl )
      bValue = _getValueForDiff( bUrl )
      if ( aUrl.ignoreTrailingWhitespaceInDiff() and
           bUrl.ignoreTrailingWhitespaceInDiff() ):
         whiteEnd = re.compile( r' +\n', re.DOTALL )
         whiteMiddle = re.compile( r'(\S )[ ]+' )
         aValue = re.sub( whiteEnd, '\n', re.sub( whiteMiddle, '\\1', aValue ) )
         bValue = re.sub( whiteEnd, '\n', re.sub( whiteMiddle, '\\1', bValue ) )

      diff = difflib.unified_diff( aValue.splitlines( 1 ),
                                   bValue.splitlines( 1 ),
                                   aUrl.url, bUrl.url )
      print( ''.join( diff ) )
   except EnvironmentError, e:
      mode.addError( "Error comparing %s to %s (%s)" % (
            aUrl.url, bUrl.url, e.strerror ) )

diffFirstUrlExpr_ = registerUrlExpression( 'First file path',
                                           lambda fs: fs.supportsRead(),
                                           notAllowed=[] )
diffSecondUrlExpr_ = registerUrlExpression( 'Second file path',
                                            lambda fs: fs.supportsRead(),
                                            notAllowed=[] )

def diffFirstUrlExpr():
   return diffFirstUrlExpr_

def diffSecondUrlExpr():
   return diffSecondUrlExpr_

def sizeFmt( num ):
   for x in [ 'bytes', 'KB', 'MB', 'GB', 'TB', 'PB' ]:
      if num < 1024.0:
         return "%3.1f %s" % ( num, x )
      num /= 1024.0
   return None

class PersistentConfig( object ):
   def __init__( self ):
      self.info_ = {}

   def _register( self, keyword, handler, description, default ):
      self.info_[ keyword ] = ( handler, description, default )

   def info( self ):
      return self.info_

   def cleanup( self, mode, delete, preserve ):
      for keyword, ( handler, _description, default ) in self.info_.iteritems():
         try:
            handler( mode,
                     bool( ( delete and keyword in delete ) or
                           ( ( ( preserve is None ) or
                             ( keyword not in preserve ) ) and
                            default ) ) )
         # pylint: disable-msg=W0703
         except Exception, e:
            mode.addError( "Error erasing %s configuration: %s" % ( keyword, e ) )

__persistentConfigInfo__ = PersistentConfig()

def registerPersistentConfiguration( keyword, handler, description,
                                     default=True ):
   """Register the existence of persistent configuration info so that the
   'write erase' command is aware of it, and can be able to delete it.
   Provide a keyword that will be used in the Cli command, a description
   string that will be part of the help string for the keyword, specify
   whether it will be erased as part of 'write erase' by default, or not, and
   finally provide a handler function that takes 2 args: mode, and a
   boolean arg, specifying whether to delete (True) or preserve (False)
   the configuration"""
   # pylint: disable-msg=W0212
   __persistentConfigInfo__._register( keyword, handler, description, default )

def chunkedHashCompute( url, hashInitializer ):
   """Computes the hash of a file pointed to by url without reading the whole file
   at once into memory (does it in chunks)."""
   chunkSize = 16384
   f = url.open()
   try:
      hashFunc = hashInitializer()
      data = f.read( chunkSize )
      while data:
         hashFunc.update( data )
         data = f.read( chunkSize )
   finally:
      f.close()
   return hashFunc.hexdigest()
