#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from IpLibConsts import DEFAULT_VRF
import SharedMem
import Toggles.RoutingLibToggleLib

def mountSmashTables( em, vrfName=DEFAULT_VRF, ipVersion=4, write=False,
                      unifiedFwdingTables=False ):
   shmemEm = SharedMem.entityManager( sysdbEm=em )
   if write:
      routeInfo = Tac.nonConst( routeStatusInfo( 'writer' ) )
      fwdInfo = Tac.nonConst( forwardingStatusInfo( 'writer' ) )
   else:
      routeInfo = Tac.nonConst( routeStatusInfo( 'reader' ) )
      fwdInfo = Tac.nonConst( forwardingStatusInfo( 'reader' ) )

   # Route status
   if ipVersion == 4:
      if vrfName == DEFAULT_VRF:
         routeStatus = shmemEm.doMount( 'routing/status', 'Smash::Fib::RouteStatus',
                                        routeInfo )
      else:
         routeStatus = \
            shmemEm.doMount( 'routing/vrf/status/%s' % vrfName,
                             'Smash::Fib::RouteStatus', routeInfo )
   elif ipVersion == 6:
      if vrfName == DEFAULT_VRF:
         routeStatus = shmemEm.doMount( 'routing6/status',
                                        'Smash::Fib6::RouteStatus', routeInfo )
      else:
         routeStatus = \
            shmemEm.doMount( 'routing6/vrf/status/%s' % vrfName,
                             'Smash::Fib6::RouteStatus', routeInfo )

   # Forwarding status
   fsType = 'Smash::Fib::ForwardingStatus'
   fs6Type = 'Smash::Fib6::ForwardingStatus'
   fsGenType = 'Smash::FibGen::ForwardingStatus'

   if unifiedFwdingTables:
      fsPath = 'forwarding/unifiedStatus'
      fs6Path = 'forwarding6/unifiedStatus'
      fsGenPath = 'forwardingGen/unifiedStatus'
   elif vrfName == DEFAULT_VRF:
      fsPath = 'forwarding/status'
      fs6Path = 'forwarding6/status'
      fsGenPath = 'forwardingGen/status'
   else:
      fsPath = 'forwarding/vrf/status/%s' % vrfName
      fs6Path = 'forwarding6/vrf/status/%s' % vrfName
      fsGenPath = 'forwardingGen/vrf/status/%s' % vrfName

   if Toggles.RoutingLibToggleLib.toggleFibGenMountPathEnabled():
      forwardingStatus = shmemEm.doMount( fsGenPath, fsGenType, fwdInfo )
   elif ipVersion == 4:
      forwardingStatus = shmemEm.doMount( fsPath, fsType, fwdInfo )
   else:
      forwardingStatus = shmemEm.doMount( fs6Path, fs6Type, fwdInfo )
   return ( routeStatus, forwardingStatus )

def unmountSmashTables( em, vrfName=DEFAULT_VRF, ipVersion=4,
                        unifiedFwdingTables=False ):
   shmemEm = SharedMem.entityManager( sysdbEm=em )
   if Toggles.RoutingLibToggleLib.toggleFibGenMountPathEnabled():
      # Do not unmount default vrf smash entities.
      if vrfName != DEFAULT_VRF:
         if ipVersion == 4:
            shmemEm.doUnmount( 'routing/vrf/status/%s' % vrfName )
         elif ipVersion == 6:
            shmemEm.doUnmount( 'routing6/vrf/status/%s' % vrfName )
         if not unifiedFwdingTables:
            shmemEm.doUnmount( 'forwardingGen/vrf/status/%s' % vrfName )
   elif ipVersion == 4:
      # Do not unmount default vrf or unified smash entities. We get same entities
      # back if we remount them. That'f fine. If we doUnmount then we have to
      # update routeStatus pointers in agents which would otherwise point to
      # orphans.
      if vrfName != DEFAULT_VRF:
         shmemEm.doUnmount( 'routing/vrf/status/%s' % vrfName )
         if not unifiedFwdingTables:
            shmemEm.doUnmount( 'forwarding/vrf/status/%s' % vrfName )
   elif ipVersion == 6:
      # Do not unmount default vrf smash entities.
      if vrfName != DEFAULT_VRF:
         shmemEm.doUnmount( 'routing6/vrf/status/%s' % vrfName )
         if not unifiedFwdingTables:
            shmemEm.doUnmount( 'forwarding6/vrf/status/%s' % vrfName )

# I realize that these don't belong here, but I am placing them here for a lack of a
# better place. Eventually we will be getting these values from Sysdb
fibInfo = Tac.Value( 'Smash::Fib::TableInfo' )

def routeStatusInfo( mode ):
   ''' Returns the Smash route table information '''
   return fibInfo.routeInfo( mode )

def forwardingStatusInfo( mode ):
   ''' Returns the Smash forwarding table information '''
   return fibInfo.forwardingInfo( mode )

def unprogrammedRouteStatusInfo( mode ):
   ''' Returns the Smash unprogrammedRoute table information '''
   return fibInfo.unprogrammedRouteInfo( mode )
