#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, print_function, division
import argparse
from collections import namedtuple
import re
import sys

import Tac
from TypeFuture import TacLazyType

FecId = TacLazyType( 'Smash::Fib::FecId' )
TunId = TacLazyType( 'Tunnel::TunnelTable::TunnelId' )
TunType = TacLazyType( 'Tunnel::TunnelTable::TunnelType' )
AdjType = TacLazyType( 'Smash::Fib::AdjType' )
NexthopGroupId = TacLazyType( 'Routing::NexthopGroup::NexthopGroupId' )
DynamicTunnelIntfId = TacLazyType( 'Arnet::DynamicTunnelIntfId' )

def castToType( val, _type ):
   if val is None:
      return None
   if not isinstance( val, _type ):
      return _type( val )
   return val

# Contains the potential ids of a given id string
_ParsedIds = namedtuple( 'ParsedIds', 'fecId,tunId,nhgId' )

def ParsedIds( fecId=None, tunId=None, nhgId=None ):
   return _ParsedIds( castToType( fecId, FecId ),
                      castToType( tunId, TunId ),
                      castToType( nhgId, NexthopGroupId ) )

class IdType( object ):
   FECID = 'FECID'
   TUNNELID = 'TUNNELID'
   NHGID = 'NHGID'

_TypedId = namedtuple( 'TypedId', 'idType,val' )

def TypedId( idType, val ):
   '''
   idType - an IdType
   val    - should be an actual FecId, TunnelId, or NexthopGroupId
   '''
   return _TypedId( idType, val )

def intStrToInt( intStr ):
   # will also parse out the ending L if provided.
   try:
      if intStr.startswith( '0x' ) or re.search( r'[a-fA-F]', intStr ):
         return long( intStr, 16 )
      else:
         return long( intStr )
   except ValueError:
      return None

def parseIds( idStr ):
   try:
      fecId = FecId.hierarchicalIntfIdToFecId( idStr )
      if fecId:
         return ParsedIds( fecId=FecId( fecId ) )
   except IndexError:
      pass

   try:
      tunId = DynamicTunnelIntfId.tunnelId( idStr )
      if tunId:
         return ParsedIds( tunId=tunId )
   except IndexError:
      pass

   # At this point, we assume that the ID is some kind of integer
   intVal = intStrToInt( idStr )
   if intVal is None:
      # ID was not a valid integer
      return None

   # Could be either FecId or TunnelId
   maybeFecId = None
   try:
      maybeFecId = FecId( intVal )
      if FecId.fecIdForAdjType( 'adjTypeMax', 0 ) <= maybeFecId.value:
         # Id is not valid
         maybeFecId = None
   except TypeError:
      # int was too big
      pass

   maybeTunId = None
   try:
      maybeTunId = TunId( intVal )
      if maybeTunId.tunnelType() != 'invalidTunnel' and \
         maybeTunId.tunnelType() != 'maxTunnelType':
         maybeTunId = maybeTunId
      else:
         maybeTunId = None
   except ( IndexError, TypeError ):
      # int was too big, or invalid value
      pass

   maybeNhgId = None
   try:
      maybeNhgId = NexthopGroupId( intVal )
   except TypeError:
      pass

   return ParsedIds( fecId=maybeFecId, tunId=maybeTunId, nhgId=maybeNhgId )

def printFecIdInfo( fecId, prnt ):
   prnt( "AdjType: %s, AdjIndex: %d" % ( fecId.adjType(), fecId.adjIndex() ) )
   prnt( "FecId: %d (0x%x)" % ( fecId.value, fecId.value ) )

   intfId = 'Invalid (unable to convert)'
   try:
      intfId = FecId.fecIdToHierarchicalIntfId( fecId.value )
   except IndexError:
      pass
   prnt( "IntfId:", intfId )

def printTunnelIdInfo( tunId, prnt ):
   prnt( "TunnelType: %s, AF: %s, TunnelIndex: %d" %
         ( tunId.tunnelType(), tunId.tunnelAf(), tunId.tunnelIndex() ) )
   prnt( "TunnelId: %d (0x%x)" % ( tunId.value, tunId.value ) )

   fecId = FecId( FecId.tunnelIdToFecId( tunId ) )
   if fecId:
      printFecIdInfo( fecId, prnt )
   else:
      intfId = DynamicTunnelIntfId.tunnelIdToIntfId( tunId.value )
      prnt( "IntfId:", intfId )
      prnt( "This TunnelId is not convertible to FecId" )

def printNhgIdInfo( nhgId, prnt ):
   prnt( "NexthopGroupId:", nhgId )

   fecId = FecId( FecId.nexthopGroupIdToFecId( nhgId.value ) )
   printFecIdInfo( fecId, prnt )

def printTypedId( typedId, prnt ):
   if typedId.idType == IdType.TUNNELID:
      printTunnelIdInfo( typedId.val, prnt )
   elif typedId.idType == IdType.NHGID:
      printNhgIdInfo( typedId.val, prnt )
   elif typedId.idType == IdType.FECID:
      printFecIdInfo( typedId.val, prnt )

def printIdInfo( idOrIdx, prnt, tunType=None, adjType=None ):
   ids = None
   assert not ( tunType and adjType )
   if tunType or adjType:
      index = intStrToInt( idOrIdx )
      if index is None:
         raise ValueError( "%s is not a valid index" % idOrIdx )
      if tunType:
         if tunType in TunType.attributes:
            ids = ParsedIds(
               tunId=TunId( TunId.convertToTunnelValue( tunType, index ) ) )
         else:
            raise ValueError(
                  "%s is not a valid tunnel type (--help for full list)" % tunType )
      elif adjType:
         if adjType in AdjType.attributes:
            ids = ParsedIds(
               fecId=FecId( FecId.fecIdForAdjType( adjType, index ) ) )
         else:
            raise ValueError( "%s is not a valid adj type (--help for full list)" %
                              adjType )
   else:
      ids = parseIds( idOrIdx )

   if ids is None or ids == ParsedIds():
      raise ValueError( "Invalid id" )

   # A list of tuples with the name of the interpretation and the TypedId
   idsToPrint = []
   if ids.fecId:
      if ids.fecId.adjType() == AdjType.tunnelFibAdj:
         tunId = TunId( FecId.fecIdToTunnelId( ids.fecId.value ) )
         typedId = TypedId( IdType.TUNNELID, tunId )
      elif ids.fecId.adjType() == AdjType.nextHopGroupAdj:
         typedId = TypedId( IdType.NHGID, NexthopGroupId( ids.fecId.adjIndex() ) )
      else:
         typedId = TypedId( IdType.FECID, ids.fecId )

      idsToPrint.append( ( "FecId", typedId ) )

   if ids.tunId:
      idsToPrint.append( ( "TunnelId", TypedId( IdType.TUNNELID, ids.tunId ) ) )

   if ids.nhgId:
      idsToPrint.append( ( "NexthopGroupId", TypedId( IdType.NHGID, ids.nhgId ) ) )

   if len( idsToPrint ) > 1:
      prnt( "ID has multiple possible interpretations" )

   for typeName, _id in idsToPrint:
      if len( idsToPrint ) > 1:
         prnt( "" )
         prnt( typeName, "Interpretation:" )
      printTypedId( _id, prnt )

def buildArgParser():
   parser = argparse.ArgumentParser( description=
"""Interprets an ID string as some kind of FecId, in its various forms
and prints the different renderings that are found.""" )

   parser.add_argument( 'id_or_index', metavar='ID_OR_INDEX',
                        help="""A representation of the FecId.
May be of various forms including the decimal or hex representation, or some other
id (eg. IntfId, TunnelId, NHGId) that can be converted into a FecId.
 """ )

   idTypeGroup = parser.add_mutually_exclusive_group()
   idTypeGroup.add_argument( '--tun-type', type=str, help="""
Tunnel type. Id must be the index if provided.
Can be any of %s""" % ','.join( TunType.attributes ) )
   idTypeGroup.add_argument( '--adj-type', type=str, help="""
Adj type. Id must be the index if provided.
Can be any of %s""" % ','.join( AdjType.attributes ) )

   return parser

class Printer( object ):
   '''A print handle. Different printers can be passed in by tests
   to avoid capturing stdout and stderr
   '''
   def __call__( self, *args, **kwargs ):
      print( *args, **kwargs )

   def err( self, *args, **kwargs ):
      print( *args, file=sys.stderr, **kwargs )

def main( prnt=None, argv=None ):
   if not prnt:
      prnt = Printer()

   if argv is None:
      argv = sys.argv[ 1 : ]

   parser = buildArgParser()
   args = parser.parse_args( argv )

   try:
      printIdInfo( args.id_or_index, prnt,
                   tunType=args.tun_type, adjType=args.adj_type )
   except ValueError as e:
      prnt.err( e )
      return 1

   return 0

if __name__ == '__main__':
   exit( main() )
